/*
 * graphics.C
 * 
 * Graphics subroutines
 *
 * In SGI's mixed model, we simply call GL functions
 *
 * $Modified: Sunday, September 18, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

#ifndef MIXED
Transform tfstack[TRANSFORM_STACK_DEPTH];
int tfsp;

advertise GC canvas_gc, popup_gc, current_gc;
advertise Drawable drawable;

XPoint *the_points, *the_last_point, *xpoint;
static XPoint the_first_point;
static Boolean flushed_already = FALSE;

advertise XGCValues xgcvalues;
#endif

//
//----------------------------------------------------------------------
//
//  INITIALIZE graphics
//

advertise void initialize_graphics(void)
{
  static s_coord T[6] = {1.0, 0.0, 0.0, -1.0, 0.5, 0.5};

  // compute screen factor:
  //  pixels per big point = #pixels / size_in_bp
  screen_factor =
    pl_vec(DisplayWidth(display, screen_num) /
	   (DisplayWidthMM(display, screen_num) * 72.0 / 25.0),
	   DisplayHeight(display, screen_num) /
	   (DisplayHeightMM(display, screen_num) * 72.0 / 25.0));

  DEBUG(DBG_GFX, "Screen factor x = ", screen_factor.x());
  DEBUG(DBG_GFX, "Screen factor y = ", screen_factor.y());
  canvas_window = XtWindow(canvasWidget);
  
#ifdef MIXED
  // install GL colormaps
  install_gl_interface();
  // GL limitation on # of vertices of polygon
  max_nvertex = getgdesc(GD_NVERTEX_POLY);
  if (max_nvertex == GD_NOLIMIT)
    max_nvertex = 65535;
  DEBUG(DBG_GL, "Max # of vertices is ", max_nvertex);

#else
  install_colormap();
  max_nvertex = (XMaxRequestSize(display) - 3) / 2;
  DEBUG(DBG_GFX, "Max request size is ", max_nvertex);

  if (max_nvertex > MAX_REQUEST_LENGTH)
    max_nvertex = MAX_REQUEST_LENGTH;
  the_points = new XPoint[max_nvertex];
  the_last_point = the_points + (max_nvertex-1);
  
  tfsp = 1;
  tfstack[tfsp] = Transform(T);
  resize_canvas();

  // create graphics context

  canvas_gc = XCreateGC(display, RootWindow(display, screen_num), 0, NIL);
  xgcvalues.arc_mode = ArcChord;
  XChangeGC(display, canvas_gc, (GCArcMode), &xgcvalues);

  popup_gc  = XCreateGC(display, RootWindow(display, screen_num), 0, NIL);
  if (1 && monochrome) {
    xgcvalues.function = GXxor;
    xgcvalues.foreground = BlackPixel(display, screen_num)
      ^ WhitePixel(display, screen_num);
    XChangeGC(display, popup_gc, (GCFunction | GCForeground), &xgcvalues);
  }
  
  // where the graphic goes...
  drawable = canvas_window;
#endif
}

#ifndef MIXED
advertise void resize_canvas(void)
{
  int x,y;
  Window root;
  unsigned int wd, ht, bw, dp;
  XGetGeometry(display, canvas_window, &root, &x, &y, &wd, &ht, &bw, &dp);
  canvas_size.x = wd;
  canvas_size.y = ht;
  DEBUG(DBG_GFX, "resize canvas to ", wd << " x " << ht);

  canvas_origin = pl_vec(rint_pp(wd / 2.0), rint_pp(ht / 2.0));
  DEBUG(DBG_GFX, "canvas origin is ", canvas_origin);

  tfstack[1].A[5] = canvas_size.y + 0.5;
}
#endif

//
//----------------------------------------------------------------------
//
// LINESTYLE maintenance
//

static charp_rr linestyle_dashes(0);

void Iobject::dash_properties(ostream& fh)
{
  fh << "% ss " << linestyle_list[dash] << " " << lwidth << "\n";
}

void Iobject::ps_dash(ostream& fh)
// outputs a dash in postscript notation
{
  if (!dash) {
    fh << "% ss 0\n" << lwidth << " [] ss\n";
  } else {
    fh << "% ss " << linestyle_list[dash] << "\n" << lwidth;
    fh << " [ ";
    char *p = linestyle_dashes[dash];
    while (*p)
      fh << int(*p++) << " ";
    fh << "] ss\n";
  }
}

static char *linestyle_to_dashes(short dash)
{
  static int p[32];
  int len = 0;
  int onoff = 1;
  unsigned int rot = dash;
  if (!(rot & 0x0001))
    rot = 0xffff ^ rot;
  int count = 0;
  for (int i = 0; i < 16; i++) {
    if (onoff != (rot & 1)) {
      p[len++] = count;
      count = 0;
      onoff = 1 - onoff;
    }
    rot >>= 1;
    count++;
  }
  p[len++] = count;
  if (onoff)
    p[len++] = 0;
  for (i = 0; i < len; i++)
    p[i+len] = p[i];
  // now determine period
  int k = 0;
  Boolean good;
  do {
    k++;
    good = TRUE;
    for (i = 0; i < len; i++)
      if (p[i] != p[i+k]) {
	good = FALSE;
      }
  } while (!good);
  // k is period of what we want
  char *out = new char[k+1];
  char *q = out;
  for (i = 0; i < k; i++)
    *q++ = char(p[i]);
  *q++ = '\0';
  return out;
}

advertise void def_linestyle(short ls, char *)
{
#ifdef MIXED
  int i = linestyle_list.size();
  if (i)
    deflinestyle(i, ls);
#endif
  linestyle_list.append(ls);
  linestyle_dashes.append(linestyle_to_dashes(ls));
}

advertise short linestyle_index(short ls)
{
  if (ls == 0)
    return 0;
  for (int i = 0; i < linestyle_list.size(); i++) {
    if (linestyle_list[i] == ls)
      return i;
  }
  // ls is not in linestyle table. Add it
  def_linestyle(ls, NIL);
  // add_to_choice(s ? s :form("%hx", ls & 0xffff));
  return i;
}

#ifndef MIXED
//
// X functions to set the linestyle
//

void set_linestyle(short idx, Boolean repeat)
// set dashes in current GC to linestyle[idx],
//     using zoom_factor as repeat factor if repeat == TRUE
{
  // set Solid line / Dashed line
  xgcvalues.line_style = (idx ? LineOnOffDash : LineSolid);
  XChangeGC(display, current_gc, (GCLineStyle), &xgcvalues);
  if (!idx)
    return;
  
  static char dash_list[32];
  // compute dashes
  if (repeat) {
    int rep = (zoom_factor >= 1.0) ? int(zoom_factor) : 1;
    DEBUG(DBG_GFX, "Dash repeat count is ", rep);
    char *p = linestyle_dashes[idx];
    char *q = dash_list;
    while (*p)
      *q++ = char(rep * (*p++));
    XSetDashes(display, current_gc, 0, dash_list,
	       strlen(linestyle_dashes[idx]));
  } else {
    XSetDashes(display, current_gc, 0, linestyle_dashes[idx],
	       strlen(linestyle_dashes[idx]));
  }
}

void set_linewidth(int lw)
// set linewidth in current GC
{
  xgcvalues.line_width = lw;
  XChangeGC(display, current_gc, (GCLineWidth), &xgcvalues);
}

//
//----------------------------------------------------------------------
//
// PUSH/POP TRANSFORMATION MATRIX
//

void push_matrix(void)
// push transformation matrix     
{
  tfsp++;
  if (tfsp == TRANSFORM_STACK_DEPTH) {
    cerr << "Fatal error: transformation stack overflow\n";
    exit(1);
  }
  tfstack[tfsp] = tfstack[tfsp-1];
}

void pop_matrix(void)
// pop transformation matrix     
{
  if (!tfsp) {
    cerr << "Fatal error: transformation stack underflow\n";
    exit(1);
  }
  tfsp--;
}

//
//----------------------------------------------------------------------
//
// SCALE, ROTATE, TRANSLATE
//

void translate(const pl_fixedvec &t)
// translate by t
{
  static s_coord T[6] = {1.0, 0.0, 0.0, 1.0, 0.0, 0.0};
  T[4] = t.x();
  T[5] = t.y();
  tfstack[tfsp].premult(Transform(T));
}

void rotate(pl_angle alpha)
// rotate with angle alpha
{
  static s_coord T[6] = {1.0, 0.0, 0.0, 1.0, 0.0, 0.0};
  T[0] = T[3] = cos_pp(alpha);
  T[1] = sin_pp(alpha);
  T[2] = -T[1];
  tfstack[tfsp].premult(Transform(T));
}

void scale(s_coord fx, s_coord fy)
// stretch in X- and Y-direction
{
  static s_coord T[6] = {1.0, 0.0, 0.0, 1.0, 0.0, 0.0};
  T[0] = fx;
  T[3] = fy;
  tfstack[tfsp].premult(Transform(T));
}

//
//----------------------------------------------------------------------
//
// DRAWING
//

inline void to_screen(const pl_vec& p, int &x, int &y) {
  pl_vec pt(p);
  tfstack[tfsp].apply(pt);
  x = int(pt.x());
  y = int(pt.y());
}

//
// these two functions draw an axis-parallel rectangle
//

void draw_rectangle(const pl_vec& p, const pl_vec& q)
{
  int x0, y0, x1, y1;
  to_screen(p, x0, y0);
  to_screen(q, x1, y1);
  int x, y, w, h;
  if (x0 < x1) {
    x = x0;
    w = x1 - x0;
  } else {
    x = x1;
    w = x0 - x1;
  }
  if (y0 < y1) {
    y = y0;
    h = y1 - y0;
  } else {
    y = y1;
    h = y0 - y1;
  }
  XDrawRectangle(display, drawable, current_gc, x, y, w, h);
}

void draw_filled_rectangle(const pl_vec& p, const pl_vec& q)
{
  int x0, y0, x1, y1;
  to_screen(p, x0, y0);
  to_screen(q, x1, y1);
  int x, y, w, h;
  if (x0 < x1) {
    x = x0;
    w = x1 - x0;
  } else {
    x = x1;
    w = x0 - x1;
  }
  if (y0 < y1) {
    y = y0;
    h = y1 - y0;
  } else {
    y = y1;
    h = y0 - y1;
  }
  XFillRectangle(display, drawable, current_gc, x, y, w, h);
}

//
// this routine draws a circle or ellipse
//  unless elliptic == TRUE, it only transforms the center
//  and obeys zoom_ and screen_factor
//

void draw_ellipse(const pl_vec& p, s_coord r, Boolean fill)
{
  // this is the nasty situation: draw a non-aligned ellipse
  // by approximating it by a polygon
  s_coord max_size = fabs_pp(tfstack[tfsp].A[0] * tfstack[tfsp].A[3]
			     - tfstack[tfsp].A[1] * tfstack[tfsp].A[2]);
  static pl_disc d;
  static pl_vecreray approx;
  d = pl_disc(p, r);
  approx.newsize(0);
  d.inner_approximation_d(approx, 1.0/max_size);
  DEBUG(DBG_GFX, "Drawing ellipse as polygon with # vertices = ",
	approx.size());
  if (approx.size() > max_nvertex && fill)
    d.outer_approximation_n(approx, int(max_nvertex - 1));
  if (fill)
    bgn_polygon();
  else
    bgn_line(TRUE);
  for (int i = 0; i < approx.size(); i++)
    put_vertex(approx[i]);
  if (fill)
    end_polygon();
  else
    end_line(TRUE);
}

void draw_generic_circle(const pl_vec& p, s_coord r,
			 Boolean fill, Boolean elliptic)
{
  pl_vec pt(p);
  tfstack[tfsp].apply(pt);
  s_coord xr;
  s_coord yr;
  if (elliptic) {
    pl_vec p1(tfstack[tfsp].A[0], tfstack[tfsp].A[1]);
    pl_vec p2(tfstack[tfsp].A[2], tfstack[tfsp].A[3]);
    if (fabs_pp(dot(p1, p2)) > SIMILAR_THRESHOLD) {
      // this is not so easy. go do it the hard way
      draw_ellipse(p, r, fill);
      return;
    }
    // it's a nice transformation, just apply it to main axis
    xr = r * p1.length();
    yr = r * p2.length();
  } else {
    // don't obey elliptification and scaling (for marks!)
    xr = screen_factor.x() * zoom_factor * r;
    yr = screen_factor.y() * zoom_factor * r;
  }
  int x = int(pt.x() - xr);
  int y = int(pt.y() - yr);
  unsigned int w = (unsigned int) (2.0 * xr);
  unsigned int h = (unsigned int) (2.0 * yr);
  if (fill)
    XFillArc(display, drawable, current_gc, x, y, w, h, 0, (360 * 64));
  else
    XDrawArc(display, drawable, current_gc, x, y, w, h, 0, (360 * 64));
}

//
// draw line connects two points p and q
//

void draw_line(const pl_vec& p, const pl_vec& q)
{
  int x0, y0, x1, y1;
  to_screen(p, x0, y0);
  to_screen(q, x1, y1);
  XDrawLine(display, drawable, current_gc, x0, y0, x1, y1);
}

//
// draw_arc draws a segment of a circular arc
//

void draw_generic_arc(const pl_vec& p, s_coord r,
		      pl_angle a0, pl_angle a1, Boolean fill)
{
  pl_vec pt(p);
  tfstack[tfsp].apply(pt);
  int x = int(pt.x() - screen_factor.x() * zoom_factor * r);
  int y = int(pt.y() - screen_factor.y() * zoom_factor * r);
  unsigned int w = (unsigned int) (2.0 * screen_factor.x() * zoom_factor * r);
  unsigned int h = (unsigned int) (2.0 * screen_factor.y() * zoom_factor * r);
  int ang0 = int(a0.value() * (180.0 * 64.0) / M_PI);
  // ang1 is difference between the two angles
  int ang1 = int(a1.value() * (180.0 * 64.0) / M_PI) - ang0;
  if (ang1 < 0)
    ang1 += (360 * 64);
  if (ang0 > (180 * 64))
    ang0 -= (360 * 64);
  if (fill)
    XFillArc(display, drawable, current_gc, x, y, w, h, ang0, ang1);
  else
    XDrawArc(display, drawable, current_gc, x, y, w, h, ang0, ang1);
}

//
//----------------------------------------------------------------------
//
// DRAWING sequences of points, lines or a polygon
//

void end_line(Boolean closed)
{
  if (closed)
    *xpoint++ = (flushed_already ? the_first_point : the_points[0]);
  flushed_already = FALSE;
  XDrawLines(display, drawable, current_gc, the_points,
	     (xpoint - the_points), CoordModeOrigin);
}

void end_point(void)
{
  flushed_already = FALSE;
  XDrawPoints(display, drawable, current_gc, the_points,
	      (xpoint - the_points), CoordModeOrigin);
}

void end_polygon(void)
{
  flushed_already = FALSE;
  XFillPolygon(display, drawable, current_gc, the_points,
	       (xpoint - the_points), Complex, CoordModeOrigin);
}

void flush_points(Boolean point_mode)
// # of segments or points exceeds the maximal request size,
//    flush buffer and start from the beginning
{
  // save first point for closed polylines
  if (!flushed_already) {
    flushed_already = TRUE;
    the_first_point = the_points[0];
  }
  if (point_mode) {
    XDrawPoints(display, drawable, current_gc, the_points,
		(xpoint - the_points), CoordModeOrigin);
  } else {
    XDrawLines(display, drawable, current_gc, the_points,
	       (xpoint - the_points), CoordModeOrigin);
  }
  xpoint = the_points;
}

#endif
