/*
 * group.C
 * 
 *  includes definition of class Group
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"


Group::~Group(void)
{
  Iobject *ob, *ob1;
  
  for (ob = first; ob; ob = ob1) {
    ob1 = ob->next;
    delete ob;
  }
}

Group::Group(Group& g)
{
  Iobject *ob, *ob1;
  
  first = last = NIL;
  for (ob = g.first; ob; ob = ob->next) {
    ob1 = ob->copy();
    add_object(ob1);
  }
}

void Group::stats(int& mem, int &nobjs, int &nvtcs)
{
  Iobject *ob;
  mem += sizeof(Group);

  for (ob = first; ob; ob = ob->next) {
    ob->stats(mem, nobjs, nvtcs);
  }
}


void Group::transform(const pl_rotra& tfm) 
{
  Iobject *ob;
  
  for (ob = first; ob; ob = ob->next) {
    ob->transform(tfm);
  }
}

void Group::setstroke(IColor c)
{
  Iobject *ob;
  
  for (ob = first; ob; ob = ob->next) {
    ob->setstroke(c);
  }
}

void Group::setfill(IColor c)
{
  Iobject *ob;
  
  for (ob = first; ob; ob = ob->next) {
    ob->setfill(c);
  }
}

void Group::setdash(short d)
{
  Iobject *ob;
  
  for (ob = first; ob; ob = ob->next) {
    ob->setdash(d);
  }
}
void Group::setlwidth(s_coord lw)
{
  Iobject *ob;
  
  for (ob = first; ob; ob = ob->next) {
    ob->setlwidth(lw);
  }
}

void Group::stretch(const s_coord xfactor, const s_coord yfactor)
{
  Iobject *ob;
  
  for (ob = first; ob; ob = ob->next) {
    ob->stretch(xfactor, yfactor);
  }
}

void Group::draw(void)
{
  Iobject *ob;
  
  for (ob = first; ob; ob = ob->next) {
    ob->draw();
  }
}

void Group::nontext_draw(void)
{
  Iobject *ob;
  
  for (ob = first; ob; ob = ob->next) {
    if (ob->type() == GROUP)
      ((Group *) ob)->nontext_draw();
    else if (ob->type() != TEXT)
      ob->draw();
  }
}

void Group::text_draw(void)
{
  Iobject *ob;
  
  for (ob = first; ob; ob = ob->next) {
    if (ob->type() == GROUP)
      ((Group *) ob)->text_draw();
    else if (ob->type() == TEXT)
      ob->draw();
  }
}

void Group::outline(void)
{
  Iobject *ob;
  
  for (ob = first; ob; ob = ob->next) {
    ob->outline();
  }
}
 
void Group::outline_rotated(const pl_vec& p, const pl_angle alpha)
{
  Iobject *ob;
  
  for (ob = first; ob; ob = ob->next) {
    ob->outline_rotated(p, alpha);
  }
}
     
s_coord Group::dist(const pl_vec& p)
{
  Iobject *ob;
  s_coord d = select_dist(), d1;
  
  for (ob = first; ob; ob = ob->next) {
    if ((d1 = ob->dist(p)) < d)
      d = d1;
  }
  return (d);
}

void Group::snap_vtx(const pl_vec& mp, pl_vec& p)
{
  Iobject *ob;
  for (ob = first; ob; ob = ob->next)
    ob->snap_vtx(mp, p);
}

void Group::snap_bnd(const pl_vec& mp, pl_vec& p)
{
  Iobject *ob;
  for (ob = first; ob; ob = ob->next)
    ob->snap_bnd(mp, p);
}

void Group::close_edge(const pl_vec& mp, Iedge_rr& s)
{
  Iobject *ob;
  for (ob = first; ob; ob = ob->next)
    ob->close_edge(mp, s);
}

void Group::save_postscript(ostream& fh)
{
  Iobject *ob;
  fh << "% Group\n\n";
  for (ob = first; ob; ob = ob->next) {
    refresh_display();
    ob->save_postscript(fh);
  }
  fh << "% End\n\n";
}

void Group::save_properties(ostream& fh)
{
  Iobject *ob;
  fh << "% Group\n";
  for (ob = first; ob; ob = ob->next) {
    refresh_display();
    ob->save_properties(fh);
  }
  fh << "% End\n\n";
}

void Group::bbox(pl_boundingbox& b)
{
  Iobject *ob;
  for (ob = first; ob; ob = ob->next) {
    ob->bbox(b);
  }
}

void Group::add_object(Iobject *n)
// add a new object to group
{
  n->next = NIL;
  n->sel = UNSELECTED;
  if (!first) {
    first = n;
    last = n;
    n->prev = n->next = NIL;
  } else {
    last->next = n;
    n->prev = last;
    last = n;
  }
}

Group::Group(istream& fh, Boolean no_segments)
{
  char *wk;
  Line *l;
  Group *gr;
  
  // initialize object, in case of read error
  first = last = NIL;
  next = prev = NIL;
  sel = UNSELECTED;
  
  while (ps_okay &&
	 (wk = ps_read_next(fh)) != NIL &&
	 strcmp(wk, "End") != 0) {
    refresh_display();
    if (!strcmp(wk, "Line")) {
      l = new Line(fh);
      if (ps_in_defd[6]) {
	// line is an old-style spline, convert
	add_object(new Spline(*l));
	delete l;
      } else
	add_object(l);
    } else if (!strcmp(wk, "Spline"))
      add_object(new Spline(fh));
    else if (!strcmp(wk, "Text"))
      add_object(new Text(fh));
    else if (!strcmp(wk, "Circle"))
      add_object(new Circle(fh));
    else if (!strcmp(wk, "Arc"))
      add_object(new Arc(fh));
    else if (!strcmp(wk, "Mark"))
	add_object(new Mark(fh));
    else if (!strcmp(wk, "Bitmap"))
      add_object(new Bitmap(fh));
    else if (!strcmp(wk, "Segments"))
      add_object(new Segments(fh));
    else if (!strcmp(wk, "Group")) {
      gr = new Group(fh);
      if (!no_segments && group_is_segments(gr->first)) {
	// the group can be compacted as a Segments object
	add_object(new Segments((Line *) gr->first));
	delete gr;
      } else {
	add_object(gr);
      }
    } else
      ps_okay = FALSE;
  }
  ps_okay &= (wk != NIL);
}

void Group::traverse(Object_type t, void (*pfunc)(Iobject*))
// Traverse the group. For all objects of type t, execute *pfunc(ob)
{
  for (Iobject *ob = first; ob; ob = ob->next) {
    if (ob->type() == GROUP)
      ((Group *) ob)->traverse(t, pfunc);
    else if (ob->type() == t)
      (*pfunc)(ob);
  }
}
    
advertise void pic_traverse(Object_type t, void (*pfunc)(Iobject*))
{
  for (Iobject *ob = pic->first; ob; ob = ob->next) {
    if (ob->sel != UNSELECTED) {
      if (ob->type() == GROUP)
	((Group *) ob)->traverse(t, pfunc);
      else if (ob->type() == t)
	(*pfunc)(ob);
    }
  }
}
