/* -*-C++-*-
 *
 * igraphics.h
 * 
 * Definitions for Ipe graphics calls
 *
 *  in mixed (openGL) mode, these are mainly inlines to GL functions
 *  otherwise, implemented in graphics.C
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".

 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

extern s_coord zoom_factor;

#ifdef MIXED
//
//----------------------------------------------------------------------
//
//   GL VERSION
//

inline void push_matrix(void)	{ pushmatrix();}
inline void pop_matrix(void)	{ popmatrix();}

inline void bgn_line(Boolean closed = FALSE) {
  if (closed)
    bgnclosedline();
  else
    bgnline();
}

inline void end_line(Boolean closed = FALSE) {
  if (closed)
    endclosedline();
  else
    endline();
}

inline void bgn_polygon(void)	{ bgnpolygon();}
inline void end_polygon(void)	{ endpolygon();}

inline void bgn_point(void)	{ bgnpoint();}
inline void end_point(void)	{ endpoint();}

inline void put_vertex(const pl_fixedvec& v, Boolean = FALSE) {
  // This code will fail if s_coord != float !!!
  v2f((float *) (&v));
}

inline void draw_circle(const pl_vec& p, s_coord radius, Boolean = FALSE) {
  circ(p.x(), p.y(), radius);
}

inline void draw_filled_circle(const pl_vec& p, s_coord radius,
			       Boolean = FALSE) {
  circf(p.x(), p.y(), radius);
}

inline void draw_rectangle(const pl_vec& p, const pl_vec& q) {
  rect(p.x(), p.y(), q.x(), q.y());
}

inline void draw_filled_rectangle(const pl_vec& p, const pl_vec& q) {
  rectf(p.x(), p.y(), q.x(), q.y());
}

inline void draw_line(const pl_vec& p, const pl_vec& q) {
  bgn_line(); put_vertex(p); put_vertex(q); end_line();
}

inline Angle gl_Angle(pl_angle alpha)
{
  return Angle(alpha/M_PI * 1800.0);
}

inline void draw_arc(const pl_vec& p, s_coord r, pl_angle a0, pl_angle a1) {
  arc(p.x(), p.y(), r, gl_Angle(a0), gl_Angle(a1));
}

inline void set_linestyle(short ls, Boolean repeat = TRUE) {
  setlinestyle(ls);
  if (repeat)
    lsrepeat((zoom_factor >= 1.0) ? int(zoom_factor) : 1);
  else
    lsrepeat(1);
}

inline void set_linewidth(int w)	{ linewidth(w);}

inline void translate(const pl_fixedvec& v) {
  translate(v.x(), v.y(), 0.0);
}

inline void rotate(pl_angle alpha) {
  rot(alpha*180.0/M_PI, 'z');
}

inline void scale(const s_coord fx, const s_coord fy) {
  scale(fx, fy, 1.0);
}

inline void scale(const pl_vec& factor) {
  scale(factor.x(), factor.y(), 1.0);
}

inline void scale(s_coord factor) {
  scale(factor, factor, 1.0);
}

inline void sub_pixel(Boolean t)	{ subpixel(t);}

inline void set_color(short i)		{ color(i); }
inline void write_mask(unsigned long m)	{ writemask((unsigned short) m); }

#else

//
//----------------------------------------------------------------------
//
//    X VERSION
//

extern XPoint *the_points, *the_last_point, *xpoint;
extern Transform tfstack[];
extern int tfsp;

extern void push_matrix(void);
extern void pop_matrix(void);

extern void translate(const pl_fixedvec& );
extern void rotate(pl_angle alpha);
extern void scale(s_coord fx, s_coord fy);
inline void scale(const pl_vec& factor)	{ scale(factor.x(), factor.y());}
inline void scale(s_coord factor)	{ scale(factor, factor);}

extern void draw_generic_circle(const pl_vec& p, s_coord radius,
				Boolean fill, Boolean elliptic);
extern void draw_rectangle(const pl_vec& p, const pl_vec& q);
extern void draw_filled_rectangle(const pl_vec& p, const pl_vec& q);
extern void draw_line(const pl_vec& p, const pl_vec& q);
extern void draw_generic_arc(const pl_vec& p, s_coord r,
			     pl_angle a0, pl_angle a1, Boolean fill);

inline void draw_circle(const pl_vec& p, s_coord radius,
			Boolean elliptic = FALSE) {
  draw_generic_circle(p, radius, FALSE, elliptic);
}

inline void draw_filled_circle(const pl_vec& p, s_coord radius,
			       Boolean elliptic = FALSE) {
  draw_generic_circle(p, radius, TRUE, elliptic);
}

inline void draw_arc(const pl_vec& p, s_coord r, pl_angle a0, pl_angle a1) {
  draw_generic_arc(p, r, a0, a1, FALSE);
}

inline void draw_filled_arc(const pl_vec& p, s_coord r,
			    pl_angle a0, pl_angle a1) {
  draw_generic_arc(p, r, a0, a1, TRUE);
}

extern void set_linestyle(short ls, Boolean repeat = TRUE);
extern void set_linewidth(int lw);
extern void set_color(short idx);      		// in colors.C
extern void write_mask(unsigned long m);	// in colors.C

// subpixel is ignored in X
inline void sub_pixel(Boolean)		{}

inline void bgn_line(Boolean = FALSE)	{ xpoint = the_points;}
inline void bgn_polygon(void)		{ xpoint = the_points;}
inline void bgn_point(void)		{ xpoint = the_points;}

extern void end_line(Boolean = FALSE);
extern void end_polygon(void);
extern void end_point(void);
extern void flush_points(Boolean point_mode);

inline void put_vertex(const pl_fixedvec& v, Boolean ptmode = FALSE) {
  pl_vec vt(v);
  tfstack[tfsp].apply(vt);
  xpoint->x = short(vt.x());
  xpoint->y = short(vt.y());
  xpoint++;
  if (xpoint == the_last_point) flush_points(ptmode);
}

//
//----------------------------------------------------------------------
//
#endif
