/*
 * interface.C
 * 
 * This module implements all aspects of the user interface of Ipe
 *
 * $Modified: Sunday, September 18, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

#include <Xm/MainW.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/RowColumn.h>
#include <Xm/FileSB.h>
#include <Xm/TextF.h>
#include <Xm/SelectioB.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/ArrowB.h>
#include <Xm/CascadeB.h>
#include <Xm/ToggleB.h>
#include <Xm/MessageB.h>
#include <Xm/CascadeB.h>

advertise Widget topLevel;

advertise IpeButton snap_vertex_bt, snap_boundary_bt, snap_crossing_bt;
advertise IpeButton snap_grid_bt, snap_directional_bt, snap_autodir_bt;
advertise IpeButton snap_self_bt;

advertise IpeButton transparent_bt;

advertise IpeButton left_arrow_bt, right_arrow_bt;

advertise IpeButton object_type_button[NUM_OBJECT_TYPES];

#include "Icons/line_icon.xbm"
#include "Icons/polygon_icon.xbm"
#include "Icons/spline_icon.xbm"
#include "Icons/splinegon_icon.xbm"
#include "Icons/circle_icon.xbm"
#include "Icons/box_icon.xbm"
#include "Icons/arc_icon.xbm"
#include "Icons/mark_icon.xbm"
#include "Icons/text_icon.xbm"

#include "Icons/left_arrow.xbm"
#include "Icons/right_arrow.xbm"

static Widget messageField, fileNameField, pageNumber;
static Widget mainWindow;
static Widget zoomFactor;
static Widget choiceBox;
static Widget messageBox;
static Widget fileSelector;
static Widget menuBar;			// used for sensitivity
static Widget objectTypeField;  	// dito.
static Widget shortcutDisplay, operationDisplay;

static Widget sizeChoice[2], fontChoice[2];

static void ipe_selection_cb(Widget, XtPointer, XtPointer);
static void ipe_fileselect_cb(Widget, XtPointer, XtPointer);

//
//----------------------------------------------------------------------
//
// BUTTONS
//

static void redraw_bt_cb(IpeButton *)
{
  redraw_canvas();
}

static void ipe_button_callback(Widget, XtPointer client_data, XtPointer)
{
  IpeButton *bt = (IpeButton *) client_data;
  bt->toggle();
}

void IpeButton::realize(Widget parent, char *name, char *label, Boolean v,
			IpeButtonCallback cb, int wd, int ht)
{
  w = 
    XtVaCreateManagedWidget(name, xmToggleButtonWidgetClass, parent, NIL);
  value = v;
  callback = cb;
  XtAddCallback(w, XmNvalueChangedCallback, ipe_button_callback,
		XtPointer(this));
  Boolean fill, indicator_on;
  unsigned foreground;
  XtVaGetValues(w,
		XmNfillOnSelect, &fill,
		XmNindicatorOn,  &indicator_on,
		XmNbackground,   &background,
		XmNforeground,	 &foreground,
		XmNselectColor,  &highlight,
		NIL);
  if (wd) {
    // label is bits, so create pixmap
    Pixmap normal_pixmap =
      XCreatePixmapFromBitmapData(display,
				  DefaultRootWindow(display),
				  label, wd, ht,
				  foreground, background,
				  DefaultDepth(display, screen_num));
    Pixmap selected_pixmap =
      XCreatePixmapFromBitmapData(display,
				  DefaultRootWindow(display),
				  label, wd, ht,
				  (fill ? foreground : highlight),
				  (fill ? highlight : background),
				  DefaultDepth(display, screen_num));
    
    XtVaSetValues(w,
		  XmNlabelPixmap,  normal_pixmap,
		  XmNselectPixmap, selected_pixmap,
		  XmNlabelType,    XmPIXMAP,
		  NIL);
    // if we use the background color for highlighting the button
    // the surrounding background should be filled as well
    if (!fill)
      // no, we use the foreground color
      highlight = background;
  } else {
    if (label)
      set_label(w, label);
    if (!fill || indicator_on)
      highlight = background;
  }
  set(value);
}

void IpeButton::set(Boolean v)
{
  value = v;
  XtVaSetValues(w, XmNset, value, NIL);
  if (highlight != background) {
    // modify appearance
    XtVaSetValues(w, XmNbackground, (value ? highlight : background), NIL);
  }
}

void IpeButton::toggle(void)
{
  set(!value);
  if (callback)
    (*callback)(this);
}

//
//----------------------------------------------------------------------
//
// CHOICE OBJECTS
//

static void redraw_cb(ChoiceArgs)
{
  redraw_canvas();
}

struct IpeChoicePacket {
  ChoiceArgs choice;
  int *current_choice;
  int entry;
  IpeChoiceCallback callback;
};

static IpeChoicePacket ipe_choice_packet;
static Widget choiceWidget, choiceMenu;

void ipe_choice_callback(Widget, XtPointer client_data, XtPointer)
{
  IpeChoicePacket *icp = (IpeChoicePacket *) client_data;

  *(icp->current_choice) = icp->entry;
  if (icp->callback)
    (*(icp->callback))(icp->choice);
}

advertise Widget create_choice(Widget parent, ChoiceArgs choice,
			       char *name, IpeChoiceCallback cb,
			       int *choice_adr)
{
  choiceMenu =
    XmCreatePulldownMenu( parent, "choiceMenu", NIL, 0);

  Arg arg[2];
  int ac = 0;
  XtSetArg(arg[ac], XmNsubMenuId, choiceMenu); ac++;
  
  choiceWidget =
    XmCreateOptionMenu( parent, name, arg, ac);
  ipe_choice_packet.choice = choice;
  ipe_choice_packet.entry = 0;
  ipe_choice_packet.callback = cb;
  ipe_choice_packet.current_choice = choice_adr;
  return choiceWidget;
}

static Widget create_managed_choice(Widget parent, ChoiceArgs choice,
				    char *name, IpeChoiceCallback cb,
				    int *choice_adr)
{
  Widget w = create_choice(parent, choice, name, cb, choice_adr);
  XtManageChild(w);
  return w;
}

advertise Widget add_to_choice(char *label)
{
  Widget button =
    XtVaCreateManagedWidget( "choiceEntry",
			     xmPushButtonWidgetClass, choiceMenu,
			     XtVaTypedArg, XmNlabelString, XmRString,
			     label, strlen(label) + 1,
			     NIL);

  IpeChoicePacket *icp = new IpeChoicePacket;
  icp->choice         = ipe_choice_packet.choice;
  icp->current_choice = ipe_choice_packet.current_choice;
  icp->entry          = ipe_choice_packet.entry++;
  icp->callback       = ipe_choice_packet.callback;

  if (icp->entry == *(icp->current_choice)) {
    // preselected entry
    set_choice(choiceWidget, button);
  }
  
  // arrange for button to call function 
  XtAddCallback(button, XmNactivateCallback, ipe_choice_callback,
		XtPointer(icp));
  return button;
}

advertise void set_choice(Widget choice, Widget button)
{
  XtVaSetValues(choice, XmNmenuHistory, button, NIL);
}

//
//----------------------------------------------------------------------
//
// INPUT VARIANT FIELDS
//

static Widget inputVariant = NIL;

advertise void manage_variant(Widget widget)
{
  if (inputVariant)
    XtUnmanageChild(inputVariant);
  inputVariant = widget;
  if (inputVariant)
    XtManageChild(inputVariant);
}
  
//
//----------------------------------------------------------------------
//
// DISPLAY FIELDS
//

advertise void put_msg(char *str)
{
  set_label(messageField, str);
}

advertise void show_filename(void)
// show filename in window
{
  char *p = filename + strlen(filename);
  while (*p != '/' && p != (filename-1))
    p--;
  p++;
  set_label(fileNameField, p);
}

advertise void show_pagenum(void)
{
  if (!mipe_mode) {
    set_label(pageNumber, "");
    hide_page_menu();
  } else {
    char buf[32];
    sprintf(buf, "%d", page_number);
    set_label(pageNumber, buf);
    show_page_menu();
  }
}

advertise void show_zoom(void)
{
  char buf[32];

  sprintf(buf,"%.2f", zoom_factor);
  set_label(zoomFactor, buf);
}

advertise void show_m_ipe(void)
// set configuration screen according to mipe/ipe mode
{
  XtUnmanageChild(sizeChoice[1 - mipe_mode]);
  XtManageChild(sizeChoice[mipe_mode]);
  XtUnmanageChild(fontChoice[1 - mipe_mode]);
  XtManageChild(fontChoice[mipe_mode]);
  show_m_ipe_configuration();
}

advertise void show_operation(const char *op, const char *key)
{
  set_label(operationDisplay, op);
  set_label(shortcutDisplay, key);
}

//
//----------------------------------------------------------------------
//
// CREATE FIELDS
//

static void fill_linestyle_field(Widget field)
{
  Widget label =
    XtVaCreateManagedWidget("linestyleLabel",
			    xmLabelWidgetClass, field,
			    NIL);
  char buf[16];
  int i;
  
  Widget linestyleChoice =
    create_managed_choice(field, CHOICE_LINESTYLE, "linestyleChoice",
			  line_choice_cb, &linestyle_choice);

  add_to_choice("Solid");
  for (i = 1; i < linestyle_list.size(); i++) {
    sprintf(buf, "%hx", linestyle_list[i] & 0xffff);
    add_to_choice(buf);
  }

  Widget lineWidthChoice =
    create_managed_choice(field, CHOICE_LINE_WIDTH, "lineWidthChoice",
			  line_choice_cb, &line_width_choice);
  
  for (i = 0; i < line_width_list.size(); i++) {
    sprintf(buf, "%g", line_width_list[i]);
    add_to_choice(buf);
  }

  left_arrow_bt.realize(field, "leftArrow", left_arrow_bits,
			((arrow_type & 1) != 0), line_arrow_cb,
			left_arrow_width, left_arrow_height);
  right_arrow_bt.realize(field, "rightArrow", right_arrow_bits,
			 ((arrow_type & 2) != 0), line_arrow_cb,
			 right_arrow_width, right_arrow_height);
}

static void fill_font_field(Widget field)
{
  Widget label =
    XtVaCreateManagedWidget("fontLabel",
			    xmLabelWidgetClass, field,
			    NIL);
  char buf[16];

  for (int mimo = 0; mimo < 2; mimo++) {
    fontChoice[mimo] =
      create_managed_choice(field, CHOICE_FONT, "fontChoice",
			    text_choice_cb, &font_choice[mimo]);
    add_to_choice("Roman");
    add_to_choice("Italic");
    add_to_choice("Bold");
    add_to_choice("$...$");

    sizeChoice[mimo] =
      create_choice(field, CHOICE_FONTSIZE, "fontsizeChoice",
		    text_choice_cb, &fontsize_choice[mimo]);
    
    for (int i = 0; i < fontsize_list[mimo].size(); i++ ) {
      sprintf(buf, "%g", fontsize_list[mimo][i]);
      add_to_choice(buf);
    }
  }
}

static void fill_zoom_field(Widget field)
{
  Widget label =
    XtVaCreateManagedWidget("zoomLabel",
			    xmLabelWidgetClass, field,
			    NIL);
  zoomFactor =
    XtVaCreateManagedWidget("zoom",
			    xmLabelWidgetClass, field,
			    NIL);
  Widget zoomButton =
    XtVaCreateManagedWidget("zoomOut",
			    xmArrowButtonWidgetClass, field,
			    XmNarrowDirection, XmARROW_LEFT,
			    NIL);
  XtAddCallback(zoomButton, XmNactivateCallback, zoom_cb, XtPointer(-1));
  zoomButton =
    XtVaCreateManagedWidget("zoomIn",
			    xmArrowButtonWidgetClass, field,
			    XmNarrowDirection, XmARROW_RIGHT,
			    NIL);
  XtAddCallback(zoomButton, XmNactivateCallback, zoom_cb, XtPointer(1));
  
  transparent_bt.realize(field, "transparentButton", NIL, FALSE,
			 redraw_bt_cb);
}

static void fill_snap_field(Widget field)
{
  Widget label =
    XtVaCreateManagedWidget("snapLabel",
			    xmLabelWidgetClass, field,
			    NIL);

  snap_vertex_bt.realize(field, "snapVertexButton",  ".");
  snap_boundary_bt.realize(field, "snapBoundaryButton", "/");
  snap_crossing_bt.realize(field, "snapCrossingButton", "x");
				 
  snap_grid_bt.realize(field, "snapGridButton", "+");
  
  Widget gridSize =
    create_managed_choice(field, CHOICE_GRIDSIZE, "gridSizeChoice",
			  redraw_cb, &grid_size_choice);
  char buf[16];
  // set choices for gridsize
  for (int i = 0; i < grid_size_list.size(); i++) {
    sprintf(buf, "%g", grid_size_list[i]);
    add_to_choice(buf);
  }

  snap_directional_bt.realize(field, "snapDirectionalButton", "-");

  Widget deltaAlpha =
    create_managed_choice(field, CHOICE_DALPHA, "deltaAlphaChoice",
			  redraw_cb, &delta_alpha_choice);
  // set choices for angular snapping
  for (i = 0; i < delta_alpha_list.size(); i++) {
    sprintf(buf, "%g", delta_alpha_list[i]);
    add_to_choice(buf);
  }
  
  snap_autodir_bt.realize(field, "snapAutodirButton", "<");

  XtVaCreateManagedWidget("snapSelfLabel",
			  xmLabelWidgetClass, field,
			  NIL);

  snap_self_bt.realize(field, "snapSelfButton", "", app.snap_self);
}

inline void add_object_button(Widget parent, Object_type t,
			      char *bits, int wd, int ht)
{
  object_type_button[t].realize(parent, 
				"objectTypeButton", bits,
				(t == curobj->type()),
				object_type_cb, wd, ht);
}

static void fill_object_type_field(Widget field)
{
  create_circle_variants(field);
  create_arc_variants(field);
  create_mark_variants(field);
  create_box_variants(field);
  create_text_variants(field);

  Widget rowColumn =
    XtVaCreateManagedWidget("objectTypeButtons",
			    xmRowColumnWidgetClass, field,
			    XmNnumColumns, 3,
			    XmNpacking, XmPACK_COLUMN,
			    XmNradioBehavior, TRUE,
			    NIL);
  
  add_object_button(rowColumn, LINE, line_icon_bits,
		    line_icon_width, line_icon_height);
  add_object_button(rowColumn, TEXT, text_icon_bits,
		    text_icon_width, text_icon_height);
  add_object_button(rowColumn, POLYGON, polygon_icon_bits,
		  polygon_icon_width, polygon_icon_height);
  add_object_button(rowColumn, SPLINE, spline_icon_bits,
		  spline_icon_width, spline_icon_height);
  add_object_button(rowColumn, SPLINEGON, splinegon_icon_bits,
		  splinegon_icon_width, splinegon_icon_height);
  add_object_button(rowColumn, CIRCLE, circle_icon_bits,
		  circle_icon_width, circle_icon_height);
  add_object_button(rowColumn, MARK, mark_icon_bits,
		    mark_icon_width, mark_icon_height);
  add_object_button(rowColumn, ARC, arc_icon_bits,
		    arc_icon_width, arc_icon_height);
  add_object_button(rowColumn, BOX, box_icon_bits,
		    box_icon_width, box_icon_height);
}

//
//----------------------------------------------------------------------
//
// Creation of all the Widgets
//

advertise void create_widgets(void)
{
  Arg arg[2];
  int ac = 0;
  XtSetArg(arg[ac], XmNunitType, Xm100TH_MILLIMETERS); ac++;
  get_ipe_colormap(arg[ac], ac);
  
  mainWindow = XtCreateManagedWidget( "mainWindow", xmMainWindowWidgetClass,
				      topLevel, arg, ac);
  
  // create menu bar along top inside of main window
  menuBar = XmCreateMenuBar( mainWindow, "menuBar",
			     (ArgList) NIL, (Cardinal) 0);
  XtManageChild(menuBar);

  Widget mainForm =
    XtVaCreateManagedWidget("mainForm", xmFormWidgetClass, mainWindow, NIL);

  //  Set MainWindow areas
  XmMainWindowSetAreas (mainWindow, menuBar, NIL, NIL, NIL, mainForm);

  Widget colorField =
    XtVaCreateManagedWidget("colorField",
			    xmBulletinBoardWidgetClass, mainForm,
			    XmNtopAttachment, XmATTACH_FORM,
			    XmNleftAttachment, XmATTACH_FORM,
			    NIL);
  
  Widget penguinIcon =
    XtVaCreateManagedWidget("penguin",
			    xmLabelWidgetClass, mainForm,
			    XmNleftAttachment, XmATTACH_WIDGET,
			    XmNleftWidget, colorField,
			    XmNtopAttachment, XmATTACH_FORM,
			    NIL);

  Widget linestyleField =
    XtVaCreateManagedWidget("linestyleField",
			    xmBulletinBoardWidgetClass, mainForm,
			    XmNleftAttachment, XmATTACH_FORM,
			    XmNtopAttachment, XmATTACH_WIDGET,
			    XmNtopWidget, colorField,
			    NIL);
  
  Widget fontField =
    XtVaCreateManagedWidget("fontField",
			    xmBulletinBoardWidgetClass, mainForm,
			    XmNleftAttachment, XmATTACH_FORM,
			    XmNtopAttachment, XmATTACH_WIDGET,
			    XmNtopWidget, linestyleField,
			    NIL);

  Widget zoomField =
    XtVaCreateManagedWidget("zoomField",
			    xmBulletinBoardWidgetClass, mainForm,
			    XmNleftAttachment, XmATTACH_FORM,
			    XmNtopAttachment, XmATTACH_WIDGET,
			    XmNtopWidget, fontField,
			    NIL);
  
  Widget snapField =
    XtVaCreateManagedWidget("snapField",
			    xmBulletinBoardWidgetClass, mainForm,
			    XmNleftAttachment, XmATTACH_FORM,
			    XmNtopAttachment, XmATTACH_WIDGET,
			    XmNtopWidget, zoomField,
			    NIL);

  objectTypeField =
    XtVaCreateManagedWidget("objectTypeField",
			    xmBulletinBoardWidgetClass, mainForm,
			    XmNleftAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_FORM,
			    NIL);

  Widget canvasFrame =
    XtVaCreateManagedWidget("canvasFrame",
			    xmFrameWidgetClass, mainForm,
			    XmNunitType, XmPIXELS,
			    XmNleftAttachment, XmATTACH_WIDGET,
			    XmNleftWidget, colorField,
			    XmNtopAttachment, XmATTACH_WIDGET,
			    XmNtopWidget, penguinIcon,
			    XmNbottomAttachment, XmATTACH_FORM,
//			    XmNbottomWidget, messageField,
			    XmNrightAttachment, XmATTACH_FORM,
			    NIL);

  messageField =
    XtVaCreateManagedWidget("message",
			    xmLabelWidgetClass, mainForm,
			    XmNleftAttachment, XmATTACH_WIDGET,
			    XmNleftWidget, objectTypeField,
			    XmNtopAttachment, XmATTACH_NONE,
//			    XmNtopWidget, canvasFrame,
			    XmNbottomAttachment, XmATTACH_FORM,
			    XmNrightAttachment, XmATTACH_FORM,
			    NIL);

  Widget mouseExplainer =
    XtVaCreateManagedWidget("mouseExplainer",
			    xmFrameWidgetClass, mainForm,
			    XmNtopAttachment, XmATTACH_FORM,
			    XmNleftAttachment, XmATTACH_NONE,
			    XmNrightAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_WIDGET,
			    XmNbottomWidget, canvasFrame,
			    NIL);

  fileNameField =
    XtVaCreateManagedWidget("fileName",
			    xmLabelWidgetClass, mainForm,
			    XmNleftAttachment, XmATTACH_NONE,
			    XmNtopAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_NONE,
			    XmNrightAttachment, XmATTACH_WIDGET,
			    XmNrightWidget, mouseExplainer,
			    NIL);
  
  pageNumber =
    XtVaCreateManagedWidget("pageNumber",
			    xmLabelWidgetClass, mainForm,
			    XmNleftAttachment, XmATTACH_NONE,
			    XmNtopAttachment, XmATTACH_NONE,
			    XmNbottomAttachment, XmATTACH_WIDGET,
			    XmNbottomWidget, canvasFrame,
			    XmNrightAttachment, XmATTACH_WIDGET,
			    XmNrightWidget, mouseExplainer,
			    NIL);

  shortcutDisplay =
    XtVaCreateManagedWidget("shortcutDisplay",
			    xmLabelWidgetClass, mainForm,
			    XmNleftAttachment, XmATTACH_WIDGET,
			    XmNleftWidget, penguinIcon,
			    XmNtopAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_NONE,
			    XmNrightAttachment, XmATTACH_NONE,
			    NIL);

  operationDisplay =
    XtVaCreateManagedWidget("operationDisplay",
			    xmLabelWidgetClass, mainForm,
			    XmNleftAttachment, XmATTACH_WIDGET,
			    XmNleftWidget, penguinIcon,
			    XmNtopAttachment, XmATTACH_NONE,
			    XmNbottomAttachment, XmATTACH_WIDGET,
			    XmNbottomWidget, canvasFrame,
			    XmNrightAttachment, XmATTACH_NONE,
			    NIL);

  fill_menu_bar(menuBar);

  fill_mouse_explainer(mouseExplainer);

  fill_linestyle_field(linestyleField);
  
  fill_font_field(fontField);

  fill_zoom_field(zoomField);

  fill_snap_field(snapField);

  fill_object_type_field(objectTypeField);
  
  //
  // create popup widgets for messages and choices
  //
  
  messageBox = XmCreateErrorDialog( topLevel, "errorBox", NIL, 0);
  Widget temp = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
  XtUnmanageChild(temp);
  temp = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
  XtUnmanageChild(temp);
  
  choiceBox = XmCreateMessageDialog( topLevel, "choiceBox", NIL, 0);
  XtVaSetValues(choiceBox,
		XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
		NIL);
  XtAddCallback(choiceBox, XmNokCallback,
		ipe_selection_cb, XtPointer(0));
  XtAddCallback(choiceBox, XmNcancelCallback,
		ipe_selection_cb, XtPointer(1));
  XtAddCallback(choiceBox, XmNhelpCallback,
		ipe_selection_cb, XtPointer(2));

  fileSelector =
    XmCreateFileSelectionDialog(topLevel, "selectionBox", NIL, 0);
  temp = XmFileSelectionBoxGetChild(fileSelector,
				    XmDIALOG_HELP_BUTTON);
  XtUnmanageChild(temp);
  XtVaSetValues(fileSelector,
		XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
		NIL);
  XtAddCallback(fileSelector, XmNokCallback, ipe_fileselect_cb, NIL);
  XtAddCallback(fileSelector, XmNcancelCallback, ipe_fileselect_cb, NIL);

  create_configuration_window();
  create_property_window();

  create_ium_parameter_windows();
  
  initialize_ipe_colors();
  fill_color_field(colorField);

  
  // canvas and penguin can only be created when colors have been set up
  create_penguin(penguinIcon);
  create_canvas_widget(canvasFrame);
  
  //
  // Realize Widgets
  //

  XtRealizeWidget(topLevel);
}

//
//----------------------------------------------------------------------
//
// MESSAGE WINDOW
//

//
// Show a message in a popup window
//

advertise void show_message(char *str1, char *str2)
// popup an (error) message window with one or two lines of text
// window stays, and is removed by its OK button
{
  char buf[MAX_LINE_LENGTH];
  strcpy(buf, str1);
  if (str2) {
    strcat(buf, "\n");
    strcat(buf, str2);
  }
  XtVaSetValues(messageBox, XtVaTypedArg,
		XmNmessageString, XmRString, buf, strlen(buf) + 1,
		NIL);
  XtManageChild(messageBox);
}

//
//----------------------------------------------------------------------
//
// CHOICE WINDOW
//

static IpeCallback choice_callback[3];

static void ipe_selection_cb(Widget, XtPointer client_data, XtPointer)
{
  int j = int(client_data);
  DEBUG(DBG_CB, "CALLBACK: Choice selection ", j);
  
  XtUnmanageChild(choiceBox);
  if (choice_callback[j])
    (*(choice_callback[j]))();
}

//
// Ask user for choice from three possibilities
//
// pops up a message window with text and two or three buttons
// (but3 may be NIL)
// continuation is possible only after the user has pressed a button
// in this window, in which case the corresponding callback is called
// the function returns immediately
//
advertise void show_choice(char *str,
		 char *but1, IpeCallback call1,
		 char *but2, IpeCallback call2,
		 char *but3, IpeCallback call3)
{
  Widget temp = XmMessageBoxGetChild(choiceBox, XmDIALOG_HELP_BUTTON);
  if (but3) {
    XtManageChild(temp);
    XtVaSetValues(choiceBox,
		  XtVaTypedArg,
		  XmNhelpLabelString, XmRString, but3, strlen(but3) + 1,
		  NIL);
  }
  else {
    XtUnmanageChild(temp);
  }
  XtVaSetValues(choiceBox,
		XtVaTypedArg,
		XmNmessageString, XmRString, str, strlen(str) + 1,
		XtVaTypedArg,
		XmNokLabelString, XmRString, but1, strlen(but1) + 1,
		XtVaTypedArg,
		XmNcancelLabelString, XmRString, but2, strlen(but2) + 1,
		NIL);

  choice_callback[0] = call1;
  choice_callback[1] = call2;
  choice_callback[2] = call3;
  
  XtManageChild(choiceBox);
}

//
//----------------------------------------------------------------------
//
// FILE SELECTOR
//
// the file selector --- returns at once, but registers callback
//

static IpeStringCallback file_callback;

static void ipe_fileselect_cb(Widget, XtPointer, XtPointer call_data)
{
  XmFileSelectionBoxCallbackStruct *scb =
    (XmFileSelectionBoxCallbackStruct *) call_data;
  XtUnmanageChild(fileSelector);

  if (scb->reason == XmCR_OK) {
    char *fname;
    XmStringGetLtoR(scb->value, XmSTRING_DEFAULT_CHARSET, &fname);
    DEBUG(DBG_CB, "CALLBACK: File selector with name ", fname);
    (*file_callback)(fname);
    XtFree(fname);
  }
}

advertise void show_file_selector(char *str, IpeStringCallback callback)
{
  XtVaSetValues(fileSelector,
		XtVaTypedArg,
		XmNselectionLabelString, XmRString, str, strlen(str) + 1,
		NIL);
  file_callback = callback;
  XtManageChild(fileSelector);
}

//
//----------------------------------------------------------------------
//
// REFRESH display while busy
//

advertise void refresh_display(void)
{
  Drawing_mode saved_drawing_mode = drawing_mode;
  drawing_mode = BUSY;

  XFlush(display);
  while (XtAppPending(app_context) & XtIMXEvent) {
    XtAppProcessEvent(app_context, XtIMXEvent);
  }

  drawing_mode = saved_drawing_mode;
}

advertise void install_colormap(void)
{
  // Install private colormap
  Window windows[1];
  windows[0] = XtWindow(mainWindow);
  XSetWMColormapWindows(XtDisplay(topLevel), XtWindow(topLevel),
			windows, 1);
}
