/* -*-C++-*-
 *
 * itypes.h
 * 
 * This header file defines the Ipe object types
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".

 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef IPE_H
#include "ipe.h"
#else

#ifndef MIXED
#undef  SPLINES_USE_NURBS
#endif

const s_coord DRAG_THRESHOLD = 5.0;
const s_coord BOX_THRESHOLD = 0.1;
const s_coord SPLINE_MULTI_THRESHOLD = 0.1;

// ============================================================

enum Drawing_mode { IDLE, DRAWING, MOVING, SELECTING, DRAGGING,
		    USERMACRO, BUSY };

typedef short Selection_mode;
const short UNSELECTED	= 0;
const short PRIMARY	= 1;
const short SECONDARY	= 2;
const short TENTATIVE	= 3;

enum Object_type { TEXT, SPLINE, LINE, MARK, SPLINEGON, POLYGON, ARC,
		   BOX, CIRCLE, BITMAP, GROUP, SEGMENTS};

const NUM_OBJECT_TYPES = 9;

// ============================================================

// functions defined here are used in inline statements in classes:

extern int mipe_mode;
extern int linestyle_choice;
extern int line_width_choice;
extern int arrow_type;
extern s_coord arrow_size[2];

extern s_coord select_distance, snap_distance;

extern pl_vec curmp;

extern s_coordreray line_width_list;
extern s_coord zoom_factor;

void manage_variant(Widget);
void shortcut(unsigned char);
void redraw_selection(void);
void put_msg(char *);

inline s_coord max(const s_coord a, const s_coord b) {
  return (a > b ? a : b); }

inline s_coord min(const s_coord a, const s_coord b) {
  return (a < b ? a : b); }

inline int max(const int a, const int b) {
  return (a > b ? a : b); }

inline int min(const int a, const int b) {
  return (a < b ? a : b); }

inline int square(const int a) {
  return (a*a); }

inline void snap_vertex(const pl_vec& mp, pl_vec& p, const pl_vec& v) {
  if (clearance(mp, v) < clearance(mp, p))
    p = v;
}

inline int operator!= (const pl_fixedvec &a, const pl_fixedvec &b) {
  return ((a.x() != b.x()) || (a.y() != b.y())); }

inline int operator== (const pl_fixedvec &a, const pl_fixedvec &b) {
  return ((a.x() == b.x()) && (a.y() == b.y())); }

inline s_coord select_dist(void) {
  return select_distance / zoom_factor; }

inline s_coord snap_dist(void) {
  return snap_distance / zoom_factor; }


////////////////////////////// IEDGE //////////////////////////////

class Iedge {
private:
  enum { EDGE, CIRCLE, ARC } type;
  pl_edge edge;
  pl_disc circle;
  pl_arc  arc;
public:
  Iedge() {};
  Iedge(pl_edge e) { edge = e; type = EDGE;}
  Iedge(pl_disc d) { circle = d; type = CIRCLE; }
  Iedge(pl_arc d)  { arc = d; type = ARC;}
  void close_intersection(const Iedge&, const pl_vec&, pl_vec&);
  void close_intersection(const pl_line&, const pl_vec&, pl_vec&);
};

DeclareReray(Iedge_rr, Iedge);
DeclareReray(char_rr, char);
typedef char *charp;
DeclareReray(charp_rr, charp);
DeclareReray(short_rr, short);

////////////////////////////// IOBJECT //////////////////////////////

class Iobject {
protected:
  IColor stroke, fill;			// color of boundary, color of interior
  short dash;				// solid, dashed etc.
  s_coord lwidth;			// linewidth of boundary
  
public:
  Iobject *next, *prev;         	// links
  Selection_mode sel;			// selected ?

  virtual ~Iobject() {};

  virtual Iobject *copy() = 0;		// return copy of yourself
  virtual void stats(int& mem, int &nobjs, int &nvtcs) = 0;
  
  // return type of object
  virtual Object_type type(void) = 0;
  virtual void setchoice(void) { manage_variant(NIL); }
  
  // set stroke and fill color
  virtual void setstroke(IColor c)		{ stroke = c; }
  virtual void setfill(IColor c)		{ fill = c;}
  virtual void setdash(short b)			{ dash = b;}
  virtual void setlwidth(s_coord lw)		{ lwidth = lw;}
  
  // return meaning of left mouse button for this object type
  virtual char *button(void) { return NIL ;}
  
  // routines to handle the mouse in creation mode
  virtual void push(pl_vec&, int, Boolean) {};
  virtual void release(pl_vec&) {};
  virtual void mouse(const pl_vec&, Boolean) {};
  virtual void redraw(void)			{ redraw_selection(); }

  virtual void key(unsigned char c)		{ shortcut(c); }
  
  virtual void edit_push(pl_vec&, int, Boolean) {};
  virtual void edit(void) {
    put_msg("EDIT for this object not implemented");
  }
  
  // transform objects
  virtual void transform(const pl_rotra&) = 0;
  virtual void stretch(const s_coord, const s_coord) = 0;
  
  // compute distance to a point, if smaller than select_dist(), otherwise
  // just return select_dist()
  virtual s_coord dist(const pl_vec&) = 0;
  virtual void snap_vtx(const pl_vec&, pl_vec&) {};
  virtual void snap_bnd(const pl_vec&, pl_vec&) {};
  virtual void close_edge(const pl_vec&, Iedge_rr &) {};
  virtual void snap_self(const pl_vec&, pl_vec&) {};
  
  // the drawing routines:
  // outline draws an outline, is used for pop up planes
  // draw should select the colors, is used only in normal plane
  // special outline modes for rotating and scaling 
  virtual void draw(void) = 0;
  virtual void outline(void) = 0;
  virtual void primary(void) { set_linewidth(2); outline(); set_linewidth(1); }
  virtual void outline_rotated(const pl_vec& p, const pl_angle alpha) {
    push_matrix(); translate(p); rotate(alpha); translate(-p); outline();
    pop_matrix();}

  virtual void outline_stretched(const s_coord xs, const s_coord ys) {
    push_matrix(); scale(xs, ys); outline(); pop_matrix();
  }
  
  virtual void save_postscript(ostream&) = 0;
  virtual void save_properties(ostream&) = 0;
  virtual void bbox(pl_boundingbox&)	 = 0;
  virtual pl_vec vertex(void)	 = 0;

  pl_vec center(void) {
    pl_boundingbox bb; bb.make_empty(); bbox(bb);
    return(pl_vec((bb.xmin() + bb.xmax())/ 2.0,
		     (bb.ymin() + bb.ymax())/ 2.0));
  }
protected:
  void ps_dash(ostream&);
  void dash_properties(ostream&);
  void setcurrent(void) {
    fill.setcurrent(TRUE); stroke.setcurrent(FALSE);
    dash   = linestyle_choice;
    lwidth = line_width_list[line_width_choice];
  }
};

struct ISortobj {
  Iobject *ob;
  s_coord ds;
  int dp;
};

typedef ISortobj *ISortp;
DeclareReray(ISortp_rr, ISortp);

////////////////////////////// SPLINE //////////////////////////////

class Line;

class Spline: public Iobject {
private:
  Boolean closed;		// spline <-> splinegon
  pl_vecreray v;		// control points
  pl_vecreray apr;           // approximation for drawing and snapping
  s_coordreray   tval;          // t values for approximation points
  short_rr       multi;         // multiplicity of control points
  int arrow;			// two bits for two arrows
  s_coord arsize;

  void unfold(pl_vecreray&);
  void apprbezier(pl_vec p[4], s_coord, s_coord);
  void apprsegment(pl_vecreray&, int, int, int, int, Boolean cubic);
  void mkapprox(void);
#ifdef SPLINES_USE_NURBS
  void drawsegment(pl_vecreray&, int, int, int, int, Boolean cubic);
#endif
  void drawcurve(void);
  void setcurrent(void) {
    Iobject::setcurrent();
    arrow = closed ? 0 : arrow_type;
    arsize = arrow_size[mipe_mode];
  }
  
public:
  Spline(void)                    { closed = FALSE; }
  Spline(Line&);
  Spline(istream&);
  Iobject *copy() {return new Spline(*this); }
  void stats(int& mem, int &nobjs, int &nvtcs) {
    mem += sizeof(Spline) + (sizeof(pl_vec) + sizeof(short)) * v.size() +
      (sizeof(pl_vec) + sizeof(s_coord)) * apr.size();
    nobjs++;
    nvtcs += v.size();
  }
  
  Object_type type(void) {return (closed ? SPLINEGON : SPLINE); }
  virtual char *button(void)	{ return ("start spline"); }
  
  void push(pl_vec&, int, Boolean);
  void mouse(const pl_vec&, Boolean);
  void release(pl_vec&);
  void redraw(void);
  void edit_push(pl_vec&, int, Boolean);
  void edit(void);
  void key(unsigned char);

  void setclosed(Boolean c)	{ closed = c; }
  void setarrow(int aw)	{
    if (!arrow && aw) arsize = arrow_size[mipe_mode];
    arrow = aw; }
  void setarsize(s_coord as)	{ arsize = as; }

  void transform(const pl_rotra&);
  void stretch(const s_coord, const s_coord);

  s_coord dist(const pl_vec& p);
  void snap_vtx(const pl_vec&, pl_vec&);
  void snap_bnd(const pl_vec&, pl_vec&);
  void snap_self(const pl_vec&, pl_vec&);

  void draw(void);
  void outline(void);

  void save_postscript(ostream&);
  void save_properties(ostream&);
  void bbox(pl_boundingbox&);
  pl_vec vertex(void)	{ return apr[0]; }
};

////////////////////////////// SEGMENTS //////////////////////////////

class Group;

class Segments: public Iobject {
private:
  pl_vecreray v;		// vertices
  char_rr  m;                   // description of path

public:
  Segments(istream&);
  Iobject *copy() {return new Segments(*this); }
  void stats(int& mem, int &nobjs, int &nvtcs) {
    mem += sizeof(Segments) + (sizeof(pl_vec) + sizeof(short)) * v.size();
    nobjs++; nvtcs += v.size();
  }
  
  Object_type type(void) { return SEGMENTS; }

  void transform(const pl_rotra&);
  void stretch(const s_coord, const s_coord);

  s_coord dist(const pl_vec& p);
  void snap_vtx(const pl_vec&, pl_vec&);
  void snap_bnd(const pl_vec&, pl_vec&);
  void close_edge(const pl_vec&, Iedge_rr&);
  pl_edge *on_edge(const pl_vec&);

  void draw(void);
  void outline(void);

  void save_postscript(ostream&);
  void save_properties(ostream&);
  void bbox(pl_boundingbox&);
  pl_vec vertex(void)	{ return v[0]; }

  Line *Lines(void);
  Segments(const Line*);
};

////////////////////////////// LINE //////////////////////////////

class Line: public Iobject {
friend class Box;
friend Line *Segments::Lines(void);
friend Segments::Segments(const Line *);
friend Boolean group_is_segments(Iobject *);

private:
  Boolean closed;		// line <-> polygon
  int arrow;			// two bits for two arrows
  s_coord arsize;

  pl_vecreray v;		// vertices
  Boolean is_box;               // true if object is an axis parallel box
  
  void setcurrent(void) {
    Iobject::setcurrent();
    arrow = closed ? 0 : arrow_type;
    arsize = arrow_size[mipe_mode];
  }
  void vertex_outline(void);
  void is_box_p(void);
  
public:
  Line(void)                    { is_box = closed = FALSE;}
  Line(istream&);
  Iobject *copy() {return new Line(*this); }
  void stats(int& mem, int &nobjs, int &nvtcs) {
    mem += sizeof(Line) + sizeof(pl_vec) * v.size();
    nobjs++; nvtcs += v.size();
  }
  
  Object_type type(void) {
    return (is_box ? BOX : (closed ? POLYGON : LINE)); }
  virtual char *button(void)	{ return ("start line"); }
  
  virtual void push(pl_vec&, int, Boolean);
  virtual void mouse(const pl_vec&, Boolean);
  virtual void release(pl_vec&);
  virtual void redraw(void);

  void edit_push(pl_vec&, int, Boolean);
  void edit(void);
  void key(unsigned char);

  void setclosed(Boolean c)	{ closed = c; }
  void setarrow(int aw)	{
    if (!arrow && aw) arsize = arrow_size[mipe_mode];
    arrow = aw; }
  void setarsize(s_coord as)	{ arsize = as; }

  void transform(const pl_rotra&);
  void stretch(const s_coord, const s_coord);

  s_coord dist(const pl_vec& p);
  void snap_vtx(const pl_vec&, pl_vec&);
  void snap_bnd(const pl_vec&, pl_vec&);
  void snap_self(const pl_vec&, pl_vec&);
  void close_edge(const pl_vec&, Iedge_rr&);
  pl_edge *on_edge(const pl_vec&);

  void draw(void);
  void outline(void);
//  void primary(void);

  void save_postscript(ostream&);
  void save_properties(ostream&);
  void bbox(pl_boundingbox&);
  pl_vec vertex(void)	{ return v[0]; }

  pl_vecreray vertices(void) { return v; }
};

////////////////////////////// BOX //////////////////////////////

class Box: public Line {
public:
  Box(void)			{ is_box = closed = TRUE; }
  Box(s_coord wd, s_coord ht);
  Box(const pl_vec& ll, const pl_vec& ur);
  
  char *button(void)		{ return ("start box"); }
  void setchoice(void);
  
  void push(pl_vec&, int, Boolean);
  void mouse(const pl_vec&, Boolean);
  void redraw(void);
};

////////////////////////////// CIRCLE //////////////////////////////

class Circle: public Iobject {
private:
  Transform tfm;		// maps unit circle to circle

public:
  Circle(void)			{}
  Circle(istream&);
  Iobject *copy() {return new Circle(*this);}
  void stats(int& mem, int &nobjs, int &nvtcs) {
    mem += sizeof(Circle); nobjs++; nvtcs++; }
  
  Object_type type(void)	{ return (CIRCLE); }
  char *button(void)		{ return ("start circle"); }
  void setchoice(void);
  
  void push(pl_vec&, int, Boolean);
  void release(pl_vec&)	{ }
  void mouse(const pl_vec& mp, Boolean) { curmp = mp ;}
  void redraw(void);

  void transform(const pl_rotra&);
  void stretch(const s_coord, const s_coord);

  s_coord dist(const pl_vec& p);
  void snap_vtx(const pl_vec& mp, pl_vec& p) {
    snap_vertex(mp, p, tfm.transl()); }
  void snap_bnd(const pl_vec&, pl_vec&);
  void close_edge(const pl_vec&, Iedge_rr&);
  s_coord radius(void);

  void draw(void);
  void outline(void);

  void save_postscript(ostream&);
  void save_properties(ostream&);
  void bbox(pl_boundingbox&);
  pl_vec vertex(void)	{ return tfm.transl(); }

};

////////////////////////////// ARC //////////////////////////////

class Arc: public Iobject {
private:
  pl_arc d;
  Arc(pl_arc a) : d(a)		{}
  int arrow;			// two bits for two arrows
  s_coord arsize;

public:
  Arc(void)			{}
  Arc(istream&);
  Iobject *copy() {return new Arc(*this);}
  void stats(int& mem, int &nobjs, int &nvtcs) {
    mem += sizeof(Arc); nobjs++; nvtcs++; }
  
  Object_type type(void)	{ return (ARC); }
  char *button(void)		{ return ("start arc"); }
  void setchoice(void);
  void setarrow(int aw)		{ arrow = aw; }
  void setarsize(s_coord as)	{ arsize = as; }
  
  void push(pl_vec&, int, Boolean);
  void release(pl_vec&)	{ }
  void mouse(const pl_vec& mp, Boolean) { curmp = mp ;}
  void redraw(void);

  void transform(const pl_rotra& tfm)	{ d.transform(tfm); }
  void stretch(const s_coord, const s_coord);

  s_coord dist(const pl_vec& p);
  void snap_vtx(const pl_vec& mp, pl_vec& p);
  void snap_bnd(const pl_vec&, pl_vec&);
  void close_edge(const pl_vec&, Iedge_rr&);

  void draw(void);
  void outline(void);
  void outline_rotated(const pl_vec& p, const pl_angle alpha);
  void outline_stretched(const s_coord xs, const s_coord ys);

  void save_postscript(ostream&);
  void save_properties(ostream&);
  void bbox(pl_boundingbox& bb)	{ bb += d.bbox(); }
  pl_vec vertex(void)	{ return d.center(); }
};

////////////////////////////// MARK //////////////////////////////

class Mark: public Iobject {
private:
  pl_vec pos;
  s_coord size;
  s_coord marker;

public:
  Mark(void)			{}
  Mark(const pl_vec&);
  Mark(istream&);
  Iobject *copy() {return new Mark(*this);}
  void stats(int& mem, int &nobjs, int &nvtcs) {
    mem += sizeof(Mark); nobjs++; nvtcs++; }
  
  Object_type type(void)	{ return (MARK); }
  char *button(void)		{ return ("set a marker"); }
  void setchoice(void);
  
  void push(pl_vec&, int, Boolean);

  void settype(s_coord m)	{ marker = m; }
  void setsize(s_coord s)	{ size = s; }

  void transform(const pl_rotra& tfm)	{ pos = tfm * pos; }
  void stretch(const s_coord xs, const s_coord ys) {
    pos.set_x(pos.x() * xs); pos.set_y(pos.y() * ys);
  }

  s_coord dist(const pl_vec&);
  void snap_vtx(const pl_vec& mp, pl_vec& p);

  void draw(void);
  void outline(void);

  void save_postscript(ostream&);
  void save_properties(ostream&);
  void bbox(pl_boundingbox&);
  pl_vec vertex(void)		{ return pos; }
};

////////////////////////////// TEXT //////////////////////////////

const int EDIT_FONT_OFFS = 4;

class Text: public Iobject {
private:
  char_rr str;
  short font;
  s_coord size;
  pl_vec pos;
  pl_vec tbox;
  Boolean parbox;
  
  void create(void);
  void start_text(const pl_vec&);
  void end_text(void);
  void mkbox(void);
public:
  Text()				{ parbox = FALSE; }
  Text(istream&);
  Iobject *copy() {return new Text(*this);}
  void stats(int& mem, int &nobjs, int &nvtcs) {
    mem += sizeof(Text) + str.size(); nobjs++; nvtcs++; }

  Object_type type(void)		{ return(TEXT); }
  char *button(void)			{ return ("start text"); }
  void setchoice(void);
  
  void push(pl_vec&, int, Boolean);
  void edit_push(pl_vec& mp, int button, Boolean p) {push(mp, button, p); }
  void mouse(const pl_vec&, Boolean);
  void redraw(void);
  void key(unsigned char);
  void edit(void);

  void setfont(short f)			{ font = f; mkbox();}
  void setsize(s_coord s)		{ size = s; mkbox();}
  
  void transform(const pl_rotra& tfm);
  void stretch(const s_coord, const s_coord);

  s_coord dist(const pl_vec&);
  
  void draw(void);
  void outline(void);
  void outline_rotated(const pl_vec& p, const pl_angle alpha);
  void outline_stretched(const s_coord, const s_coord);

  void save_latex(ostream&, const pl_vec& save_offset);
  void save_postscript(ostream&);
  void save_properties(ostream&);
  void bbox(pl_boundingbox& b)	{
    b.add_point(pos);
    b.add_point(pos + pl_vec(tbox.x(), (parbox ? -tbox.y() : tbox.y())));}
  pl_vec vertex(void)		{ return pos; }
};

////////////////////////////// BITMAP //////////////////////////////

class BitmapBits {
private:
  short refcount;
  unsigned long *_bits;
public:
  short width, height;

  BitmapBits(unsigned long *b) { _bits = b;   refcount = 1; }
  ~BitmapBits()                { if (_bits) delete [] _bits; }
  Boolean unlink(void)         { refcount--; return (refcount <= 0); }
  unsigned long *bits(void)    { return _bits; }
  BitmapBits *copy()           { refcount++; return this; }
  int memsize()		       { return sizeof(unsigned long) * width*height;}
};

class Bitmap: public Iobject {
private:
  pl_vec pos;                       // lower left corner of bitmap
  pl_vec ur;                        // upper right corner
  
  Boolean in_color;
  BitmapBits *bits;
  BitmapBits *screen;
  
public:
  Bitmap(void)       { bits = screen = NIL; }
  ~Bitmap()          {if (bits && bits->unlink()) delete bits;
		      if (screen && screen->unlink()) delete screen; }
  Bitmap(Bitmap&);
  Bitmap(istream&);
  Object_type type(void)       { return (BITMAP); }
  Iobject *copy() {return new Bitmap(*this);}
  void stats(int& mem, int &nobjs, int &nvtcs) {
    mem += sizeof(Bitmap) + (bits ? bits->memsize() : 0) +
      (screen ? screen->memsize() : 0); nobjs++; nvtcs++; }

  void transform(const pl_rotra& tfm);
  void stretch(const s_coord xs, const s_coord ys);
  s_coord dist(const pl_vec&);

  void draw(void);
  void outline(void);
  void outline_rotated(const pl_vec& p, const pl_angle alpha);
  
  void save_postscript(ostream&);
  void save_properties(ostream&);
  void bbox(pl_boundingbox& bb)       { bb.add_point(pos); bb.add_point(ur);}
  pl_vec vertex(void)              { return pos; }
};

////////////////////////////// GROUP //////////////////////////////

class Group: public Iobject {
public:
  Iobject *first, *last;
  Group()				{ first = last = NIL; }
  Group(Group &);
  Iobject *copy() {return new Group(*this);}
  Group(istream&, Boolean no_segments = FALSE);
  ~Group();

  void add_object(Iobject *);
  void stats(int& mem, int &nobjs, int &nvtcs);
  
  Object_type type(void)		{ return(GROUP); }

  void setstroke(IColor);
  void setfill(IColor);
  void setdash(short);
  void setlwidth(s_coord);

  void traverse(Object_type, void (*pfunc)(Iobject*));

  void transform(const pl_rotra&);
  void stretch(const s_coord, const s_coord);

  s_coord dist(const pl_vec& p);
  void snap_vtx(const pl_vec& mp, pl_vec& p);
  void snap_bnd(const pl_vec&, pl_vec&);
  void close_edge(const pl_vec&, Iedge_rr&);

  void draw(void);
  void nontext_draw(void);
  void text_draw(void);
  void outline(void);
  void outline_rotated(const pl_vec&, const pl_angle);

  void save_postscript(ostream&);
  void save_properties(ostream&);
  void bbox(pl_boundingbox&);
  pl_vec vertex(void)		{ return first->vertex(); }

};

typedef Group *Groupp;
DeclareReray(Groupp_rr, Groupp);

#endif
