/*
 * ium.C
 * 
 * Calling an Ipe user macro
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"
#include <Xm/FileSB.h>
#include <Xm/SelectioB.h>

static void show_file_box(char *prompt, char *str);
static void show_prompt_box(char *title, char *str);

/////////////////////////// IUMRC INTERFACE ////////////////////////////

static Boolean iumrc_read = FALSE;
static Boolean iumrc_modified = FALSE;
static char iumrc_name[64];

static Iumrc_rr iumrc_contents;

static void read_iumrc(void)
{
  char *home = getenv("HOME");
  char lbuf[MAX_LINE_LENGTH];
  ipestream ifd;
  char *p;

  iumrc_contents.newsize(0);
  iumrc_read = TRUE;
  if (!home)
    return;
  sprintf(iumrc_name, "%s/.iumrc", home);
  if (!(ifd.open(iumrc_name, "r")))
    return;
  IumrcEntry ie;
  while (ifd.getline(lbuf, MAX_LINE_LENGTH)) {
    for (p = lbuf; *p && *p != ':'; p++)
      ;
    if (!*p)
      continue;
    *p = '\0';
    ie.keyword = strdup(lbuf);
    for (p++; *p == ' ' || *p == '\t'; p++)
      ;
    if (p[strlen(p) - 1] == '\n')
      p[strlen(p) - 1] = '\0';
    ie.contents = strdup(p);
    iumrc_contents.append(ie);
  }
  ifd.close();
  if (app.debug & DBG_IUM) {
    cerr << "Iumrc contents read from file:\n";
    for (int k = 0; k < iumrc_contents.size(); k++)
      cerr << iumrc_contents[k].keyword << " -> "
	   << iumrc_contents[k].contents << "\n";
  }
}
    
    
static char *iumrc_get(char *keyword)
/* find entry in iumrc file with keyword */     
{
  if (!iumrc_read) read_iumrc();
  if (!keyword || !iumrc_contents.size())
    return NIL;
  for (int k = 0; k <  iumrc_contents.size(); k++) {
    if (!strcmp(keyword, iumrc_contents[k].keyword)) {
      return iumrc_contents[k].contents;
    }
  }
  return NIL;
}

static void iumrc_put(char *keyword, char *contents)
// save entry in iumrc file
{
  if (!keyword || !contents) return;
  if (!iumrc_read) read_iumrc();
  iumrc_modified = TRUE;
  for (int k = 0; k < iumrc_contents.size(); k++) {
    if (!strcmp(keyword, iumrc_contents[k].keyword)) {
      delete [] iumrc_contents[k].contents;
      iumrc_contents[k].contents = strdup(contents);
      return;
    }
  }
  IumrcEntry ie;
  ie.contents = strdup(contents);
  ie.keyword =  strdup(keyword);
  iumrc_contents.append(ie);
}

static void write_iumrc(void)
{
  ipestream ifd;

  if (!iumrc_modified) return;
  if (!(ifd.open(iumrc_name, "w")))
    return;
  for (int k = 0; k < iumrc_contents.size(); k++) {
    ifd << iumrc_contents[k].keyword << ": "
	<< iumrc_contents[k].contents << "\n";
    // and free storage
    delete [] iumrc_contents[k].keyword;
    delete [] iumrc_contents[k].contents;
  }
  ifd.close();
  iumrc_contents.newsize(0);
  iumrc_read = iumrc_modified = FALSE;
}
    

//
//----------------------------------------------------------------------
//
//  CALLING a IUM
//

static char* ium_exec;
static const IumDescription *ium_description;

advertise void ium_operation(const IumDescription &ium)
{
  ium_exec = search_a_ium(ium.exec);
  if (!ium_exec) {
    // no executable found
    cerr << "no executable found with name " << ium.exec << "\n";
    put_msg("Could not find file for Ipe user macro");
    return;
  }
  if (ium.flags & IUM_IS_IPE) {
    insert_file(ium_exec, TRUE);
  } else {
    if (ium.prompt) {
      // we have to ask for a parameter to send to the ium
      ium_description = &ium;
      char *persist = NIL;
      if (ium.persist) {
	// the parameter should be persistent
	persist = iumrc_get(ium.persist);
      }
      if (ium.flags & IUM_FILENAME) {
	// ask for a filename
	show_file_box(ium.prompt, persist);
	return;
      } else {
	// ask for a string
	show_prompt_box(ium.prompt, persist);
	return;
      }
    }
    call_ium(ium_exec, ium.argument, ium.flags, NIL);
  }
}

static char *spacify(char *str)
{
  for (char *p = str; *p; p++)
    if (*p == '_')
      *p = ' ';
  return(str);
}

//
//----------------------------------------------------------------------
//
// Setting up Iums
//

static Boolean first_ium_in_menu[LAST_MENU] = {
  TRUE, TRUE, TRUE, TRUE, TRUE, TRUE, TRUE };

static Boolean add_ium_operation(char *word[], int num_words)
//  add an Ipe User Macro to the ipe_operation table
{
if (app.debug & DBG_IUM) {
    cerr << "Ipe User Macro description:\n   ";
    for (int i = 0; i < num_words; i++)
      cerr << "`" << word[i] << "' ";
    cerr << "\n";
  }

  IpeOperation op;

  if (num_words < 2)
    return FALSE;

  op.type = IUM_OP;
  op.label = spacify(word[1]);
  op.keys = NIL;
  op.in_drawing = FALSE;

  op.ium.exec  = word[0];
  op.ium.argument = "0";
  op.ium.prompt = NIL;
  op.ium.persist = NIL;

  int j = strlen(word[0]);
  if (j < 4)
    return FALSE;
  if (!strcmp(".ium", word[0] + j - 4))
    op.ium.flags = 0;
  else if (!strcmp(".ipe", word[0] + j - 4))
    op.ium.flags = IUM_IS_IPE;
  else
    return FALSE;
  
  Menus m = LAST_MENU;
  Boolean underline = FALSE;

  for (j = 2; j < num_words; j++) {
    if (!strcmp(word[j], "-key")) {
      if (++j == num_words) return FALSE;
      op.keys = spacify(word[j]);
    } else if (!strcmp(word[j], "-norawbits")) {
      op.ium.flags |= IUM_NO_RAW;
    } else if (!strcmp(word[j], "-nodata")) {
      op.ium.flags |= IUM_NO_DATA;
    } else if (!strcmp(word[j], "-nofile")) {
      op.ium.flags |= IUM_NO_FILE;
    } else if (!strcmp(word[j], "-sep")) {
      underline = TRUE;
    } else if (!strcmp(word[j], "-menu")) {
      if (++j == num_words) return FALSE;
      int val = atoi(word[j]) - 1;
      if (val < 0 || val >= LAST_MENU)
	return FALSE;
      m = Menus(val);
    } else if (!strcmp(word[j], "-arg")) {
      if (++j == num_words) return FALSE;
      op.ium.argument = word[j];
    } else if (!strcmp(word[j], "-input")) {
      if (++j == num_words) return FALSE;
      if (op.ium.prompt)
	return FALSE;
      op.ium.prompt = spacify(word[j]);
    } else if (!strcmp(word[j], "-file")) {
      if (++j == num_words) return FALSE;
      if (op.ium.prompt)
	return FALSE;
      op.ium.prompt = spacify(word[j]);
      op.ium.flags |= IUM_FILENAME;
    } else if (!strcmp(word[j], "-persist")) {
      if (++j == num_words) return FALSE;
      if (op.ium.persist)
	return FALSE;
      op.ium.persist = word[j];
    } else
      return FALSE;
  }
  
  ipe_operation.append(op);
  if (m != LAST_MENU) {
    if (first_ium_in_menu[m])
      add_menu_separator(m);
    first_ium_in_menu[m] = FALSE;
    add_menu_button(m);
    if (underline)
      add_menu_separator(m);
  }
  return TRUE;
}

advertise void decode_iums(char *iums)
// decode Ipe User Macro description
{
  char *word[32];
  char *p = iums;
  int i = 0;
  char ch;
 
  while (TRUE) {
    // wind forward to next non-blank
    while (*p == ' ' || *p == '\t')
      p++;

    ch = *p;

    // if this is a word, add it
    if (ch && ch != ';') {
      word[i++] = p;
      while (*p && *p != ' ' && *p != '\t' && *p != ';')
	p++;
      // at end of ium?
      ch = *p;
      // set 0 at end of word
      *p = '\0';
    }
    p++;
    
    // now ch contains the character we have to use
    if ((!ch || ch == ';') && i > 0) {
      // this is the end of the ium description, so we add ium
      if (!add_ium_operation(word, i)) {
	cerr << "Cannot parse Ipe user macro description:\n   ";
	for (int j = 0; j < i; j++)
	  cerr << word[j] << " ";
	cerr << "\n";
      }
      i = 0;
    }

    // if it's the end of the string, we are done
    if (!ch)
      return;
  }
}

//
//----------------------------------------------------------------------
//
// FILE SELECTOR
// PROMPT BOX
//

Widget fileSelector;
Widget promptBox;

static void ium_parameter_cb(Widget, XtPointer, XtPointer call_data)
{
  XmSelectionBoxCallbackStruct *scb =
    (XmSelectionBoxCallbackStruct *) call_data;
  XtUnmanageChild(fileSelector);

  if (scb->reason == XmCR_OK) {
    char *fname;
    XmStringGetLtoR(scb->value, XmSTRING_DEFAULT_CHARSET, &fname);
    DEBUG(DBG_CB, "CALLBACK: ium_parameter_cb ", fname);
    // if the parameter is persistent, save it
    if (ium_description->persist) {
      iumrc_put(ium_description->persist, fname);
      write_iumrc();
    }
    // now call the Ium!
    call_ium(ium_exec, ium_description->argument,
	     ium_description->flags, fname);
    XtFree(fname);
  }
}

static void show_file_box(char *prompt, char *str)
{
  XtVaSetValues(fileSelector,
		XtVaTypedArg,
		XmNselectionLabelString, XmRString, prompt, strlen(prompt) + 1,
		NIL);
  if (str) {
    // we need to set directory and filename
    char buf[MAX_FILENAME_LENGTH];
    strcpy(buf, str);
    for (char *p = buf + strlen(buf) - 1; p > buf && *p != '/'; p--)
      ;
    if (*p == '/')
      // found directory
      *p = '\0';
    else
      strcpy(buf, ".");
    XtVaSetValues(fileSelector,
		  XtVaTypedArg,
		  XmNdirSpec, XmRString, str, strlen(str) + 1,
		  XtVaTypedArg,
		  XmNdirectory, XmRString, buf, strlen(buf) + 1,
		  NIL);
  }
  XtManageChild(fileSelector);
}

static void show_prompt_box(char *title, char *str)
{
  if (!str)
    str = "";
  XtVaSetValues(promptBox,
		XtVaTypedArg,
		XmNselectionLabelString, XmRString, title, strlen(title) + 1,
		XtVaTypedArg,
		XmNtextString, XmRString, str, strlen(str) + 1,
		NIL);
  XtManageChild(promptBox);
}

//
//----------------------------------------------------------------------
//

advertise void create_ium_parameter_windows(void)
{
  fileSelector =
    XmCreateFileSelectionDialog(topLevel, "iumSelectionBox", NIL, 0);
  Widget temp = XmFileSelectionBoxGetChild(fileSelector,
					   XmDIALOG_HELP_BUTTON);
  XtUnmanageChild(temp);
  XtVaSetValues(fileSelector,
		XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
		NIL);
  XtAddCallback(fileSelector, XmNokCallback,
		ium_parameter_cb, XtPointer(0));
  XtAddCallback(fileSelector, XmNcancelCallback,
		ium_parameter_cb, XtPointer(0));

  promptBox =
    XmCreatePromptDialog(topLevel, "iumPromptBox", NIL, 0);
  temp = XmSelectionBoxGetChild(promptBox,
				XmDIALOG_HELP_BUTTON);
  XtUnmanageChild(temp);
  XtVaSetValues(promptBox,
		XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
		NIL);
  XtAddCallback(promptBox, XmNokCallback, ium_parameter_cb, XtPointer(1));
  XtAddCallback(promptBox, XmNcancelCallback, ium_parameter_cb, XtPointer(1));
}

