/*
 * line.C
 * 
 * Drawing lines
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

static Boolean moving = FALSE;
static int pos;

advertise void draw_arrow(const pl_vec& from, const pl_vec& to, s_coord size)
// draw an arrow
{
  pl_angle alpha = angle_of(to - from);
  push_matrix();
  translate(to);
  rotate(alpha);
  bgn_polygon();
  put_vertex(pl_vec(0,0));
  put_vertex(pl_vec(-size, size/3.0));
  put_vertex(pl_vec(-size, -size/3.0));
  end_polygon();
  pop_matrix();
}

void Line::outline(void)
{
  bgn_line(closed);
  for (int i = 0; i < v.size(); i++)
    put_vertex(v[i]);
  end_line(closed);
}

void Line::draw(void)
{
  if (this == curobj)
    return;

  int i;

  // draw filled interior
  if (!fill.is_empty() && v.size() <= max_nvertex && !transparent_bt.on()) {
    fill.setcol();
    bgn_polygon();
    for (i = 0; i < v.size(); i++)
      put_vertex(v[i]);
    end_polygon();
  }
  
  // draw outline
  if (!stroke.is_empty()) {
    stroke.setcol();
    set_linestyle(dash);
    set_linewidth((lwidth*zoom_factor < 1.0) ? 1 : int(lwidth*zoom_factor));
    // draw the polygonal chain
    bgn_line(closed);
    for (i = 0; i < v.size(); i++)
      put_vertex(v[i]);
    end_line(closed);
  }
  if (arrow && (!stroke.is_empty() || !fill.is_empty())) {
    if (stroke.is_empty())
      fill.setcol();
    if (arrow & 1) {
      draw_arrow(v[1], v[0], arsize);
    }
    if (arrow & 2) {
      draw_arrow(v[v.size()-2], v[v.size()-1], arsize);
    }
  }
}

void Line::is_box_p(void)
// test whether object is (still) a box
{
  is_box = (v.size() == 4) &&
    fabs_pp(v[0].x() - v[3].x()) < BOX_THRESHOLD &&
    fabs_pp(v[1].x() - v[2].x()) < BOX_THRESHOLD &&
    fabs_pp(v[0].y() - v[1].y()) < BOX_THRESHOLD &&
    fabs_pp(v[2].y() - v[3].y()) < BOX_THRESHOLD;
}

void Line::transform(const pl_rotra& tfm)
{
  for (int i = 0; i < v.size(); i++)
    v[i] = tfm * v[i];
  is_box_p();
}

void Line::stretch(const s_coord xfactor, const s_coord yfactor)
{
  for (int i = 0; i < v.size(); i++) {
    v[i].set_x(v[i].x() * xfactor);
    v[i].set_y(v[i].y() * yfactor);
  }
}

s_coord Line::dist(const pl_vec& p)
{
  if (!fill.is_empty() && enable_interior_bt.on())
    return(clearance(p, pl_polygon(v), select_dist()));
  s_coord d = select_dist(), d1;
  for (int i = 0; i < v.size()-1; i++)
    if ((d1 = clearance(p, pl_edge(v[i], v[i+1]), select_dist())) < d)
      d = d1;
  if (closed) {
    d1 = clearance(p, pl_edge(v[i], v[0]), select_dist());
    if (d1 < d)
      d = d1;
  }
  return(d);
}

void Line::close_edge(const pl_vec& p, Iedge_rr& s)
{
  s_coord d = snap_dist();
  pl_edge e;
  
  for (int i = 0; i < v.size()-1; i++) {
    e = pl_edge(v[i], v[i+1]);
    if (clearance(p, e, d) < d) {
      s.append(Iedge(e));
    }
  }
  e = pl_edge(v[i], v[0]);
  if (closed && clearance(p, e, d) < d) {
    s.append(Iedge(e));
  }
}

pl_edge *Line::on_edge(const pl_vec& mp)
// find an edge under the position mp
{
  pl_edge e;
  for (int i = 0; i < v.size() - 1; i++) {
    e = pl_edge(v[i], v[i+1]);
    if (clearance(mp, e) < 1.0)
      return(new pl_edge(e));
  }
  e = pl_edge(v[i], v[0]);
  if (closed && clearance(mp, e) < 1.0) {
    return(new pl_edge(e));
  }
  return NIL;
}
      
advertise void snap_edge(const pl_vec& mp, pl_vec& p, const pl_edge& e)
{
  s_coord d = clearance(mp, p);
  pl_vec v = placeproject(mp, e.supporting_line());
  pl_vec u = v - e.begin_vt();
  if (dot(u, e.direction()) < 0)
    v = e.begin_vt();
  else if (e.length() < u.length())
    v = e.end_vt();
  if (clearance(mp, v) < d)
    p = v;
}

void Line::snap_bnd(const pl_vec& mp, pl_vec& p)
{
  for (int i = 0; i < v.size()-1; i++)
    snap_edge(mp, p, pl_edge(v[i], v[i+1]));
  if (closed)
    snap_edge(mp, p, pl_edge(v[v.size()-1], v[0]));
}

void Line::snap_vtx(const pl_vec& mp, pl_vec& p)
{
  for (int i = 0; i < v.size(); i++)
    snap_vertex(mp, p, v[i]);
}

void Line::snap_self(const pl_vec& mp, pl_vec& p)
{
  for (int i = 0; i < v.size(); i++)
    if (i != pos)
      snap_vertex(mp, p, v[i]);
}  

void Line::bbox(pl_boundingbox& b)
{
  for (int i = 0; i < v.size(); i++)
    b.add_point(v[i]);
}

/////////////////////////// MOUSE //////////////////////////////

void Line::push(pl_vec& mousep, int button, Boolean)
{
  pl_vec mp;

  if (drawing_mode != DRAWING) {
    // left mouse button pressed, starting line mode
    clear_selection();
    snap_base = mousep;
    v.newsize(2);
    v[0] = mousep;
    v[pos = 1] = mousep;
    nonidle(DRAWING);
    moving = TRUE;
    mouse_msg("next point", "final point", "delete last");
    redraw_pup();
  } else {
    switch (button) {
    case LEFT_MOUSE:
      // add a point to current polyline
      snap_base = mousep;
      v[pos++] = mousep;
      v.append(mousep);
      redraw_pup();
      return;
    case MIDDLE_MOUSE:
      // final point!
      v[pos] = mousep;
      {
	Line *l = new Line();
	l->closed = closed;
	l->setcurrent();
	if (closed) {
	  if (!l->fill.is_empty() && !enable_stroke_bt.on())
	    l->stroke.mkempty();
	} else
	  l->fill.mkempty();
	l->v = v;
	l->is_box_p();
	pic->add_object(l);
	object_added();
      }
      v.newsize(0);
      idlefy(2);
      return;
    case RIGHT_MOUSE:
      // delete_last_point
      if (v.size() > 2) {
	v.resize(v.size() - 1);
	snap_base = v[v.size() - 1];
	v[--pos] = mousep;
	redraw_pup();
      } else {
	put_msg("polyline cancelled");
	idlefy(1);
      }
      return;
    }
  }
}

void Line::vertex_outline(void)
{
  s_coord mf = 2.0/zoom_factor;
  int i, ppos;

  for (i = 0; i < v.size(); i++)
    draw_filled_rectangle(v[i] + pl_vec(-mf, -mf),
			  v[i] + pl_vec( mf, mf));
  if (pos < 0) {
    outline();
    return;
  }
  draw_circle(v[pos], 8.0/zoom_factor);
  ppos = (pos == v.size() - 1) ? 0 : pos + 1;
  bgn_line();
  for (i = 0; i <= pos; i++)
    put_vertex(v[i]);
  end_line();
  if (ppos || closed) {
    set_linestyle(linestyle_index(0x00ff), FALSE);
    draw_line(v[pos], v[ppos]);
    set_linestyle(0, FALSE);
  }
  if (ppos) {
    bgn_line();
    for (i = ppos; i < v.size(); i++)
      put_vertex(v[i]);
    if (closed)
      put_vertex(v[0]);
    end_line();
  }
}

/*
  void Line::primary(void)
{
  pos = -1;
  vertex_outline();
}
*/

void Line::key(unsigned char c)
// handle keypress
{
  int i;
  
  if (edit_object) {
    switch ((unsigned char) c) {
    case ('e' + 0x80):
      // end of edit
      is_box_p();
      idlefy(2);
      undo_changed();
      return;
    case ('d' - 0x60):
    case ('h' - 0x60):
      // delete vertex
      if (v.size() < 3) {
	put_msg("cannot remove the last vertices!");
	return;
      }
      for (i = pos; i < v.size()-1; i++)
	v[i] = v[i+1];
      v.resize(v.size() - 1);
      if (pos == v.size())
	pos--;
      moving = FALSE;
      redraw_pup();
      return;
    case ('r'):
      // reverse polyline
      {
	pl_vec w;
	for (i = 0; i < v.size() / 2; i++) {
	  w = v[i];
	  v[i] = v[v.size() - 1 - i];
	  v[v.size() - 1 - i] = w;
	}
	pos = v.size() - 1 - pos;
      }
      if (((arrow & 2) >> 1) != (arrow & 1))
	arrow = 3 - arrow;
      moving = FALSE;
      redraw_pup();
      return;
    default:
      shortcut(c);
      return;
    }
  } else if ((c == 'd' - 0x60) ||
	     (c == 'h' - 0x60)) {
    // kill last vertex
    pl_vec mousep = get_mouse();
    snap(mousep);
    push(mousep, RIGHT_MOUSE, FALSE);
    return;
  } else
    shortcut(c);
}

void Line::edit(void)
{
  saved_curobj = curobj;
  edit_object = TRUE;
  curobj = this;
  nonidle(DRAWING);
  mouse_msg("insert vertex", "move vertex", "select vertex");
  put_msg("use \"C-d\" to delete the current vertex");
  pos = v.size() - 1;
  moving = FALSE;
  redraw_canvas();
}
    

void Line::edit_push(pl_vec& mousep, int button, Boolean)
{
  int i;
  s_coord d, d1;
  
  switch (button) {
  case RIGHT_MOUSE:
    // right mouse key: select a vertex:
    d = clearance(mousep, v[0]);
    pos = 0;
    for (i = 1; i < v.size(); i++) {
      if ((d1 = clearance(mousep, v[i])) < d) {
	d = d1;
	pos = i;
      }
    }
    if (d > select_dist()) {
      // end of edit
      is_box_p();
      idlefy(2);
      undo_changed();
      return;
    } else
      moving = FALSE;
    break;
  case MIDDLE_MOUSE:
    // now move the vertex!
    d = clearance(mousep, v[0]);
    pos = 0;
    for (i = 1; i < v.size(); i++) {
      if ((d1 = clearance(mousep, v[i])) < d) {
	d = d1;
	pos = i;
      }
    }
    moving = TRUE;
    v[pos] = mousep;
    break;
  case LEFT_MOUSE:
    // insert a vertex at mouseposition
    v.resize(v.size() + 1);
    pos++;
    for (i = v.size() -1; i > pos; i--)
      v[i] = v[i-1];
    v[pos] = mousep;
    moving = TRUE;
    break;
  }
  redraw_pup();
  put_msg("use \"C-d\" to delete the current vertex");
}

void Line::mouse(const pl_vec& mp, Boolean)
{
  if (moving) {
    v[pos] = mp;
  }
}

void Line::release(pl_vec& mp)
{
  if (edit_object && moving) {
    v[pos] = mp;
    moving = FALSE;
    redraw_pup();
  }
}

void Line::redraw(void)
// redraw selection plane
{
  if (edit_object) {
    canvas(COL_EDIT);
    vertex_outline();
  } else {
    canvas(COL_CREATION);
    outline();
  }
}

////////////////////////////// SAVING //////////////////////////////

inline int degrees(pl_angle alpha)
{
  return int(alpha.value() * 180.0/ M_PI);
}

void Line::save_postscript(ostream& fh)
// save (multi)line in postscript format
{
  fh << "% Line\n";
  ps_dash(fh);
  if (arrow) {
    fh << "% ar " << arrow << " " << arsize << "\n";
    if (stroke.is_empty())
      fill.save(fh, NIL, " ");
    else 
      stroke.save(fh, NIL, " ");
  }
  fh << "np % # " << v.size() << "\n";
  for (int i = 0; i < v.size(); i++) {
    fh << v[i].x() << " " << v[i].y();
    if (i == 0) {
      if ((arrow & 1) && !(stroke.is_empty() && fill.is_empty()))
	fh << " " << arsize << " " << (v[0]-v[1]) << " af";
      else
	fh << " mt";
    } else {
      if (i == v.size()-1 && (arrow & 2) &&
	  !(stroke.is_empty() && fill.is_empty()))
	fh << " " << arsize << " " << (v[v.size()-1]-v[v.size()-2]) << " at";
      else
	fh << " lt";
    }
    fh << "\n";
  }
  if (closed)
    fh << "cl % cl\n";
  // don't fill if only two vertices
  if (v.size() < 3)
    fill.save(fh, "fi", NIL);
  else
    fill.save(fh, "fi", (stroke.is_empty() ? " fi\n" : " sfi\n"));
  stroke.save(fh, "sk", " sk\n");
  fh << "% End\n\n";
}

void Line::save_properties(ostream& fh)
// save (multi)line as properties
{
  fh << "% Line\n";
  dash_properties(fh);
  if (arrow) {
    fh << "% ar " << arrow << " " << arsize << "\n";
  }
  fh << "% # "
     << v.size() << "\n";
  for (int i = 0; i < v.size(); i++)
    fh << v[i] << "\n";
  if (closed)
    fh << "% cl\n";
  stroke.save(fh, "sk", NIL);
  fill.save(fh, "fi", NIL);
  fh << "% End\n\n";
}

static char *fieldlist[] = {
  "?2ss", "..", "!##", "?0cl", "?2ar", "..", "?0spl" };

Line::Line(istream& fh)
// read line item from file
{
  // set  defaults
  ps_in_value[0] = 65535.0;
  ps_in_value[1] = 1.0;
  ps_in_value[4] = 0;
  ps_in_value[5] = 1.0;

  ps_okay &= ps_read_entry(fh, 7, fieldlist);
  
  if (!ps_okay)
    return;
  
  //
  // be careful with changing indices in ps_in_whatever,
  // because Spline::Spline(Line&) uses them!
  // (to fixup old-style splines)
  // ps_in_defd[6] is tested by Group::Group(istream&)
  // (this is a hack, and can hopefully disappear in Ipe 6.0 :-) )
  //
  stroke.read(0);
  fill.read(1);

  dash = linestyle_index(short(ps_in_value[0]));
  lwidth = ps_in_value[1];
  closed = ps_in_defd[3];
  arrow = int(ps_in_value[4]);
  arsize = ps_in_value[5];

  v = ps_in_array;
  ps_okay = v.size() > 1;
  is_box_p();
}

