/*
 * mark.C
 * 
 * Placing little markers
 *
 * $Modified: Friday, October 28, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"


static Widget markVariant;
advertise int mark_variant;

static void mark_set_type(Iobject *ob)
{
  ((Mark *) ob)->settype(mark_variant + 1);
}

static void variant_choice_callback(ChoiceArgs)
{
  if (selection) {
    pic_traverse(MARK, mark_set_type);
    undo_changed();
    redraw_canvas();
  }
}

advertise void create_mark_variants(Widget parent)
{
  markVariant = 
    create_choice(parent, CHOICE_MODE, "inputVariant",
		  variant_choice_callback, &mark_variant);
  add_to_choice("circle");
  add_to_choice("disc");
  add_to_choice("box");
  add_to_choice("square");
  add_to_choice("cross");
}

void Mark::setchoice(void)
{
  manage_variant(markVariant);
}

void Mark::outline(void)
{
  draw_rectangle(pos + pl_vec(-size, -size),
		 pos + pl_vec(size, size));
}

void Mark::draw(void)
{
  pl_vec v;

  if (stroke.is_empty())
    return;
  
  stroke.setcol();
  sub_pixel(TRUE);
  set_linestyle(0, FALSE);
  set_linewidth(1);
  switch (int(marker)) {
  case 1:
    draw_circle(pos, size);
    break;
  case 2:
    draw_filled_circle(pos, size);
    break;
  case 3:
    draw_rectangle(pos + pl_vec(-size, -size),
		   pos + pl_vec(size, size));
    break;
  case 4:
    draw_filled_rectangle(pos + pl_vec(-size, -size),
			  pos + pl_vec(size, size));
    break;
  case 5:
    draw_line(pos + pl_vec(-size, -size),
	      pos + pl_vec( size,  size));
    draw_line(pos + pl_vec(-size,  size),
	      pos + pl_vec( size, -size));
    break;
  }
  sub_pixel(FALSE);
}

void Mark::push(pl_vec& mousep, int, Boolean)
{
  // left mouse button pressed, place mark
  clear_selection();
  Mark *m = new Mark;
  m->marker = mark_variant + 1;
  m->pos = mousep;
  m->size = mark_size[mipe_mode];
  m->fill.mkempty();
  m->stroke.setcurrent(FALSE);
  m->dash = 0;
  m->lwidth = 1.0;
  pic->add_object(m);
  object_added();
  redraw_canvas();
}

void Mark::save_postscript(ostream& fh)
// save Box in postscript format
{
  fh << "% Mark\n";
  stroke.save(fh, "sk", "\n");
  fh << "% ty " << marker << "\n% sz\n" << size;
  if (stroke.is_empty())
    fh << "\n% xy " << pos << "\n";
  else 
    fh << "\n% xy\n" << pos << " m" << int(marker) << "\n";
  fh << "% End\n\n";
}

void Mark::save_properties(ostream& fh)
// save Box in properties format
{
  fh << "% Mark\n";
  stroke.save(fh, "sk", NIL);
  fh << "% ty " << marker
     << "\n% sz " << size
     << "\n% xy " << pos
     << "\n% End\n\n";
}

s_coord Mark::dist(const pl_vec& mp)
{
  s_coord d = clearance(mp, pos) - size;
  if (d < 0)
    d = 0.0;
  return d;
}

void Mark::bbox(pl_boundingbox& b)
{
  b.add_point(pos);
}

void Mark::snap_vtx(const pl_vec& mp, pl_vec& p)
{
  if (clearance(mp, pos) < clearance(mp, p))
    p = pos;
}

static char *fieldlist[] = {
  "!2xy", "..", "?1ty", "?1sz" };

Mark::Mark(istream& fh)
// read Mark item  from file
{
  // set  defaults
  ps_in_value[2] = app.mark_variant + 1;
  ps_in_value[3] = app.mark_size[mipe_mode];
  ps_okay &= ps_read_entry(fh, 4, fieldlist);
  if (!ps_okay)
    return;

  stroke.read(0);
  fill.mkempty();
  marker = ps_in_value[2];
  size = ps_in_value[3];
  pos.set_x(ps_in_value[0]);
  pos.set_y(ps_in_value[1]);
}

Mark::Mark(const pl_vec& v)
// create mark at position
{
  marker = mark_variant + 1;
  size = mark_size[mipe_mode];
  pos = v;
  fill.mkempty();
  stroke.setcurrent(FALSE);
  dash = 0;
  lwidth = 1.0;
}

