/*
 * menus.C
 * 
 * This module implements the menu bar of Ipe
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

#include <Xm/RowColumn.h>
#include <Xm/PushB.h>
#include <Xm/CascadeB.h>
#include <Xm/Separator.h>

static char *menu_name[] = {
  "fileMenu", "strucMenu", "alignMenu", "miscMenu", "pageMenu",
  "firstIumMenu", "secondIumMenu",
};

static void (*menu_function[])(int) = {
  file_function, struc_function, align_function, misc_function,
    page_function, NIL, NIL
  };

advertise IpeOperation_rr ipe_operation;

static Widget menu[LAST_MENU];
static Widget pageMenuButton;

advertise void hide_page_menu(void)
{
  XtVaSetValues(pageMenuButton, XmNsensitive, FALSE, NIL);
}

advertise void show_page_menu(void)
{
  XtVaSetValues(pageMenuButton, XmNsensitive, TRUE, NIL);
}

//
//----------------------------------------------------------------------
// 
// CALLBACKS
//

void help_callback(Widget, XtPointer, XtPointer)
{
  system(app.help_command);
}

static void menu_callback(Widget, XtPointer client_data, XtPointer)
{
  int entry = int(client_data);
  ipe_op_exec(entry);
}

//
//----------------------------------------------------------------------
//
// TABLES with standard menu definitions
//

static struct {
  char *keys;
  char *label;
  Menus menu;
  MenuFunction menuf;
} std_menu_definition[] = {
  { "C-x C-f", "find file",     FILE_MENU,   FILE_FIND },
  { "C-x C-s", "save file",     FILE_MENU,   FILE_SAVE },
  { "C-x C-w", "write as...",   FILE_MENU,   FILE_WRITE },
  { "C-x i",   "_insert file",  FILE_MENU,   FILE_INSERT },
  { "C-c C-c", "preview",       FILE_MENU,   FILE_PREVIEW_PG },
  { NIL,       "preview all",   FILE_MENU,   FILE_PREVIEW_ALL },
  { NIL,       "print",         FILE_MENU,   FILE_PRINT_PG },
  { NIL,       "_print all",    FILE_MENU,   FILE_PRINT_ALL },
  { "C-x C-c", "quit",          FILE_MENU,   FILE_QUIT },
  
  { "C-_",     "_undo",         STRUC_MENU,  STRUC_UNDO },
  { "g",       "group",         STRUC_MENU,  STRUC_GROUP },
  { "u",       "ungroup",       STRUC_MENU,  STRUC_UNGROUP },
  { "f",       "front",         STRUC_MENU,  STRUC_FRONT },
  { "b",       "_back",         STRUC_MENU,  STRUC_BACK },
  { "M-a",     "sel all",       STRUC_MENU,  STRUC_SELECT_ALL },
  { "M-C-a",   "_sel type",     STRUC_MENU,  STRUC_SELECT_TYPE },
  { "C-d",     "delete",        STRUC_MENU,  STRUC_DELETE },
  { "C-w",     "cut",           STRUC_MENU,  STRUC_CUT },
  { "M-w",     "copy",          STRUC_MENU,  STRUC_COPY },
  { "C-y",     "paste",         STRUC_MENU,  STRUC_YANK },
  { "d",       "_duplicate",    STRUC_MENU,  STRUC_DUPLICATE },
  { "M-e",     "edit",          STRUC_MENU,  STRUC_EDIT },
  { "M-p",     "property",      STRUC_MENU,  STRUC_PROPERTIES },
  
  { "L",       "left",          ALIGN_MENU,  ALIGN_LEFT },
  { "R",       "right",         ALIGN_MENU,  ALIGN_RIGHT },
  { "T",       "top",           ALIGN_MENU,  ALIGN_TOP },
  { "B",       "bottom",        ALIGN_MENU,  ALIGN_BOTTOM },
  { "H",       "H center",      ALIGN_MENU,  ALIGN_HCENTER },
  { "V",       "V center",      ALIGN_MENU,  ALIGN_VCENTER },
  { "C",       "center",        ALIGN_MENU,  ALIGN_CENTER },

  { "n",       "normal size",   MISC_MENU,   MISC_NORMAL },
  { "=",       "zoom to fit",   MISC_MENU,   MISC_FIT_ALL },
  { "@",       "fit selected",  MISC_MENU,   MISC_FIT_SELECTED },
  { "?",       "_grid visible", MISC_MENU,   MISC_GRID_VISIBLE },
  { "C-x C-k", "kill preview",  MISC_MENU,   MISC_GS_KILL },
  { "M-c",     "configuration", MISC_MENU,   MISC_CONFIGURATION },
  { "M-S",     "statistics",    MISC_MENU,   MISC_STATS },

  { "C-n",     "next page",     PAGE_MENU,   PAGE_NEXT },
  { "C-p",     "previous page", PAGE_MENU,   PAGE_PREVIOUS },
  { "M-<",     "first page",    PAGE_MENU,   PAGE_FIRST },
  { "M->",     "_last page",    PAGE_MENU,   PAGE_LAST },
  { "C-o",     "_new page",     PAGE_MENU,   PAGE_NEW },
  { NIL,      "cut page",       PAGE_MENU,   PAGE_CUT },
  { NIL,      "paste page",     PAGE_MENU,   PAGE_YANK },

  { NIL,      NIL,              PAGE_MENU,   END_ENTRY }
};

//
// Table with keyboard activated functions
//

static struct {
  IpeOperationType type;
  char *keys;
  char *label;
  Boolean in_drawing;
  void (*func)(int);
  int argument;
} keyboard_shortcut[] = {
  { MENU_OP,    "C-x u","undo",			FALSE, struc_function,
						       int(STRUC_UNDO) },
  { MENU_OP,    "C-h",  "delete",		FALSE, struc_function,
						       int(STRUC_DELETE)},
  { MISC_OP,    "C-l",  "redraw canvas",	TRUE,  zoom_op,  0 },
  { SELMODE_OP, "l",    "line mode",		FALSE, NIL,      LINE },
  { SELMODE_OP, "w",    "polygon mode",		FALSE, NIL,      POLYGON },
  { SELMODE_OP, "s",    "spline mode",		FALSE, NIL,      SPLINE },
  { SELMODE_OP, "q",    "splinegon mode",	FALSE, NIL,      SPLINEGON },
  { SELMODE_OP, "c",    "circle mode",		FALSE, NIL,      CIRCLE },
  { SELMODE_OP, "r",    "rectangle mode",	FALSE, NIL,      BOX },
  { SELMODE_OP, "m",    "mark mode",		FALSE, NIL,      MARK },
  { SELMODE_OP, "a",    "arc mode",		FALSE, NIL,      ARC },
  { MISC_OP,	"t",    "text mode",		FALSE, textmode_op, 0 },
  { MISC_OP,	"p",    "minipage mode",	FALSE, textmode_op, 1 },
  { MISC_OP,    "M-,",  "zoom in",		TRUE,  zoom_op,  1 },
  { MISC_OP,    "M-.",  "zoom out",		TRUE,  zoom_op,  -1 },
  { MISC_OP,    "o",    "set pan",		TRUE,  pan_op,   0 },
  { MISC_OP,    "M-`",  "reset origin",		TRUE,  fixpoint_op, 0 },
  { MISC_OP,    "`",    "set origin",		TRUE,  fixpoint_op, 1 },
  { MISC_OP,    "C-`",  "set origin & snap",	TRUE,  fixpoint_op, 2 },
  { MISC_OP,    ">",    "set direction",	TRUE,  fixpoint_op, 3 },
  { MISC_OP,    "~",    "set line",		TRUE,  fixpoint_op, 4 },
  { MISC_OP,    "C-~",  "set line & snap",	TRUE,  fixpoint_op, 5 },
  { PREFIX_OP,  "C-x",  "C-x prefix",		FALSE, NIL,      0 },
  { PREFIX_OP,  "C-c",  "C-c prefix",		FALSE, NIL,      0 },
  { MISC_OP,    NIL,    NIL,			FALSE, NIL,      0 }
};

static struct {
  char	    *keys;
  IpeButton *button;
} button_shortcut[] = {
  { ".",    &snap_vertex_bt },
  { "/",    &snap_boundary_bt },
  { "x",    &snap_crossing_bt },
  { "+",    &snap_grid_bt },
  { "<",    &snap_autodir_bt },
  { ",",    &snap_autodir_bt },
  { "z",    &snap_self_bt },
  { "-",    &snap_directional_bt },
  { NIL,    NIL }
};

static void set_in_drawing(char *keys)
{
  for (int i = 0; i < ipe_operation.size(); i++) {
    if (ipe_operation[i].keys && !strcmp(ipe_operation[i].keys, keys)) {
      ipe_operation[i].in_drawing = TRUE;
      return;
    }
  }
}

//
//----------------------------------------------------------------------
//
// SET UP IPE OPERATION TABLE
//

advertise void add_menu_separator(Menus m)
{
  XtVaCreateManagedWidget("menuEntry",
			  xmSeparatorWidgetClass, menu[m],
			  NIL);
}

advertise void add_menu_button(Menus m)
{
  static char buf[MAX_LINE_LENGTH];
  strcpy(buf, ipe_operation[ipe_operation.size() -1].label);
  if (app.show_shortcuts && ipe_operation[ipe_operation.size() -1].keys) {
    strcat(buf, "   ");
    strcat(buf, ipe_operation[ipe_operation.size() -1].keys);
  }
  Widget button =
    XtVaCreateManagedWidget( "menuEntry",
			     xmPushButtonWidgetClass, menu[m],
			     XtVaTypedArg,
			     XmNlabelString, XmRString, buf, strlen(buf) + 1,
			     NIL);
  // arrange for button to call function 
  XtAddCallback(button, XmNactivateCallback, menu_callback,
		XtPointer(ipe_operation.size() - 1));
}

static void set_menu_strings(void)
{
  int i;
  Menus m;
  IpeOperation op;

  ipe_operation.newsize(0);

  // add standard menu definitions

  Boolean underline;
  
  for (i = 0; std_menu_definition[i].menuf != END_ENTRY; i++) {
    // add entry to operation table
    m = std_menu_definition[i].menu;
    op.type = MENU_OP;
    op.keys = std_menu_definition[i].keys;
    underline = FALSE;
    op.label = std_menu_definition[i].label;
    if (op.label[0] == '_') {
      underline = TRUE;
      op.label++;
    }
    op.function.argument = int(std_menu_definition[i].menuf);
    op.in_drawing = FALSE;
    op.function.func = menu_function[m];
    ipe_operation.append(op);
    // and add label to menu
    add_menu_button(m);
    if (underline)
      add_menu_separator(m);
  }
  
  // add keyboard shortcuts
  
  for (i = 0 ; keyboard_shortcut[i].keys; i++) {
    op.type = keyboard_shortcut[i].type;
    op.keys = keyboard_shortcut[i].keys;
    op.label = keyboard_shortcut[i].label;
    op.function.argument = keyboard_shortcut[i].argument;
    op.function.func = keyboard_shortcut[i].func;
    op.in_drawing = keyboard_shortcut[i].in_drawing;
    ipe_operation.append(op);
  }

  for (i = 0 ; button_shortcut[i].keys; i++) {
    op.type  = BUTTON_OP;
    op.keys  = button_shortcut[i].keys;
    op.label = "";
    op.in_drawing = TRUE;
    op.button     = button_shortcut[i].button;
    ipe_operation.append(op);
  }

  // the following menu operations are allowed in drawing mode
  
  set_in_drawing("n");
  set_in_drawing("=");
  set_in_drawing("?");
}

advertise void fill_menu_bar(Widget menuBar)
// creates the buttons and menu widgets, and fills them from the
// tables    
{
  Widget button[LAST_MENU];

  for (int k = FILE_MENU; k < LAST_MENU; k++) {
    button[k] =
      XtVaCreateManagedWidget(menu_name[k],
			      xmCascadeButtonWidgetClass, menuBar,
			      NIL);
    menu[k] = XmCreatePulldownMenu( menuBar, "pulldownMenu", NIL, 0);
    XtVaSetValues(button[k], XmNsubMenuId, menu[k], NIL); 
  }
  pageMenuButton = button[PAGE_MENU];

  //
  // insert entries
  //
  set_menu_strings();
  
  //
  // the HELP menu
  //

  Widget helpButton =
    XtVaCreateManagedWidget( "helpMenu",
			     xmCascadeButtonWidgetClass, menuBar,
			     NIL);
  
  // tell menuBar that this is the help button (will be specially positioned)
  XtVaSetValues(menuBar, XmNmenuHelpWidget, helpButton, NIL);
  
  Widget helpMenu = XmCreatePulldownMenu( menuBar, "pulldownMenu", NIL, 0);
  XtVaSetValues(helpButton, XmNsubMenuId, helpMenu, NIL);
  
  // create the help button up in the menu
  Widget help =
    XtVaCreateManagedWidget( "helpMenuEntry",
			     xmPushButtonWidgetClass, helpMenu, NIL);
  // arrange for Help button to pop up Help
  XtAddCallback(help, XmNactivateCallback, help_callback, 0);

  decode_iums(app.iums);
  decode_iums(app.more_iums);
}
