/*
 * moving.C
 * 
 * move, rotate, scale, and stretch objects
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

enum Moving_mode { MOVE_PANNING = 0, MOVE_TRANSLATE,
		      MOVE_ROTATE, MOVE_SCALE, MOVE_STRETCH };

static char *message[] = {
  "end panning", "end move", "end rotate", "end scale", "end stretch" };

Moving_mode moving_mode;

static pl_vec orig;
static pl_angle orig_alpha;

///////////////////////// CALC TRANSFORM //////////////////////////////

inline pl_angle calc_angle(const pl_vec& p)
{
  pl_vec tmp = p - fixpoint;
  return angle_of(tmp);
}

static pl_vec calc_factors(const pl_vec& p)
{
  pl_vec u = p - fixpoint;
  pl_vec uo = orig - fixpoint;
  
  if (fabs_pp(u.x()) < 1.0/zoom_factor)
    u.set_x(u.x() < 0 ? -1.0/zoom_factor : 1.0/zoom_factor);
  if (fabs_pp(u.y()) < 1.0/zoom_factor)
    u.set_y(u.y() < 0 ? -1.0/zoom_factor : 1.0/zoom_factor);

  if (moving_mode == MOVE_SCALE) {
    s_coord f = u.length() / uo.length();
    return (pl_vec(f,f));
  } else {
    u.set_x(u.x() / uo.x());
    u.set_y(u.y() / uo.y());
    return u;
  }
}

///////////////////////// RUBBER //////////////////////////////

static void stretch_outline(const pl_vec fact)
{
  Iobject *ob;
  for (ob = pic->first; ob; ob=ob->next)
    if (ob->sel) {
      push_matrix();
      translate(pl_vec(fixpoint.x() * (1 - fact.x()),
		       fixpoint.y() * (1 - fact.y())));
      ob->outline_stretched(fact.x(), fact.y());
      pop_matrix();
    }
}

static void rotate_outline(const pl_angle alpha)
{
  Iobject *ob;
  pl_vec fp;
  for (ob = pic->first; ob; ob=ob->next)
    if (ob->sel) {
      if (fixpoint_set) 
	fp = fixpoint;
      else
	fp = ob->center();
      ob->outline_rotated(fp, alpha);
    }
}

inline void move_outline(void)
{
  Iobject *ob;
  for (ob = pic->first; ob; ob=ob->next)
    if (ob->sel || moving_mode == MOVE_PANNING)
      ob->outline();
}

advertise void moving_redraw(void)
{
  pl_angle alpha;
  pl_vec factors;

  if (moving_mode == MOVE_PANNING)
    canvas(COL_PANNING);
  else
    canvas(COL_MOVING);

  switch (moving_mode) {
  case MOVE_SCALE:
  case MOVE_STRETCH:
    factors = calc_factors(curmp);
    stretch_outline(factors);
    break;
  case MOVE_ROTATE:
    alpha = calc_angle(curmp) - orig_alpha;
    rotate_outline(alpha);
    break;
  case MOVE_TRANSLATE:
  case MOVE_PANNING:
    translate(curmp - orig);
    move_outline();    
    break;
  }
}

////////////////////////////// PERFORM MOVE //////////////////////////////

static void move_selection(const pl_vec offs)
// move selected object by offs
{
  for (Iobject *ob = pic->first; ob; ob=ob->next)
    if (ob->sel)
      ob->transform(offs);
}

static void stretch_selection(const pl_vec fact)
// stretch selected objects 
{
  Iobject *ob;
  for (ob = pic->first; ob; ob=ob->next)
    if (ob->sel) {
      ob->transform(-fixpoint);
      ob->stretch(fact.x(), fact.y());
      ob->transform(fixpoint);
    }
}

static void rotate_selection(const pl_angle alpha)
// rotate selected objects by alpha
{
  Iobject *ob;
  pl_vec fp;

  for (ob = pic->first; ob; ob=ob->next)
    if (ob->sel) {
      if (fixpoint_set)
	fp = fixpoint;
      else
	fp = ob->center();
      pl_rotra tfm = pl_rotra(alpha, fp) * pl_rotra(-fp);
      ob->transform(tfm);
    }
}

///////////////////////// MOVE WHAT //////////////////////////////

static Boolean any_close_selection(const pl_vec &mousep)
// true if any selected object close enough to mouse
// if there is none, selects the closest object
// returns FALSE if there is no object close enough
{
  for (Iobject *ob = pic->first; ob; ob = ob->next) {
    if (ob->sel && ob->dist(mousep) < select_dist()) {
      return TRUE;
    }
  }
  s_coord d, ds = select_dist();
  Iobject *ob1;
  for (ob = pic->first; ob; ob = ob->next) {
    if ((d = ob->dist(mousep)) < ds) {
      ob1 = ob;
      ds = d;
    }
  }
  if (ds < select_dist()) {
    clear_selection();
    ob1->sel = PRIMARY;
    return TRUE;
  }
  return (selection() != NIL);
}

//====================================================================

// these routines are called by the canvas handler

advertise void moving_push(pl_vec &mousep, Boolean shft, Boolean ctl, char key)
// mouse button pushed and gone into MOVING mode
{
  // S-Left   : Scale
  // C-Left   : Stretch
  // Middle   : Move
  // S-Middle : Pan
  // C-Middle : Rotate

  if (key == MIDDLE_MOUSE && !ctl && shft) {
    moving_mode = MOVE_PANNING;
  } else if (ctl && shft) {
    return;
  } else {
    // is there something to move?
    if (!any_close_selection(mousep)) {
      put_msg("there is nothing near the mouse");
      return;
    }
    // determine type of moving
    moving_mode = ((key == LEFT_MOUSE) ?
		   (ctl ? MOVE_STRETCH : MOVE_SCALE ) :
		   (ctl ? MOVE_ROTATE  : MOVE_TRANSLATE ));
  }  

  // start MOVING mode
  snap(mousep);
  nonidle(MOVING);
  if (key == LEFT_MOUSE)
    mouse_msg(message[moving_mode], "", "");
  else
    mouse_msg("", message[moving_mode], "");

  if (!fixpoint_set) {
    if (moving_mode == MOVE_ROTATE) {
      fixpoint = selection()->center();
    } else if (moving_mode == MOVE_SCALE || moving_mode == MOVE_STRETCH) {
      pl_boundingbox bb;
      bb.make_empty();
      selection()->bbox(bb);
      fixpoint = pl_vec((mousep.x() > (bb.xmin()+bb.xmax())/2.0 ?
			 bb.xmin() : bb.xmax()),
			(mousep.y() > (bb.ymin()+bb.ymax())/2.0 ?
			 bb.ymin() : bb.ymax()));
    }
  }

  // for ROTATE mode
  orig_alpha = calc_angle(mousep);

  orig = mousep;
  curmp = mousep;
  redraw_pup();
}
  
////////////////////////////// MOUSE //////////////////////////////

advertise void moving_mouse(const pl_vec &mousep)
// the mouse has been moved
{
  curmp = mousep;
}

////////////////////////////// DONE //////////////////////////////

advertise void moving_done(pl_vec &mousep)
{
  snap(mousep);
  switch (moving_mode) {
  case MOVE_PANNING:
    // modify offset
    canvas_pan = canvas_pan + (mousep - orig);
    if (enable_zoom_pan_bt.on())
      gs_show();
    break;
  case MOVE_TRANSLATE:
    move_selection(mousep - orig);
    break;
  case MOVE_SCALE:
  case MOVE_STRETCH:
    stretch_selection(calc_factors(mousep));
    break;
  case MOVE_ROTATE:
    rotate_selection(calc_angle(mousep) - orig_alpha);
    break;
  }
  if (moving_mode != MOVE_PANNING)
    undo_changed();
  idlefy(2);
}

////////////////////////////// PASTE //////////////////////////////

advertise void move_for_paste(void)
// move the selection to mouse position, and setup pasting
{
  pl_vec mousep = get_mouse();
  snap(mousep);

  // move object here first
  pl_vec offs = mousep - pic->last->vertex();
  for (Iobject *ob = pic->first; ob; ob = ob->next)
    if (ob->sel)
      ob->transform(offs);

  nonidle(MOVING);
  moving_mode = MOVE_TRANSLATE;
  mouse_msg("", message[moving_mode], "");
  curmp = mousep;
  orig = mousep;
  redraw_pup();
}

advertise void paste_selection(void)
// paste the contents of the cut_buffer back into the picture
{
  if (!cut_buffer) {
    put_msg("Cut buffer is empty");
    return;
  }
  clear_selection();

  Iobject *ob, *ob1;
  for (ob = cut_buffer; ob; ob = ob->next) {
    ob1 = ob->copy();
    ob1->sel = SECONDARY;
    if (!pic->first) {
      // pasting into empty picture
      pic->first = ob1;
      ob1->prev = NIL;
    } else {
      pic->last->next = ob1;
      ob1->prev = pic->last;
    }
    pic->last = ob1;
  }
  pic->last->next = NIL;
  pic->last->sel = PRIMARY;

  move_for_paste();
}

