/*
 * operations.C
 * 
 * Ipe Operations
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

static void do_zoom(int arg)
{
  s_coord new_zoom_factor = zoom_factor * ((arg > 0) ? 1.2 : (1.0/1.2));
  if (0.1 <= new_zoom_factor && new_zoom_factor <= 100.0) {
    zoom_factor = new_zoom_factor;
    show_zoom();
  }
  if (enable_zoom_pan_bt.on()) 
    gs_show();
}

advertise void zoom_op(int zf)
{
  if (zf)
    do_zoom(zf);
  redraw_canvas();
}

advertise void pan_op(int)
{
  pl_vec mp = get_mouse();
  simple_snap(mp);
  canvas_pan = -mp;
  if (enable_zoom_pan_bt.on()) 
    gs_show();
  redraw_canvas();
}

static void zoom_to_fit(Boolean all)
// zoom to fit all or selected objects     
{
  pl_boundingbox bb;
  bb.make_empty();
  // if fit all: include template!
  if (all && page_number)
    pages[0]->bbox(bb);
  for (Iobject *ob = pic->first; ob; ob = ob->next)
    if (all || ob->sel != UNSELECTED)
      ob->bbox(bb);
  if (bb.is_empty())
    return;

  pl_vec offs(bb.xmax() + bb.xmin(), bb.ymax() + bb.ymin());
  offs *= 0.5;
  canvas_pan = -offs;
  
  s_coord xfac = (canvas_size.x / screen_factor.x()) / (bb.xmax() - bb.xmin());
  s_coord yfac = (canvas_size.y / screen_factor.y()) / (bb.ymax() - bb.ymin());
  s_coord newzoom = ((xfac > yfac) ? yfac : xfac);
  newzoom = 0.01 * floor_pp(newzoom/0.01);
  if (newzoom < 0.1)
    newzoom = 0.1;
  if (newzoom > 100.0)
    newzoom = 100.0;
  zoom_factor = newzoom;
  show_zoom();
  
  if (enable_zoom_pan_bt.on()) 
    gs_show();
  
  redraw_canvas();
}

// ============================================================

static void find_file_cb(char *fname)
{
  DEBUG(DBG_CB, "CALLBACK: find_file", fname);
  find_file(fname);
}

advertise void find_file(char *fn)
// find file with given name and replace memory image by it
{
  int npages, i;
  Group *gr, *ob;

  if (!fn || !fn[0]) {
    show_file_selector("Open picture", find_file_cb);
    return;
  }
  put_msg("loading file ... ");
  switch (npages = ps_read_file(fn, gr)) {
  case -1:
    // could not read
    return;
  case 0:
    // read an IPE format figure
    for (i = 0; i < pages.size(); i++)
      delete pages[i];
    pages.newsize(2);
    mipe_mode = FALSE;
    pages[0] = new Group;
    pic = pages[1] = gr;
    page_number = 1;
    break;
  default:
    // MIPE format
    for (i = 0; i < pages.size(); i++)
      delete pages[i];
    pages.newsize(npages + 1);
    page_number = 0;
    for (ob = (Group *) gr->first; ob; ob = (Group *) ob->next) {
      pages[page_number++] = ob;
    }
    gr->first = gr->last = NIL;
    delete gr;
    page_number = 1;
    pic = pages[page_number];
    mipe_mode = TRUE;
    break;
  }
  maybe_turnoff_undo();
  undo_reset();
  strcpy(filename, newfilename);
  set_ps_preamble(new_pspreamble.size() ? &new_pspreamble[0] : "");
  show_filename();
  show_pagenum();
  show_m_ipe();
  put_msg("");
  gs_lostsync = TRUE;
  zoom_to_fit(TRUE);
}

static void insert_file_cb(char *fname)
{
  DEBUG(DBG_CB, "CALLBACK: insert_file", fname);
  insert_file(fname);
}

advertise void insert_file(char *fn, Boolean movable DefaultsTo(TRUE))
// find file with given name and replace memory image by it
{
  int npages;
  Group *gr;
  if (!fn || !fn[0]) {
    show_file_selector("Insert picture", insert_file_cb);
    return;
  }
  put_msg("inserting file ... ");
  switch (npages = ps_read_file(fn, gr)) {
  case -1:
    // could not read
    return;
  case 0:
    // read an IPE format figure
    clear_selection();
    pic->add_object(gr);
    object_added();
    maybe_turnoff_undo();
    if (movable) {
      move_for_paste();
    } else {
      put_msg("file inserted");
      redraw_canvas();
    }
    return;
  default:
    // MIPE format
    put_msg("cannot insert MIPE format file");
    delete gr;
    return;
  }
}


static void exit_anyway_cb(void)
{
  DEBUG(DBG_CB, "CHOICE CALLBACK: Exit anyway", "");
  undo_reset();
  cleanup_preview();
  exit(0);
}

static void save_and_exit_cb(void)
{
  DEBUG(DBG_CB, "CHOICE CALLBACK: Save and Exit", "");
  if (! ps_save_picture(exit_anyway_cb)) return;
  undo_reset();
  cleanup_preview();
  exit(0);
}

static void open_anyway_cb(void)
{
  undo_reset();
  find_file(NIL);
}

static void save_and_open_cb(void)
{
  if (! ps_save_picture(open_anyway_cb)) return;
  undo_reset();
  find_file(NIL);
}

static void file_properly_saved(void)
{
  char buf[MAX_FILENAME_LENGTH];
  strcpy(buf, "Wrote ");
  strcpy(filename, newfilename);
  strcat(buf, filename);
  put_msg(buf);
  show_filename();
  undo_reset();
}

advertise void file_function(int arg)
// execute a file function
{
  switch (MenuFunction(arg)) {
  case FILE_FIND:
    // find file
    if (undo_p()) {
      show_choice("WARNING\nChanges have not been saved.",
		  "Save",         save_and_open_cb,
		  "Open Anyway",  open_anyway_cb,
		  "Cancel",       NIL);
      return;
    }
    find_file(NIL);
    return;
  case FILE_INSERT:
    // insert an Ipe file
    insert_file(NIL);
    return;
  case FILE_SAVE:
  case FILE_WRITE:
    // save file
    ps_save_picture(((arg == FILE_SAVE) ? filename : (char *) NIL),
		    0, TRUE, file_properly_saved);
    return;
  case FILE_PREVIEW_PG:
    // preview single page in background ghostscript previewer
    preview_picture(FALSE, FALSE);
    break;
  case FILE_PREVIEW_ALL:
    // preview all pages in ghostview or similar
    preview_picture(FALSE, TRUE);
    break;
  case FILE_PRINT_PG:
    // print single page
    preview_picture(TRUE, FALSE);
    break;
  case FILE_PRINT_ALL:
    // print all pages
    preview_picture(TRUE, TRUE);
    break;
  case FILE_QUIT:
    // quit ipe
    if (undo_p()) {
      show_choice("WARNING\nChanges have not been saved.",
		  "Save and Exit", save_and_exit_cb,
		  "Exit Anyway",   exit_anyway_cb,
		  "Cancel",        NIL);
      return;
    }
    cleanup_preview();
    exit(0);
  }
}

advertise void align_function(int arg)
{
  pl_boundingbox bprim, bsec;
  Iobject *ob;
  s_coord mx, my;
  pl_vec v;

  Iobject *sel = selection();

  if (!sel) {
    put_msg("nothing selected");
    return;
  }
  bprim.make_empty();
  sel->bbox(bprim);
  for (ob = pic->first; ob; ob = ob->next) {
    if (ob->sel == SECONDARY) {
      bsec.make_empty();
      ob->bbox(bsec);
      mx = my = 0.0;
      switch (arg) {
      case ALIGN_LEFT:
	mx = bprim.xmin() - bsec.xmin();
	break;
      case ALIGN_RIGHT:
	mx = bprim.xmax() - bsec.xmax();
	break;
      case ALIGN_TOP:
	my = bprim.ymax() - bsec.ymax();
	break;
      case ALIGN_BOTTOM:
	my = bprim.ymin() - bsec.ymin();
	break;
      case ALIGN_HCENTER:
	mx = ((bprim.xmin() + bprim.xmax())
	      - (bsec.xmin() + bsec.xmax())) / 2.0;
	break;
      case ALIGN_VCENTER:
	my = ((bprim.ymin() + bprim.ymax())
	      - (bsec.ymin() + bsec.ymax())) / 2.0;
	break;
      case ALIGN_CENTER:
	mx = ((bprim.xmin() + bprim.xmax())
	      - (bsec.xmin() + bsec.xmax())) / 2.0;
	my = ((bprim.ymin() + bprim.ymax())
	      - (bsec.ymin() + bsec.ymax())) / 2.0;
	break;
      }
      ob->transform(pl_vec(mx, my));
    }
  }
  redraw_canvas();
  undo_changed();
}  

advertise void misc_function(int arg)
{
  switch (MenuFunction(arg)) {
  case MISC_NORMAL:
    // normal size
    zoom_factor = 1.0;
    show_zoom();
    if (enable_zoom_pan_bt.on())
      gs_show();
    redraw_canvas();
    return;
  case MISC_FIT_ALL:
    // zoom to fit all
    zoom_to_fit(TRUE);
    break;
  case MISC_FIT_SELECTED:
    // zoom to fit selected
    zoom_to_fit(FALSE);
    break;
  case MISC_GRID_VISIBLE:
    // toggle grid visibility
    grid_visible = !grid_visible;
    redraw_canvas();
    return;
  case MISC_GS_KILL:
    // kill preview window
    gs_kill();
    return;
  case MISC_CONFIGURATION:
    // show configuration window
    show_configuration_window();
    return;
  case MISC_STATS:
    // show statistics
    {
      int mem = 0, nobjs = 0, nvtcs= 0;
      pic->stats(mem, nobjs, nvtcs);
      char buf1[128], buf2[128];
      sprintf(buf1, "%d objects with %d vertices", nobjs, nvtcs);
      sprintf(buf2, "%d bytes of memory used", mem );
      show_message(buf1, buf2);
    }
    return;
  }
}  

advertise void struc_function(int arg)
{
  switch (MenuFunction(arg)) {
  case STRUC_UNDO:
    // undo
    undo_operation();
    return;
  case STRUC_GROUP:
    // group
    group_selection();
    break;
  case STRUC_UNGROUP:
    // ungroup
    ungroup_selection();
    break;
  case STRUC_FRONT:
    // front
    front_selection();
    break;
  case STRUC_BACK:
    // back
    back_selection();
    break;
  case STRUC_SELECT_ALL:
    select_all(FALSE);
    break;
  case STRUC_SELECT_TYPE:
    select_all(TRUE);
    break;
  case STRUC_CUT:
    cut_selection(TRUE);
    break;
  case STRUC_DELETE:
    if (enable_undo_bt.on())
      cut_selection(FALSE);
    else
      put_msg("UNDO is off---you better cut than delete...");
    break;
  case STRUC_COPY:
    copy_selection();
    break;
  case STRUC_YANK:
    paste_selection();
    break;
  case STRUC_DUPLICATE:
    duplicate_selection();
    break;
  case STRUC_EDIT:
    if (!selection())
      put_msg("select object to EDIT first");
    else if (selection()->type() == GROUP)
      put_msg("cannot edit group");
    else {
      selection()->edit();
    }
    return;
  case STRUC_PROPERTIES:
    if (!selection()) {
      put_msg("select an object first");
      return;
    }
    show_properties();
    return;
  }
}  

advertise void page_function(int iarg)
{
  MenuFunction arg = MenuFunction(iarg);
  int m = pages.size();
  int i;

  if (!mipe_mode)
    return;
  if (PAGE_NEXT <= arg && arg <= PAGE_LAST) {
    switch (arg) {
    case PAGE_NEXT:
      if (page_number < m - 1) {
	page_number++;
      }
      break;
    case PAGE_PREVIOUS:
      if (page_number > 0) {
	page_number--;
      }
      break;
    case PAGE_FIRST:
      page_number = 1;
      break;
    case PAGE_LAST:
      page_number = m - 1;
      break;
    }
    clear_selection();
    pic = pages[page_number];
    maybe_turnoff_undo();
    undo_new_page();
    show_pagenum();
    redraw_canvas();
    return;
  }

  gs_lostsync = TRUE;
  switch (arg) {
  case PAGE_NEW:
    pages.resize(m+1);
    for (i = m; i > page_number+1; i--)
      pages[i] = pages[i-1];
    pages[++page_number] = new Group;
    clear_selection();
    pic = pages[page_number];
    undo_reset(FALSE);
    break;
  case PAGE_CUT:
    // cut page to cut buffer
    if (page_number == 0 )
      put_msg("cannot remove template");
    else if (m == 2)
      put_msg("cannot remove only page of document");
    else {
      if (cut_page)
	delete cut_page;
      cut_page = pic;
      if (page_number == m - 1)
	page_number--;
      else {
	for (int i = page_number; i < m - 1; i++)
	  pages[i] = pages[i+1];
      }
      pages.resize(m - 1);
      clear_selection();
      pic = pages[page_number];
      undo_reset(FALSE);
    }
    break;
  case PAGE_YANK:
    // yank page
    if (cut_page != NIL) {
      pages.resize(m+1);
      for (int i = m; i > page_number + 1; i--)
	pages[i] = pages[i-1];
      clear_selection();
      page_number++;
      pic = pages[page_number] = cut_page;
      cut_page = NIL;
      undo_reset(FALSE);
    } else
      put_msg("Nothing in page cutbuffer");
    break;
  }
  show_pagenum();
  redraw_canvas();
}

//
// ========================= KEYBOARD SHORTCUTS ======================
//

static char some_keys[10] = "";

advertise void ipe_op_exec(int i)
{
  IpeOperation &op = ipe_operation[i];
  
  if (op.type == PREFIX_OP) {
    put_msg(some_keys);
    return;
  }

  // not a prefix any more, reset key sequence
  some_keys[0] = '\0';
  
  if (drawing_mode != IDLE && !op.in_drawing) {
    // key sequence not legal in drawing mode
    put_msg("cannot execute this command while drawing");
    return;
  }

  // legal operation, display name and shortcut
  show_operation(op.label, (op.keys ? op.keys : "nil"));

  // and execute it
  switch (op.type) {
  case IUM_OP:
    // call an Ipe User Macro, or include object
    ium_operation(op.ium);
    return;
  case BUTTON_OP:
    // toggle a button
    op.button->toggle();
    return;
  case SELMODE_OP:
    // select a drawing mode
    sel_mode(Object_type(op.function.argument));
    return;
  case MISC_OP:
  case MENU_OP:
    // call a function
    (*op.function.func)(op.function.argument);
  case PREFIX_OP:
    return;
  }
}

advertise void shortcut(unsigned char key)
{
  char keystr[2];

  if (some_keys[0])
    strcat(some_keys, " ");
  if (key & 0x80) {
    strcat(some_keys, "M-");
    key &= 0x7f;
  }
  if (key < 0x1b || key == 0x1e) {
    strcat(some_keys, "C-");
    key += 0x60;
  } else if (key < 0x20) {
    strcat(some_keys, "C-");
    key += 0x40;
  }
  keystr[0] = key;
  keystr[1] = '\0';
  strcat(some_keys, keystr);
  
  for (int i = 0; i < ipe_operation.size(); i++) {
    if (ipe_operation[i].keys && !strcmp(ipe_operation[i].keys, some_keys)) {
      // found shortcut, now execute it
      ipe_op_exec(i);
      return;
    }
  }
  // key sequence unknown, reset prefix
  char buf[128];
  sprintf(buf, "unknown command: `%s'", some_keys);
  put_msg(buf);
  some_keys[0] = '\0';
  return;
}

