/*
 * preview.C
 * 
 * run LaTeX on picture, and preview or print it
 *
 * $Modified: Wednesday, November  2, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>

static Boolean dir_made = FALSE;
static char dir_name[128];

static pid_t latex_pid = 0;
static pid_t gs_pid = 0;

static void sig_killed(SIGNAL_HANDLER_ARGUMENT)
{
  // if current file has not been saved, do it now!
  cleanup_preview();
  if (undo_p()) {
    ps_save_picture((mipe_mode ? "#unsaved.mipe#" :"#unsaved.ipe#"), 0);
  }
  exit(1);
}

static void make_latex_dir(void)
{
  if (!dir_made) {
    sprintf(dir_name, "%s/ipe%d", app.tmp_directory, getpid());
    if (mkdir(dir_name, 0700) < 0) {
      show_message("cannot create temporary directory", dir_name);
      return;
    }
    dir_made = TRUE;
    DEBUG(DBG_LATEX, "Made temporary directory ", dir_name);
  }
}  

static int run_latex(Boolean print = FALSE, Boolean all_pages = FALSE)
// run latex on picture
{
  char buf[512], dvipsbuf[512];
  put_msg("saving file ... ");
  refresh_display();
  make_latex_dir();

  char pageno[32];
  if (all_pages) {
    // save pages in separate files
    sprintf(pageno, "\\def\\IPEPVpages{%d}", pages.size() - 1);
    for (int i = 1; i < pages.size(); i++) {
      // save page i
      sprintf(buf, "%s/page%d.ipe", dir_name, i);
      if (! ps_save_picture(buf, i)) {
	show_message("could not save picture in", buf);
	return -1;
      }
    }
  } else {
    // save only current page in an ipe file
    sprintf(buf, "%s/page1.ipe", dir_name);
    sprintf(pageno, "\\def\\IPEPVmipage{%d}", page_number);
    if (! ps_save_picture(buf, page_number ? page_number : 1)) {
      show_message("could not save picture in", buf);
      return -1;
    }
  }
  // if gs window previewing and no text -> no Latex
  if (!all_pages && !print && !text_objects_saved) {
    sprintf(dvipsbuf, "%s/ipe_page.ps", dir_name);
    return rename(buf, dvipsbuf);
  }
  // now run latex
  put_msg("running LaTeX ... ");
  refresh_display();
  sprintf(buf,
	  "cd %s; %s '\\def\\IPEPVpreamble{%s}%s%s\\input{ipe_pages.tex}'",
	  dir_name, app.latex_command, get_preamble(),
	  ((!print && !all_pages) ? "\\def\\IPEGS{}" : ""),
	  pageno);
  
  for (char *p = buf; *p; p++) {
    if (*p == '\n')
      *p = ' ';
  }

  DEBUG(DBG_LATEX, "Running Latex: ", buf);
  int latex_ret = system(buf);
  if (latex_ret != 0)
    return latex_ret;

  put_msg("running dvips ... ");
  refresh_display();
  sprintf(dvipsbuf, "cd %s; ", dir_name);
  sprintf(buf, app.dvips_command,
	  (all_pages ? "ipe_pages.ps" : "ipe_page.ps"));
  strcat(dvipsbuf, buf);
  DEBUG(DBG_LATEX, "Running Dvips: ", buf);
  latex_ret = system(dvipsbuf);
  put_msg("running dvips ... done");
  
  return latex_ret;
}

static ipestream to_gs;
static Boolean gs_with_text;
static pl_vec gs_save_offset;

// when the pipe gets broken, we just continue sending data into
// the failing channel for the current session.
// then, we behave as if it had been properly cancelled

static void sig_pipe_broken(SIGNAL_HANDLER_ARGUMENT)
{
  gs_pid = 0;
  signal(SIGPIPE, SIG_IGN);
  to_gs.close();
  //  to_gs.clear(ios::failbit);
}

advertise void gs_show(void)
{
  if (!gs_pid)
    return;

  if (gs_lostsync)
    return;

  // clear previous ghostscript page
  to_gs << "\n";

  // translate such that origin is in center of window
  to_gs << canvas_origin.x() << " " << canvas_origin.y() << " translate ";
  // show picture with current zoom factor
  to_gs << (zoom_factor*screen_factor.x()) << " "
	<< (zoom_factor*screen_factor.y()) << " scale ";
  to_gs << canvas_pan << " translate 0 setgray ";
  // now show it:
  if (gs_with_text)
    // (500,500) is the magic position from ipe_pages.tex
    to_gs << -(gs_save_offset + pl_vec(500,500))
	  << " translate (ipe_page.ps) run\n";
  else
    // directly from ipe file!
    to_gs << -gs_save_offset << " translate (ipe_page.ps) run showpage\n";
  to_gs.flush();
}

advertise void preview_picture(Boolean printflag, Boolean all_pages)
// run LaTeX and previewer/printer on image
// printflag: false -> preview   true -> print
// all_pages determines whether to print current page or full document
{
  if (run_latex(printflag, all_pages) != 0) {
    put_msg("LaTeX error");
    return;
  }

  char buf[256], psname[128];
  if (printflag || all_pages) {
    strcpy(psname, dir_name);
    strcat(psname, (all_pages ? "/ipe_pages.ps" : "/ipe_page.ps"));
    sprintf(buf, get_command(printflag), psname);
    system(buf);
  } else {
    // previewing on background Ghostscript window

    // first we want to figure out whether Ghostscript is still alive
    if (gs_pid) {
      for (int rept = 0; rept < 100; rept++) {
	to_gs << "\n";
	to_gs.flush();
      }
      // by now we should have a broken pipe...
    }
    if (!gs_pid) {
      // startup ghostscript
      put_msg("starting Ghostscript in background ...");
      refresh_display();
      int to_ghs[2];
      if (pipe(to_ghs)) {
	show_message("could not open pipes to ghostscript", "");
	return;
      }
      if ((gs_pid = fork()) == 0) {
	// this is the child process. run gs and connect to pipe
	chdir(dir_name);
	close(to_ghs[1]);
	dup2(to_ghs[0], 0);
	// set window size to same size as current canvas
	sprintf(buf, "-g%dx%d", canvas_size.x, canvas_size.y);
	DEBUG(DBG_GS, "Starting Ghostscript", buf);
	if (app.debug & DBG_GS)
	  execl(app.ghostscript_name, "gs", buf, "-r72", NIL);
	else
	  execl(app.ghostscript_name, "gs", buf, "-r72", "-q", NIL);
	cerr << "could not start ghostscript\n";
	exit(1);
      }
      
      // this is the main process
      // ghostscript is running now

      // setup stream to ghostscript
      close(to_ghs[0]);

      to_gs.close();
      // catch broken pipes from now on
      signal(SIGPIPE, sig_pipe_broken);
      to_gs.openfd(to_ghs[1], "w");

      put_msg("starting Ghostscript in background ... done");
    }

    // ghostscript should be up and running now, show drawing
    gs_lostsync = FALSE;
    gs_with_text = text_objects_saved;
    gs_save_offset = save_offset;
    gs_show();
  }
}

advertise void gs_kill(void)
// kill ghostscript process
{
  if (!gs_pid)
    return;
  to_gs << "\nquit\n\n";
  gs_pid = 0;
  to_gs.close();
  return;
}

advertise void parm_noundo_cb(void *, long)
// turn off/on undo system
{
  undo_reset(FALSE);
}

advertise void cleanup_preview(void)
// try to unlink the garbage in /usr/tmp
{
  char buf[256];

  if (dir_made) {
    sprintf(buf, "rm -r %s", dir_name);
    system(buf);
  }
  if (latex_pid) {
    // latex is running, stop it
    kill(latex_pid, SIGKILL);
  }
  if (gs_pid) {
    // ghostscript is running, stop it
    kill(gs_pid, SIGKILL);
  }
  kill_ium();
}

advertise void show_properties(void)
{
  static char propname[MAX_FILENAME_LENGTH];
    
  ipestream fh;
  make_latex_dir();
  sprintf(propname, "%s/ipe_property", dir_name);
  if (!fh.open(propname, "w")) {
    show_message("Cannot open file for writing", propname);
    return;
  }
  fh << "Properties of selected object:\n\n";
  selection()->save_properties(fh);
  fh << "% End\n";
  fh.close();
  // Show properties
  show_property_window(propname);
}

static ipestream to_latex;
static ipestream from_latex;

static void kill_latex(void)
{
  to_latex.close();
  from_latex.close();
  kill(latex_pid, SIGKILL);
  latex_pid = 0;
}

advertise Boolean latex_box(const char *bstr,
			    const char_rr& dstr, const char *fstr,
			    s_coord& wd, s_coord& ht, s_coord& dp)
// compute latex box for string str
{
  char linebuf[1024];
  static char *old_preamble = NIL;

  strcpy(linebuf, get_preamble());
  if (use_new_preamble)
    strcpy(linebuf, &new_latexpreamble[0]);
  
  if (!latex_pid || strcmp(old_preamble, linebuf)) {
    // remember LaTeX preamble
    if (old_preamble) delete [] old_preamble;
    old_preamble = strdup(linebuf);
    if (latex_pid) kill_latex();
    
    put_msg("starting LaTeX in background ...");
    refresh_display();

    // startup latex in background, connect by pipe
    int to_ltx[2], from_ltx[2];
    
    make_latex_dir();
    if (pipe(to_ltx) || pipe(from_ltx)) {
      show_message("could not open pipes to latex", "");
      return FALSE;
    }
    if ((latex_pid = fork()) == 0) {
      // this is the child process. run latex and connect to pipe
      chdir(dir_name);
      close(to_ltx[1]);
      close(from_ltx[0]);
      dup2(to_ltx[0], 0);
      dup2(from_ltx[1], 1);
      (void) execlp(app.latex_command, app.latex_command, NIL);
      cerr << "could not start latex\n";
      exit(1);
    }

    // this is the main process
    // latex is running now
    // setup streams to/from latex

    close(to_ltx[0]);
    close(from_ltx[1]);

    to_latex.openfd(to_ltx[1], "w");
    from_latex.openfd(from_ltx[0], "r");
    
    // would like to use \\nonstopmode, but that does not work from a pipe
    to_latex << "\\scrollmode\n";
    to_latex << linebuf << "\n"
	     << "\\makeatletter\n"
	     << "\\@ifundefined{selectfont}"
	     << "{\\let\\selectfont\\relax\\def\\fontsize#1#2{}}{}"
	     << "\\@ifundefined{IPEfs}"
	     << "{\\def\\IPEfs#1{\\dimen0=#1pt\\fontsize{#1}{1.2\\dimen0}"
	     << "\\selectfont}}{}"
	     << "\\makeatother\n"
	     << "\\newcounter{mipage}\\setcounter{mipage}{1}\n"
	     << "\\newcount\\bigpoint\n"
	     << "\\dimen0=0.01bp\n"
	     << "\\bigpoint=\\dimen0\n"
	     << "\\newbox\\Ipebox\n"
	     << "\\def\\display#1{\\count0=#1\\divide\\count0 by \\bigpoint\n"
	     << "  \\message{\\the\\count0\\space}}\n"
	     << "\\long\\def\\bbox#1{\\global\\setbox\\Ipebox=\\hbox{#1}}\n"
	     << "\\def\\IpeErr{\\errmessage{some error in string }}\n"
	     << "\\def\\showbbox{\\message{ IPEbox }%\n"
	     << "\\display{\\wd\\Ipebox}\n"
	     << "\\display{\\ht\\Ipebox}\n"
	     << "\\display{\\dp\\Ipebox}}\n"
	     << "\\newif\\ifX\n"
	     << "\\newif\\ifZ\n"
	     << "\\Xfalse\\Zfalse\n";

    // now everything is open and initialized
  }
  // latex is running, send data there
  put_msg("computing bounding box ...");
  refresh_display();

  to_latex << "\\Xfalse\\Zfalse{\\Xtrue{{{{{{\\Ztrue\n";
  to_latex << "\\bbox{" << bstr << (&dstr[0]) << fstr << "}\n";
  to_latex <<
    "\\ifmmode\\IpeErr$\\fi\\ifZ\\else\\IpeErr\\fi}\\ifZ\\IpeErr\\fi\n";
  for (int i = 0; i < 11; i++)
    to_latex << "\\ifX}\\fi";
  to_latex << "\\showbbox\n";
  to_latex.flush();
  Boolean latex_error = FALSE;
  int fatal_latex_error = 0;
  do {
    from_latex >> linebuf;
    if (!strcmp(linebuf, "<*>"))
      latex_error = TRUE;
    if (!strcmp(linebuf, "another"))
      fatal_latex_error++;
    if (!strcmp(linebuf, "file"))
      fatal_latex_error++;
    if (!strcmp(linebuf, "input"))
      fatal_latex_error++;
    DEBUG(DBG_LATEX, "LATEX says ", linebuf);
  } while (fatal_latex_error < 4 &&
	   strcmp(linebuf, "IPEbox") != 0 &&
	   from_latex.good());
  if (fatal_latex_error > 3 || !from_latex.good()) {
    // read error
    show_message("error reading from latex",
		 "something completely messed up");
    kill_latex();
    enable_latex_box_bt.set(FALSE);
    put_msg("fatal LaTeX error");
    return FALSE;
  }
  if (latex_error) {
    put_msg("LaTeX error");
    return FALSE;
  }
    
  // everything is fine, now read measurements of box
  int w, h, d;
  from_latex >> w >> h >> d;
  wd = w/100.0;
  ht = h/100.0;
  dp = d/100.0;
  ht += dp;
  put_msg("computing bounding box ... done");
  return TRUE;
}

//
// calling Ipe user macros
//
// We fork, and run the process in background. Foreground process only does
// redraws, and waits for SIGUSR1 (good) or SIGUSR2 (bad)
//

static int ium_wait_state = 0;
static pid_t ium_pid = 0;
// the number of objects sent to the user macro
static int no_obj = 0;
static char ium_string[MAX_LINE_LENGTH] = "";
static char ium_file[MAX_FILENAME_LENGTH] = "";

//
// The following code somehow isn't portable enough.
//
// static void sig_child(int ... )
// // a child has died
// {
//   pid_t pid, status;
// 
//   // is it the Ium process?
//   pid = wait(&status);
//   if (pid == ium_pid && !WIFSTOPPED(status)) {
//     ium_wait_state = (WIFEXITED(status) && !WEXITSTATUS(status)) ? 1 : -1;
//     qenter(IUM_EVENT, 0);
//   }
//   sigset(SIGCLD, sig_child);
// }

static void sig_ium_ended_well(SIGNAL_HANDLER_ARGUMENT);
static void sig_ium_ended_badly(SIGNAL_HANDLER_ARGUMENT);

static void ium_has_ended(int iws)
{
  ium_wait_state = iws;
  signal(SIGUSR1, sig_ium_ended_well);
  signal(SIGUSR2, sig_ium_ended_badly);
  // when Ipe is idle, it will find time to look here
}

static void sig_ium_ended_well(SIGNAL_HANDLER_ARGUMENT)
// (***) signal from child that ium has ended
{
  ium_has_ended(1);
}

static void sig_ium_ended_badly(SIGNAL_HANDLER_ARGUMENT)
// (***) signal from child that ium has ended
{
  ium_has_ended(-1);
}

advertise char *search_a_ium(char *ium_name)
// looks in IumDirectories for the given name and returns first match or NIL
{
  static char ium_path[MAX_FILENAME_LENGTH];
  static struct stat statbuf;

  for (int i = 0; i < ium_directory.size(); i++) {
    // test whether file exists
    strcpy(ium_path, ium_directory[i]);
    strcat(ium_path, "/");
    strcat(ium_path, ium_name);
    printf( "ium_path: [%s]\n", ium_path );
    if (!stat(ium_path, &statbuf)) {
      return ium_path;
    }
  }
  return NIL;
}

static Boolean ium_ended_test(XtPointer)
// this is a Xt work procedure to test whether a Ium has terminated
// if ium_wait_state is no longer zero, the Ium has finished
{
  // get rid of spurious events---this should not happen
  if (drawing_mode != USERMACRO) {
    // unregister myself
    return TRUE;
  }
  if (!ium_wait_state) {
    // has not terminated, stay active
    return FALSE;
  }

  int ium_return = ium_wait_state;
  ium_wait_state = 0;
  
  DEBUG(DBG_CB, "Ipe user macro has terminated with status ", ium_return);
  
  // we are out of USERMACRO mode
  idlefy(0);

  if (ium_return < 0) {
    show_message("Error when calling IUM", ium_string);
    cerr << "Error calling an Ipe user macro as\n" << ium_string << "\n";
    return TRUE;
  }
  
  // try to read file
  ipestream fh;
  if (!fh.open(ium_file, "r")) {
    show_message("cannot open file for reading", ium_file);
    return TRUE;
  }
  char linebuf[MAX_LINE_LENGTH];

  fh >> linebuf;
  if (strncmp(linebuf, "%\\IUMid", 7)) {
    show_message("this is not a valid IUM output", ium_file);
    fh.close();
    return TRUE;
  }
  char *kw = ps_read_next(fh);
  if (!kw || strcmp(kw, "Interface")) {
    show_message("IUM file corrupted", ium_file);
    fh.close();
    return TRUE;
  }
  int ium_mode;
  fh >> ium_mode;
  fh.getline(linebuf, MAX_LINE_LENGTH);
  for (char *msg = linebuf; *msg && (*msg == ' ' || *msg == '\t'); msg++)
    ;
  
  ps_okay = TRUE;
  Group *gr = new Group(fh);
  fh.close();
  if (!ps_okay) {
    show_message("could not read IUM output file", ium_file);
    delete gr;
    return TRUE;
  }

  put_msg(msg);

  // ium_mode :
  //  0 just add new things, do not select them
  //  1 deselect old, select new
  //  2 select new, select old
  //  3 replace old by new (must be same number of objects)
  //  4 delete old objects, add and select new
  
  if (ium_mode == 1)
    clear_selection();

  if (ium_mode == 1 || ium_mode == 2 || ium_mode == 4) {
    for (Iobject *ob = gr->first; ob; ob = ob->next)
      ob->sel = SECONDARY;
  }

  if (ium_mode == 4) {
    Iobject *ob, *ob1, *nf, *nl;
    make_selection_list(nf, nl);
    for (ob = nf; ob; ob = ob1) {
      ob1 = ob->next;
      delete ob;
    }
  }
  
  if (ium_mode != 3) {
    if (gr->first) {
      // there actually is something to add
      if (pic->first) {
	pic->last->next = gr->first;
	gr->first->prev = pic->last;
	pic->last = gr->last;
      } else {
	pic->first = gr->first;
	pic->last = gr->last;
      }
    }
    gr->first = gr->last = NIL;
    delete gr;
  } else {
    // replace selection by new objects
    for (Iobject *ob = gr->first; ob; ob = ob->next)
      no_obj--;
    if (no_obj) {
      //  not equal number !
      delete gr;
      show_message("IUM output has not same number of objects",
		   "cannot replace old objects");
      return TRUE;
    }
    // now do the replacement:
    Iobject *nob = gr->first, *nnob;
    for (ob = pic->first; ob; ob = ob->next) {
      if (ob->sel != UNSELECTED) {
	// replace object with *nob
	nnob = nob->next;
	nob->sel = ob->sel;
	nob->prev = ob->prev;
	nob->next = ob->next;
	delete ob;
	ob = nob;
	if (ob->next)
	  ob->next->prev = ob;
	else
	  pic->last = ob;
	if (ob->prev)
	  ob->prev->next = ob;
	else
	  pic->first = ob;
	nob = nnob;
      }
    }
    gr->first = gr->last = NIL;
    delete gr;
  }
    
  find_new_primary();
  maybe_turnoff_undo();
  undo_changed();
  redraw_canvas();
  return TRUE;
}

advertise void call_ium(char *exec, char *ium_arg,
			unsigned long flags, char *ium_parm)
// call an Ipe User Macro
{
  ipestream ofh;
  make_latex_dir();
  sprintf(ium_file, "%s/ium.ipe", dir_name);

  if (!(flags & IUM_NO_FILE)) {
    // write interface file for Ium
    if (!ofh.open(ium_file, "w")) {
      show_message("Cannot open file for writing", ium_file);
      return;
    }
    
    IColor col;
    
    ofh << "%\\IUMid{" << ipe_version << "}\n"
	<< "% Settings\n"
	<< "% ss " << linestyle_list[linestyle_choice]
	<< " " << line_width_list[line_width_choice]
	<< "\n% ar " << arrow_type
	<< " " << arrow_size[mipe_mode]
	<< "\n% ty " << mark_variant
	<< "\n% sz " << mark_size[mipe_mode]
	<< "\n% f " << font_choice[mipe_mode]
	<< " " << fontsize_list[mipe_mode][fontsize_choice[mipe_mode]]
	<< "\n% grid " << grid_size_list[grid_size_choice]
	<< " " << delta_alpha_list[delta_alpha_choice]
	<< "\n";
    col.setcurrent(FALSE);
    col.save(ofh, "sk", NIL);
    col.setcurrent(TRUE);
    col.save(ofh, "fi", NIL);
    if (fixpoint_set) {
      ofh << "% axis " << fixpoint << " " << axis_dir.value() << "\n";
    }
    ofh << "% End\n\n";
    if (!(flags & IUM_NO_DATA)) {
      // write selection
      if (!(flags & IUM_NO_RAW))
	write_raw_bitmaps = 1;
      no_obj = 0;
      for (Iobject *ob = pic->first; ob; ob = ob->next) {
	if (ob->sel != UNSELECTED) {
	  if (ob->sel == PRIMARY)
	    ofh << "% Primary\n";
	  ob->save_properties(ofh);
	  no_obj++;
	}
      }
    }
    ofh << "% End\n";
    ofh.close();
    write_raw_bitmaps = 0;
  }
  
  ium_wait_state = 0;
  if (ium_parm)
    sprintf(ium_string, "%s %s %s \"%s\"", exec, ium_file, ium_arg, ium_parm);
  else
    sprintf(ium_string, "%s %s %s", exec, ium_file, ium_arg);
  DEBUG(DBG_IUM, "Calling Ium as \n", ium_string);
  
  if (!(ium_pid = fork())) {
    // this is the child process.
    setsid();
    signal(SIGINT, SIG_DFL);
    signal(SIGQUIT, SIG_DFL);
    int ium_ret = system(ium_string);
    // (***) execl(ium_path, ium_exec, ium_file, buf, NULL);
    // ium has returned, send signal to parent
    // (this seems to be much more portable than catching SIG_CHILD)
    kill(getppid(), (ium_ret ? SIGUSR2 : SIGUSR1));
    exit(0);
  }
  // main process
  if (ium_pid < 0) {
    put_msg("Cannot start Ipe User macro.");
  } else {
    // we go in USERMACRO mode, and wait for the things to come
    mouse_msg("", "", "");
    put_msg("waiting for Ipe user macro ...");
    nonidle(USERMACRO);
    XtAppAddWorkProc(app_context, ium_ended_test, NIL);
  }
  return;
}

advertise void kill_ium(void)
{
  if (ium_pid) {
    kill(-ium_pid, SIGINT);
    ium_pid = 0;
  }
}

advertise void signal_setup(void)
// setup signal handlers and event callback for SIGUSR interface
{
  // catch killing signals, to save current drawing and remove temp dir
  signal(SIGINT, sig_killed);
  signal(SIGQUIT, sig_killed);

  // and this is for the IUM interface
  // (***) sigset(SIGCLD, sig_child);
  signal(SIGUSR1, sig_ium_ended_well);
  signal(SIGUSR2, sig_ium_ended_badly);
}

