/*
 * segments.C
 * 
 * The segments object type
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

enum { LINETO, BGN_OPEN, END_OPEN, BGN_CLOSED, END_CLOSED };

void Segments::outline(void)
{
  for (int i = 0; i < v.size(); i++) {
    switch (m[i]) {
    case BGN_OPEN:
      bgn_line(FALSE);
      put_vertex(v[i]);
      break;
    case END_OPEN:
      put_vertex(v[i]);
      end_line(FALSE);
      break;
    case BGN_CLOSED:
      bgn_line(TRUE);
      put_vertex(v[i]);
      break;
    case END_CLOSED:
      put_vertex(v[i]);
      end_line(TRUE);
      break;
    case LINETO:
      put_vertex(v[i]);
      break;      
    }
  }
}

void Segments::draw(void)
{
  int i;

  // draw filled interior
  if (!fill.is_empty() && !transparent_bt.on()) {
    fill.setcol();
    for (i = 0; i < v.size(); i++) {
      int j;
      switch (m[i]) {
      case BGN_OPEN:
      case BGN_CLOSED:
	for (j = i+1; m[j] != END_OPEN && m[j] != END_CLOSED; j++)
	  ;
	if (j - i + 1 <= max_nvertex) {
	  bgn_polygon();
	  put_vertex(v[i]);
	} else
	  i = j;
	break;

      case LINETO:
	put_vertex(v[i]);
	break;
	
      case END_OPEN:
      case END_CLOSED:
	put_vertex(v[i]);
	end_polygon();
	break;
      }
    }
  }
  
  // draw outline
  if (!stroke.is_empty()) {
    stroke.setcol();
    set_linestyle(dash);
    set_linewidth((lwidth*zoom_factor < 1.0) ? 1 : int(lwidth*zoom_factor));
    // draw the polygonal chain
    outline();
  }
}

void Segments::transform(const pl_rotra& tfm)
{
  for (int i = 0; i < v.size(); i++)
    v[i] = tfm * v[i];
}

void Segments::stretch(const s_coord xfactor, const s_coord yfactor)
{
  for (int i = 0; i < v.size(); i++) {
    v[i].set_x(v[i].x() * xfactor);
    v[i].set_y(v[i].y() * yfactor);
  }
}

s_coord Segments::dist(const pl_vec& p)
{
  s_coord d = select_dist(), d1;

  if (fill.is_empty() || !enable_interior_bt.on()) {

    // test distance to boundaries
    short beg;
    
    for (int i = 0; i < v.size(); i++) {
      switch (m[i]) {
      case BGN_CLOSED:
      case BGN_OPEN:
	beg = i;
	break;
      case END_CLOSED:
	if ((d1 = clearance(p, pl_edge(v[i], v[beg]), select_dist())) < d)
	  d = d1;
      case LINETO:
      case END_OPEN:
	if ((d1 = clearance(p, pl_edge(v[i-1], v[i]), select_dist())) < d)
	  d = d1;
      break;
      }
    }
    return(d);
  }

  
  // need to test whether a filled region contains mouse position
  pl_vecreray w;
  
  for (int i = 0; i < v.size(); i++) {
    switch (m[i]) {
    case BGN_CLOSED:
    case BGN_OPEN:
      w.newsize(1);
      w[0] = v[i];
      break;
    case LINETO:
      w.append(v[i]);
      break;
    case END_CLOSED:
    case END_OPEN:
      if ((d1 = clearance(p, pl_polygon(w), select_dist())) < d)
	d = d1;
      break;
    }
  }
  return(d);
}

void Segments::close_edge(const pl_vec& p, Iedge_rr& s)
{
  s_coord d = snap_dist();
  pl_edge e;
  short beg;
    
  for (int i = 0; i < v.size(); i++) {
    switch (m[i]) {
    case BGN_CLOSED:
    case BGN_OPEN:
      beg = i;
      break;
    case END_CLOSED:
      e = pl_edge(v[i], v[beg]);
      if (clearance(p, e, d) < d) {
	s.append(Iedge(e));
      }
    case LINETO:
    case END_OPEN:
      e = pl_edge(v[i-1], v[i]);
      if (clearance(p, e, d) < d) {
	s.append(Iedge(e));
      }
      break;
    }
  }
}

pl_edge *Segments::on_edge(const pl_vec& mp)
// find an edge under the position mp
{
  pl_edge e;
  short beg;
    
  for (int i = 0; i < v.size(); i++) {
    switch (m[i]) {
    case BGN_CLOSED:
    case BGN_OPEN:
      beg = i;
      break;
    case END_CLOSED:
      e = pl_edge(v[i], v[beg]);
      if (clearance(mp, e) < 1.0)
	return(new pl_edge(e));
    case LINETO:
    case END_OPEN:
      e = pl_edge(v[i-1], v[i]);
      if (clearance(mp, e) < 1.0)
	return(new pl_edge(e));
      break;
    }
  }
  return NIL;
}
      
void Segments::snap_bnd(const pl_vec& mp, pl_vec& p)
{
  short beg;
    
  for (int i = 0; i < v.size(); i++) {
    switch (m[i]) {
    case BGN_CLOSED:
    case BGN_OPEN:
      beg = i;
      break;
    case END_CLOSED:
      snap_edge(mp, p, pl_edge(v[i], v[beg]));
    case LINETO:
    case END_OPEN:
      snap_edge(mp, p, pl_edge(v[i-1], v[i]));
      break;
    }
  }
}

void Segments::snap_vtx(const pl_vec& mp, pl_vec& p)
{
  for (int i = 0; i < v.size(); i++)
    snap_vertex(mp, p, v[i]);
}

void Segments::bbox(pl_boundingbox& b)
{
  for (int i = 0; i < v.size(); i++)
    b.add_point(v[i]);
}

////////////////////////////// SAVING //////////////////////////////

void Segments::save_postscript(ostream& fh)
// save Segments in postscript format
{
  fh << "% Segments\n";
  ps_dash(fh);
  fh << "% # " << v.size() << "\n";
  for (int i = 0; i < v.size(); i++) {
    fh << v[i].x() << " " << v[i].y() << " ";
    switch (m[i]) {
    case BGN_OPEN:
    case BGN_CLOSED:
      fh << "N\n";
      break;
    case LINETO:
      fh << "L\n";
      break;
    case END_OPEN:
      fh << "E\n";
      break;
    case END_CLOSED:
      fh << "C\n";
      break;
    }
  }
  fill.save(fh, "fi", (stroke.is_empty() ? " fi\n" : " sfi\n"));
  stroke.save(fh, "sk", " sk\n");
  fh << "% End\n\n";
}

void Segments::save_properties(ostream& fh)
// save Segments as properties = Group of Line
{
  fh << "% Group %% many similar Line segments\n";
  for (int i = 0; i < v.size(); i++) {
    int j;
    
    switch (m[i]) {
    case BGN_OPEN:
    case BGN_CLOSED:
      fh << "% Line\n";
      dash_properties(fh);
      for (j = i+1; m[j] != END_OPEN && m[j] != END_CLOSED; j++)
	;
      fh << "% # " << (j - i + 1) << "\n";
      fh << v[i] << "\n";
      break;
    case LINETO:
      fh << v[i] << "\n";
      break;
    case END_CLOSED:
    case END_OPEN:
      fh << v[i] << "\n";
      if (m[i] == END_CLOSED)
	fh << "% cl\n";
      stroke.save(fh, "sk", NIL);
      fill.save(fh, "fi", NIL);
      fh << "% End\n";
      break;
    }
  }
  fh << "% End %% of similar Line segments\n";
}

static char *fieldlist[] = {"?2ss", "..", "!@#" };

Segments::Segments(istream& fh)
// read line item from file
{
  // set  defaults
  ps_in_value[0] = 65535.0;
  ps_in_value[1] = 1.0;

  ps_okay &= ps_read_entry(fh, 3, fieldlist);
  
  if (!ps_okay)
    return;

  stroke.read(0);
  fill.read(1);

  dash = linestyle_index(short(ps_in_value[0]));
  lwidth = ps_in_value[1];

  v = ps_in_array;

  // now decode segments

  if (v.size() < 2) {
    ps_okay = FALSE;
    return;
  }
  Boolean active = FALSE;
  short beg;
  m.newsize(v.size());
    
  for (int i = 0; i < v.size(); i++) {
    if ((ps_in_string[i] == 'N' && active) ||
	(ps_in_string[i] != 'N' && !active)) {
      ps_okay = FALSE;
      return;
    }
    switch (ps_in_string[i]) {
    case 'N':
      beg = i;
      m[i] = BGN_OPEN;
      active = TRUE;
      break;
    case 'L':
      m[i] = LINETO;
      break;
    case 'E':
      m[i] = END_OPEN;
      active = FALSE;
      break;
    case 'C':
      m[i] = END_CLOSED;
      m[beg] = BGN_CLOSED;
      active = FALSE;
      break;
    }
  }
  ps_okay = !active;
}

///////////////////////// CONVERSION TO LINES ///////////////////////////

Line *Segments::Lines(void)
// return Segments as a linked list of Lines     
{
  Line *l, *fl, *pl;

  fl = pl = NIL;
  for (int i = 0; i < v.size(); i++) {
    switch (m[i]) {
    case BGN_OPEN:
    case BGN_CLOSED:
      l = new Line;
      if (pl) {
	pl->next = l;
	l->prev = pl;
	pl = l;
      } else {
	fl = pl = l;
	l->prev = NIL;
      }
      l->next = NIL;
      l->sel = UNSELECTED;
      l->arrow = 0;
      l->arsize = 1.0;
      l->closed = (m[i] == BGN_CLOSED);
      l->stroke = stroke;
      l->fill = fill;
      l->dash = dash;
      l->lwidth = lwidth;
      l->v.newsize(1);
      l->v[0] = v[i];
      break;
    case LINETO:
    case END_CLOSED:
    case END_OPEN:
      l->v.append(v[i]);
      break;
    }
  }
  return fl;
}

Segments::Segments(const Line* l)
// make Segments from a linked list of Lines
{
  v.newsize(0);
  m.newsize(0);
  stroke = l->stroke;
  fill = l->fill;
  dash = l->dash;
  lwidth = l->lwidth;
  next = prev = NIL;
  sel = UNSELECTED;

  int i = 0, j, n;
  
  for ( ; l; l = (Line *) l->next) {
    n = l->v.size();
    v.resize(i + n);
    m.resize(i + n);
    for (j = 0; j < n; j++) {
      v[i + j] = l->v[j];
      m[i + j] = LINETO;
    }
    if (l->closed) {
      m[i] = BGN_CLOSED;
      m[i + n - 1] = END_CLOSED;
    } else {
      m[i] = BGN_OPEN;
      m[i + n - 1] = END_OPEN;
    }
    i += n;
  }
}

advertise void ungroup_segments(void)
{
  Iobject *ob = selection();
  Iobject *nf, *nl, *ob1;
  
  // retrieve list of Lines
  nl = nf = ((Segments *) ob)->Lines();
  
  while (nl->next) {
    nl->sel = SECONDARY;
    nl = nl->next;
  }
  nl->sel = SECONDARY;
  nf->sel = PRIMARY;
  
  // unlink selection
  ob1 = ob->next;
  if (!ob1 && !ob->prev) {
    // only one object present
    pic->first = nf;
    pic->last = nl;
  } else {
    if (!ob->prev) {
      // first object
      pic->first = ob1;
      ob1->prev = NIL;
    } else if (!ob1) {
      // last object
      pic->last = ob->prev;
      ob->prev->next = NIL;
    } else {
      // elements before and behind me
      ob->prev->next = ob1;
      ob1->prev = ob->prev;
    }
    // unlinked group, add its elements at front
    pic->last->next = nf;
    nf->prev = pic->last;
    pic->last = nl;
  }
  // destroy group object
  delete ob;
  undo_changed();
  redraw_canvas();
}

advertise Boolean group_is_segments(Iobject *ob)
{
  if (!ob ||
      (ob->type() != BOX && ob->type() != LINE && ob->type() != POLYGON))
    return FALSE;

  // must group at least three elements
  if (!ob->next || !ob->next->next)
    return FALSE;
  
  Line *l = (Line *) ob;
  if (l->arrow)
    return FALSE;

  IColor stroke = l->stroke;
  IColor fill = l->fill;
  short dash = l->dash;
  s_coord lwidth = l->lwidth;

  while (l->next) {
    l = (Line *) l->next;
    if ((l->type() != BOX && l->type() != LINE && l->type() != POLYGON) ||
	(stroke != l->stroke || fill != l->fill ||
	 dash != l->dash || lwidth != l->lwidth) ||
	l->arrow)
      return FALSE;
  }
  return TRUE;
}
