/*
 * selecting.C
 * 
 * maintaining the current selection
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

static ISortp_rr selqueue;
static int cursel;

advertise void clear_selection(void)
{
  for (Iobject *ob = pic->first; ob; ob = ob->next)
    ob->sel = UNSELECTED;
}

advertise void find_new_primary(void)
// tests whether there is a primary selection, and creates a new one if not
{
  if (selection())
    return;

  for (Iobject *ob = pic->last; ob; ob = ob->prev) {
    if (ob->sel) {
      ob->sel = PRIMARY;
      return;
    }
  }
}

advertise Iobject *selection(void)
// return primary selection
{
  for ( Iobject *ob = pic->last; ob; ob = ob->prev ) {
    if (ob->sel == PRIMARY)
      return ob;
  }
  return NIL; 
}

static int sobjcompare(ISortp ob1, ISortp ob2)
{
  return ((ob1->ds == ob2->ds) ?
	  ((ob1->dp > ob2->dp) ? -1 : 1) :
	  ((ob1->ds < ob2->ds) ? -1 : 1));
}

static void create_selection(const pl_vec& mousep, Boolean fixed_type)
{
  ISortp sob;
  Iobject *ob;
  s_coord d;
  int lev = 0;
  Object_type type;

  if (fixed_type)
    type = curobj->type();

  selqueue.newsize(0);
  for (ob = pic->first; ob; ob=ob->next) {
    if ((d = ob->dist(mousep)) < select_dist()) {
      if (!fixed_type || ob->type() == type) {
	sob = new ISortobj;
	sob->ob = ob;
	sob->ds = d;
	sob->dp = lev++;
	selqueue.append(sob);
      }
    }
  }
  selqueue.sort(sobjcompare);
  cursel = 0;
}

static pl_vec orig;

advertise void dragging_redraw(void)
{
  canvas(COL_DRAGGING);
  draw_rectangle(orig, curmp);
}

//====================================================================

// these routines are called by the canvas handler

static Boolean shift_mode;
static Boolean ctl_mode;
static char *select_msg[2][2] = {
  { "select", "select type" },
  { "select more", "sl more type" }};

advertise void selecting_push(pl_vec &mousep, Boolean shft, Boolean ctl)
// mouse button pushed and gone into selecting mode
{
  // start SELECTING mode
  nonidle(SELECTING);
  mouse_msg("", "", select_msg[shft][ctl]);
  shift_mode = shft;
  ctl_mode = ctl;

  create_selection(mousep, ctl);
  orig = mousep;
  if (!selqueue.size()) {
    // there is no object in the neighbourhood
    if (!shft)
      clear_selection();
    return;
  }    

  if (shft && selection()) {
    // add secondary selection
    selqueue[cursel]->ob->sel += TENTATIVE;
  } else {
    // update primary selection only
    clear_selection();
    selqueue[cursel]->ob->sel = PRIMARY;
  }
  if (selqueue.size() > 1) {
    // more that one object found near mouse
    put_msg("use SPACE to select another object");
  }
  redraw_pup();
}

advertise void selecting_key(void)
// space key pressed in SELECTING mode
{
  if (selqueue[cursel]->ob->sel >= TENTATIVE) {
    selqueue[cursel]->ob->sel -= TENTATIVE;
    cursel++;
    if (cursel >= selqueue.size())
      cursel = 0;
    selqueue[cursel]->ob->sel += TENTATIVE;
  } else {
    selqueue[cursel]->ob->sel = UNSELECTED;
    cursel++;
    if (cursel >= selqueue.size())
      cursel = 0;
    selqueue[cursel]->ob->sel = PRIMARY;
  }
  put_msg("use SPACE to select another object");
  redraw_pup();
}

advertise void selecting_mouse(const pl_vec &mousep)
// mouse moved in SELECTING mode
{
  // if mouse is moved away, start dragging mode
  if (clearance(orig, mousep) > DRAG_THRESHOLD) {
    put_msg("");
    if (!shift_mode) {
      clear_selection();
    } else if (selqueue.size() && selqueue[cursel]->ob->sel >= TENTATIVE){
      selqueue[cursel]->ob->sel -= TENTATIVE;
    }
    drawing_mode = DRAGGING;
    curmp = mousep;
  }
}

advertise void dragging_mouse(const pl_vec &mousep)
// drag selection box
{
  curmp = mousep;
}

advertise void selecting_done(void)
// mouse button pressed or released while in SELECTING mode
{
  put_msg("");
  // selection made
  if (selqueue.size() && selqueue[cursel]->ob->sel >= TENTATIVE) {
    // tentative case
    Selection_mode m = selqueue[cursel]->ob->sel;
    selqueue[cursel]->ob->sel =  (m > TENTATIVE) ? UNSELECTED : SECONDARY;
    if (m == PRIMARY+TENTATIVE)
      find_new_primary();
  }
  idlefy(1);
}

advertise void dragging_done(const pl_vec& mp)
// mouse button pressed or released in DRAGGING mode
{
  pl_boundingbox bdrag, bobj;
  Object_type curtype = curobj->type();
  
  bdrag.add_first_point(orig);
  bdrag.add_non_first_point(mp);
  for (Iobject *ob = pic->first; ob; ob = ob->next) {
    if (!ob->sel && (!ctl_mode || ob->type() == curtype)) {
      bobj.make_empty();
      ob->bbox(bobj);
      if (lies_inside(pl_vec(bobj.xmin(), bobj.ymin()), bdrag) &&
	  lies_inside(pl_vec(bobj.xmax(), bobj.ymax()), bdrag))
	ob->sel = SECONDARY;
    }
  }
  find_new_primary();
  idlefy(1);
}

advertise void redraw_selection(void)
// redraw the selection in the pop up plane
{
  for (Iobject *ob = pic->first; ob; ob=ob->next)
    if (ob->sel != UNSELECTED) {
      if (ob->sel > SECONDARY) {
	set_linewidth(4);
	set_linestyle(linestyle_index(0x0f0f), FALSE);
      }
      if (ob->sel == PRIMARY)
	ob->primary();
      else
	ob->outline();
      set_linewidth(1);
      set_linestyle(0, FALSE);
    }
}

