/*
 * snapping.C
 * 
 * contains all functions to snap the mouse to context or grid
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

advertise void draw_grid(void)
{
  s_coord step = grid_size_list[grid_size_choice];
  s_coord sstep = step * zoom_factor;

  if (sstep < 2.0)
    return;
  
  int vfactor = int(1.0 + 16.0/sstep);
  s_coord vstep = step * vfactor;

  pl_vec ll = get_mouse(0.0, 0.0);
  ll.set_x(vstep * int((ll.x() / vstep) - 1.0));
  ll.set_y(vstep * int((ll.y() / vstep) - 1.0));
  pl_vec ur = get_mouse(canvas_size.x, canvas_size.y);
//  ur.set_x(vstep * int((ur.x() / vstep) + 1.0));
//  ur.set_y(vstep * int((ur.y() / vstep) + 1.0));
  
  pl_vec v;

  sys_color[COL_GRID].setcol();
  bgn_point();
  for (v.set_x(ll.x()); v.x() < ur.x(); v.set_x(v.x() + vstep))
    for (v.set_y(ll.y()); v.y() < ur.y(); v.set_y(v.y() + step))
      put_vertex(v, TRUE);
  if (vfactor != 1) {
    for (v.set_y(ll.y()); v.y() < ur.y(); v.set_y(v.y() + vstep))
      for (v.set_x(ll.x()); v.x() < ur.x(); v.set_x(v.x() + step))
	put_vertex(v, TRUE);
  }
  end_point();
}

static void snap_closest_intersection(const pl_vec& mp, pl_vec& v,
				      int i,
				      const pl_vec& u, const pl_vec& w)
// snap to closer of u and w, depending on i
{
  if (i & 1 && clearance(mp, u) < clearance(mp, v))
    v = u;
  if (i & 2 && clearance(mp, w) < clearance(mp, v))
    v = w;
}    
  
void Iedge::close_intersection(const pl_line& l, const pl_vec& mp,
			       pl_vec& v)
// compute closest intersection point of this and l
{
  int i;
  pl_vec u, w;

  switch (type) {
  case EDGE:
    if (intersect(u, edge, l) && clearance(mp,u) < clearance(mp,v))
      v = u;
    return;
  case CIRCLE:
    i = intersect(u, w, circle, l);
    snap_closest_intersection(mp, v, i, u, w);
    return;
  case ARC:
    i = intersect(u, w, arc, l);
    snap_closest_intersection(mp, v, i, u, w);
    return;
  }
}
  
void Iedge::close_intersection(const Iedge& t,
			       const pl_vec& mp, pl_vec& v)
// compute closest intersection point of this and t
{
  int i;
  pl_vec u, w;
  
  switch (type) {
  case EDGE:
    switch (t.type) {
    case EDGE:
      if (intersect(u, edge, t.edge) && clearance(mp,u) < clearance(mp,v))
	v = u;
      return;
    case CIRCLE:
      i = intersect(u, w, edge, t.circle);
      snap_closest_intersection(mp, v, i, u, w);
      return;
    case ARC:
      i = intersect(u, w, edge, t.arc);
      snap_closest_intersection(mp, v, i, u, w);
      return;
    }
  case CIRCLE:
    switch (t.type) {
    case EDGE:
      i = intersect(u, w, circle, t.edge);
      snap_closest_intersection(mp, v, i, u, w);
      return;
    case CIRCLE:
      i = intersect(u, w, circle, t.circle);
      snap_closest_intersection(mp, v, i, u, w);
      return;
    case ARC:
      i = intersect(u, w, circle, t.arc);
      snap_closest_intersection(mp, v, i, u, w);
      return;
    }
  case ARC:
    switch (t.type) {
    case EDGE:
      i = intersect(u, w, arc, t.edge);
      snap_closest_intersection(mp, v, i, u, w);
      return;
    case CIRCLE:
      i = intersect(u, w, arc, t.circle);
      snap_closest_intersection(mp, v, i, u, w);
      return;
    case ARC:
      i = intersect(u, w, arc, t.arc);
      snap_closest_intersection(mp, v, i, u, w);
      return;
    }
  }
}

static void snap_intersection(const pl_vec& mp, pl_vec& snapp)
// snap to intersection point
{
  Iedge_rr s(0);

  for (Iobject *ob = pic->first; ob; ob = ob->next) {
    if (ob != curobj)
      ob->close_edge(mp, s);
  }
  
  for (int i = 0; i < s.size(); i++) {
    for (int j = i+1; j < s.size(); j++) {
      s[i].close_intersection(s[j], mp, snapp);
    }
  }
}

static void snap_angular_intersection(pl_vec& snapp, const pl_line& l)
{
  Iedge_rr s(0);

  for (Iobject *ob = pic->first; ob; ob = ob->next) {
    if (ob != curobj)
      ob->close_edge(snapp, s);
  }
  pl_vec finp = snapp + pl_vec(snap_dist() + 1.0, 0);
  
  for (int i = 0; i < s.size(); i++) {
    s[i].close_intersection(l, snapp, finp);
  }
  if (clearance(snapp, finp) < snap_dist())
    snapp = finp;
}


static void get_line(const pl_vec& mp, const pl_fixedvec &base, pl_line &l)
// compute line through base with suitable slope
{
  pl_angle alpha = axis_dir;

  if (clearance(mp, base) > 2.0) {
    alpha = angle_of(mp - base) - axis_dir;
    alpha.normalise();
    double delta_alpha =
      delta_alpha_list[delta_alpha_choice] / 180.0 * M_PI;
    alpha = delta_alpha * rint_pp(alpha / delta_alpha) + axis_dir;
    alpha.normalise();
  }
  l = pl_line(base, pl_unitvec(alpha));
}

advertise void simple_snap(pl_vec& mp)
{
  Iobject *ob;
  s_coord snpd = snap_dist();
  pl_vec snapp = mp + pl_vec(snpd + 1.0, 0);
  //
  // highest priority: vertex/intersection snapping
  //
  if (snap_vertex_bt.on()) {
    if (page_number > 0) {
      for (ob = pages[0]->first; ob; ob = ob->next) {
	ob->snap_vtx(mp, snapp);
      }
    }
    for (ob = pic->first; ob; ob = ob->next) {
      if (ob != curobj)
	ob->snap_vtx(mp, snapp);
    }
    if (snap_self_bt.on() && drawing_mode == DRAWING)
      curobj->snap_self(mp, snapp);
  }
  if (snap_crossing_bt.on())
    snap_intersection(mp, snapp);
  if (clearance(mp, snapp) < snpd) {
    mp = snapp;
    return;
  }
  //
  // snap to an object's boundary
  //
  if (snap_boundary_bt.on()) {
    if (page_number > 0) {
      for (ob = pages[0]->first; ob; ob = ob->next) {
	ob->snap_bnd(mp, snapp);
      }
    }
    for (ob = pic->first; ob; ob = ob->next) {
      if (ob != curobj)
	ob->snap_bnd(mp, snapp);
    }
    if (clearance(mp, snapp) < snpd) {
      mp = snapp;
      return;
    }
  }
  
  // remains to try grid snapping
  if (snap_grid_bt.on()) {
    s_coord grid = grid_size_list[grid_size_choice];
    mp.set_x(grid * rint_pp(mp.x() / grid));
    mp.set_y(grid * rint_pp(mp.y() / grid));
  }
}

advertise void snap(pl_vec& mp)
// snap point mp to context/ grid / angle as desired
{
  pl_vec snapp;
  Boolean ang_snap = drawing_mode == DRAWING &&
    snap_autodir_bt.on() && !edit_object &&
    (curobj->type() == LINE || curobj->type() == POLYGON);

  // angular snapping and line snapping both on?
  if (ang_snap && snap_directional_bt.on()) {
    // only one possible point!
    pl_line ang_snap, line_snap;
    get_line(mp, snap_base, ang_snap);
    get_line(mp, fixpoint, line_snap);
    if (intersect(snapp, ang_snap, line_snap)) {
      mp = snapp;
      return;
    }
    // if the two lines do not intersect, use following case
  }

  // case of only one one-dimensional snapping mode
  if (ang_snap || snap_directional_bt.on()) {
    pl_line l;
    if (ang_snap)
      get_line(mp, snap_base, l);
    else
      get_line(mp, fixpoint, l);
    mp = placeproject(mp, l);
    if (snap_boundary_bt.on())
      snap_angular_intersection(mp, l);
    return;
  }

  // we are not in one-dimensional mode
  simple_snap(mp);
}

static pl_edge *edge;

static void on_edge(Iobject *ob)
{
  if (!edge)
    if (ob->type() == SEGMENTS)
      edge = ((Segments *) ob)->on_edge(fixpoint);
    else
      edge = ((Line *) ob)->on_edge(fixpoint);
}

advertise void fixpoint_op(int arg)
{
  // get mouse position
  pl_vec mousep = get_mouse();
  simple_snap(mousep);
  
  switch (arg) {
  case 0:
    // reset fixpoint
    fixpoint_set = FALSE;
    fixpoint = pl_vec(0,0);
    snap_directional_bt.set(FALSE);
    axis_dir = 0.0;
    break;

  case 2:
    // start mode
    snap_directional_bt.set(TRUE);
  case 1:
    // set fixpoint to mouse position
/*    if (drawing_mode == MOVING)
      return; */
    fixpoint = mousep;
    fixpoint_set = TRUE;
    break;

  case 3:
    // set direction
    if (fixpoint_set) {
      axis_dir = angle_of(mousep - fixpoint);
    }
    break;

  case 5:
    // start mode
    snap_directional_bt.set(TRUE);
  case 4:
    // set line
/*  if (drawing_mode == MOVING)
      return; */
    edge = NIL;
    fixpoint = mousep;
    fixpoint_set = TRUE;
    pic->traverse(LINE, on_edge);
    pic->traverse(POLYGON, on_edge);
    pic->traverse(BOX, on_edge);
    pic->traverse(SEGMENTS, on_edge);
    if (edge) {
      fixpoint = edge->begin_vt();
      axis_dir = angle_of(edge->direction());
      delete edge;
    }
    break;
  }
  redraw_canvas();
}

advertise void draw_fixpoint(void)
{
  if (fixpoint_set) {
    sys_color[COL_FIXPOINT].setcol();
    push_matrix();
    translate(fixpoint);
    s_coord dalpha = delta_alpha_list[delta_alpha_choice] / 180.0 * M_PI;

    s_coord alpha = 0.0;
    pl_vec bp(0,0),
      ep((canvas_size.x + canvas_size.y) / (zoom_factor*screen_factor.x()),
	 0);

    rotate(axis_dir);
    draw_arrow(bp, pl_vec(100.0/zoom_factor, 0), 10.0/zoom_factor);
    while (alpha < 360.0) {
      draw_line(bp, ep);
      rotate(pl_angle(dalpha));
      alpha += dalpha;
    }
    
    if (dalpha > (0.9 * M_PI)) {
      sub_pixel(TRUE);
      draw_circle(pl_vec(0, 0), 3.0/zoom_factor);
      sub_pixel(FALSE);
    }
    pop_matrix();
  }
}
