/*
 * spline.C
 * 
 * Uniform Nonrational B-splines
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

static Boolean moving = FALSE;
static int pos;

//////////////////////////// DRAW SPLINE //////////////////////////////

inline pl_vec midpoint(const pl_vec& p, const pl_vec& q)
{
  return 0.5 * (p + q);
}

inline pl_vec thirdpoint(const pl_vec& p, const pl_vec& q)
{
  return (1.0/3.0) * ((2 * p) + q);
}

#ifdef SPLINES_USE_NURBS
static double spline_knots[8] = { 0, 0, 0, 0, 1, 1, 1, 1};

static double sub_ctlpts[4][3];

void Spline::drawsegment(pl_vecreray& w,
			 int i0, int i1, int i2, int i3, Boolean cubic )
//  draw a segment of a CUBIC or QUADRATIC B-spline
{
  pl_vec p[4];
  if (cubic) {
    p[1] = thirdpoint(w[i1], w[i2]);
    p[2] = thirdpoint(w[i2], w[i1]);
    p[0] = midpoint(thirdpoint(w[i1], w[i0]), p[1]);
    p[3] = midpoint(thirdpoint(w[i2], w[i3]), p[2]);
  } else {
    p[0] = midpoint(w[i1], w[i0]);
    p[3] = midpoint(w[i1], w[i2]);
    p[1] = thirdpoint(w[i1], p[0]);
    p[2] = thirdpoint(w[i1], p[3]);
  }
  for (int i = 0; i < 4; i++) {
    sub_ctlpts[i][0] = p[i].x();
    sub_ctlpts[i][1] = p[i].y();
    sub_ctlpts[i][2] = 0.0;
  }
  bgncurve();
  nurbscurve(8, spline_knots, 3*sizeof(double), &sub_ctlpts[0][0], 4, N_V3D);
  endcurve();
}
#endif

void Spline::drawcurve(void)
// draw spline curves
{
#ifdef SPLINES_USE_NURBS
  pl_vecreray w;
  unfold(w);
  
  int n = w.size();
  setnurbsproperty(N_PIXEL_TOLERANCE, 10.0);
  if (n == 2) {
    draw_line(w[0], w[1]);
  } else if (n == 3) {
    // QUADRATIC B-spline
    drawsegment(w, 0, 1, 2, 0, FALSE);
    if (closed) {
      drawsegment(w, 1, 2, 0, 0, FALSE);
      drawsegment(w, 2, 0, 1, 0, FALSE);
    }
  } else {
    // CUBIC B-spline
    for (int i = 3; i < n; i++)
      drawsegment(w, i-3, i-2, i-1, i, TRUE);
    if (closed) {
      drawsegment(w, n-3, n-2, n-1, 0, TRUE);
      drawsegment(w, n-2, n-1,   0, 1, TRUE);
      drawsegment(w, n-1,   0,   1, 2, TRUE);
    }
  }
#else
  bgn_line(closed);
  for (int i = 0; i < apr.size(); i++)
    put_vertex(apr[i]);
  end_line(closed);
#endif
}

///////////////////////// APPROXIMATE /////////////////////////

void Spline::apprbezier(pl_vec p[4], s_coord t0, s_coord t1)
// approximate a bezier curve by a polygonal chain
{
  pl_line l(p[0], p[3]);
  s_coord d1 = clearance(p[1], l);
  s_coord d2 = clearance(p[2], l);
  if (d1 < app.spline_precision && d2 < app.spline_precision) {
    // approximation is good enough, use segment p[0]..p[3]
    apr.append(p[3]);
    tval.append(s_coord(t1));
  } else {
    // subdivide
    pl_vec l[4], r[4], h;
    l[0] = p[0];
    l[1] = 0.5 * (p[0] + p[1]);
    h = 0.5 * (p[1] + p[2]);
    l[2] = 0.5 * (l[1] + h);
    r[2] = 0.5 * (p[2] + p[3]);
    r[1] = 0.5 * (h + r[2]);
    r[0] = 0.5 * (l[2] + r[1]);
    l[3] = r[0];
    r[3] = p[3];
    s_coord tm = 0.5 * (t0 + t1);
    apprbezier(l, t0, tm);
    apprbezier(r, tm, t1);
  }
}

void Spline::apprsegment(pl_vecreray &w,
			 int i0, int i1, int i2, int i3, Boolean cubic )
//  approximate a segment of a CUBIC or QUADRATIC B-spline
{
  pl_vec p[4];
  if (cubic) {
    p[1] = thirdpoint(w[i1], w[i2]);
    p[2] = thirdpoint(w[i2], w[i1]);
    p[0] = midpoint(thirdpoint(w[i1], w[i0]), p[1]);
    p[3] = midpoint(thirdpoint(w[i2], w[i3]), p[2]);
  } else {
    p[0] = midpoint(w[i1], w[i0]);
    p[3] = midpoint(w[i1], w[i2]);
    p[1] = thirdpoint(w[i1], p[0]);
    p[2] = thirdpoint(w[i1], p[3]);
  }
  if (!i0) {
    apr.append(p[0]);
    tval.append(0.0);
  }
  apprbezier(p, s_coord(i0), s_coord(i0) + 1.0);
}

void Spline::mkapprox(void)
// make an approximation of the spline by a polygonal chain
{
  pl_vecreray w;
  unfold(w);
  
  int n = w.size();
  apr.newsize(0);
  tval.newsize(0);
  if (n == 2) {
    apr.append(w[0]);
    tval.append(0.0);
    apr.append(w[1]);
    tval.append(1.0);
  } else if (n == 3) {
    // QUADRATIC B-spline
    apprsegment(w, 0, 1, 2, 0, FALSE);
    if (closed) {
      apprsegment(w, 1, 2, 0, 0, FALSE);
      apprsegment(w, 2, 0, 1, 0, FALSE);
    } 
  } else {
    // CUBIC B-spline
    for (int i = 3; i < n; i++)
      apprsegment(w, i-3, i-2, i-1, i, TRUE);
    if (closed) {
      apprsegment(w, n-3, n-2, n-1, 0, TRUE);
      apprsegment(w, n-2, n-1,   0, 1, TRUE);
      apprsegment(w, n-1,   0,   1, 2, TRUE);
    }
  }
}

void Spline::unfold(pl_vecreray& w)
{
  w.newsize(0);
  int i, j;
  for (i = 0; i < v.size(); i++) {
    for (j = 0; j < multi[i]; j++)
      w.append(v[i]);
  }
}

////////////////////////////// DRAW //////////////////////////////

void Spline::draw(void)
{
  if (this == curobj)
    return;

  // draw filled interior
  if (!fill.is_empty() && apr.size() <= max_nvertex && !transparent_bt.on()) {
    fill.setcol();
    bgn_polygon();
    for (int i = 0; i < apr.size(); i++)
      put_vertex(apr[i]);
    end_polygon();
  }
  
  // draw outline
  if (!stroke.is_empty()) {
    stroke.setcol();
    set_linestyle(dash);
    set_linewidth((lwidth*zoom_factor < 1.0) ? 1 : int(lwidth*zoom_factor));
    drawcurve();

    if (arrow) {
      if (arrow & 1 && multi[0] == 3) {
	draw_arrow(v[1], v[0], arsize);
      }
      if (arrow & 2 && multi[v.size()-1] == 3) {
	draw_arrow(v[v.size()-2], v[v.size()-1], arsize);
      }
    }
  }
}

void Spline::outline(void)
{
  drawcurve();
}

//////////////////////////// TRANSFORMATIONS /////////////////////////

void Spline::transform(const pl_rotra& tfm)
{
  for (int i = 0; i < v.size(); i++)
    v[i] = tfm * v[i];
  for (i = 0; i < apr.size(); i++)
    apr[i] = tfm * apr[i];
}

void Spline::stretch(const s_coord xfactor, const s_coord yfactor)
{
  for (int i = 0; i < v.size(); i++) {
    v[i].set_x(v[i].x() * xfactor);
    v[i].set_y(v[i].y() * yfactor);
  }
  for (i = 0; i < apr.size(); i++) {
    apr[i].set_x(apr[i].x() * xfactor);
    apr[i].set_y(apr[i].y() * yfactor);
  }
}


s_coord Spline::dist(const pl_vec& p)
{
  if (!fill.is_empty() && enable_interior_bt.on())
    return(clearance(p, pl_polygon(apr), select_dist()));
  s_coord d = select_dist(), d1;
  for (int i = 0; i < apr.size()-1; i++)
    if ((d1 = clearance(p, pl_edge(apr[i], apr[i+1]), select_dist())) < d)
      d = d1;
  return(d);
}

void Spline::snap_bnd(const pl_vec& mp, pl_vec& p)
{
  if (v.size() == 2) {
    snap_edge(mp, p, pl_edge(v[0], v[1]));
    return;
  }

  s_coord d1, d = clearance(mp, p);
  int j = -1;
  for (int i = 0; i < apr.size()-1; i++) {
    if ((d1 = clearance(mp, pl_edge(apr[i], apr[i+1]))) < d) {
      j = i; d = d1;
    }
  }
  if (j < 0) return;

  pl_edge e(apr[j], apr[j+1]);
  s_coord s = dot(e.direction(), (mp - apr[j])) / e.length();
  if (s < 0.0) s = 0.0;
  if (s > 1.0) s = 1.0;
  
  // figure out spline segment
  int i0, i1, i2, i3;
  s_coord t = tval[j] + s * (tval[j+1] - tval[j]);
  pl_vecreray ww;
  unfold(ww);
  i0 = int(t);
  i1 = i0 + 1;
  if (i1 == ww.size()) i1 = 0;
  i2 = i1 + 1;
  if (i2 == ww.size()) i2 = 0;
  i3 = i2 + 1;
  if (i3 == ww.size()) i3 = 0;
  // now choose point on spline segment
  t = t - i0;
  s_coord t2 = t * t, t3 = t2 * t;
  pl_vec w;
  if (ww.size() == 3) {
    w.set_x(((1.0 - 2*t + t2) * ww[i0].x() +
	    (1.0 + 2*t - 2*t2) * ww[i1].x() + t2 * ww[i2].x()) / 2.0);
    w.set_y(((1.0 - 2*t + t2) * ww[i0].y() +
	    (1.0 + 2*t - 2*t2) * ww[i1].y() + t2 * ww[i2].y()) / 2.0);
  } else {
    w.set_x(((1.0 - 3*t + 3*t2 - t3) * ww[i0].x() +
	    (4.0 - 6*t2 + 3*t3) * ww[i1].x() +
	    (1.0 + 3*t + 3*t2 - 3*t3) * ww[i2].x() + t3 * ww[i3].x()) / 6.0);
    w.set_y(((1.0 - 3*t + 3*t2 - t3) * ww[i0].y() +
	    (4.0 - 6*t2 + 3*t3) * ww[i1].y() +
	    (1.0 + 3*t + 3*t2 - 3*t3) * ww[i2].y() + t3 * ww[i3].y()) / 6.0);
  } 
  snap_vertex(mp, p, w);
}

void Spline::snap_vtx(const pl_vec& mp, pl_vec& p)
{
  // snap to all ctl points with multiplicity 3, plus start and endpoint
  for (int i = 0; i < v.size(); i++)
    if (multi[i] == 3)
      snap_vertex(mp, p, v[i]);
  // closed splinegon: no start or endpoint
  if (closed)
    return;
  // otherwise: just start and endpoint
  snap_vertex(mp, p, apr[0]);
  snap_vertex(mp, p, apr[apr.size() - 1]);
}

void Spline::snap_self(const pl_vec& mp, pl_vec& p)
{
  for (int i = 0; i < v.size(); i++)
    if (!moving || i != pos)
      snap_vertex(mp, p, v[i]);
}  

void Spline::bbox(pl_boundingbox& b)
{
  for (int i = 0; i < apr.size(); i++)
    b.add_point(apr[i]);
}

/////////////////////////////// MOUSE //////////////////////////////

void Spline::push(pl_vec& mousep, int button, Boolean)
{
  if (drawing_mode != DRAWING) {
    // left mouse button pressed, starting spline mode
    clear_selection();
    v.newsize(2);
    multi.newsize(2);
    snap_base = mousep;
    v[0] = mousep;
    v[pos = 1] = mousep;
    multi[1] = multi[0] = (!closed && enable_3_spline_bt.on()) ? 3 : 1;
    nonidle(DRAWING);
    moving = TRUE;
    mouse_msg("next ctl point", "final ctl point", "delete last");
  } else {
    switch (button) {
    case LEFT_MOUSE:
      // add a control point
      multi.append(multi[pos]);
      multi[pos] = 1;
      snap_base = mousep;
      v[pos++] = mousep;
      v.append(mousep);
      break;
      
    case MIDDLE_MOUSE:
      // final control point
      v[pos] = mousep;
      {
	Spline *l = new Spline();
	l->closed = closed;
	l->setcurrent();
	if (closed) {
	  if (!l->fill.is_empty() && !enable_stroke_bt.on())
	    l->stroke.mkempty();
	} else
	  l->fill.mkempty();
	l->v = v;
	l->multi = multi;
	l->mkapprox();
	pic->add_object(l);
	object_added();
      }
      idlefy(2);
      return;
      
    case RIGHT_MOUSE:
      // delete_last_point
      if (v.size() > 2) {
	v.resize(v.size() - 1);
	v[--pos] = mousep;
	multi[pos] = multi[pos + 1];
	multi.resize(v.size());
	snap_base = v[v.size() - 2];
      } else {
	put_msg("spline cancelled");
	idlefy(1);
	return;
      }
      break;
    }
  }
  put_msg("\"2\" and \"3\" set multiplicity");
#ifndef SPLINES_USE_NURBS
  mkapprox();
#endif
  redraw_pup();
  return;
}


void Spline::edit_push(pl_vec& mousep, int button, Boolean)
{
  int i;
  s_coord d, d1;
  
  switch (button) {
  case RIGHT_MOUSE:
    // right mouse key: select a control point:
    d = clearance(mousep, v[0]);
    pos = 0;
    for (i = 1; i < v.size(); i++) {
      if ((d1 = clearance(mousep, v[i])) < d) {
	d = d1;
	pos = i;
      }
    }
    if (d > select_dist()) {
      // end of edit
      mkapprox();
      idlefy(2);
      undo_changed();
      return;
    } else
      moving = FALSE;
    break;
  case MIDDLE_MOUSE:
    // now move control point
    d = clearance(mousep, v[0]);
    pos = 0;
    for (i = 1; i < v.size(); i++) {
      if ((d1 = clearance(mousep, v[i])) < d) {
	d = d1;
	pos = i;
      }
    }
    moving = TRUE;
    v[pos] = mousep;
    break;
  case LEFT_MOUSE:
    // insert a control point at mouseposition
    v.resize(v.size() + 1);
    multi.resize(v.size());
    pos++;
    for (i = v.size() -1; i > pos; i--) {
      v[i] = v[i-1];
      multi[i] = multi[i-1];
    }
    v[pos] = mousep;
    multi[pos] = 1;
    moving = TRUE;
    break;
  }
#ifndef SPLINES_USE_NURBS
  mkapprox();
#endif
  redraw_pup();
  put_msg("\"C-d\" deletes, \"r\" reverses, \"1\",\"2\", \"3\" set multiplicity");
}

void Spline::mouse(const pl_vec& mp, Boolean)
{
  if (moving) {
#ifndef SPLINES_USE_NURBS
    mkapprox();
#endif
    v[pos] = mp;
  }
}

void Spline::release(pl_vec& mp)
{
  if (edit_object && moving) {
    v[pos] = mp;
    moving = FALSE;
#ifndef SPLINES_USE_NURBS
    mkapprox();
#endif
    redraw_pup();
  }
}

void Spline::key(unsigned char c)
// handle keypress
{
  int i;
  
  if (edit_object) {
    switch (c) {
    case ('e' + 0x80):
      // end of edit
      mkapprox();
      idlefy(2);
      undo_changed();
      return;
    case ('1'):
    case ('2'):
    case ('3'):
      multi[pos] = c - '0';
#ifndef SPLINES_USE_NURBS
      mkapprox();
#endif
      redraw_pup();
      return;
    case ('d' - 0x60):
    case ('h' - 0x60):
      // delete vertex
      if (v.size() <  3) {
	put_msg("cannot remove the last vertices!");
	return;
      }
      for (i = pos; i < v.size()-1; i++) {
	v[i] = v[i+1];
	multi[i] = multi[i+1];
      }
      v.resize(v.size() - 1);
      multi.resize(v.size());
      if (pos == v.size())
	pos--;
      moving = FALSE;
#ifndef SPLINES_USE_NURBS
      mkapprox();
#endif
      redraw_pup();
      return;
    case ('r'):
      // reverse direction
      {
	pl_vec w;
	short m;
	for (i = 0; i < v.size() / 2; i++) {
	  w = v[i];
	  m = multi[i];
	  v[i] = v[v.size() - 1 - i];
	  multi[i] = multi[v.size() - 1 - i];
	  v[v.size() - 1 - i] = w;
	  multi[v.size() - 1 - i] = m;
	}
	pos = v.size() - 1 - pos;
      }
      if (((arrow & 2) >> 1) != (arrow & 1))
	arrow = 3 - arrow;
      moving = FALSE;
#ifndef SPLINES_USE_NURBS
      mkapprox();
#endif
      redraw_pup();
      return;
    default:
      shortcut(c);
      return;
    }
  } else {
    switch (c) {
    case ('1'):
    case ('2'):
    case ('3'):
      multi[pos-1] = c - '0';
      redraw_pup();
#ifndef SPLINES_USE_NURBS
      mkapprox();
#endif
      return;
    case ('d' - 0x60):
    case ('h' - 0x60):
      {
	// kill last vertex
	pl_vec mousep = get_mouse();
	snap(mousep);
	push(mousep, RIGHT_MOUSE, FALSE);
      }
#ifndef SPLINES_USE_NURBS
      mkapprox();
#endif
      return;
    default:
      shortcut(c);
      return;
    }
  }
}

void Spline::edit(void)
{
  saved_curobj = curobj;
  edit_object = TRUE;
  curobj = this;
  nonidle(DRAWING);
  mouse_msg("insert ctl pt", "move ctl point", "select ctl pt");
  put_msg("\"C-d\" deletes, \"r\" reverses, \"1\",\"2\", \"3\" set multiplicity");
  pos = v.size() - 1;
  moving = FALSE;
  redraw_canvas();
}
    

///////////////////////// REDRAW //////////////////////////////

static void draw_ctlpoint(const pl_vec& pos, s_coord size, short multi)
{
  switch (multi) {
  case 1:
    draw_filled_rectangle(pos + pl_vec(-size, -size),
			  pos + pl_vec(size, size));
    return;
  case 2:
    bgn_polygon();
    size *= 2.0;
    put_vertex(pos + pl_vec(0, size));
    put_vertex(pos + pl_vec(-size, -size));
    put_vertex(pos + pl_vec(size, -size));
    end_polygon();
    return;
  case 3:
    draw_filled_circle(pos, 1.6 * size);
    return;
  }
}

void Spline::redraw(void)
// redraw selection plane
{
  int i;
  if (edit_object) {
    canvas(COL_EDIT);
  } else {
    canvas(COL_CREATION);
  }
  // show spline
  drawcurve();
  // show ctl points
  set_linestyle(linestyle_index(0x5555), FALSE);
  bgn_line();
  for (i = 0; i <= pos; i++)
    put_vertex(v[i]);
  end_line();
  if (edit_object)
    set_linestyle(linestyle_index(0x00ff), FALSE);
  if (pos < v.size() - 1 || (edit_object && closed)) {
    draw_line(v[pos],
	      v[(pos == v.size() - 1) ? 0 : pos + 1]);
  }
  set_linestyle(linestyle_index(0x5555), FALSE);
  bgn_line();
  for (i = pos + 1; i < v.size(); i++)
    put_vertex(v[i]);
  end_line();
  set_linestyle(0, FALSE);
  
  s_coord mf = 2.0/zoom_factor;
  sub_pixel(TRUE);
  for (i = 0; i < v.size() - (edit_object ? 0 : 1); i++)
    draw_ctlpoint(v[i], mf, multi[i]);
  if (edit_object)
    draw_circle(v[pos], 7.0/zoom_factor);
  sub_pixel(FALSE);
}
  
////////////////////////////// SAVING //////////////////////////////

inline int degrees(pl_angle alpha)
{
  return int(alpha.value() * 180.0/ M_PI);
}

void Spline::save_postscript(ostream& fh)
// save spline in postscript format
{
  pl_vecreray w;
  unfold(w);
  
  fh << "% Spline\n";
  ps_dash(fh);
  if (arrow) {
    fh << "% ar " << arrow << " " << arsize << "\n";
    if (!stroke.is_empty()) {
      stroke.save(fh, NIL, " ");
      if (arrow & 1 && multi[0] == 3) {
	fh << v[0] << " " << arsize << " " << (v[0] - v[1]) << " arw\n";
      }
      if (arrow & 2 && multi[v.size() - 1] == 3) {
	fh << v[v.size() - 1] << " " << arsize << " "
	   << (v[v.size() - 1] - v[v.size() - 2]) << " arw\n";
      }
    }	
  }
  fh << "np % # " << w.size() << "\n";
  if (w.size() == 2) {
    // line segment
    fh << w[0] << " mt\n";
    fh << w[1] << " lt\n";
  } else if (w.size() == 3) {
    // QUADRATIC B-spline
    fh << w[0] << "\n" << w[1] << "\n" << w[2]
       << (closed ? " cqspl\n" : " qspl\n" );
  } else {    
    for (int i = 0; i < w.size(); i++) {
      fh << w[i];
      if (i < 3)
	fh << "\n";
      else if (i == 3)
	fh << " fspl\n";
      else 
	fh << " spl\n";
    }
    if (closed) {
      fh << w[0] << " spl\n";
      fh << w[1] << " spl\n";
      fh << w[2] << " spl\n";
    }
    fh << "xspl\n";
  }
  if (closed)
    fh << "% cl\n";
  fill.save(fh, "fi", (stroke.is_empty() ? " fi\n" : " sfi\n"));
  stroke.save(fh, "sk", " sk\n");
  fh << "% End\n\n";
}

void Spline::save_properties(ostream& fh)
// save spline as properties
{
  pl_vecreray w;
  unfold(w);
  
  fh << "% Spline\n";
  dash_properties(fh);
  if (arrow) {
    fh << "% ar " << arrow << " " << arsize << "\n";
  }
  fh << "% # "
     << w.size() << "\n";
  for (int i = 0; i < w.size(); i++)
    fh << w[i] << "\n";
  if (closed)
    fh << "% cl\n";
  stroke.save(fh, "sk", NIL);
  fill.save(fh, "fi", NIL);
  fh << "% End\n\n";
}

static char *fieldlist[] = {
  "?2ss", "..", "!##", "?0cl", "?2ar", ".." };

Spline::Spline(istream& fh)
// read spline item from file
{
  // set  defaults
  ps_in_value[0] = 65535.0;
  ps_in_value[1] = 1.0;
  ps_in_value[4] = 0;
  ps_in_value[5] = 1.0;

  ps_okay &= ps_read_entry(fh, 6, fieldlist);
  ps_okay &= (ps_in_array.size() > 1);
  
  if (!ps_okay)
    return;

  stroke.read(0);
  fill.read(1);

  dash = linestyle_index(short(ps_in_value[0]));
  lwidth = ps_in_value[1];
  closed = ps_in_defd[3];
  arrow = int(ps_in_value[4]);
  arsize = ps_in_value[5];

  v.newsize(0);
  multi.newsize(0);
  short one = 1;
  int i;
  for (i = 0; i < ps_in_array.size(); i++) {
    if (v.size() &&
	multi[v.size() - 1] < 3 &&
	clearance(ps_in_array[i], v[v.size() - 1]) < SPLINE_MULTI_THRESHOLD) {
      multi[v.size() - 1]++;
    } else {
      v.append(ps_in_array[i]);
      multi.append(one);
    }
  }
  mkapprox();
}

Spline::Spline(Line &)
// make a spline with the specifications read in by Line::Line(istream&)
// only used to read in old-style splines
{
  closed = ps_in_defd[3];
  stroke.read(0);
  fill.read(1);
  
  dash   = linestyle_index(short(ps_in_value[0]));
  lwidth = ps_in_value[1];
  arrow  = int(ps_in_value[4]);
  arsize = ps_in_value[5];

  sel    = UNSELECTED;
  next = prev = NIL;

  v      = ps_in_array;
  multi.newsize(v.size());
  for (int i = 0; i < multi.size(); )
    multi[i++] = 1;
  if (!closed) {
    multi[0] = 3;
    multi[multi.size() - 1] = 3;
  }
  mkapprox();
}
