/*
 * struc.C
 * 
 * Structure modification: grouping, front, back etc.
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

advertise void ungroup_selection(void)
{
  Iobject *ob, *ob1, *nf, *nl;

  ob = selection();

  if (ob && ob->type() == SEGMENTS) {
    ungroup_segments();
    return;
  }
    
  if (!ob || ob->type() != GROUP) {
    put_msg("Must select a group to ungroup");
    return;
  }
  
  // retrieve list of group elements
  nl = nf = ((Group *)ob)->first;
  while (nl->next) {
    nl->sel = SECONDARY;
    nl = nl->next;
  }
  nl->sel = SECONDARY;
  nf->sel = PRIMARY;
  
  // unlink selection
  ob1 = ob->next;
  if (!ob1 && !ob->prev) {
    // only one object present
    pic->first = nf;
    pic->last = nl;
  } else {
    if (!ob->prev) {
      // first object
      pic->first = ob1;
      ob1->prev = NIL;
    } else if (!ob1) {
      // last object
      pic->last = ob->prev;
      ob->prev->next = NIL;
    } else {
      // elements before and behind me
      ob->prev->next = ob1;
      ob1->prev = ob->prev;
    }
    // unlinked group, add its elements at front
    pic->last->next = nf;
    nf->prev = pic->last;
    pic->last = nl;
  }
  // destroy group object
  ((Group *)ob)->first = ((Group *)ob)->last = NIL;
  delete ob;
  undo_changed();
  redraw_canvas();
}

advertise void make_selection_list(Iobject*& nf, Iobject*& nl)
// make a list of all selected objects, and  unlink from main list
// returns NIL if nothing selected
{
  Iobject *ob, *ob1;

  nf = nl = NIL;
  for (ob = pic->first; ob; ob = ob1) {
    ob1 = ob->next;
    if (ob->sel != UNSELECTED) {
      // found one, unlink it
      if (!ob1 && !ob->prev) {
	// nothing left: unlinking last element
	pic->first = pic->last = NIL;
      } else if (!ob->prev) {
	// first object
	pic->first = ob1;
	ob1->prev = NIL;
      } else if (!ob1) {
	// last object
	pic->last = ob->prev;
	ob->prev->next = NIL;
      } else {
	// elements before and behind me
	ob->prev->next = ob1;
	ob1->prev = ob->prev;
      }
      // okay, now put into new list
      if (!nf) {
	// first one in new list
	nf = nl = ob;
	ob->prev = ob->next = NIL;
      } else {
	// not first one, nl is initialized
	nl->next = ob;
	ob->prev = nl;
	ob->next = NIL;
	nl = ob;
      }
    }
  }
  return;
}

advertise void front_selection(void)
{
  Iobject *nf, *nl;
  
  make_selection_list(nf, nl);
  if (!nf)
    // nothing to do
    return;
  if (!pic->first) {
    pic->first = nf;
    pic->last = nl;
  } else {
    pic->last->next = nf;
    nf->prev = pic->last;
    pic->last = nl;
  }
  undo_changed();
  redraw_canvas();
}

advertise void back_selection(void)
{
  Iobject *nf, *nl;
  
  make_selection_list(nf, nl);
  if (!nf)
    // nothing to do
    return;
  if (!pic->first) {
    pic->first = nf;
    pic->last = nl;
  } else {
    pic->first->prev = nl;
    nl->next = pic->first;
    pic->first = nf;
  }
  undo_changed();
  redraw_canvas();
}

advertise void group_selection(void)
{
  Iobject *nf, *nl;

  int n = 0;
  for (nf = pic->first; nf; nf = nf->next)
    if (nf->sel) n++;

  if (n < 2) {
    put_msg("you have to group at least two elements");
    return;
  }

  make_selection_list(nf, nl);

  if (group_is_segments(nf)) {
    Segments *sg = new Segments((Line *) nf);
    // simply delete list
    Iobject *ob, *ob1;
    for (ob = nf; ob; ob = ob1) {
      ob1 = ob->next;
      delete ob;
    }
    pic->add_object(sg);
  } else {
    Group *gr = new Group;
    gr->first = nf;
    gr->last = nl;
    for ( ; nf; nf= nf->next )
      nf->sel = UNSELECTED;
    pic->add_object(gr);
  }
  object_added();
  redraw_canvas();
}

advertise void select_all(Boolean fixed_type)
{
  Object_type type;

  type = curobj->type();

  for (Iobject *ob = pic->first; ob; ob = ob->next)
    if (ob && (!fixed_type || ob->type() == type))
      ob->sel = SECONDARY;
  find_new_primary();
  redraw_pup();
}

advertise void cut_selection(Boolean save)
{
  Iobject *nf, *nl, *ob, *ob1;

  if (!selection()) {
    put_msg("nothing selected");
    return;
  }
  make_selection_list(nf, nl);

  // save in cut buffer?
  if (save) {
    // delete old contents of cut_buffer
    for (ob = cut_buffer; ob; ob = ob1) {
      ob1 = ob->next;
      delete ob;
    }
    // just put list nf..nl into cutbuffer (already unlinked from pic)
    cut_buffer = nf;
  } else {
    // simply delete list
    for (ob = nf; ob; ob = ob1) {
      ob1 = ob->next;
      delete ob;
    }
  }
  undo_changed();
  redraw_canvas();
}
      
advertise void copy_selection(void)
{
  Iobject *nf, *nl, *ob, *ob1;

  if (!selection()) {
    put_msg("Nothing selected");
    return;
  }
  nf = nl = NIL;
  for (ob = pic->first; ob; ob = ob->next) {
    if (ob->sel != UNSELECTED) {
      // okay, copy it into new list
      ob1 = ob->copy();
      if (!nf) {
	// first one in new list
	nf = nl = ob1;
	ob1->prev = ob1->next = NIL;
      } else {
	// not first one, nl is initialized
	nl->next = ob1;
	ob1->prev = nl;
	ob1->next = NIL;
	nl = ob1;
      }
    }
  }
  // delete old contents of cut_buffer
  for (ob = cut_buffer; ob; ob = ob1) {
    ob1 = ob->next;
    delete ob;
  }
  // put stuff in cut buffer
  cut_buffer = nf;
}

advertise void duplicate_selection(void)
{
  Iobject *nf, *nl, *ob, *ob1;

  if (!selection()) {
    put_msg("nothing selected");
    return;
  }
  nf = nl = NIL;
  for (ob = pic->first; ob; ob = ob->next) {
    if (ob->sel != UNSELECTED) {
      ob->sel = UNSELECTED;
      // okay, copy it into new list
      ob1 = ob->copy();
      if (!nf) {
	// first one in new list
	nf = nl = ob1;
	ob1->prev = ob1->next = NIL;
      } else {
	// not first one, nl is initialized
	nl->next = ob1;
	ob1->prev = nl;
	ob1->next = NIL;
	nl = ob1;
      }
    }
  }
  // now stuff it back into arena
  for (ob = nf; ob; ob = ob->next) {
    ob1 = ob->copy();
    ob1->sel = SECONDARY;
    pic->last->next = ob1;
    ob1->prev = pic->last;
    pic->last = ob1;
  }
  pic->last->next = NIL;
  object_added();
  redraw_canvas();
}

advertise void object_added(void)
// this should be called after creating an object and adding it
// to the pic group. It makes it the primary selection, and
// calls undo

{
  pic->last->sel = PRIMARY;
  undo_changed();
}


// ============================ undo mechanism ========================

static Group *undo_buffer[MAX_UNDO];
static int undo_pagenum[MAX_UNDO];
static int changes;

advertise void undo_init(void)
{
  for (int i = 0; i < MAX_UNDO; i++)
    undo_buffer[i] = NIL;
  undo_reset();
}

advertise void undo_reset(Boolean fresh DefaultsTo(TRUE) )
{
  for (int i = 0; i < MAX_UNDO; i++) {
    if (undo_buffer[i]) {
      delete undo_buffer[i];
      undo_buffer[i] = NIL;
    }
  }
  if (enable_undo_bt.on()) {
    undo_buffer[0] = new Group(*pic);
    undo_pagenum[0] = page_number;
  }
  if (fresh) {
    changes = 0;
// FORMS    fl_set_object_color(bitmap_obj, 0, 47);
  }
//  else {
//    changes++;
//    fl_set_object_color(bitmap_obj, 1, 47);
//  }
}
		      
advertise void undo_operation(void)
{
  if (!enable_undo_bt.on()) {
    put_msg("UNDO disabled");
    return;
  }
  if (!undo_p()) {
    put_msg("nothing to UNDO");
    return;
  }
  if (undo_buffer[1] == NIL) {
    put_msg("no more UNDO information");
    return;
  }
  delete undo_buffer[0];
  for (int i = 1; i < MAX_UNDO; i++) {
    undo_buffer[i-1] = undo_buffer[i];
    undo_pagenum[i-1] = undo_pagenum[i];
  }
  undo_buffer[MAX_UNDO-1] = NIL;
  page_number = undo_pagenum[0];
  delete pages[page_number];
  pic = pages[page_number] = new Group(*(undo_buffer[0]));
  clear_selection();
  show_pagenum();
  redraw_canvas();
  changes--;
//  if (!undo_p())
// FORMS    fl_set_object_color(bitmap_obj, 0, 47);
}

advertise Boolean undo_p(void)
// returns true if picture has been touched since last load or save
{
  return (changes > 0);
}

advertise void undo_changed(void)
// record change in data structure
{
  if (enable_undo_bt.on()) {
    if (undo_buffer[MAX_UNDO - 1])
      delete undo_buffer[MAX_UNDO -1];
    for (int i = MAX_UNDO-1; i > 0; i--) {
      undo_buffer[i] = undo_buffer[i-1];
      undo_pagenum[i] = undo_pagenum[i-1];
    }
    undo_buffer[0] = new Group(*pic);
    undo_pagenum[0] = page_number;
  }
// FORMS  fl_set_object_color(bitmap_obj, 1, 47);
  changes++;
}

advertise void undo_new_page(void)
// switched to a new page, replace last entry
{
  // lost sync with ghostscript window
  gs_lostsync = TRUE;
  if (enable_undo_bt.on()) {
    delete undo_buffer[0];
    undo_buffer[0] = new Group(*pic);
    undo_pagenum[0] = page_number;
  }
}

advertise void maybe_turnoff_undo(void)
// turn off undo, if picture is too large
{
  if (enable_undo_bt.on()) {
    int mem = 0, nobjs = 0, nvtcs =0;
    pic->stats(mem, nobjs, nvtcs);
    if (mem > app.undo_limit * 1024) {
      enable_undo_bt.set(FALSE);
      put_msg("This is a BIG drawing. Undo has been turned off.");
      undo_reset(FALSE);
    }
  }
}
