/*
 * text.c
 * 
 * text objects
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ipe.h"

static char_rr undo_buf[MAX_UNDO];
static int undo_position[MAX_UNDO];
static int undo_marker[MAX_UNDO];
static int undo_pos = 0;
static int undo_start = 0;

static int position = 0;
static int text_marker = -1;

static Boolean dragging = FALSE;

static char *latex_font_string[4] = {
  "\\rm ", "\\it ", "\\bf ", "$" };

///////////////////////////// UNDO //////////////////////////////////////

static void text_undo_init(char_rr &str)
// init undo mechanism
{
  undo_pos = undo_start = 0;
  undo_buf[0] = str;
  undo_position[0] = position;
  undo_marker[0] = text_marker;
}

static void text_undo_changed(char_rr &str)
{
  if (strcmp(&str[0], &undo_buf[undo_pos][0])) {
    undo_pos = (undo_pos + 1) & 0x0f;
    if (undo_pos == undo_start)
      undo_start = (undo_start + 1) & 0x0f;
    undo_buf[undo_pos] = str;
  }
  undo_position[undo_pos] = position;
  undo_marker[undo_pos] = text_marker;
}

///////////////////////// TEXT VARIANTS //////////////////////////////

static Widget textVariant, textVariantButton[2];
advertise int text_variant InitializeWith(0);

advertise void create_text_variants(Widget parent)
{
  textVariant = 
    create_choice(parent, CHOICE_MODE, "inputVariant", NIL, &text_variant);
  textVariantButton[0] = add_to_choice("text label");
  textVariantButton[1] = add_to_choice("mini page");
}

advertise void textmode_op(int parbox)
{
  text_variant = parbox;
  set_choice(textVariant, textVariantButton[text_variant]);
  sel_mode(TEXT);
}

void Text::setchoice(void)
{
  manage_variant(textVariant);
}

///////////////////////// DRAWING TEXT /////////////////////////

void Text::outline(void)
{
  draw_rectangle(pos, pos + pl_vec(tbox.x(),
				   (parbox ? -tbox.y() : tbox.y())));
}

void Text::outline_rotated(const pl_vec& p, const pl_angle alpha)
// outline the box with origin rotated
{
  push_matrix();
  translate((pl_rotra(alpha, p) * (pos - p)) - pos);
  outline();
  pop_matrix();
}

void Text::outline_stretched(const s_coord xs, const s_coord ys)
{
  if (parbox)
    Iobject::outline_stretched(xs, ys);
  else {
    pl_vec newpos(pos.x() * xs, pos.y() * ys);
    
    push_matrix();
    translate(newpos - pos);
    outline();
    pop_matrix();
  }
}

void Text::draw(void)
{
  // do not draw the object currently edited
  if (this == curobj)
    return;
  if (!stroke.is_empty()) {
    stroke.setcol();
    draw_string(pos, font, size, -1, str, parbox);
  }
}

///////////////////////// MAKE BOUNDING BOX /////////////////////////

void Text::mkbox(void)
// make a new bounding box for the text item, something has changed
{
  s_coord height, width, depth;
  pl_vecreray v(4);
  int nlines = 0;
  char bstr[128], fstr[16];

  strcpy(fstr, ((font == 3) ? "$" : ""));
  if (parbox) {
    sprintf(bstr, "\\begin{minipage}{%gbp}\\IPEfs{%g}%s", tbox.x(),
	    size, latex_font_string[font]);
    strcat(fstr, "\\end{minipage}");
  } else {
    sprintf(bstr, "\\IPEfs{%g}%s", size, latex_font_string[font]);
  }
  if (enable_latex_box_bt.on()
      && latex_box(bstr, str, fstr, width, height, depth))
    nlines = 1;
  else {
    int sol = 0, eol;
    width = 1.0;
    s_coord wd;
    do {
      for (eol = sol; str[eol] && str[eol] != '\n'; eol++)
	;
      get_string_size(str, sol, eol, font, size, wd, height, depth);
      if (wd > width)
	width = wd;
      sol = eol + 1;
      nlines++;
    } while (str[eol] != '\0');
  }

  if (!parbox) {
    tbox.set_x(width);
  }
  tbox.set_y(nlines * height);
}

///////////////////////// CREATE/EDIT FONT /////////////////////////

void Text::create(void)
{
  Text *n = new Text;
  
  n->str = str;
  n->font = font_choice[mipe_mode];
  n->size = fontsize_list[mipe_mode][fontsize_choice[mipe_mode]];
  n->pos = pos;
  n->stroke.setcurrent(FALSE);
  n->fill.mkempty();
  n->parbox = parbox;
  n->tbox = tbox;
  n->mkbox();
  pic->add_object(n);
  object_added();
}

void Text::start_text(const pl_vec& mousep)
{
  position = 0;
  text_marker = -1;
  str.newsize(1);
  str[0] = '\0';
  text_undo_init(str);
  font = font_choice[mipe_mode];
  size = fontsize_list[mipe_mode][fontsize_choice[mipe_mode]];
  curmp = mousep;
  pos = mousep;
  dragging = parbox;
  redraw_pup();
}

void Text::end_text(void)
{
  if (edit_object) {
    if (str.size() > 1) {
      put_msg("EDIT done");
      mkbox();
      undo_changed();
    } else {
      put_msg("use KILL to delete the empty object");
    }
  } else {
    if (str.size() > 1)
      create();
    else
      put_msg("empty string ignored");
  }
  idlefy(2);
}

void Text::redraw(void)
{
  if (edit_object)
    canvas(COL_EDIT);
  else
    canvas(COL_CREATION);
  if (dragging)
    draw_rectangle(pos, curmp);
  else {
    if (parbox) {
      set_linestyle(linestyle_index(0x5555), FALSE);
      draw_rectangle(pos, pos + pl_vec(tbox.x(),
				       (parbox ? -tbox.y() : tbox.y())));
      set_linestyle(0, FALSE);
    }
    draw_string(pos, font + EDIT_FONT_OFFS, size, position, str, parbox);
  }
}

void Text::edit(void)
// place object in EDIT mode
{
  saved_curobj = curobj;
  edit_object = TRUE;
  curobj = this;
  nonidle(DRAWING);
  
  mouse_msg("set cursor", "paste", "mark & copy");
  position = 0;
  text_marker = -1;
  text_undo_init(str);
  dragging = FALSE;
  
  redraw_canvas();
}

static void delete_piece(char_rr &str, int beginrange, int endrange)
// Removes a piece of the string
{
  if (text_marker > beginrange)
    if (text_marker <= endrange)
      text_marker = beginrange;
    else
      text_marker -= (endrange-beginrange)+1;
  int i = 0;
  do {
    i++;
    str[beginrange+i-1] = str[endrange+i];
  } while (str[endrange+i] != '\0');
  str.resize(str.size() - (endrange - beginrange + 1));
  position = beginrange;
}

inline Boolean is_word(char c) {
  return (c == '\''
	  || ('0' <= c && c <= '9')
	  || ('a' <= c && c <= 'z')
	  || ('A' <= c && c <= 'Z'));
}

static s_coord goalwid = -1.0;

static void copy_to_cutbuffer(char_rr &str, int cursor, int marker)
{
  if (marker < 0 || cursor == marker)
    return;
  XStoreBytes(XtDisplay(topLevel), &str[min(cursor, marker)],
	      (marker > cursor ? marker - cursor : cursor - marker));
}

static void copy_from_cutbuffer(char_rr &str, int& position)
{
  char *badr;
  int slen, len, i;

  if (!(badr = XFetchBytes(XtDisplay(topLevel), &len)))
    return;
  if (len > 0) {
    slen = str.size();
    str.resize(slen + len);
    for (i = slen+len-1; i >= position+len; i--)
      str[i] = str[i-len];
    for (i = 0; i < len; i++) {
      str[position + i] = badr[i];
    }
    position += len;
  }
  XFree(badr);
}

static void fill_parbox(char_rr &str, s_coord width, short font, s_coord size)
{
  int sol = 0;
  int lastspace = -1;
  for (int i = 0; str[i]; i++) {
    if (str[i] == ' ' || str[i] == '\n') {
      // test if box width is exceeded
      if (get_substring_width(str, sol, i, font, size) > width) {
	if (lastspace > sol) {
	  str[lastspace] = '\n';
	  sol = lastspace+1;
	}
      }
      lastspace = i;
      str[i] = ' ';
    }
  }
}

void Text::key(unsigned char key)
// Handles a key press
{
  if ((!parbox && key == 0x0d) || key == ('e' + 0x80)) {
    // finish string
    end_text();
    return;
  }

  int slen = str.size() - 1;
  for (int sol = position-1; sol >= 0 && str[sol] != '\n'; sol--)
    ;
  sol++;
  for (int eol = position; eol < slen && str[eol] != '\n'; eol++)
    ;
  int i;

  if (key != ('p' - 0x60) && key != ('n' - 0x60))
    goalwid = -1.0;

  switch (key) {
  case 'a' - 0x60:
    position = sol;
    break;
  case 'e' - 0x60:
    position = eol;
    break;
  case 'b' - 0x60:
    if (position >0)
      position--;
    break;
  case 'f' - 0x60:
    if (position < slen)
      position++;
    break;
  case 'd' - 0x60:
    if (position < slen)
      delete_piece(str, position,position);
    break;
  case 'h' - 0x60:		/* Backspace-Key */
  case 0x7f:			/* Delete-Key */
    if (position >0)
      delete_piece(str, position-1,position-1);
    break;
  case 'k' - 0x60:
    if (str[position] == '\n')
      delete_piece(str, position, position);
    else if (eol > position)
      delete_piece(str, position, eol-1);
    break;
  case ('b' + 0x80):
    if (position > 0)
      position--;
    while (position > 0 && !is_word(str[position]))
      position--;
    while (position > 0 && is_word(str[position]))
      position--;
    if (position > 0)
      position++;
    break;
  case ('d' + 0x80):           /* Meta-D */
    i = position;
    while (i < slen && !is_word(str[i]))
      i++;
    while (i < slen && is_word(str[i]))
      i++;
    delete_piece(str, position, i-1);
    break;
  case ('f' + 0x80):           /* Meta-F */
    while (position < slen && !is_word(str[position]))
      position++;
    while (position < slen && is_word(str[position]))
      position++;
    break;
  case 0xff:                   /* Meta-Delete */
  case (0x08 + 0x80):          /* Meta-Control-H */
    i = position;
    if (position > 0)
      position--;
    while (position > 0 && !is_word(str[position]))
      position--;
    while (position > 0 && is_word(str[position]))
      position--;
    if (position != i)
      delete_piece(str, position, i-1);
    break;
  case ('t' - 0x60):
    if (position > 0) {
      char t;
      if (position < slen) {
	t = str[position-1];
	str[position-1] = str[position];
	str[position] = t;
	position++;
      } else {
	t = str[position-2];
	str[position-2] = str[position-1];
	str[position-1] = t;
      }
    }
    break;
  case 'w' + 0x80:
    // copy to cutbuffer
    copy_to_cutbuffer(str, position, text_marker);
    break;
  case 'w' -0x60:
    // cut to cutbuffer
    copy_to_cutbuffer(str, position, text_marker);
    delete_piece(str, min(position,text_marker), max(position, text_marker)-1);
    break;
  case 0x0:
    // set marker
    text_marker = position;
    break;
  case 'y' -0x60:
    // yank back
    copy_from_cutbuffer(str, position);
    break;
  case 'n' - 0x60:
    if (goalwid < 0)
      goalwid = get_substring_width(str, sol, position,
				    font + EDIT_FONT_OFFS, size);
    if (str[eol] != '\0') {
      position = eol+1;
      while (position < slen && str[position] != '\n' &&
	     get_substring_width(str, eol+1, position,
				 font + EDIT_FONT_OFFS, size)
	     < goalwid)
	position++;
      if ((get_substring_width(str, eol+1, position-1,
			       font + EDIT_FONT_OFFS, size) +
	   get_substring_width(str, eol+1, position,
			       font + EDIT_FONT_OFFS, size))/2.0
	  > goalwid)
	position--;
    }
    break;
  case 'p' - 0x60:
    if (goalwid < 0)
      goalwid = get_substring_width(str, sol, position,
				    font + EDIT_FONT_OFFS, size);
    if (sol == 1)
      position = 0;
    else if (sol > 1) {
      for (position = sol-2; position >= 0 && str[position] != '\n';
	   position--)
	;
      i = ++position;
      while (str[position] != '\n'
	     && get_substring_width(str, i, position,
				    font + EDIT_FONT_OFFS, size)
	     < goalwid)
	position++;
      if ((get_substring_width(str, i, position-1,
			       font + EDIT_FONT_OFFS, size) +
	   get_substring_width(str, i, position,
			       font + EDIT_FONT_OFFS, size))/2.0
	  > goalwid)
	position--;
    }
    break;
  case 'g' + 0x80:
    if (parbox) {
      fill_parbox(str, tbox.x(), font, size);
    }
    break;
  case 'x' - 0x60:
    i = position;
    position = text_marker;
    text_marker = i;
    break;
  case '>' + 0x80:
    position = slen;
    break;
  case '<' + 0x80:
    position = 0;
    break;
  case 0x1f:			// C-_
    // undo last change:
    if (undo_pos == undo_start) {
      put_msg("nothing to undo");
    } else {
      undo_pos = (undo_pos - 1) & 0x0f;
      str = undo_buf[undo_pos];
      position = undo_position[undo_pos];
      text_marker = undo_marker[undo_pos];
    }
    break;
  case '\r':
    key = '\n';
  default:
    if ( key >= 0x20 && key < 0x7f || key == '\n') {
      // Add the character
      str.resize(slen+2);
      for (i=slen+1; i>position; i--)
	str[i] = str[i-1];
      str[position] = key;
      if (text_marker >= position)
	text_marker++;
      position++;
    }
    break;
  }
  text_undo_changed(str);
  redraw_pup();
}

void Text::push(pl_vec& mousep, int button, Boolean)
{
  if (drawing_mode == IDLE) {
    parbox = text_variant;
    // selected a position, start drawing here
    clear_selection();
    nonidle(DRAWING);
    put_msg("type Alt+e to end text");
    start_text(mousep);
    if (parbox)
      mouse_msg("", "set minipage", "");
    else
      mouse_msg("set cursor", "paste", "mark & copy");
  } else if (dragging) {
    if (button != MIDDLE_MOUSE) return;

    snap(mousep);
    s_coord left = min(pos.x(), mousep.x());
    s_coord right = max(pos.x(), mousep.x());
    s_coord bot = min(pos.y(), mousep.y());
    s_coord top = max(pos.y(), mousep.y());
    
    tbox = pl_vec(right - left, top - bot);
    pos = pl_vec(left, top);

    dragging = FALSE;
    redraw_pup();
  } else {
    // in text mode
    switch (button) {
    case LEFT_MOUSE:
      // set cursor
      position = mouse_position(str, mousep - pos,
				font + EDIT_FONT_OFFS, size);
      break;
    case MIDDLE_MOUSE:
      // paste at mouse position
      position = mouse_position(str, mousep - pos,
				font + EDIT_FONT_OFFS, size);
      copy_from_cutbuffer(str, position);
      text_undo_changed(str);
      break;
    case RIGHT_MOUSE:
      // set marker
      text_marker = mouse_position(str, mousep - pos,
				   font + EDIT_FONT_OFFS, size);
      copy_to_cutbuffer(str, position, text_marker);
      break;
    }
    redraw_pup();
  }
}

void Text::mouse(const pl_vec& mousep, Boolean)
{
  curmp = mousep;
}

void Text::transform(const pl_rotra& tfm)
{
  pl_vec oldpos = pos;
  
  pos = tfm * pos;
}

s_coord Text::dist(const pl_vec& p)
{
  s_coord left = pos.x();
  s_coord right = pos.x() + tbox.x();
  s_coord top = (parbox ? pos.y() : (pos.y() + tbox.y()));
  s_coord bot = (parbox ? (pos.y() - tbox.y()) : pos.y());

  if (p.x() <= left)
    return clearance(p, pl_edge(pl_vec(left, top), pl_vec(left, bot)));
  if (right <= p.x())
    return clearance(p, pl_edge(pl_vec(right, top), pl_vec(right, bot)));
  if (p.y() <= bot)
    return (bot - p.y());
  if (top <= p.y())
    return (p.y() - top);
  return 0.0;
}
 
void Text::stretch(const s_coord xfac, const s_coord yfac)
{
  pos.set_x(pos.x() * xfac);
  pos.set_y(pos.y() * yfac);
  if (parbox) {
    tbox.set_x(tbox.x() * xfac);
    tbox.set_y(tbox.y() * yfac);
    mkbox();
  }
}

void Text::save_latex(ostream& fh, const pl_vec& save_offset)
{
  if (!stroke.is_empty()) {
    fh << "%%\\put(" << pos.x() + save_offset.x()
       << "," << pos.y() + save_offset.y() << "){";
    if (parbox) {
      fh << "\\IPEmp{" << tbox.x() << "}{";
    } else {
      fh << "\\IPEtext{";
    }
    stroke.save_latex(fh);
    fh << "{" << size << "}" << latex_font_string[int(font)];
    for (int i = 0; i < str.size() - 1; i++) {
      if (str[i] != '\n')
	fh << str[i];
      else
	fh << "\n%%";
    }
    if (font == 3)
      fh << "$";
    fh << "}}\n";
  }
}

  
void Text::save_postscript(ostream& fh)
{
  fh << "% Text\n";
  if (parbox)
    fh << "% xy " << pos
       << "\n% bb " << tbox.x() << "\n";
  else
    fh << "% xy " << pos << "\n";
  stroke.save(fh, "sk", NIL);
  fh << "% f " << font << " " << size << "\n";
  fh << "% s ";
  for (int i = 0; i < str.size() - 1; i++) {
    if (str[i] != '\n')
      fh << str[i];
    else
      fh << "\n% s ";
  }
  fh << "\n% End\n\n";
}

void Text::save_properties(ostream& fh)
{
  fh << "% Text\n";
  fh << "% xy " << pos
     << "\n% tbb " << tbox << " " << (parbox ? tbox.y() : 0.0) << "\n";
  if (parbox) {
    fh << "% bb " << tbox.x() << "\n";
  }
  stroke.save(fh, "sk", NIL);
  fh << "% f " << font << " " << size << "\n% s ";
  for (int i = 0; i < str.size() - 1; i++) {
    if (str[i] != '\n')
      fh << str[i];
    else
      fh << "\n% s ";
  }
  fh << "\n% End\n\n";
}

static char *fieldlist[] = {
  "!2xy", "..", "?2f", "..", "?1bb", "?3tbb", "..", "..", "!*s"};

Text::Text(istream& fh)
// read text item from file
{
  // default font
  ps_in_value[2] = 0.0;
  ps_in_value[3] = 10.0;
  
  ps_okay &= ps_read_entry(fh, 9, fieldlist);
  if (!ps_okay)
    return;

  stroke.read(0);
  fill.mkempty();

  str = ps_in_string;

  font = short(ps_in_value[2]);
  size = ps_in_value[3];

  pos = pl_vec(ps_in_value[0], ps_in_value[1]);
  if (ps_in_defd[4]) {
    // is a parbox
    parbox = TRUE;
    tbox = pl_vec(ps_in_value[4], 1);
    mkbox();
  } else {
    parbox = FALSE;
    mkbox();
  }
}
