/*
 * transform.C
 * 
 * A planar affine transformation
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "transform.h"

Transform::Transform(void)
// setup identity transform
{
  A[0] = A[3] = 1.0;
  A[1] = A[2] = A[4] = A[5] = 0.0;
}

Transform::Transform(const s_coord *init)
// setup transform
{
  for (int i = 0; i < 6; i++)
    A[i] = init[i];
}

void Transform::apply(pl_vec& v)
// transform a vertex
{
  s_coord x = v.x();
  s_coord y = v.y();
  v.set_x(A[0] * x + A[2] * y + A[4]);
  v.set_y(A[1] * x + A[3] * y + A[5]);
}

void Transform::lapply(pl_vec& v)
// transform a vertex, without translation
{
  s_coord x = v.x();
  s_coord y = v.y();
  v.set_x(A[0] * x + A[2] * y);
  v.set_y(A[1] * x + A[3] * y);
}

void Transform::stretch(const s_coord xs, const s_coord ys)
// postmultiply matrix with stretch matrix
{
  A[0] *= xs;
  A[2] *= xs;
  A[4] *= xs;
  A[1] *= ys;
  A[3] *= ys;
  A[5] *= ys;
}

void Transform::transform(const pl_rotra& tf)
{
  pl_vec t(A[4], A[5]);
  pl_vec p1(A[0], A[1]);
  pl_vec p2(A[2], A[3]);
  p1 = tf * (p1 + t);
  p2 = tf * (p2 + t);
  t = tf * t;
  p1 -= t;
  p2 -= t;
  A[0] = p1.x() ;
  A[1] = p1.y();
  A[2] = p2.x();
  A[3] = p2.y();
  A[4] = t.x();
  A[5] = t.y();
}

Transform Transform::inverse(void)
// return inverse transformation
{
  s_coord t4 = 1/(A[0]*A[3]-A[2]*A[1]);
  Transform inv;
  
  inv.A[0] = A[3]*t4;
  inv.A[1] = -A[1]*t4;
  inv.A[2] = -A[2]*t4;
  inv.A[3] = A[0]*t4;
  inv.A[4] = (A[2]*A[5]-A[3]*A[4])*t4;
  inv.A[5] = -(A[0]*A[5]-A[1]*A[4])*t4;
  return inv;
}

void Transform::premult(const Transform& tf)
// premultiply with matrix tf
{
  s_coord B[6];
  
  for (int i = 0; i < 6; i++)
    B[i] = A[i];
  const s_coord *AA = &tf.A[0];
  A[0] = AA[0]*B[0] + AA[1]*B[2];
  A[1] = AA[0]*B[1] + AA[1]*B[3];
  A[2] = AA[2]*B[0] + AA[3]*B[2];
  A[3] = AA[2]*B[1] + AA[3]*B[3];
  A[4] = AA[4]*B[0] + AA[5]*B[2] + B[4];
  A[5] = AA[4]*B[1] + AA[5]*B[3] + B[5];
}

int Transform::is_similar(void)
// true if transform is a similar transformation, i.e. keeps a circle
{
  pl_vec p1(A[0], A[1]);
  pl_vec p2(A[2], A[3]);
  return (fabs_pp(dot(p1, p2)) < SIMILAR_THRESHOLD
	  && fabs_pp(p1.length() - p2.length()) < SIMILAR_THRESHOLD);
}

