/*
 * goodies.C
 * 
 * Ipe User Macros for several cute little functions, like
 *  precise scale and rotate, precise boxes, marking circle centers
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <string.h>

#include <plageo.h>

#define IUM_PLAGEO
#define IUM_TRANSFORM
#include "ium.h"

void mark_circle_center(void)
// mark center of circle or arc in primary selection
{
  for (IpeObject *ob = ium_input; ob && !ob->primary; ob = ob->next)
    ;
  if (!ob || (ob->type != IPE_CIRCLE && ob->type != IPE_ARC)) {
    ium_message = "select Circle or Arc first";
    return;
  }
  ium_output = new IpeObject;
  ium_output->type = IPE_MARK;
  ium_output->next = NULL;
  ium_output->stroke = ipe_environment.stroke;
  ium_output->w.mark = new Mark;
  ium_output->w.mark->type = ipe_environment.marktype;
  ium_output->w.mark->size = ipe_environment.marksize;
  ium_output->w.mark->pos = (ob->type == IPE_CIRCLE) ?
    ob->w.circle->tfm.transl() : ob->w.arc->arc.center();
  ium_mode = IUM_SELECT_BOTH;
  return;
}

static pl_box bounding_box(void)
// compute bounding box for all objects
// this box is correct, i.e. computes the same box as Ipe itself, except
// for spline objects, where it simply takes the bounding box of the 
// control points.
{
  int i;
  pl_vec r;
  pl_boundingbox bb;

  bb.make_empty();
  
  for (IpeObject *ob=ium_input; ob; ob = ob->next) {
    // add object to bounding box
    switch (ob->type) {
    case IPE_LINE:
    case IPE_SPLINE:
      for (i = 0; i < ob->w.line->v.size(); i++)
	bb.add_point(ob->w.line->v[i]);
      break;
    case IPE_CIRCLE:
      // this is a bit complicated, and uses the Transform class,
      // which represents the circle/ellipse
      // (for circles alone it would be much easier)
      {
	pl_angle beta;
	pl_vec dir;
	pl_unitvec exdir;
	pl_vec ex;
	for (beta = 0; beta < M_PI; beta = beta + M_PI_2) {
	  dir = pl_vec(cos_pp(beta), sin_pp(beta));
	  ob->w.circle->tfm.inverse().lapply(dir);
	  exdir = normalised(dir).normal();
	  ex = exdir;
	  ob->w.circle->tfm.apply(ex);
	  bb.add_point(ex);
	  ex = -exdir;
	  ob->w.circle->tfm.apply(ex);
	  bb.add_point(ex);
	}
      }
      break;
    case IPE_ARC:
      bb += ob->w.arc->arc.bbox();
      break;
    case IPE_MARK:
      bb.add_point(ob->w.mark->pos);
      break;
    case IPE_TEXT:
      bb.add_point(ob->w.text->ll);
      bb.add_point(ob->w.text->ur);
      break;
    case IPE_BITMAP:
      bb.add_point(ob->w.bitmap->ll);
      bb.add_point(ob->w.bitmap->ur);
      break;
    case IPE_BEGINGROUP:
    case IPE_ENDGROUP:
      break;
    }
  }
  return bb;
}

static void transform_object(IpeObject *iobj, const pl_rotra &tfm)
// apply tfm to object
{
  int i;
  switch (iobj->type) {
  case IPE_LINE:
  case IPE_SPLINE:
    for (i = 0; i < iobj->w.line->v.size(); i++)
      iobj->w.line->v[i] = tfm * iobj->w.line->v[i];
    break;
  case IPE_CIRCLE:
    iobj->w.circle->tfm.transform(tfm);
    break;
  case IPE_ARC:
    iobj->w.arc->arc.transform(tfm);
    break;
  case IPE_MARK:
    iobj->w.mark->pos = tfm * iobj->w.mark->pos;
    break;
  case IPE_TEXT:
    // do not rotate text, just apply operation to text position
    {
      pl_vec oldpos = iobj->w.text->pos;
      iobj->w.text->pos = tfm * iobj->w.text->pos;
      pl_vec offs = iobj->w.text->pos - oldpos;
      iobj->w.text->ll += offs;
      iobj->w.text->ur += offs;
    }
    break;
  case IPE_BITMAP:
    // do not rotate bitmap, just apply operation to bbox
    {
      pl_vec oldpos = iobj->w.bitmap->ll;
      iobj->w.bitmap->ll = tfm * iobj->w.bitmap->ll;
      pl_vec offs = iobj->w.bitmap->ll - oldpos;
      iobj->w.bitmap->ur += offs;
    }
    break;
  }
}

inline void stretch_vertex(pl_vec &v, const pl_vec &f)
{
  v.set_x(v.x() * f.x());
  v.set_y(v.y() * f.y());
}
  
static void stretch_object(IpeObject *iobj,
			   const pl_vec fp, const pl_vec &factor)
// apply stretch to object
{
  int i;

  transform_object(iobj, pl_rotra(-fp));
  switch (iobj->type) {
  case IPE_LINE:
  case IPE_SPLINE:
    for (i = 0; i < iobj->w.line->v.size(); i++) 
      stretch_vertex(iobj->w.line->v[i], factor);
    break;
  case IPE_CIRCLE:
    iobj->w.circle->tfm.stretch(factor.x(), factor.y());
    break;
  case IPE_ARC:
    {
      pl_vec p(iobj->w.arc->arc.center());
      stretch_vertex(p, factor);
      if (factor.x() != factor.y()) {
	ium_message = "cannot stretch a circular arc";
      }
      iobj->w.arc->arc = pl_arc(p, (iobj->w.arc->arc.radius() * factor.x()),
				iobj->w.arc->arc.begin_angle(),
				iobj->w.arc->arc.enclosed_angle());
    }      
    break;
  case IPE_MARK:
    stretch_vertex(iobj->w.mark->pos, factor);
    break;
  case IPE_BITMAP:
    stretch_vertex(iobj->w.bitmap->ll, factor);
    stretch_vertex(iobj->w.bitmap->ur, factor);
    break;
  case IPE_TEXT:
    // do not stretch text, just apply operation to text position
    {
      pl_vec oldpos = iobj->w.text->pos;
      stretch_vertex(iobj->w.text->pos, factor);
      pl_vec offs = iobj->w.text->pos - oldpos;
      iobj->w.text->ll += (offs);
      iobj->w.text->ur += (offs);
    }
    break;
  }
  transform_object(iobj, pl_rotra(fp));
}
  
void precise_rotate(void)
{
  if (!ium_input) {
    ium_message = "nothing to rotate";
    return;
  }

  char *p = ium_parameter;
  s_coord alpha = strtod(ium_parameter, &p);
  if (p == ium_parameter) {
    ium_message = "no angle recognized";
    return;
  }

  alpha = M_PI * alpha / 180.0;
  pl_box bb = bounding_box();
  pl_vec fp((bb.xmax() + bb.xmin()) / 2.0,
	    (bb.ymax() + bb.ymin()) / 2.0);
  if (ipe_environment.axisset)
    fp = ipe_environment.origin;
  pl_rotra tfm = pl_rotra(pl_angle(alpha), fp) * pl_rotra(-fp);
  
  for (IpeObject *ob = ium_input; ob; ob = ob->next) {
    if (ob->type != IPE_BEGINGROUP && ob->type != IPE_ENDGROUP)
      transform_object(ob, tfm);
  }
  ium_output = ium_input;
  ium_mode = IUM_REPLACE;
  return;
}

void precise_stretch(void)
{
  if (!ium_input) {
    ium_message = "nothing to stretch";
    return;
  }

  pl_vec factor;

  char *p = ium_parameter;
  factor.set_x(strtod(ium_parameter, &p));
  if (p == ium_parameter) {
    ium_message = "no factor recognized";
    return;
  }
  char *q = p;
  factor.set_y(strtod(q, &p));
  if (p == q) {
    // no y factor?
    if (!*p)
      factor.set_y(factor.x());
    else {
      ium_message = "no factor recognized";
      return;
    }
  }    
  // recognized factors
  pl_box bb = bounding_box();
  pl_vec fp((bb.xmax() + bb.xmin()) / 2.0,
	    (bb.ymax() + bb.ymin()) / 2.0);
  if (ipe_environment.axisset)
    fp = ipe_environment.origin;
  
  for (IpeObject *ob = ium_input; ob; ob = ob->next) {
    if (ob->type != IPE_BEGINGROUP && ob->type != IPE_ENDGROUP)
      stretch_object(ob, fp, factor);
  }
  ium_output = ium_input;
  ium_mode = IUM_REPLACE;
  return;
}

IpeObject *make_box(const pl_vec &ll, const pl_vec &ur)
// create a box with given measurements
{
  IpeObject *b = new IpeObject;
  b->type = IPE_LINE;
  b->next = NULL;
  b->stroke = ipe_environment.stroke;
  b->fill = ipe_environment.fill;
  b->linestyle = ipe_environment.linestyle;
  b->linewidth = ipe_environment.linewidth;
  b->w.line = new Line;
  b->w.line->closed = 1;
  b->w.line->arrow = 0;
  b->w.line->v.newsize(4);
  b->w.line->v[0] = ll;
  b->w.line->v[1] = pl_vec(ur.x(), ll.y());
  b->w.line->v[2] = ur;
  b->w.line->v[3] = pl_vec(ll.x(), ur.y());
  return b;
}

void make_bounding_box(void)
{
  // make bounding box for selection
  if (!ium_input) {
    ium_message = "please select some objects first";
    return;
  }
  pl_box bb = bounding_box();

  ium_output = make_box(pl_vec(bb.xmin(), bb.ymin()),
			pl_vec(bb.xmax(), bb.ymax()));
  ium_mode = IUM_SELECT_NEW;
}


void precise_box(void)
{
  char *ls = ium_parameter;
  char *p = ls;
  s_coord wd = strtod(ls, &p);
  if (p != ls) {
    ls = p;
    s_coord ht = strtod(ls, &p);
    if (p != ls) {
      // recognized width and height
      // convert to postscript points
      wd *= 72.0 / 25.0;
      ht *= 72.0 / 25.0;
      pl_vec ll(-wd/2.0, -ht/2.0);
      pl_vec ur = ll + pl_vec(wd, ht);
      // now make box
      ium_output = make_box(ll, ur);
      ium_mode = IUM_SELECT_NEW;
      return;
    }
  }
  ium_message = "dimensions not recognized";
}

static void test_and_add(IpeObject* &last, IpeObject *ob, int no, int noi,
			 const pl_pgn &pgn)
// test edge, and clip it against box
{
  pl_edge e(ob->w.line->v[no], ob->w.line->v[noi]);
  pl_ray ray(e.begin_vt(), e.direction());

  intervals iv = intersect(ray, pgn);
  s_coordreray ivbnd = iv.boundaries;
  ivbnd.resize(iv.find_index_after(e.length()));
  if (iv.lies_in(e.length()))
    ivbnd.append(e.length());
  
  for (int i = 0; i < ivbnd.size(); i += 2) {
    // make edge
    last->next = new IpeObject;
    last = last->next;
    last->next = NULL;
    last->type = IPE_LINE;
    last->linestyle = ob->linestyle;
    last->linewidth = ob->linewidth;
    last->stroke = ob->stroke;
    last->fill.red = -1;
    last->w.line = new Line;
    last->w.line->closed = 0;
    last->w.line->arrow = 0;
    last->w.line->v.newsize(2);
    last->w.line->v[0] = e.begin_vt() + ivbnd[i] * e.direction();
    last->w.line->v[1] = e.begin_vt() + ivbnd[i+1] * e.direction();
  }
}

static void clip_objects(void)
// clip all objects to the primary selection, which must be a box
{
  for (IpeObject *ob = ium_input; ob && !ob->primary; ob = ob->next)
    ;
  if (!ob || (ob->type != IPE_LINE) || (!ob->w.line->closed)) {
    ium_message = "primary selection must be a simple polygon for clipping";
    return;
  }
  pl_pgn pgn(ob->w.line->v);

  if (!pgn.is_ok()) {
    ium_message = "Clip polygon must be a SIMPLE polygon";
    return;
  }
  
  // now walk through selection, and do the clipping

  ium_output = new IpeObject;
  ium_output->type = IPE_BEGINGROUP;
  ium_output->next = NULL;
  
  IpeObject *last = ium_output;
  int i;

  for (ob = ium_input; ob; ob = ob->next) {
    if (!ob->primary) {
      switch (ob->type) {
      case IPE_BEGINGROUP:
      case IPE_ENDGROUP:
	break;
      case IPE_MARK:
	if (lies_inside(ob->w.mark->pos, pgn)) {
	  // add mark to output
	  last->next = new IpeObject;
	  last = last->next;
	  last->next = NULL;
	  last->type = IPE_MARK;
	  last->stroke = ob->stroke;
	  last->w.mark = ob->w.mark;
	}
	break;
      case IPE_LINE:
	for (i = 0; i < ob->w.line->v.size() - 1; i++)
	  test_and_add(last, ob, i, i+1, pgn);
	if (ob->w.line->closed)
	  test_and_add(last, ob, ob->w.line->v.size() - 1, 0, pgn);
	break;
      default:
	ium_message = "cannot clip anything but segments and marks";
	ium_output = NULL;
	return;
      }
    }
  }
  last->next = new IpeObject;
  last = last->next;
  last->next = NULL;
  last->type = IPE_ENDGROUP;
  ium_message = "clipped all segments and marks";
  ium_mode = IUM_SELECT_NEW;
}

int main(int argc, char **argv)
{
  ium_begin(argc, argv);
  
  switch (atoi(ium_argument)) {

  case 0:
    // mark circle center
    mark_circle_center();
    break;

  case 1:
    // precise rotate
    precise_rotate();
    break;

  case 2:
    // precise stretch
    precise_stretch();
    break;

 case 3:
    // precise box
   precise_box();
   break;

  case 4:
    // bounding box
    make_bounding_box();
    break;
    
  case 5:
    // clip objects to box or disc
    clip_objects();
    break;
    
  default:
    // illegal argument
//  cerr << "IUM failed: Illegal argument " << argv[2] << "\n";
    exit(1);
  }
  ium_end();
}
