/*
 * iumio.c
 * 
 * This file contains routines for Ipe User Macros,
 * to read and write the interface file from/to Ipe
 *
 *  Call
 *    ium_begin(argc, argv)
 *  at the beginning of the IUM to set ium_input.
 *  Create ium_output, set ium_mode and ium_message, and call
 *    ium_end()
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#ifdef IUM_PLAGEO
#include <plageo.h>
#endif

#include "ium.h"

#define MAX_LINE_LENGTH 1024

#define FALSE 0
#define TRUE 1

#ifdef __cplusplus
#define NEWOBJECT(Type) (new Type)
#define NEWARRAY(Type, Size) (new Type[Size])
#define FREEARRAY(Ptr) delete [] Ptr
#else
#define NEWOBJECT(Type) ((Type *) malloc(sizeof(Type)))
#define NEWARRAY(Type, Size) ((Type *) malloc(Size * sizeof(Type)))
#define FREEARRAY(Ptr) free(Ptr)
#endif

#ifdef sun
#define BITS char
#else
#define BITS void
#endif

#ifdef IUM_PLAGEO
#define SETXY(v, x, y) v = pl_vec(x,y)
#define XCOORD x()
#define YCOORD y()
#else
#define SETXY(v, x, y) v.x = x; v.y = y
#define XCOORD x
#define YCOORD y
#endif

/* the current Ipe environment when IUM is called */

IpeEnvironment ipe_environment;

/* IUM input and output */

IpeObject *ium_input;
IpeObject *ium_output;

char *ium_message = NULL;
int  ium_mode = 0;
char *ium_argument = NULL;
char *ium_parameter = NULL;

static char *fname;
static FILE *fh;
static char linebuf[MAX_LINE_LENGTH];
static int grouplevel = 0;
static bool in_settings = TRUE;

/******************** reading ******************************************/

static char *read_next()
{
  int ch2, ch1, ch;
  char *p;
  
  /* read words until we find a sole "%" */
  ch2 = ch1 = ch = ' ';
  
  while ((ch = fgetc(fh)) != EOF &&
	 !(ch1 == '%' && isspace(ch2) && isspace(ch))) {
    ch2 = ch1;
    ch1 = ch;
  }

  if (ch == EOF) {
    /* read error (could be EOF, then it\'s a format error) */
    fprintf(stderr,
	    "IUM failed: error reading interface file %s\n", fname);
    exit(9);
  }

  /* next word is keyword */

  p = linebuf;
  while ((*p = fgetc(fh)), !isspace(*p))
    p++;
  *p++ = '\0';

  return linebuf;
}

/* temporary storage for read_env and read_entry */

static struct {
  bool closed;
  vertex xy;
  bool minipage;
  float wd, ht, dp;
  float radius;
  float begangle, endangle;
  char *str;
  int n;
  vertex *v;
  bool ellipse;
  float tfm[4];
  unsigned long *words;
  int xbits, ybits;
  bool bmcolor;
} rd;

static void read_env(IpeEnvironment *ienv)
{
  char *wk;
  float x, y;
  int i;
  
  ienv->stroke.red = ienv->stroke.green = ienv->stroke.blue = -1;
  ienv->fill.red = ienv->fill.green = ienv->fill.blue = -1;
  ienv->arrow = 0;
  ienv->axisset = FALSE;
  rd.closed = FALSE;
  /* init transformation to identity */
  rd.tfm[0] = rd.tfm[3] = 1.0;
  rd.tfm[1] = rd.tfm[2] = 0.0;
  rd.minipage = FALSE;
  rd.ellipse = FALSE;
  rd.str = NULL;
  rd.v = NULL;
  
  while (TRUE) {
    wk = read_next();
    if (!strcmp(wk, "sk")) {
      fscanf(fh, "%f", &ienv->stroke.red);
      ienv->stroke.blue = ienv->stroke.green = ienv->stroke.red;
    } else if (!strcmp(wk, "fi")) {
      fscanf(fh, "%f", &ienv->fill.red);
      ienv->fill.blue = ienv->fill.green = ienv->fill.red;
    } else if (!strcmp(wk, "skc")) {
      fscanf(fh, "%f%f%f",
	     &ienv->stroke.red, &ienv->stroke.green, &ienv->stroke.blue);
    } else if (!strcmp(wk, "fic")) {
      fscanf(fh, "%f%f%f",
	     &ienv->fill.red, &ienv->fill.green, &ienv->fill.blue);
    } else if (!strcmp(wk, "ss")) {
      fscanf(fh, "%hu%f", &ienv->linestyle, &ienv->linewidth);
    } else if (!strcmp(wk, "ar")) {
      fscanf(fh, "%hu%f", &ienv->arrow, &ienv->arsize);
    } else if (!strcmp(wk, "cl")) {
      rd.closed = TRUE;
    } else if (!strcmp(wk, "f")) {
      fscanf(fh, "%hu%f", &ienv->font, &ienv->fontsize);
    } else if (!strcmp(wk, "grid")) {
      fscanf(fh, "%f%f", &ienv->gridsize, &ienv->snapangle);
    } else if (!strcmp(wk, "ty")) {
      fscanf(fh, "%hu", &ienv->marktype);
    } else if (!strcmp(wk, "sz")) {
      fscanf(fh, "%f",  &ienv->marksize);
    } else if (!strcmp(wk, "xy")) {
      fscanf(fh, "%f%f", &x, &y);
      SETXY(rd.xy, x, y);
    } else if (!strcmp(wk, "px")) {
      fscanf(fh, "%d%d", &rd.xbits, &rd.ybits);
    } else if (!strcmp(wk, "bb")) {
      rd.minipage = TRUE;
      fscanf(fh, "%f%f", &rd.wd, &rd.ht);
      rd.dp = rd.ht;
    } else if (!strcmp(wk, "tbb")) {
      rd.minipage = FALSE;
      fscanf(fh, "%f%f%f", &rd.wd, &rd.ht, &rd.dp);
    } else if (!strcmp(wk, "ang")) {
      fscanf(fh, "%f%f", &x, &y);
      rd.begangle = x * M_PI / 180.0;
      rd.endangle = y * M_PI / 180.0;
    } else if (!strcmp(wk, "r")) {
      fscanf(fh, "%f", &rd.radius);
    } else if (!strcmp(wk, "tfm")) {
      rd.ellipse = TRUE;
      fscanf(fh, "%f%f%f%f", &rd.tfm[0], &rd.tfm[1], &rd.tfm[2], &rd.tfm[3]);
    } else if (!strcmp(wk, "axis")) {
      ienv->axisset = TRUE;
      fscanf(fh, "%f%f%f", &x, &y, &ienv->axisdir);
      SETXY(ienv->origin, x, y);
    } else if (!strcmp(wk, "#")) {
      /* vertices of a polyline */
      fscanf(fh, "%d", &rd.n);
      rd.v = NEWARRAY(vertex, rd.n);
      for (i = 0; i < rd.n; i++ ) {
	fscanf(fh, "%g%g", &x, &y);
	SETXY(rd.v[i], x, y);
      }
    } else if (!strcmp(wk, "s")) {
      /* get string */
      fgets(linebuf, MAX_LINE_LENGTH, fh);
      linebuf[strlen(linebuf) - 1] = '\0';
      if (!rd.str) {
	/* first string */
	rd.str = strdup(linebuf);
      } else {
	char *ns = NEWARRAY(char, (strlen(rd.str) + strlen(linebuf) + 2));
	strcpy(ns, rd.str);
	strcat(ns, "\n");
	strcat(ns, linebuf);
	free(rd.str);
	rd.str = ns;
      }
    } else if (!strcmp(wk, "bits")) {
      /* get bitmap */
      long i, nwords, incolor, nchars;
      int ch, mode;
      char *p, *strbits, buf[3];
      short red, green, blue;
      
      fscanf(fh, "%ld%d", &nwords, &mode);
      incolor = mode & 1;
      if (mode & 0x8) {
	/* read RAW bitmap */
	do {
	  if ((ch = fgetc(fh)) == EOF) {
	    fprintf(stderr, "IUM failed: cannot read RAW bitmap\n");
	    exit(9);
	  }
	} while (ch != '\n');
	rd.bmcolor = incolor ? TRUE : FALSE;
	rd.words = NEWARRAY(unsigned long, nwords);
	if (mode & 0x01) {
	  /* color bitmap: 32 bits per pixel */
	  if (fread((BITS *) rd.words, sizeof(unsigned long),
		    ((unsigned int) nwords), fh)
	      != nwords) {
	    fprintf(stderr, "IUM failed: cannot read RAW bitmap\n");
	    exit(9);
	  }
	} else {
	  /* gray bitmap: 8 bits per pixel */
	  register char *inp, *end;
	  register unsigned long *out;
	  char *pix = NEWARRAY(char, nwords);
	  if (fread((BITS *) pix, sizeof(char), ((unsigned int) nwords), fh)
	      != nwords) {
	    fprintf(stderr, "IUM failed: cannot read RAW bitmap\n");
	    exit(9);
	  }
	  /* convert to unsigned longs */
	  inp = pix;
	  end = pix + nwords;
	  out = rd.words;
	  while (inp < end) {
	    *out++ = (*inp << 16) | (*inp << 8) | (*inp);
	    inp++;
	  }
	  FREEARRAY(pix);
	}	  
      } else {
	/* read Postscript style bitmap */
	nchars = (incolor ? 6 : 2) * nwords;
	p = strbits = NEWARRAY(char, nchars);
	for (i = 0; i < nchars; i++) {
	  do {
	    if ((ch = fgetc(fh)) == EOF) {
	      fprintf(stderr, "IUM failed: cannot read bitmap\n");
	      exit(9);
	    }
	  } while (!(('0' <= ch && ch <= '9') || ('a' <= ch && ch <= 'f')));
	  *p++ = ch;
	}
	p = strbits;
	rd.words = NEWARRAY(unsigned long, nwords);
	rd.bmcolor = incolor ? TRUE : FALSE;
	buf[2] = '\0';
	for (i = 0; i < nwords; ) {
	  buf[0] = *p++;
	  buf[1] = *p++;
	  red = ((short) strtol(buf, NULL, 16));
	  if (incolor) {
	    buf[0] = *p++;
	    buf[1] = *p++;
	    green = ((short) strtol(buf, NULL, 16));
	    buf[0] = *p++;
	    buf[1] = *p++;
	    blue = ((short) strtol(buf, NULL, 16));
	    rd.words[i++] = (blue * 0x10000) | (green * 0x100) | red;
	  } else {
	    rd.words[i++] = (red * 0x10000) | (red * 0x100) | red;
	  }
	}
	free(strbits);
      }
    } else if (!strcmp(wk, "End")) {
      return;
    } else {
      if (in_settings) {
	/* unknown keyword in settings: ignore this line */
	fgets(linebuf, MAX_LINE_LENGTH, fh);
      } else {
	/* unknown keyword in an object: this is serious */
	fprintf(stderr,
		"IUM failed: illegal keyword %s in interface file %s\n",
		wk, fname);
	exit(9);
      }
    }
  }
}

static IpeObject *read_entry()
{
  char *wk;
  int i;
  IpeObject *iobj;
  IpeEnvironment ienv;

  iobj = NEWOBJECT(IpeObject);
  
  iobj->primary = FALSE;
  iobj->next = NULL;
  wk = read_next();
  if (!strcmp(wk, "Primary")) {
    wk = read_next();
    iobj->primary = TRUE;
  }
  
  if (!strcmp(wk, "Group")) {
    iobj->type = IPE_BEGINGROUP;
    grouplevel++;
    return iobj;
  } else if (!strcmp(wk, "End")) {
    iobj->type = IPE_ENDGROUP;
    grouplevel--;
    return ((grouplevel >= 0) ? iobj : NULL);
  } else if (!strcmp(wk, "Line")) {
    iobj->type = IPE_LINE;
  } else if (!strcmp(wk, "Spline")) {
    iobj->type = IPE_SPLINE;
  } else if (!strcmp(wk, "Text")) {
    iobj->type = IPE_TEXT;
  } else if (!strcmp(wk, "Circle")) {
    iobj->type = IPE_CIRCLE;
  } else if (!strcmp(wk, "Arc")) {
    iobj->type = IPE_ARC;
  } else if (!strcmp(wk, "Mark")) {
    iobj->type = IPE_MARK;
  } else if (!strcmp(wk, "Bitmap")) {
    iobj->type = IPE_BITMAP;
  } else {
    fprintf(stderr,
	    "IUM failed: illegal keyword %s in interface file %s\n",
	    wk, fname);
    exit(9);
  }

  /* read header, now read data */
  read_env(&ienv);

  /* read data, now fill in object */
  iobj->stroke = ienv.stroke;
  iobj->fill = ienv.fill;
  iobj->linestyle = ienv.linestyle;
  iobj->linewidth = ienv.linewidth;
  
  switch (iobj->type) {
    /* we treat polylines and splines alike */
  case IPE_LINE:
  case IPE_SPLINE:
    iobj->w.line = NEWOBJECT(Line);
    iobj->w.line->closed = rd.closed;
    iobj->w.line->arrow = ienv.arrow;
    iobj->w.line->arsize = ienv.arsize;
#ifdef IUM_PLAGEO
    iobj->w.line->v.newsize(rd.n);
    for (i = 0; i < rd.n; i++) {
      iobj->w.line->v[i] = rd.v[i];
    }
#else
    iobj->w.line->n = rd.n;
    iobj->w.line->v = rd.v;
#endif
    break;

  case IPE_ARC:
    iobj->w.arc = NEWOBJECT(Arc);
    iobj->w.arc->arrow = ienv.arrow;
    iobj->w.arc->arsize = ienv.arsize;
#ifdef IUM_PLAGEO
    {
      pl_angle alpha0(rd.begangle);
      pl_angle alpha1(rd.endangle);
      iobj->w.arc->arc = pl_arc(rd.xy, rd.radius, alpha0,
				pl_angle(alpha1-alpha0).normalised_value(0.0));
    }
#else
    iobj->w.arc->center = rd.xy;
    iobj->w.arc->radius = rd.radius;
    iobj->w.arc->begangle = rd.begangle;
    iobj->w.arc->endangle = rd.endangle;
#endif
    break;

  case IPE_CIRCLE:
    iobj->w.circle = NEWOBJECT(Circle);
#ifdef IUM_TRANSFORM
    for (i = 0; i < 4; i++) {
      iobj->w.circle->tfm.A[i] = rd.tfm[i];
    }
    iobj->w.circle->tfm.A[4] = iobj->w.circle->tfm.A[5] = 0.0;
    iobj->w.circle->tfm.stretch(rd.radius, rd.radius);
    iobj->w.circle->tfm.transform(pl_vec(rd.xy.x(), rd.xy.y()));
#else
    iobj->w.circle->center = rd.xy;
    iobj->w.circle->radius = rd.radius;
    iobj->w.circle->ellipse = rd.ellipse;
    for (i = 0; i < 4; i++) {
      iobj->w.circle->tfm[i] = rd.tfm[i];
    }
#endif
    break;
    
  case IPE_MARK:
    iobj->w.mark = NEWOBJECT(Mark);
    iobj->w.mark->pos = rd.xy;
    iobj->w.mark->type = ienv.marktype;
    iobj->w.mark->size = ienv.marksize;
    break;

  case IPE_BITMAP:
    iobj->w.bitmap = NEWOBJECT(Bitmap);
    iobj->w.bitmap->ll = rd.xy;
    iobj->w.bitmap->width = rd.xbits;
    iobj->w.bitmap->height = rd.ybits;
    iobj->w.bitmap->words = rd.words;
    iobj->w.bitmap->in_color = rd.bmcolor;
#ifdef IUM_PLAGEO
    iobj->w.bitmap->ur = rd.xy + pl_vec(rd.wd, rd.ht);
#else
    iobj->w.bitmap->ur.x = rd.xy.x + rd.wd;
    iobj->w.bitmap->ur.y = rd.xy.y + rd.ht;
#endif
    break;
    
  case IPE_TEXT:
    iobj->w.text = NEWOBJECT(Text);
    iobj->w.text->pos = rd.xy;
    iobj->w.text->font = ienv.font;
    iobj->w.text->fontsize = ienv.fontsize;
    iobj->w.text->minipage = rd.minipage;
#ifdef IUM_PLAGEO
    iobj->w.text->ll = pl_vec(rd.xy.x(), rd.xy.y() - rd.dp);
    iobj->w.text->ur = iobj->w.text->ll + pl_vec(rd.wd, rd.ht);
#else
    iobj->w.text->ll = rd.xy;
    iobj->w.text->ll.y -= rd.dp;
    iobj->w.text->ur = iobj->w.text->ll;
    iobj->w.text->ur.x += rd.wd;
    iobj->w.text->ur.y += rd.ht;
#endif
    iobj->w.text->str = rd.str;
    break;
  }

  /* now set all values in IpeObject, return it */
  return iobj;
}

/* ium_init: test argc count, decode argument */

void ium_init(int argc, char **argv)
{
  ium_input = NULL;
  ium_output = NULL;

  if (argc < 3) {
    /* something is wrong here, we should have at least two arguments */
    fprintf(stderr, "IUM %s failed: interface is broken\n", argv[0]);
    exit(9);
  }
  fname = argv[1];
  ium_argument = argv[2];
  ium_parameter = NULL;
  if (argc >= 4)
    ium_parameter = argv[3];
}

/* ium_begin: initialize, then read interface file */

void ium_begin(int argc, char **argv)
{
  IpeObject *last, *iobj;

  ium_init(argc, argv);

  /* read IUM file */

  if (!(fh = fopen(fname, "r"))) {
    fprintf(stderr,
	    "IUM failed: cannot open interface file %s\n", fname);
    exit(9);
  }
  
  grouplevel = 0;
  
  if (!fgets(linebuf, MAX_LINE_LENGTH, fh) ||
      strncmp(linebuf, "%\\IUMid", 7) ||
      strcmp(read_next(), "Settings")) {
    fprintf(stderr,
	    "IUM failed: interface file %s has wrong format\n", fname);
    exit(9);
  }
  
  read_env(&ipe_environment);
  last = NULL;
  while ((iobj = read_entry()) != NULL) {
    if (last)
      last->next = iobj;
    else
      ium_input = iobj;
    last = iobj;
  }
  fclose(fh);
}

/******************** iumrc interface **********************************/

static bool iumrc_read = FALSE;
static bool iumrc_modified = FALSE;
static char iumrc_name[64];

typedef struct _iumrc_entry {
  struct _iumrc_entry *next;
  char *keyword;
  char *contents;
} IumrcEntry;

static IumrcEntry *iumrc_contents = NULL;

static void read_iumrc(void)
{
  char *home = getenv("HOME");
  char lbuf[MAX_LINE_LENGTH];
  FILE *ifd;
  IumrcEntry *ie;
  char *p;
  
  iumrc_read = TRUE;
  if (!home)
    return;
  sprintf(iumrc_name, "%s/.iumrc", home);
  if (!(ifd = fopen(iumrc_name, "r")))
    return;
  while (fgets(lbuf, MAX_LINE_LENGTH, ifd)) {
    ie = NEWOBJECT(IumrcEntry);
    ie->next = iumrc_contents;
    iumrc_contents = ie;
    for (p = lbuf; *p && *p != ':'; p++)
      ;
    if (!p) {
      fprintf(stderr, "Error in .iumrc file\n");
      return;
    }
    *p = '\0';
    ie->keyword = NEWARRAY(char, strlen(lbuf) + 1);
    strcpy(ie->keyword, lbuf);
    for (p++; *p == ' ' || *p == '\t'; p++)
      ;
    if (p[strlen(p) - 1] == '\n')
      p[strlen(p) - 1] = '\0';
    ie->contents = NEWARRAY(char, strlen(p) + 1);
    strcpy(ie->contents, p);
  }
  fclose(ifd);
/* for (ie = iumrc_contents; ie; ie = ie->next)
   fprintf(stderr, "%s -> %s\n", ie->keyword, ie->contents);
   */
}
    
    
char *iumrc_get(char *keyword)
/* find entry in iumrc file with keyword */     
{
  IumrcEntry *ie;
  
  if (!iumrc_read) read_iumrc();
  if (!keyword || !iumrc_contents)
    return NULL;
  for (ie = iumrc_contents; ie; ie = ie->next) {
    if (!strcmp(keyword, ie->keyword)) {
      return ie->contents;
    }
  }
  return NULL;
}

void iumrc_put(char *keyword, char *contents)
/* save entry in iumrc file */
{
  IumrcEntry *ie;
  
  if (!keyword || !contents) return;
  if (!iumrc_read) read_iumrc();
  iumrc_modified = TRUE;
  for (ie = iumrc_contents; ie; ie = ie->next) {
    if (!strcmp(keyword, ie->keyword)) {
      FREEARRAY(ie->contents);
      ie->contents = NEWARRAY(char, strlen(contents) + 1);
      strcpy(ie->contents, contents);
      return;
    }
  }
  ie = NEWOBJECT(IumrcEntry);
  ie->next = iumrc_contents;
  iumrc_contents = ie;
  ie->keyword = NEWARRAY(char, strlen(keyword) + 1);
  strcpy(ie->keyword, keyword);
  ie->contents = NEWARRAY(char, strlen(contents) + 1);
  strcpy(ie->contents, contents);
}

static void write_iumrc(void)
{
  FILE *ifd;
  IumrcEntry *ie;
  
  if (!iumrc_modified) return;
  if (!(ifd = fopen(iumrc_name, "w")))
    return;
  for (ie = iumrc_contents; ie; ie = ie->next) {
    fprintf(ifd, "%s: %s\n", ie->keyword, ie->contents);
  }
  fclose(ifd);
}
    

/******************** writing ******************************************/

static void write_colors(IpeObject *iobj)
{
  if (iobj->stroke.red != -1) {
    fprintf(fh, "%% skc %g %g %g\n",
	    iobj->stroke.red, iobj->stroke.green, iobj->stroke.blue);
  }
  if (iobj->fill.red != -1) {
    fprintf(fh, "%% fic %g %g %g\n",
	    iobj->fill.red, iobj->fill.green, iobj->fill.blue);
  }
}

static void write_linestyle(IpeObject *iobj)
{
  fprintf(fh, "%% ss %u %g\n", iobj->linestyle, iobj->linewidth);
}

static void write_entry(IpeObject *iobj)
/*  write a single Ipe Object to output file */
{
  int i;
  char *p;
  
  switch (iobj->type) {
  case IPE_BEGINGROUP:
    fprintf(fh, "%% Group\n");
    return;
  case IPE_ENDGROUP:
    break;

  case IPE_LINE:
  case IPE_SPLINE:
    fprintf(fh, (iobj->type == IPE_SPLINE ? "%% Spline\n" : "%% Line\n"));
    write_colors(iobj);
    write_linestyle(iobj);
    if (iobj->w.line->closed)
      fprintf(fh, "%% cl\n");
    /* splines cannot have arrow --- you better don't set them */
    if (iobj->w.line->arrow)
      fprintf(fh, "%% ar %d %g\n", iobj->w.line->arrow, iobj->w.line->arsize);
#ifdef IUM_PLAGEO
    fprintf(fh, "%% # %d\n", iobj->w.line->v.size());
    for (i = 0; i < iobj->w.line->v.size(); i++ ) {
      fprintf(fh, "%g %g\n", iobj->w.line->v[i].x(), iobj->w.line->v[i].y() );
    }
#else
    fprintf(fh, "%% # %d\n", iobj->w.line->n);
    for (i = 0; i < iobj->w.line->n; i++ ) {
      fprintf(fh, "%g %g\n", iobj->w.line->v[i].x, iobj->w.line->v[i].y );
    }
#endif
    break;

  case IPE_CIRCLE:
    fprintf(fh, "%% Circle\n");
    write_colors(iobj);
    write_linestyle(iobj);
#ifdef IUM_TRANSFORM
    fprintf(fh, "%% xy %g %g\n",
	    iobj->w.circle->tfm.transl().x(),
	    iobj->w.circle->tfm.transl().y());
    fprintf(fh, "%% r 1.0\n%% tfm %g %g %g %g\n",
	    iobj->w.circle->tfm.A[0], iobj->w.circle->tfm.A[1],
	    iobj->w.circle->tfm.A[2], iobj->w.circle->tfm.A[3]);
#else
    fprintf(fh, "%% xy %g %g\n",
	    iobj->w.circle->center.XCOORD, iobj->w.circle->center.YCOORD);
    fprintf(fh, "%% r %g\n", iobj->w.circle->radius);
    if (iobj->w.circle->ellipse) {
      fprintf(fh, "%% tfm %g %g %g %g\n",
	      iobj->w.circle->tfm[0], iobj->w.circle->tfm[1],
	      iobj->w.circle->tfm[2], iobj->w.circle->tfm[3]);
    }
#endif
    break;
    
  case IPE_MARK:
    fprintf(fh, "%% Mark\n");
    iobj->fill.red = -1;
    write_colors(iobj);
    fprintf(fh, "%% xy %g %g\n",
	    iobj->w.mark->pos.XCOORD, iobj->w.mark->pos.YCOORD);
    fprintf(fh, "%% ty %d\n", iobj->w.mark->type);
    fprintf(fh, "%% sz %g\n", iobj->w.mark->size);
    break;
   
  case IPE_ARC:
    fprintf(fh, "%% Arc\n");
    write_colors(iobj);
    write_linestyle(iobj);
#ifdef IUM_PLAGEO
    fprintf(fh, "%% xy %g %g\n",
	    iobj->w.arc->arc.center().x(), iobj->w.arc->arc.center().y());
    fprintf(fh, "%% r %g\n", iobj->w.arc->arc.radius());
    fprintf(fh, "%% ang %g %g\n",
	    (iobj->w.arc->arc.small_angle() / M_PI * 180.0),
	    (iobj->w.arc->arc.big_angle() / M_PI * 180.0));
#else
    fprintf(fh, "%% xy %g %g\n",
	    iobj->w.arc->center.x, iobj->w.arc->center.y);
    fprintf(fh, "%% r %g\n", iobj->w.arc->radius);
    fprintf(fh, "%% ang %g %g\n",
	    (iobj->w.arc->begangle / M_PI * 180.0),
	    (iobj->w.arc->endangle / M_PI * 180.0));
#endif
    fprintf(fh, "%% ar %d %g\n", iobj->w.arc->arrow, iobj->w.arc->arsize);
    break;

  case IPE_TEXT:
    fprintf(fh, "%% Text\n");
    iobj->fill.red = -1;
    write_colors(iobj);
    fprintf(fh, "%% xy %g %g\n",
	    iobj->w.text->pos.XCOORD, iobj->w.text->pos.YCOORD);
    fprintf(fh, "%% f %d %g\n", iobj->w.text->font, iobj->w.text->fontsize);
    fprintf(fh, "%% s ");
    for (p = iobj->w.text->str; *p; p++) {
      if (*p == '\n') {
	fprintf(fh, "\n%% s ");
      } else {
	fputc(*p, fh);
      }
    }
    fprintf(fh, "\n");
    if (iobj->w.text->minipage) {
      fprintf(fh, "%% bb %g %g\n",
	      (iobj->w.text->ur.XCOORD - iobj->w.text->ll.XCOORD),
	      (iobj->w.text->ur.YCOORD - iobj->w.text->ll.YCOORD));
    }
    break;

  case IPE_BITMAP:
    fprintf(fh, "%% Bitmap\n");
    fprintf(fh, "%% xy %g %g\n",
	    iobj->w.bitmap->ll.XCOORD, iobj->w.bitmap->ll.YCOORD);
    fprintf(fh, "%% bb %g %g\n",
	    iobj->w.bitmap->ur.XCOORD - iobj->w.bitmap->ll.XCOORD,
	    iobj->w.bitmap->ur.YCOORD - iobj->w.bitmap->ll.YCOORD);
    /* write bitmap in raw format */
    fprintf(fh, "%% px %d %d\n%% bits %ld %d\n",
	    iobj->w.bitmap->width, iobj->w.bitmap->height,
	    iobj->w.bitmap->width * iobj->w.bitmap->height,
	    iobj->w.bitmap->in_color + 8);
    if (iobj->w.bitmap->in_color) {
      /* write a color bitmap: 32 bits per pixel */
      fwrite((BITS *) iobj->w.bitmap->words, sizeof(unsigned long),
	     iobj->w.bitmap->width * iobj->w.bitmap->height, fh);
    } else {
      /* gray bitmap: 8 bits per pixel */
      char *pix = NEWARRAY(char,
			   iobj->w.bitmap->width * iobj->w.bitmap->height);
      register char *out;
      register unsigned long *inp, *end;
      inp = iobj->w.bitmap->words;
      end = inp + iobj->w.bitmap->width * iobj->w.bitmap->height;
      out = pix;
      while (inp < end)
	*out++ = ((char) *inp++);
      fwrite((BITS *) pix, sizeof(char),
	     iobj->w.bitmap->width * iobj->w.bitmap->height, fh);
      FREEARRAY(pix);
    }
    fprintf(fh, "\n\n");
    break;
    
  default:
    /* this should never happen */
    fprintf(stderr, "IUM failed: trying to write unknown type %d\n",
	    iobj->type);
    exit(1);
  }
  fprintf(fh, "%% End\n");
  return;
}

/* finish IUM: write interface file, and exit to Ipe */

void ium_end(void)
{
  IpeObject *iobj;
  
  /* write file for Ipe */

  if (!(fh = fopen(fname, "w"))) {
    fprintf(stderr,
	    "IUM failed: cannot open interface file %s for writing\n", fname);
    exit(9);
  }

  fprintf(fh, "%%\\IUMid{4.1}\n%% Interface %d %s\n",
	  ium_mode, ium_message ? ium_message : "");
  for (iobj = ium_output; iobj; iobj = iobj->next) {
    write_entry(iobj);
  }
  fprintf(fh, "%% End\n");
  
  if (fclose(fh) == EOF) {
    fprintf(stderr,
	    "IUM failed: write error on interface file %s\n", fname);
    exit(9);
  }
  write_iumrc();
  exit(0);
}

