/*
 * spline.C
 * 
 * Ipe User Macros to implement some spline functions
 * currently provides parabolas, defined by points and lines
 *
 * $Modified: Sunday, September 11, 1994 by otfried $
 *
 * This file is part of the extendible drawing editor Ipe
 * Copyright (C) 1994 Otfried Schwarzkopf
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *    
 * A copy of the GNU General Public License is available on the World
 * Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
 * You can also obtain it by writing to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <plageo.h>

#define IUM_PLAGEO
#include "ium.h"

static void add_parabola(IpeObject* &last, pl_rotra& tfm,
			 pl_vec p, s_coord xmax)
// compute the parabola defined by mark at p     
{
  p = tfm.inverse() * p;
  s_coord a = -p.x();
  s_coord b = xmax - p.x();

  // the following are the three control points for the unit parabola
  // between x = a and x = b
  pl_vec q0(1.5*a - 0.5*b, 2*a*a - a*b);
  pl_vec q1(0.5*(a + b), a*b);
  pl_vec q2(1.5*b - 0.5*a, 2*b*b - a*b);
  
  // now stretch in y-direction and move
  s_coord stretch = 2.0 * p.y();
  pl_vec offs(p.x(), p.y() / 2.0);
  q0.set_y(q0.y() / stretch);
  q1.set_y(q1.y() / stretch);
  q2.set_y(q2.y() / stretch);
  q0 += offs;
  q1 += offs;
  q2 += offs;
  
  // and transform back to position of line segment
  q0 = tfm * q0;
  q1 = tfm * q1;
  q2 = tfm * q2;

  // and make parabola
  last->next = new IpeObject;
  last = last->next;
  last->next = NULL;
  last->type = IPE_SPLINE;
  last->linestyle = ipe_environment.linestyle;
  last->linewidth = ipe_environment.linewidth;
  last->stroke = ipe_environment.stroke;
  last->fill.red = -1;
  last->w.line = new Line;
  last->w.line->closed = 0;
  last->w.line->arrow = 0;
  last->w.line->v.newsize(3);
  last->w.line->v[0] = q0;
  last->w.line->v[1] = q1;
  last->w.line->v[2] = q2;
}


static void make_parabolas(void)
// make parabolas defined by line in primary selection and marks
{
  for (IpeObject *ob = ium_input; ob && !ob->primary; ob = ob->next)
    ;
  if (!ob || (ob->type != IPE_LINE) || (ob->w.line->closed) ||
      (ob->w.line->v.size() != 2)) {
    ium_message = "primary selection must be a line segment";
    return;
  }

  // this transformation maps the positive x-axis on the segment
  
  pl_rotra tfm(angle_of(ob->w.line->v[1] - ob->w.line->v[0]),
	       ob->w.line->v[0]);
  s_coord xmax = clearance(ob->w.line->v[0], ob->w.line->v[1]);
  
  // now walk through selection, and compute parabolas

  ium_output = new IpeObject;
  ium_output->type = IPE_BEGINGROUP;
  ium_output->next = NULL;
  
  IpeObject *last = ium_output;
  pl_vec p;

  for (ob = ium_input; ob; ob = ob->next) {
    if (!ob->primary) {
      switch (ob->type) {
      case IPE_BEGINGROUP:
      case IPE_ENDGROUP:
	break;
      case IPE_MARK:
	p = ob->w.mark->pos;
	// compute parabola
	add_parabola(last, tfm, p, xmax);
	break;
      default:
	ium_message = "cannot treat anything but marks";
	ium_output = NULL;
	return;
      }
    }
  }
  if (last == ium_output) {
    ium_message = "nothing computed";
    return;
  }
  if (last == ium_output->next) {
    // only one parabola, ungroup
    ium_output = last;
    last->next = NULL;
  } else {
    last->next = new IpeObject;
    last = last->next;
    last->next = NULL;
    last->type = IPE_ENDGROUP;
  }
  ium_message = "computed parabolas";
  ium_mode = IUM_SELECT_NEW;
}

int main(int argc, char **argv)
{
  ium_begin(argc, argv);
  
  switch (atoi(ium_argument)) {

  case 0:
    // make parabolas defined by points and line
    make_parabolas();
    break;
    
  default:
    // illegal argument
//  cerr << "IUM failed: Illegal argument " << argv[2] << "\n";
    exit(1);
  }
  ium_end();
}
