#include <plageo.h>

const pl_unitvec pl_xdir = normalised(pl_fixedvec(1, 0));
const pl_unitvec pl_ydir = normalised(pl_fixedvec(0, 1));

void pl_angle::normalise(s_coord lowlimit)
{
    while (_a >= lowlimit+2*M_PI)
	_a -= 2*M_PI;
    while (_a < lowlimit)
	_a += 2*M_PI;
}

s_coord pl_angle::normalised_value(s_coord lowlimit) const
{
    s_coord result = _a;
    while (result >= lowlimit+2*M_PI)
	result -= 2*M_PI;
    while (result < lowlimit)
	result += 2*M_PI;
    return result;
}

int pl_angle::lies_between(pl_angle small, pl_angle large) const
{
    large.normalise(value());
    small.normalise(large - 2*M_PI);
    return (value() >= small);
}

int compare(const pl_fixedvec &a, const pl_fixedvec &b)
{
    if (a.x()<b.x())
	return -1;
    if (a.x()>b.x())
	return 1;
    if (a.y()<b.y())
	return -1;
    if (a.y()>b.y())
	return 1;
    return 0;
}

pl_vec to_polar(const pl_fixedvec &xy)
{
    pl_vec result;
    result.set_x(hypot_pp(xy.x(), xy.y()));
    result.set_y(atan2_pp(xy.y(), xy.x()));
    return result;
}

pl_vec from_polar(const pl_fixedvec &rtheta)
{
    pl_vec result;
    result.set_x(rtheta.x()*cos_pp(rtheta.y()));
    result.set_y(rtheta.x()*sin_pp(rtheta.y()));
    return result;
}


pl_unitvec normalised(const pl_fixedvec &v)
{
    s_coord l = v.length();
    if (l == 0)
	return pl_unitvec();
    else
	if (l == 1)
	    return pl_unitvec(v);
	else
	    return  pl_unitvec((1/l)*v);
}

s_coord factorise(pl_unitvec &uv, const pl_fixedvec &vec)
{
    s_coord l = vec.length();
    if (l == 0)
	uv = pl_unitvec();
    else
	if (l == 1)
	    uv = pl_unitvec(vec);
	else
	    uv = pl_unitvec((1/l)*vec);
    return l;
}

pl_unitvec::pl_unitvec(pl_angle angle)
    : pl_fixedvec(cos_pp(angle.value()), sin_pp(angle.value()))
{ }

pl_angle angle_between(const pl_unitvec &uv1, const pl_unitvec &uv2)
{
    return atan2_pp(cross(uv1, uv2), dot(uv1, uv2));
}

pl_angle angle_of(const pl_fixedvec &vec)
{
    return atan2_pp(vec.y(), vec.x());
}

void pl_object::rotate(pl_angle theta)
{
    transform(pl_rotation(theta));
}

void pl_object::translate(const pl_fixedvec &vec)
{
    transform(pl_translation(vec));
}

void pl_object::scale(s_coord fac)
{
    transform(pl_uniscale(fac));
}

pl_object::~pl_object()
{ }

pl_object_type pl_point::type() const
{
    return PL_POINT;
}

pl_box pl_point::bbox() const
{
    return pl_box(_vec);
}

int lies_inside(const pl_box &innerbb, const pl_box &outerbb)
{
    if (innerbb.is_empty()) return 1;
    if (outerbb.is_empty()) return 0;
    if (    innerbb.xmin() >= outerbb.xmin() &&
	    innerbb.xmax() <= outerbb.xmax() &&
	    innerbb.ymin() >= outerbb.ymin() &&
	    innerbb.ymax() <= outerbb.ymax() )
	return 1;
    else
	return 0;
}

void pl_point::scale(s_coord fac)
{
    _vec *= fac;
}

void pl_point::translate(const pl_fixedvec &disp)
{
    _vec += disp;
}

void pl_point::transform(const pl_transform &tfm)
{
    _vec = tfm * _vec;
}

s_coord pl_boundingbox::area() const
{
    if (is_empty())
	return 0;
    return (xmax()-xmin())*(ymax()-ymin());
}

void pl_boundingbox::scale(s_coord fac)
{
    if (is_empty())
	return ;
    max.set_x(fac*max.x());
    max.set_y(fac*max.y());
    min.set_x(fac*min.x());
    min.set_y(fac*min.y());
}

void pl_boundingbox::add_point(const pl_fixedvec &c)
{
    if (is_empty()) {
	max = c;
	min = c;
    } else {
	add_non_first_point(c);
    }
}


void pl_boundingbox::add_non_first_point(const pl_fixedvec &c)
{
    if (c.x() > max.x())
	max.set_x(c.x());
    else if (c.x() < min.x())
	min.set_x(c.x());
    if (c.y() > max.y())
	max.set_y(c.y());
    else if (c.y() < min.y())
	min.set_y(c.y());
}

void pl_boundingbox::translate(const pl_fixedvec &disp)
{
    if (is_empty())
	return ;
    max.set_x(max.x() + disp.x());
    min.set_x(min.x() + disp.x());
    max.set_y(max.y() + disp.y());
    min.set_y(min.y() + disp.y());
}

// take the union of two bounding boxes
pl_boundingbox operator+(pl_boundingbox const & bb1,
			    pl_boundingbox const & bb2)
{
    pl_boundingbox     result;
    if (bb1.is_empty())
	return bb2;
    if (bb2.is_empty())
	return bb1;
    if (bb1.min.x() < bb2.min.x())
	result.min.set_x(bb1.min.x());
    else
	result.min.set_x(bb2.min.x());
    if (bb1.max.x() > bb2.max.x())
	result.max.set_x(bb1.max.x());
    else
	result.max.set_x(bb2.max.x());
    if (bb1.min.y() < bb2.min.y())
	result.min.set_y(bb1.min.y());
    else
	result.min.set_y(bb2.min.y());
    if (bb1.max.y() > bb2.max.y())
	result.max.set_y(bb1.max.y());
    else
	result.max.set_y(bb2.max.y());
    return result;
}

void pl_boundingbox::operator+=(const pl_boundingbox &bb)
{
    if (bb.is_empty())
	return;
    if (is_empty()) {
	*this = bb;
	return;
    }
    if (bb.min.x() < min.x())
	min.set_x(bb.min.x());
    if (bb.max.x() > max.x())
	max.set_x(bb.max.x());
    if (bb.min.y() < min.y())
	min.set_y(bb.min.y());
    if (bb.max.y() > max.y())
	max.set_y(bb.max.y());
}
