#include <plageo.h>


static s_coord big = max_of_type(s_coord(0));
static s_coord littl = min_of_type(s_coord(0));

static void pgn_out_of_box(pl_pgn &pgn, const pl_box &box)
{
// assert: !box.is_empty()
    static pl_vecreray vrr(4);
    vrr[0] = pl_vec(box.xmax(), box.ymax());
    vrr[1] = pl_vec(box.xmin(), box.ymax());
    vrr[2] = pl_vec(box.xmin(), box.ymin());
    vrr[3] = pl_vec(box.xmax(), box.ymin());
    pgn = pl_pgn(vrr);
}

static s_coord signed_dist(const pl_vec &vt, const pl_line &line)
{
    return dot(vt - line.ref_vt(), line.normal());
}


// take the intersection of two bounding boxes
pl_boundingbox intersect(pl_boundingbox const & bb1,
			    pl_boundingbox const & bb2)
{
    pl_boundingbox     result;
    if (bb1.min.x() > bb2.min.x())
	result.min.set_x(bb1.min.x());
    else
	result.min.set_x(bb2.min.x());
    if (bb1.max.x() < bb2.max.x())
	result.max.set_x(bb1.max.x());
    else
	result.max.set_x(bb2.max.x());
    if (bb1.min.y() > bb2.min.y())
	result.min.set_y(bb1.min.y());
    else
	result.min.set_y(bb2.min.y());
    if (bb1.max.y() < bb2.max.y())
	result.max.set_y(bb1.max.y());
    else
	result.max.set_y(bb2.max.y());
    if (result.max.y() < result.min.y()) {
	result.make_empty();
    }
    return result;
}


// ***************************

static int intersect(volatile s_coord &lam, const pl_line &l, const pl_line &m)
{
    const pl_unitvec &dir2 = m.direction();
    s_coord denom =  cross(dir2, l.direction());
    lam = cross((l.ref_vt() - m.ref_vt()), dir2) / denom;
    // now test on overflow. If lam is Infinity
    s_coord zero = lam-lam;	// then lam - lam is Not a Number (NaN)
    return zero == zero;	// and only a NaN is not equal to itself
}

// do two lines intersect ? if so return intersection-point
int intersect(pl_vec &result, const pl_line &l, const pl_line &m)
{
    s_coord lam;
    if (intersect(lam, l, m)) {
	result = l.placevec(lam);
	return 1;
    }
    return 0;
}

int intersect(pl_vec &result, const pl_ray &ray, const pl_line &line)
{
    s_coord lam;
    const pl_line &l = ray.supporting_line();
    if (intersect(lam, l, line)) {
	if (lam < 0)
	    return 0;
	result = l.placevec(lam);
	return 1;
    }
    return 0;
}

int intersect(pl_vec &result, const pl_edge &edge, const pl_line &line)
{
    s_coord lam;
    const pl_line &l = edge.supporting_line();
    if (intersect(lam, l, line)) {
	if (lam < 0 || lam > edge.length())
	    return 0;
	result = l.placevec(lam);
	return 1;
    }
    return 0;
}

int intersect(pl_vec &witness1, pl_vec &witness2, 
		const pl_disc &circ, const pl_line &line)
// witness1 is the point where the line enters the disc.
// witness2 is the point where the line leaves the disc.
{
    pl_vec p;    // projection
    s_coord sqd, sqx, x;
    pl_vec center(circ.center());
    int result = 0;
    p = placeproject(center, line);
    sqd = (p - center).sqlength();
    s_coord sqradius = circ.radius() * circ.radius();
    sqx = sqradius-sqd;
    if (sqx < 0)   // no intersection between line and disc
	return 0;
    x = sqrt_pp(sqx);
    witness1 = p + (-x)*line.direction();
    witness2 = p + x*line.direction();
    return 3;
}

int intersect(pl_vec &witness1, pl_vec &witness2, 
		const pl_disc &disc, const pl_ray &ray)
{
    pl_vec p;    // projection
    pl_line line = ray.supporting_line();
    int result;
    result = intersect(witness1, witness2, disc, line);
    if (result == 0)
	return 0;
    result = 0;
    if (dot(witness1-(ray.begin_vt()), line.direction()) >= 0) {
	result++;
    }
    if (dot(witness2-(ray.begin_vt()), line.direction()) >= 0) {
	result += 2;
    }
    return result;
}

int intersect(pl_vec &witness1, pl_vec &witness2, 
		const pl_disc &disc, const pl_edge &edge)
{
    pl_vec p;    // projection
    pl_line line = edge.supporting_line();
    int result;
    result = intersect(witness1, witness2, disc, line);
    if (result == 0)
	return 0;
    result = 0;
    if (dot(witness1-(edge.begin_vt()), line.direction()) >= 0 &&
	    dot((edge.end_vt())-witness1, line.direction()) >= 0) {
	result++;
    }
    if (dot(witness2-(edge.begin_vt()), line.direction()) >= 0 &&
	    dot((edge.end_vt())-witness2, line.direction()) >= 0) {
	result += 2;
    }
    return result;
}

int intersect(pl_vec &witness1, pl_vec &witness2, 
		const pl_edge &edge, const pl_arc &arc)
{
    pl_vec p;    // projection
    pl_line line = edge.supporting_line();
    pl_angle startangle, endangle;
    int result;
    result = intersect(witness1, witness2, arc.supporting_disc(), line);
    if (result == 0)
	return 0;
    startangle = arc.small_angle();
    endangle = arc.big_angle();
    result = 0;
    if (dot(witness1-(edge.begin_vt()), line.direction()) >= 0 &&
	    dot((edge.end_vt())-witness1, line.direction()) >= 0) {
	pl_angle theta = angle_of(witness1 - arc.center());
	if (theta.lies_between(startangle, endangle))
	    result += 1;
    }
    if (dot(witness2-(edge.begin_vt()), line.direction()) >= 0 &&
	    dot((edge.end_vt())-witness2, line.direction()) >= 0) {
	pl_angle theta = angle_of(witness2 - arc.center());
	if (theta.lies_between(startangle, endangle))
	    result += 2;
    }
    return result;
}

int intersect(pl_vec &witness1, pl_vec &witness2, 
		const pl_ray &ray, const pl_arc &arc)
{
    pl_vec p;    // projection
    pl_line line = ray.supporting_line();
    pl_angle startangle, endangle;
    int result;
    result = intersect(witness1, witness2, arc.supporting_disc(), line);
    if (result == 0)
	return 0;
    startangle = arc.small_angle();
    endangle = arc.big_angle();
    result = 0;
    if (dot(witness1-(ray.begin_vt()), line.direction()) >= 0) {
	pl_angle theta = angle_of(witness1 - arc.center());
	if (theta.lies_between(startangle, endangle))
	    result += 1;
    }
    if (dot(witness2-(ray.begin_vt()), line.direction()) >= 0) {
	pl_angle theta = angle_of(witness2 - arc.center());
	if (theta.lies_between(startangle, endangle))
	    result += 2;
    }
    return result;
}

int intersect(pl_vec &witness1, pl_vec &witness2, 
		const pl_line &line, const pl_arc &arc)
{
    pl_vec p;    // projection
    pl_angle startangle, endangle, theta;
    int result;
    result = intersect(witness1, witness2, arc.supporting_disc(), line);
    if (result == 0)
	return 0;
    startangle = arc.small_angle();
    endangle = arc.big_angle();
    result = 0;
    theta = angle_of(witness1 - arc.center());
    if (theta.lies_between(startangle, endangle))
	result += 1;
    theta = angle_of(witness2 - arc.center());
    if (theta.lies_between(startangle, endangle))
	result += 2;
    return result;
}

int intersect(pl_vec &witness1, pl_vec &witness2, 
		const pl_disc &disc1, const pl_disc &disc2)
{
    pl_vec joinvec;
    pl_unitvec joindir;
    s_coord l, d, h, sqrd;
    joinvec = disc2.center()-disc1.center();
    l = factorise(joindir, joinvec);
    if (l==0) {
// ??????? als disc1.radius() == disc2.radius() dan vallen samen ???????
	return 0;
    }
    if (l >= disc1.radius() + disc2.radius())
	return 0;
    if (disc1.radius() > l + disc2.radius())
	return 0;
    if (disc2.radius() > l + disc1.radius())
	return 0;
    s_coord sqrad1 = disc1.radius() * disc1.radius();
    s_coord sqrad2 = disc2.radius() * disc2.radius();
    d = (l*l+sqrad1-sqrad2)/(2*l);
    sqrd = sqrad1-d*d;
    h = (sqrd<=0) ? 0 : sqrt_pp(sqrd);
    witness1 = disc1.center() + d*joindir + h*joindir.normal();
    witness2 = disc1.center() + d*joindir - h*joindir.normal();
    return 3;
}

int intersect(pl_vec &wn1, pl_vec &wn2, const pl_disc &disc, const pl_arc &arc)
{
    if (!intersect(wn1, wn2, disc, arc.supporting_disc()))
	return 0;
    pl_angle angle;
    int result = 0;
    angle = angle_of(wn1-arc.center());
    if (angle.lies_between(arc.small_angle(), arc.big_angle())) {
	result++;
    }
    angle = angle_of(wn2-arc.center());
    if (angle.lies_between(arc.small_angle(), arc.big_angle())) {
	result += 2;
    }
    return result;
}

int intersect(pl_vec &wn1, pl_vec &wn2, const pl_arc &arc1, const pl_arc &arc2)
{
    if (!intersect(wn1, wn2, arc1.supporting_disc(), arc2.supporting_disc()))
	return 0;
    pl_angle angle1, angle2;
    int result = 0;
    angle1 = angle_of(wn1-arc1.center());
    angle2 = angle_of(wn1-arc2.center());
    if (angle1.lies_between(arc1.small_angle(), arc1.big_angle()) &&
	    angle2.lies_between(arc2.small_angle(), arc2.big_angle())) {
	result++;
    }
    angle1 = angle_of(wn2-arc1.center());
    angle2 = angle_of(wn2-arc2.center());
    if (angle1.lies_between(arc1.small_angle(), arc1.big_angle()) &&
	    angle2.lies_between(arc2.small_angle(), arc2.big_angle())) {
	result += 2;
    }
    return result;
}

intervals intersect(const pl_pgn &pgn, const pl_line &line)
{
    int i;
    pl_vec isvt;
    intervals result;	// result starts empty
    if (clearance(pgn.bbox(), line) > 0)
	return result;		// empty
    for (i=0; i<pgn.size(); i++) {
	if (intersect(isvt, pgn.edge(i), line)) {
	    s_coord lam = dot(isvt-line.ref_vt(), line.direction());
	    result.toggle_after(lam);
	}
    }
    return result;
}

intervals intersect(const pl_pgn &pgn, const pl_ray &ray)
{
    intervals fiv1;
    fiv1.toggle_after(0.0);
    intervals fiv2(intersect(pgn, pl_line(ray.begin_vt(), ray.direction())));
    return fiv1 & fiv2;
}


/***** 2x2 Matrix routines  *****/
// Matrix should be class

inline int ind2(int i, int j) { return 2*i + j; }

static s_coord determinant(s_coord M[4])
{
    return M[ind2(0,0)] * M[ind2(1,1)] - M[ind2(0,1)] * M[ind2(1,0)];
}

static int invertmatrix(s_coord  inverse[4], s_coord M[4])
{
    s_coord det = determinant(M);
    s_coord zero;
    volatile register s_coord temp;
    if (det == 0)
	return 0;
    temp =   M[ind2(1,1)] / det;
    zero = temp-temp; if (zero != zero) return 0;
    inverse[ind2(0,0)] =   temp;
    temp = - M[ind2(0,1)] / det;
    zero = temp-temp; if (zero != zero) return 0;
    inverse[ind2(0,1)] = temp;
    temp = - M[ind2(1,0)] / det;
    zero = temp-temp; if (zero != zero) return 0;
    inverse[ind2(1,0)] = temp;
    temp =   M[ind2(0,0)] / det;
    zero = temp-temp; if (zero != zero) return 0;
    inverse[ind2(1,1)] =   temp;
    return 1;
}


// gegeven een vector v en drie onafhankelijke vectoren e0, e1 en e2,
// vind factoren fac zo dat v == fac[0]*e0 + fac[1]*e1 + fac[2]*e2.
int pl__ontbind2(s_coord fac[], const pl_vec &e0, const pl_vec &e1,
                 const pl_vec &v)
{
    s_coord M[4];
    s_coord invM[4];
// initialiseer matrix M
    M[ind2(0,0)] = e0.x();
    M[ind2(0,1)] = e1.x();
    M[ind2(1,0)] = e0.y();
    M[ind2(1,1)] = e1.y();
    if (!invertmatrix(invM, M))
	return 0;
    fac[0] = invM[ind2(0,0)]*v.x() + invM[ind2(0,1)]*v.y();
    fac[1] = invM[ind2(1,0)]*v.x() + invM[ind2(1,1)]*v.y();
    return 1;
}



int intersect(pl_vec &result, const pl_ray &ray0, const pl_ray &ray1)
{
    pl_unitvec joindir;
    pl_unitvec e0=ray0.direction(), e1=ray1.direction();
    pl_vec joinvec = ray1.begin_vt() - ray0.begin_vt();
    if ( cross(e0,e1) == 0.0F) {
        // parallel lines
	return 0;
    } else {
        s_coord eta[2];
        pl__ontbind2(eta, e0, e1, joinvec); // ?????? can probably fail (overflow)
        if (eta[0] >= 0 && eta[1] <= 0) {
	    result = ray0.begin_vt() + eta[0] * e0;
            return 1;
	}
	return 0;
    }
}

int intersect(pl_vec &result, const pl_edge & edge, const pl_ray & ray)
{
    pl_boundingbox bb = edge.bbox();
    // look if bounding boxes overlap.
    if (ray.direction().x() >= 0 && ray.begin_vt().x() > bb.xmax())
        return 0;
    if (ray.direction().x() <= 0 && ray.begin_vt().x() < bb.xmin())
        return 0;
    if (ray.direction().y() >= 0 && ray.begin_vt().y() > bb.ymax())
        return 0;
    if (ray.direction().y() <= 0 && ray.begin_vt().y() < bb.ymin())
        return 0;
    // compute intersection of supporting lines.
    if (!intersect(result, ray, edge.supporting_line()))
        return 0;
    // decide if intersectionpoint lies on edge.
    if (bb.xmax()-bb.xmin() > bb.ymax()-bb.ymin()) {
        if (result.x() > bb.xmax() || result.x() < bb.xmin())
            return 0;
    } else {
        if (result.y() > bb.ymax() || result.y() < bb.ymin())
            return 0;
    }
    // intersectionpoint lies on both.
    return 1;
}

// do two edges intersect ? If so, return intersection-point
int intersect(pl_vec &result, const pl_edge & e1, const pl_edge & e2)
{
    pl_boundingbox bb1 = e1.bbox();
    pl_boundingbox bb2 = e2.bbox();
    // look if bounding boxes overlap.
    if (!do_overlap(bb1,bb2))
        return 0;
    // compute intersection of supporting lines.
    if (!intersect(result, e1.supporting_line(), e2.supporting_line()))
        return 0;
    // decide if intersectionpoint lies on e1.
    if (bb1.xmax()-bb1.xmin() > bb1.ymax()-bb1.ymin()) {
        if (result.x() > bb1.xmax() || result.x() < bb1.xmin())
            return 0;
    } else {
        if (result.y() > bb1.ymax() || result.y() < bb1.ymin())
            return 0;
    }
    // decide if intersectionpoint lies on e2.
    if (bb2.xmax()-bb2.xmin() > bb2.ymax()-bb2.ymin()) {
        if (result.x() > bb2.xmax() || result.x() < bb2.xmin())
            return 0;
    } else {
        if (result.y() > bb2.ymax() || result.y() < bb2.ymin())
            return 0;
    }
    // intersectionpoint lies on both.
    return 1;
}

// ***************************

struct edgelist_link {
    edgelist_link	*next;		// next element of list
    int		end,		// the index of the endpoint vertex
		id,		// the ident of the polygon (1 or 2)
		side;		// this edge is an upside or downside of pgn
    pl_line	line;		// supporting line of edge
};

typedef edgelist_link *edgelistp;

static const    up = 0;
static const    down = 1;

// sorted list of edges intersecting current sweep line
static edgelistp el = 0;
static edgelistp freelist = 0;

static edgelistp new_edgelist()
{
    if (freelist == 0)
	return new edgelist_link;
    edgelistp result;
    result = freelist;
    freelist = freelist->next;
    return result;
}

static void delete_edgelist(edgelistp elist)
{
    elist->next = freelist;
    freelist = elist;
}

static void clear_edgelist()
{
    edgelistp tmp;
    while (el != NULL) {
	tmp = el;
	el = el->next;
	delete_edgelist(tmp);
    }
}

/*
edgelistp *search_edgelist(edgelistp *lp, const pl_vec &vt)
{
    while (*lp != NULL && is_below(vt, (*lp)->line))
	lp = &((*lp)->next);
    return lp;
}
*/

edgelistp *search_edgelist(edgelistp *lp, const pl_vec &vt)
{
    int n = 1;
    edgelistp *ep = lp;
    while (*ep != NULL) {
	ep = &((*ep)->next);
	n++;
    }
// binary search on insertion points
    while (n>1) {
	int n1 = n/2;
	int i;
	for (i=1, ep=lp; i<n1; i++)
	    ep = &((*ep)->next);
	if (is_below(vt, (*ep)->line)) {
	    lp = &((*ep)->next); n = n-n1;
	} else {
	    n = n1;
	}
    }
    return lp;
}


// handle a vertex encountered by the sweep line
// check if overlap takes place
static int overlap_event(const pl_pgn & pg, int pol_id, int vnum)
{
    edgelistp       cur, prev;
    int             nb1, nb2;
    int             vertextype = 0;
    pl_edge	    e;
    pl_vec	    vec;
// nb1 and nb2 are the vertices preceding and succeedeing 'vnum' in 'pg'
    nb1 = vnum == 0 ? pg.size() - 1 : vnum - 1;
    nb2 = vnum == pg.size() - 1 ? 0 : vnum + 1;
// decide what type of vertex 'vnum' is
// if both edges go to right then insertionpoint (type 0)
// if both edges come from the left then deletionpoint (type 3)
// otherwise a transitionpoint (type 1 or 2)
    if (pg.order(nb1) < pg.order(vnum))
	vertextype++;
    if (pg.order(nb2) < pg.order(vnum))
	vertextype += 2;
    switch (vertextype) {
    case 0:
    // vnum is insertionpoint
	edgelistp *lp;
	edgelistp       e1, e2;
    // search between which edges 'vnum' lies
/*
	for (lp = &el;
	     *lp != NULL && is_below(pg.vertex(vnum), (*lp)->line);
	     lp = &((*lp)->next));	// EMPTY STATEMENT
*/
	lp = search_edgelist(&el, pg.vertex(vnum));
    // check if vnum lies inside the other polygon
	if (*lp != NULL && (*lp)->side == down && (*lp)->id != pol_id)
	    return 1;			// overlap found
    // insert the two edges starting in vertex vnum
	e1 = new_edgelist();
	e2 = new_edgelist();
	e2->next = *lp;
	e1->next = e2;
	*lp = e1;
	e1->id = e2->id = pol_id;
	if (e2->next == NULL || e2->next->side == up) {
	    e1->side = up;
	    e2->side = down;
	} else {
	    e1->side = down;
	    e2->side = up;
	}
	if (cross(pg.vertex(nb1) - pg.vertex(vnum),
		pg.vertex(nb2) - pg.vertex(vnum)) < 0) {
	// vnum-nb1 lies above vnum-nb2
	    e1->line = pl_line(pg.vertex(nb1), pg.direction(nb1));
	    e1->end = nb1;
	    e2->line = pl_line(pg.vertex(vnum), pg.direction(vnum));
	    e2->end = nb2;
	} else {	
	// vnum-nb1 lies below vnum-nb2
	    e1->line = pl_line(pg.vertex(vnum), pg.direction(vnum));
	    e1->end = nb2;
	    e2->line = pl_line(pg.vertex(nb1), pg.direction(nb1));
	    e2->end = nb1;
	}
	break;
    case 1:
    // vnum is transitionpoint: nb1->vnum->nb2
    // zoek in lijst tot je edge met vnum als eindpunt vindt.
	cur = el;
	prev = NULL;
	while (cur->end != vnum || cur->id != pol_id) {
	    prev = cur;
	    cur = cur->next;
	}
    // als dit een upper edge is, check dan of is_below(vnum,bovenbuur)
    // anders check of above(vnum,benedenbuur)
    // als check faalt return 1;(overlap)
	if (cur->side == up) {
	    if (prev != NULL && is_above(pg.vertex(vnum), prev->line))
		return 1;
	} else {
	    if (cur->next != NULL && is_below(pg.vertex(vnum), cur->next->line))
		return 1;
	}
    // vervang edge door edge vnum->nb2;
	cur->end = nb2;
	cur->line = pl_line(pg.vertex(vnum), pg.direction(vnum));
	break;
    case 2:	
    // vnum is transitionpoint: nb2->vnum->nb1
	cur = el;
	prev = NULL;
	while (cur->end != vnum || cur->id != pol_id) {
	    prev = cur;
	    cur = cur->next;
	}
	if (cur->side == up) {
	    if (prev != NULL && is_above(pg.vertex(vnum), prev->line))
		return 1;
	} else {
	    if (cur->next != NULL && is_below(pg.vertex(vnum), cur->next->line))
		return 1;
	}
	cur->end = nb1;
	cur->line = pl_line(pg.vertex(nb1), pg.direction(nb1));
	break;
    case 3:
    // vnum is deletionpoint
    // zoek in lijst tot je edge met vnum als eindpunt vindt.
	cur = el;
	prev = NULL;
	while (cur->end != vnum || cur->id != pol_id) {
	    prev = cur;
	    cur = cur->next;
	}
    // check of benedenbuur ook vnum als eindpunt heeft
	if (cur->next->end != vnum || cur->next->id != pol_id)
	    return 1;
	edgelistp       next;
	next = cur->next->next;
    // check of is_below(vnum,bovenbuur)
	if (prev != NULL && is_above(pg.vertex(vnum), prev->line))
	    return 1;
    // check of above(vnum,2*benedenbuur)
	if (next != NULL && is_below(pg.vertex(vnum), next->line))
	    return 1;
    // laat twee edges uit list weg
	delete_edgelist(cur->next);
	delete_edgelist(cur);
	if (prev == NULL)
	    el = next;
	else
	    prev->next = next;
	break;
    }
    return 0;
}

// ****** OBJECT ******

int do_cross(const pl_object &, const pl_fixedvec &)
{ return 0; }

int do_cross(const pl_object &, const pl_point &)
{ return 0; }

int do_cross(const pl_object &obj, const pl_line &line)
{
    switch (obj.type()) {
    case PL_POINT:
	return 0;
    case PL_LINE:
	return 1;
    case PL_RAY:
	return do_cross(line, *(pl_ray *) &obj);
    case PL_EDGE:
	return do_cross(line, *(pl_edge *) &obj);
    case PL_POLYGON:
	return do_cross(line, *(pl_pgn *) &obj);
    case PL_DISC:
	return do_cross(line, *(pl_disc *) &obj);
    case PL_ARC:
	return do_cross(line, *(pl_arc *) &obj);
    }
}

int do_cross(const pl_object &obj, const pl_ray &ray)
{
    switch (obj.type()) {
    case PL_POINT:
	return 0;
    case PL_LINE:
	return do_cross(ray, *(pl_line *) &obj);
    case PL_RAY:
	return do_cross(ray, *(pl_ray *) &obj);
    case PL_EDGE:
	return do_cross(ray, *(pl_edge *) &obj);
    case PL_POLYGON:
	return do_cross(ray, *(pl_pgn *) &obj);
    case PL_DISC:
	return do_cross(ray, *(pl_disc *) &obj);
    case PL_ARC:
	return do_cross(ray, *(pl_arc *) &obj);
    }
}

int do_cross(const pl_object &obj, const pl_edge &edge)
{
    switch (obj.type()) {
    case PL_POINT:
	return 0;
    case PL_LINE:
	return do_cross(edge, *(pl_line *) &obj);
    case PL_RAY:
	return do_cross(edge, *(pl_ray *) &obj);
    case PL_EDGE:
	return do_cross(edge, *(pl_edge *) &obj);
    case PL_POLYGON:
	return do_cross(edge, *(pl_pgn *) &obj);
    case PL_DISC:
	return do_cross(edge, *(pl_disc *) &obj);
    case PL_ARC:
	return do_cross(edge, *(pl_arc *) &obj);
    }
}

int do_cross(const pl_object &obj, const pl_arc &arc)
{
    switch (obj.type()) {
    case PL_POINT:
	return 0;
    case PL_LINE:
	return do_cross(arc, *(pl_line *) &obj);
    case PL_RAY:
	return do_cross(arc, *(pl_ray *) &obj);
    case PL_EDGE:
	return do_cross(arc, *(pl_edge *) &obj);
    case PL_POLYGON:
	return do_cross(arc, *(pl_pgn *) &obj);
    case PL_DISC:
	return do_cross(arc, *(pl_disc *) &obj);
    case PL_ARC:
	return do_cross(arc, *(pl_arc *) &obj);
    }
}

int do_cross(const pl_object &obj, const pl_box &box)
{
    switch (obj.type()) {
    case PL_POINT:
	return 0;
    case PL_LINE:
	return do_cross(box, *(pl_line *) &obj);
    case PL_RAY:
	return do_cross(box, *(pl_ray *) &obj);
    case PL_EDGE:
	return do_cross(box, *(pl_edge *) &obj);
    case PL_POLYGON:
	return do_cross(box, *(pl_pgn *) &obj);
    case PL_DISC:
	return do_cross(box, *(pl_disc *) &obj);
    case PL_ARC:
	return do_cross(box, *(pl_arc *) &obj);
    }
}

int do_cross(const pl_object &obj, const pl_pgn &pgn)
{
    switch (obj.type()) {
    case PL_POINT:
	return 0;
    case PL_LINE:
	return do_cross(pgn, *(pl_line *) &obj);
    case PL_RAY:
	return do_cross(pgn, *(pl_ray *) &obj);
    case PL_EDGE:
	return do_cross(pgn, *(pl_edge *) &obj);
    case PL_POLYGON:
	return do_cross(pgn, *(pl_pgn *) &obj);
    case PL_DISC:
	return do_cross(pgn, *(pl_disc *) &obj);
    case PL_ARC:
	return do_cross(pgn, *(pl_arc *) &obj);
    }
}

int do_cross(const pl_object &obj, const pl_disc &disc)
{
    switch (obj.type()) {
    case PL_POINT:
	return 0;
    case PL_LINE:
	return do_cross(disc, *(pl_line *) &obj);
    case PL_RAY:
	return do_cross(disc, *(pl_ray *) &obj);
    case PL_EDGE:
	return do_cross(disc, *(pl_edge *) &obj);
    case PL_POLYGON:
	return do_cross(disc, *(pl_pgn *) &obj);
    case PL_DISC:
	return do_cross(disc, *(pl_disc *) &obj);
    case PL_ARC:
	return do_cross(disc, *(pl_arc *) &obj);
    }
}

int do_cross(const pl_object &obj, const pl_object &obj2)
{
    switch (obj.type()) {
    case PL_POINT:
	return 0;
    case PL_LINE:
	return do_cross(obj2, *(pl_line *) &obj);
    case PL_RAY:
	return do_cross(obj2, *(pl_ray *) &obj);
    case PL_EDGE:
	return do_cross(obj2, *(pl_edge *) &obj);
    case PL_POLYGON:
	return do_cross(obj2, *(pl_pgn *) &obj);
    case PL_DISC:
	return do_cross(obj2, *(pl_disc *) &obj);
    case PL_ARC:
	return do_cross(obj2, *(pl_arc *) &obj);
    }
}


int do_overlap(const pl_object &obj, const pl_fixedvec &vec)
{
    switch (obj.type()) {
    case PL_POINT:
    case PL_LINE:
    case PL_RAY:
    case PL_EDGE:
    case PL_ARC:
	return 0;
    case PL_POLYGON:
	return do_overlap(vec, *(pl_pgn *) &obj);
    case PL_DISC:
	return do_overlap(vec, *(pl_disc *) &obj);
    }
}

int do_overlap(const pl_object &obj, const pl_point &vt)
{
    switch (obj.type()) {
    case PL_POINT:
    case PL_LINE:
    case PL_RAY:
    case PL_EDGE:
    case PL_ARC:
	return 0;
    case PL_POLYGON:
	return do_overlap(vt.vec(), *(pl_pgn *) &obj);
    case PL_DISC:
	return do_overlap(vt.vec(), *(pl_disc *) &obj);
    }
}

int do_overlap(const pl_object &obj, const pl_line &line)
{
    switch (obj.type()) {
    case PL_POINT:
	return 0;
    case PL_LINE:
	return do_overlap(line, *(pl_line *) &obj);
    case PL_RAY:
	return do_overlap(line, *(pl_ray *) &obj);
    case PL_EDGE:
	return do_overlap(line, *(pl_edge *) &obj);
    case PL_POLYGON:
	return do_overlap(line, *(pl_pgn *) &obj);
    case PL_DISC:
	return do_overlap(line, *(pl_disc *) &obj);
    case PL_ARC:
	return do_overlap(line, *(pl_arc *) &obj);
    }
}

int do_overlap(const pl_object &obj, const pl_ray &ray)
{
    switch (obj.type()) {
    case PL_POINT:
	return 0;
    case PL_LINE:
	return do_overlap(ray, *(pl_line *) &obj);
    case PL_RAY:
	return do_overlap(ray, *(pl_ray *) &obj);
    case PL_EDGE:
	return do_overlap(ray, *(pl_edge *) &obj);
    case PL_POLYGON:
	return do_overlap(ray, *(pl_pgn *) &obj);
    case PL_DISC:
	return do_overlap(ray, *(pl_disc *) &obj);
    case PL_ARC:
	return do_overlap(ray, *(pl_arc *) &obj);
    }
}

int do_overlap(const pl_object &obj, const pl_edge &edge)
{
    switch (obj.type()) {
    case PL_POINT:
	return 0;
    case PL_LINE:
	return do_overlap(edge, *(pl_line *) &obj);
    case PL_RAY:
	return do_overlap(edge, *(pl_ray *) &obj);
    case PL_EDGE:
	return do_overlap(edge, *(pl_edge *) &obj);
    case PL_POLYGON:
	return do_overlap(edge, *(pl_pgn *) &obj);
    case PL_DISC:
	return do_overlap(edge, *(pl_disc *) &obj);
    case PL_ARC:
	return do_overlap(edge, *(pl_arc *) &obj);
    }
}

int do_overlap(const pl_object &obj, const pl_arc &arc)
{
    switch (obj.type()) {
    case PL_POINT:
	return 0;
    case PL_LINE:
	return do_overlap(arc, *(pl_line *) &obj);
    case PL_RAY:
	return do_overlap(arc, *(pl_ray *) &obj);
    case PL_EDGE:
	return do_overlap(arc, *(pl_edge *) &obj);
    case PL_POLYGON:
	return do_overlap(arc, *(pl_pgn *) &obj);
    case PL_DISC:
	return do_overlap(arc, *(pl_disc *) &obj);
    case PL_ARC:
	return do_overlap(arc, *(pl_arc *) &obj);
    }
}

int do_overlap(const pl_object &obj, const pl_box &box)
{
    switch (obj.type()) {
    case PL_POINT:
	return do_overlap(box, *(pl_point *) &obj);
    case PL_LINE:
	return do_overlap(box, *(pl_line *) &obj);
    case PL_RAY:
	return do_overlap(box, *(pl_ray *) &obj);
    case PL_EDGE:
	return do_overlap(box, *(pl_edge *) &obj);
    case PL_POLYGON:
	return do_overlap(box, *(pl_pgn *) &obj);
    case PL_DISC:
	return do_overlap(box, *(pl_disc *) &obj);
    case PL_ARC:
	return do_overlap(box, *(pl_arc *) &obj);
    }
}

int do_overlap(const pl_object &obj, const pl_pgn &pgn)
{
    switch (obj.type()) {
    case PL_POINT:
	return do_overlap(pgn, *(pl_point *) &obj);
    case PL_LINE:
	return do_overlap(pgn, *(pl_line *) &obj);
    case PL_RAY:
	return do_overlap(pgn, *(pl_ray *) &obj);
    case PL_EDGE:
	return do_overlap(pgn, *(pl_edge *) &obj);
    case PL_POLYGON:
	return do_overlap(pgn, *(pl_pgn *) &obj);
    case PL_DISC:
	return do_overlap(pgn, *(pl_disc *) &obj);
    case PL_ARC:
	return do_overlap(pgn, *(pl_arc *) &obj);
    }
}

int do_overlap(const pl_object &obj, const pl_disc &disc)
{
    switch (obj.type()) {
    case PL_POINT:
	return do_overlap(disc, *(pl_point *) &obj);
    case PL_LINE:
	return do_overlap(disc, *(pl_line *) &obj);
    case PL_RAY:
	return do_overlap(disc, *(pl_ray *) &obj);
    case PL_EDGE:
	return do_overlap(disc, *(pl_edge *) &obj);
    case PL_POLYGON:
	return do_overlap(disc, *(pl_pgn *) &obj);
    case PL_DISC:
	return do_overlap(disc, *(pl_disc *) &obj);
    case PL_ARC:
	return do_overlap(disc, *(pl_arc *) &obj);
    }
}

int do_overlap(const pl_object &obj, const pl_object &obj2)
{
    switch (obj.type()) {
    case PL_POINT:
	return do_overlap(obj2, *(pl_point *) &obj);
    case PL_LINE:
	return do_overlap(obj2, *(pl_line *) &obj);
    case PL_RAY:
	return do_overlap(obj2, *(pl_ray *) &obj);
    case PL_EDGE:
	return do_overlap(obj2, *(pl_edge *) &obj);
    case PL_POLYGON:
	return do_overlap(obj2, *(pl_pgn *) &obj);
    case PL_DISC:
	return do_overlap(obj2, *(pl_disc *) &obj);
    case PL_ARC:
	return do_overlap(obj2, *(pl_arc *) &obj);
    }
}



// ******   VERTEX   ******

// does a vertex lie inside a bounding box ?
int lies_inside(const pl_fixedvec &vt, const pl_boundingbox &bb)
{
    return vt.x() >= bb.xmin() && vt.x() <= bb.xmax() &&
    vt.y() >= bb.ymin() && vt.y() <= bb.ymax();
}

int lies_inside(const pl_fixedvec &vt, const pl_pgn &pg)
{
// eerst bounding-box-test
    if (!lies_inside(vt, pg.bbox()))
	return 0;
    int             l1 = 0;
// cleanup list of edges crossing sweepline
    clear_edgelist();
    while (l1 < pg.size() ) {
	if (pg.velement(l1) < vt ) {
	    if (overlap_event(pg, 1, pg.element(l1)))
		return 1;
	    l1++;
	} else {
	    edgelistp *lp;
	// search between which edges vt lies
/*
	    for (lp = &el;
		 *lp != NULL && is_below(vt , (*lp)->line);
		 lp = &((*lp)->next));	// EMPTY STATEMENT
*/
	lp = search_edgelist(&el, vt);
	// check if vt lies inside the polygon
	    if (*lp != NULL && (*lp)->side == down )
		return 1;			// overlap found
	    else
		return 0;
	}
    }
    return 0;
}

int lies_inside(const pl_fixedvec &vt, const pl_disc &disc)
{
    return (clearance(vt, disc.center()) <= disc.radius());
}

// ******   EDGE   ******

int do_intersect(const pl_edge &e1, const pl_edge &e2)
{
    if (!do_overlap(e1.bbox(), e2.bbox()))
	return 0;
    return (do_intersect(e1, e2.supporting_line()) &&
	    do_intersect(e2, e1.supporting_line()));
}

int do_intersect(const pl_edge &e1, const pl_ray &ray)
{
    return (do_intersect(e1, ray.supporting_line()) &&
	    do_intersect(ray, e1.supporting_line()));
}

int do_intersect(const pl_edge &edge, const pl_line &line)
{
    int side1, side2;
    side1 = is_to_the_left(edge.begin_vt(), line);
    side2 = is_to_the_left(edge.end_vt(), line);
    return (side1 && !side2) || (!side1 && side2);
}

int nonant_of(const pl_vec &vt, const pl_box &box)
{
    s_coord x, y;
    int result = 0;
    x = vt.x();
    if (x>=box.xmin()) {
	if (x>box.xmax()) {
	    result +=2;
	} else {
	    result++;
	}
    }
    y = vt.y();
    if (y>=box.ymin()) {
	if (y>box.ymax()) {
	    result += 6;
	} else {
	    result += 3;
	}
    }
    return result;
}

int do_cross(const pl_edge & edge, const pl_box & box)
{
    if (!do_overlap(edge.bbox(), box))
	return 0;
    pl_vec leftvt, rightvt;
    int nonant;
    nonant = nonant_of(edge.begin_vt(), box);
    if (nonant == 4)
	return (!lies_inside(edge.end_vt(), box));
    switch (nonant) {
    case 0:
	leftvt = pl_vec(box.xmin(), box.ymax());
	rightvt = pl_vec(box.xmax(), box.ymin());
	break;
    case 1:
	leftvt = pl_vec(box.xmin(), box.ymin());
	rightvt = pl_vec(box.xmax(), box.ymin());
	break;
    case 2:
	leftvt = pl_vec(box.xmin(), box.ymin());
	rightvt = pl_vec(box.xmax(), box.ymax());
	break;
    case 3:
	leftvt = pl_vec(box.xmin(), box.ymax());
	rightvt = pl_vec(box.xmin(), box.ymin());
	break;
    case 5:
	leftvt = pl_vec(box.xmax(), box.ymin());
	rightvt = pl_vec(box.xmax(), box.ymax());
	break;
    case 6:
	leftvt = pl_vec(box.xmax(), box.ymax());
	rightvt = pl_vec(box.xmin(), box.ymin());
	break;
    case 7:
	leftvt = pl_vec(box.xmax(), box.ymax());
	rightvt = pl_vec(box.xmin(), box.ymax());
	break;
    case 8:
	leftvt = pl_vec(box.xmax(), box.ymin());
	rightvt = pl_vec(box.xmin(), box.ymax());
	break;
    }
    return is_to_the_left(edge.end_vt(), pl_line(edge.begin_vt(), rightvt))
       &&  is_to_the_right(edge.end_vt(), pl_line(edge.begin_vt(), leftvt));
}

int do_overlap(const pl_edge & edge, const pl_box & box)
{
    if (!do_overlap(edge.bbox(), box))
	return 0;
    pl_vec leftvt, rightvt;
    int nonant;
    nonant = nonant_of(edge.begin_vt(), box);
    if (nonant == 4)
	return (1);
    switch (nonant) {
    case 0:
	leftvt = pl_vec(box.xmin(), box.ymax());
	rightvt = pl_vec(box.xmax(), box.ymin());
	break;
    case 1:
	leftvt = pl_vec(box.xmin(), box.ymin());
	rightvt = pl_vec(box.xmax(), box.ymin());
	break;
    case 2:
	leftvt = pl_vec(box.xmin(), box.ymin());
	rightvt = pl_vec(box.xmax(), box.ymax());
	break;
    case 3:
	leftvt = pl_vec(box.xmin(), box.ymax());
	rightvt = pl_vec(box.xmin(), box.ymin());
	break;
    case 5:
	leftvt = pl_vec(box.xmax(), box.ymin());
	rightvt = pl_vec(box.xmax(), box.ymax());
	break;
    case 6:
	leftvt = pl_vec(box.xmax(), box.ymax());
	rightvt = pl_vec(box.xmin(), box.ymin());
	break;
    case 7:
	leftvt = pl_vec(box.xmax(), box.ymax());
	rightvt = pl_vec(box.xmin(), box.ymax());
	break;
    case 8:
	leftvt = pl_vec(box.xmax(), box.ymin());
	rightvt = pl_vec(box.xmin(), box.ymax());
	break;
    }
    return is_to_the_left(edge.end_vt(), pl_line(edge.begin_vt(), rightvt))
       &&  is_to_the_right(edge.end_vt(), pl_line(edge.begin_vt(), leftvt));
}

int do_cross(const pl_edge & edge, const pl_pgn & pgn)
{
    int i;
    static intreray whichside;
    if (!do_overlap(edge.bbox(), pgn.bbox()))
	return 0;
    pl_vec start, end;
    if (edge.begin_vt() < edge.end_vt()) {
	start = edge.begin_vt();
	end = edge.end_vt();
    } else {
	start = edge.end_vt();
	end = edge.begin_vt();
    }
    pl_line line(edge.supporting_line());
    whichside.newsize(pgn.size());
// compute for all vertices on which side of the line they lie
    for (i=0; i<pgn.size(); i++)
	whichside[i] = (is_to_the_left(pgn.vertex(i), line) ? 1 : -1);
// now look for two consecutive vertices on opposite sides
    int prev = pgn.size()-1;
    for(i=0; i<pgn.size(); i++) {
	if (whichside[i] != whichside[prev]) {
	    pl_line pline(pgn.vertex(prev), pgn.direction(prev));
	    if (is_to_the_left(start, pline)) {
		if (!is_to_the_left(end, pline))
		    return 1;
	    } else {
		if (is_to_the_left(end, pline))
		    return 1;
	    }
	}
	prev = i;
    }
    return 0;
}

/*
 * do_intersect ontdekt of een edge een polygon overlapt.
 * De routine werkt hetzelfde als do_overlap, waarbij de edge als een
 * gedegenereerd polygon wordt opgevat.
 * De edge wordt daarom twee keer in de scanlijst opgenomen (een keer als
 * bovenkant en een keer als onderkant).
 * De code blinkt niet uit in helderheid.
 */
int do_overlap(const pl_edge & e, const pl_pgn & pg)
{
// eerst bounding-box-test
    if (!do_overlap(e.bbox(), pg.bbox()))
	return 0;
    pl_vec vt[2];
    if (e.begin_vt() < e.end_vt() ) {
	vt[0] = e.begin_vt();
	vt[1] = e.end_vt();
    } else {
	vt[0] = e.end_vt();
	vt[1] = e.begin_vt();
    }
    int             l1 = 0, l2 = 0;
// cleanup list of edges crossing sweepline
    clear_edgelist();
    while (l1 < pg.size() && l2 < 2) {
	if (pg.velement(l1) < vt[l2]) {
	    if (overlap_event(pg, 1, pg.element(l1)))
		return 1;
	    l1++;
	} else {
	    if (l2 == 0) {		// startpoint of edge
		edgelistp *lp;
		edgelistp       e1, e2;
	    // search between which edges startpoint lies
/*
		for (lp = &el;
		     *lp != NULL && is_below(vt[0], (*lp)->line);
		     lp = &((*lp)->next));	// EMPTY STATEMENT
*/
	    lp = search_edgelist(&el, vt[0]);
	    // check if startpoint lies inside the other polygon
		if (*lp != NULL && (*lp)->side == down)
		    return 1;
		e1 = new_edgelist();
		e2 = new_edgelist();
		e2->next = *lp;
		e1->next = e2;
		*lp = e1;
		e1->id = e2->id = 2;
		e1->side = up;
		e2->side = down;
		e1->line = e2->line = e.supporting_line();
		e1->end = e2->end = 1;
	    } else {			// endpoint of edge
		edgelistp cur = el;
		edgelistp prev = 0;
		while (cur->end != 1 || cur->id != 2) {
		    prev = cur;
		    cur = cur->next;
		}
	    // check of benedenbuur ook vnum als eindpunt heeft
		if (cur->next->end != 1 || cur->next->id != 2)
		    return 1;
		edgelistp       next = cur->next->next;
	    // check of is_below(vnum,bovenbuur)
		if (prev != NULL && is_above(vt[1], prev->line))
		    return 1;
	    // check of is_above(vnum,2*benedenbuur)
		if (next != NULL && is_below(vt[1], next->line))
		    return 1;
		return 0;
	    }
	    l2++;
	}
    }
    return 0;
}

int do_cross(const pl_edge &edge, const pl_disc &disc)
{
    s_coord l, d1, d2, xx;
    pl_unitvec edgedir;
    pl_vec beginvec, projectvec;
    l = factorise(edgedir, edge.end_vt()-edge.begin_vt());
    beginvec = edge.begin_vt()-disc.center();
    d1 = dot(beginvec, edgedir);
    projectvec = beginvec-d1*edgedir;
    s_coord sqrad = disc.radius() * disc.radius();
    xx = sqrad - dot(projectvec, projectvec);
    if (xx < 0)
	return 0;   // edge outside disc
    d2 = d1 + l;
    if (d1<0) {
	if (d2<0) {
	    if (d2*d2 > xx)
		return 0;   // edge outside disc
	    if (d1*d1 < xx)
		return 0;   // edge inside disc
	} else {
	    if (-d1 > d2) {
		if (d1*d1 < xx)
		    return 0;   // edge inside disc
	    } else {
		if (d1*d1 < xx)
		    return 0;   // edge inside disc
	    }
	}
    } else {
	if (d1*d1 > xx)
	    return 0;   // edge outside disc
	if (d2*d2 < xx)
	    return 0;   // edge inside disc
    }
    return 1;
}

int do_overlap(const pl_edge &edge, const pl_disc &disc)
{
    return (clearance(edge, disc)<= 0);
}

// ******   RAY   ******

int do_intersect(const pl_ray &ray1, const pl_ray &ray2)
{
    return (do_intersect(ray1, ray2.supporting_line()) &&
	    do_intersect(ray2, ray1.supporting_line()));
}

int do_intersect(const pl_ray &ray, const pl_line &line)
{
    int side1, side2;
    s_coord endsign = cross(line.direction(), ray.direction());
    if (endsign>0) side2 = 1;
    else if (endsign<0) side2 = 0;
    else return 0;	// parallel
    side1 = is_to_the_left(ray.begin_vt(), line);
    return (side1 && !side2) || (!side1 && side2);
}

int do_intersect(const pl_ray &ray, const pl_arc &arc)
{
    pl_vec dummy1, dummy2;
    return intersect(dummy1, dummy2, ray, arc);
}

int do_cross(const pl_ray & ray, const pl_box & box)
{
    if (box.is_empty())
	return 0;
    pl_vec leftvt, rightvt;
    int nonant;
    nonant = nonant_of(ray.begin_vt(), box);
    if (nonant == 4)
	return (1);
    switch (nonant) {
    case 0:
	leftvt = pl_vec(box.xmin(), box.ymax());
	rightvt = pl_vec(box.xmax(), box.ymin());
	break;
    case 1:
	leftvt = pl_vec(box.xmin(), box.ymin());
	rightvt = pl_vec(box.xmax(), box.ymin());
	break;
    case 2:
	leftvt = pl_vec(box.xmin(), box.ymin());
	rightvt = pl_vec(box.xmax(), box.ymax());
	break;
    case 3:
	leftvt = pl_vec(box.xmin(), box.ymax());
	rightvt = pl_vec(box.xmin(), box.ymin());
	break;
    case 5:
	leftvt = pl_vec(box.xmax(), box.ymin());
	rightvt = pl_vec(box.xmax(), box.ymax());
	break;
    case 6:
	leftvt = pl_vec(box.xmax(), box.ymax());
	rightvt = pl_vec(box.xmin(), box.ymin());
	break;
    case 7:
	leftvt = pl_vec(box.xmax(), box.ymax());
	rightvt = pl_vec(box.xmin(), box.ymax());
	break;
    case 8:
	leftvt = pl_vec(box.xmax(), box.ymin());
	rightvt = pl_vec(box.xmin(), box.ymax());
	break;
    }
    return (is_to_the_left(leftvt, ray.supporting_line())
       &&  is_to_the_right(rightvt, ray.supporting_line()));
}

int do_overlap(const pl_ray & ray, const pl_box & box)
{ return do_cross(ray, box); }

int do_cross(const pl_ray &ray, const pl_pgn &pgn)
{
    return !intersect(pgn, ray).is_empty();
}

int do_overlap(const pl_ray &ray, const pl_pgn &pgn)
{
    return !intersect(pgn, ray).is_empty();
}

int do_cross(const pl_ray &ray, const pl_disc &disc)
{
    return (clearance(ray, disc)<= 0);
}

int do_overlap(const pl_ray &ray, const pl_disc &disc)
{
    return (clearance(ray, disc)<= 0);
}

// ******   LINE   ******

int do_intersect(const pl_line &line, const pl_arc &arc)
{
    pl_vec dummy1, dummy2;
    return intersect(dummy1, dummy2, line, arc);
}

int do_cross(const pl_line &line, const pl_box &box)
{
    return (clearance(line, box) <= 0);
}

int do_overlap(const pl_line &line, const pl_box &box)
{
    return (clearance(line, box) <= 0);
}

int do_cross(const pl_line &line, const pl_pgn &pgn)
{
    return !intersect(pgn, line).is_empty();
}

int do_overlap(const pl_line &line, const pl_pgn &pgn)
{
    return !intersect(pgn, line).is_empty();
}

int do_cross(const pl_line &line, const pl_disc &disc)
{
    return (clearance(line, disc)<= 0);
}

int do_overlap(const pl_line &line, const pl_disc &disc)
{
    return (clearance(line, disc)<= 0);
}

// ******   ARC   ******

int do_intersect(const pl_arc &arc1, const pl_arc &arc2)
{
    pl_vec wn1, wn2;
    return intersect(wn1, wn2, arc1, arc2);
}

int do_cross(const pl_arc &arc, const pl_box &box)
{
    pl_pgn boxpgn;
    if (!do_overlap(arc.bbox(), box))
	return 0;
    pgn_out_of_box(boxpgn, box);
    return do_cross(arc, boxpgn);
}

int do_overlap(const pl_arc &arc, const pl_box &box)
{
    pl_pgn boxpgn;
    if (!do_overlap(arc.bbox(), box))
	return 0;
    pgn_out_of_box(boxpgn, box);
    return do_overlap(arc, boxpgn);
}

int do_cross(const pl_arc &arc, const pl_pgn &pgn)
{
    int i;
    if (!do_overlap(arc.bbox(), pgn.bbox()))
	return 0;
    for (i=0; i<pgn.size(); i++)
	if (do_intersect(pgn.edge(i), arc))
	    return 1;
    return 0;
}

int do_overlap(const pl_arc &arc, const pl_pgn &pgn)
{
    if (!do_overlap(arc.bbox(), pgn.bbox()))
	return 0;
    if (do_cross(arc, pgn))
	return 1;
    if (lies_inside(arc.begin_vt(), pgn))
	return 1;
    return 0;
}

int do_cross(const pl_arc &arc, const pl_disc &disc)
{
    pl_vec wn1, wn2;
    return intersect(wn1, wn2, disc, arc);
}

int do_overlap(const pl_arc &arc, const pl_disc &disc)
{
    if (do_cross(disc, arc))
	return 1;
    if (lies_inside(arc.begin_vt(), disc))
	return 1;
    return 0;
}

// ******   BOX   ******

int do_cross(const pl_boundingbox &bb1, const pl_boundingbox &bb2)
{
    if (!do_overlap(bb1, bb2))
	return 0;
    if (bb2.xmin() > bb1.xmin()) {
	if (bb2.xmax() > bb1.xmax())
	    return 1;
    } else {
	if (bb2.xmax() < bb1.xmax())
	    return 1;
    }
    if (bb2.ymin() > bb1.ymin()) {
	if (bb2.ymax() > bb1.ymax())
	    return 1;
    } else {
	if (bb2.ymax() < bb1.ymax())
	    return 1;
    }
    return 0;
}

int do_overlap(const pl_boundingbox &bb1, const pl_boundingbox &bb2)
{
    if (bb1.xmin() > bb2.xmax())
	return 0;
    if (bb2.xmin() > bb1.xmax())
	return 0;
    if (bb1.ymin() > bb2.ymax())
	return 0;
    if (bb2.ymin() > bb1.ymax())
	return 0;
    if (bb1.is_empty() || bb2.is_empty())
	return 0;
    return 1;
}

int do_cross(const pl_box &box, const pl_pgn &pgn)
{
    pl_pgn boxpgn;
    if (!do_overlap(pgn.bbox(), box))
	return 0;
    pgn_out_of_box(boxpgn, box);
    return do_cross(pgn, boxpgn);
}

int do_overlap(const pl_box &box, const pl_pgn &pgn)
{
    pl_pgn boxpgn;
    if (!do_overlap(pgn.bbox(), box))
	return 0;
    pgn_out_of_box(boxpgn, box);
    return do_overlap(pgn, boxpgn);
}

int do_cross(const pl_box &box, const pl_disc &disc)
{
    int nonant;
    if (box.is_empty())
	return 0;
    s_coord sqradius = disc.radius() * disc.radius();
    pl_vec center(disc.center());
    s_coord d1, d2;
    nonant = nonant_of(disc.center(), box);
    switch (nonant) {
    case 0:
	if (sqclearance(center, pl_vec(box.xmin(), box.ymin())) > sqradius)
	    return 0;	// separate
	if (sqclearance(center, pl_vec(box.xmax(), box.ymax())) < sqradius)
	    return 0;	// box inside disc
	return 1;
    case 2:
	if (sqclearance(center, pl_vec(box.xmax(), box.ymin())) > sqradius)
	    return 0;	// separate
	if (sqclearance(center, pl_vec(box.xmin(), box.ymax())) < sqradius)
	    return 0;	// box inside disc
	return 1;
    case 6:
	if (sqclearance(center, pl_vec(box.xmin(), box.ymax())) > sqradius)
	    return 0;	// separate
	if (sqclearance(center, pl_vec(box.xmax(), box.ymin())) < sqradius)
	    return 0;	// box inside disc
	return 1;
    case 8:
	if (sqclearance(center, pl_vec(box.xmax(), box.ymax())) > sqradius)
	    return 0;	// separate
	if (sqclearance(center, pl_vec(box.xmin(), box.ymin())) < sqradius)
	    return 0;	// box inside disc
	return 1;
    case 1:
	if (box.ymin()-center.y() > disc.radius())
	    return 0;	// separate
	d1 = center.x()-box.xmin();
	d2 = box.xmax()-center.x();
	if (d2 > d1) d1 = d2;
	d2 = box.ymax()-center.y();
	if ((d1*d1+d2*d2) < sqradius)
	    return 0;	// box inside disc
	return 1;
    case 3:
	if (box.xmin()-center.x() > disc.radius())
	    return 0;	// separate
	d1 = center.y()-box.ymin();
	d2 = box.ymax()-center.y();
	if (d2 > d1) d1 = d2;
	d2 = box.xmax()-center.x();
	if ((d1*d1+d2*d2) < sqradius)
	    return 0;	// box inside disc
	return 1;
    case 5:
	if (center.x()-box.xmax() > disc.radius())
	    return 0;	// separate
	d1 = center.y()-box.ymin();
	d2 = box.ymax()-center.y();
	if (d2 > d1) d1 = d2;
	d2 = center.x()-box.xmin();
	if ((d1*d1+d2*d2) < sqradius)
	    return 0;	// box inside disc
	return 1;
    case 7:
	if (center.y()-box.ymax() > disc.radius())
	    return 0;	// separate
	d1 = center.x()-box.xmin();
	d2 = box.xmax()-center.x();
	if (d2 > d1) d1 = d2;
	d2 = center.y()-box.ymin();
	if ((d1*d1+d2*d2) < sqradius)
	    return 0;	// box inside disc
    case 4:
	{ s_coord x1, x2, y1, y2;
	d1 = center.x()-box.xmin();
	d2 = box.xmax()-center.x();
	if (d1<d2) { x1 = d1; x2 = d2; }
	else { x1 = d2; x2 = d1; }
	d1 = center.y()-box.ymin();
	d2 = box.ymax()-center.y();
	if (d1<d2) { y1 = d1; y2 = d2; }
	else { y1 = d2; y2 = d1; }
	if (x1<y1) d1 = x1; else d1 = y1;
	if (disc.radius() < d1)
	    return 0;	// disc inside box
	d2 = x2*x2 + y2*y2;
	if (disc.radius() > d2)
	    return 0;	// box inside disc
	return 1;
	}
    }
}

int do_overlap(const pl_box &box, const pl_disc &disc)
{
    if (box.is_empty())
	return 0;
    return (clearance(box, disc) <= 0);
}


// ******   POLYGON   ******

/*
 * do_overlap test op overlappen van twee polygons middels een sweepline alg.
 * de polygonen moeten simpel zijn. tijdens de sweep wordt een lijst met
 * edges bijgehouden. bij vertices worden edges toegevoegd/weggehaald en
 * gecheckt op overlap. als er overlap optreedt kan de ordening van edges in
 * de war raken. dit kan echter geen kwaad. overlap wordt dan toch
 * ontdekt.(linkste overlap)
 */
int do_overlap(const pl_pgn & pgn1, const pl_pgn & pgn2)
{
// eerst bounding-box-test
    if (!do_overlap(pgn1.bbox(), pgn2.bbox()))
	return 0;
    int l1 = 0, l2 = 0;
// cleanup list of edges crossing sweepline
    clear_edgelist();
    while (l1 < pgn1.size() && l2 < pgn2.size()) {
	if (pgn1.velement(l1) < pgn2.velement(l2)) {
	    if (overlap_event(pgn1, 1, pgn1.element(l1)))
		return 1;
	    l1++;
	} else {
	    if (overlap_event(pgn2, 2, pgn2.element(l2)))
		return 1;
	    l2++;
	}
    }
    return 0;
}

// handle a vertex encountered by the sweep line
// check if overlap takes place
static int cross_event(const pl_pgn & pg, int pol_id, int vnum)
{
    edgelistp       cur, prev;
    int             nb1, nb2;
    int             vertextype = 0;
    pl_edge	    e;
    pl_vec	    vec;
// nb1 and nb2 are the vertices preceding and succeedeing 'vnum' in 'pg'
    nb1 = vnum == 0 ? pg.size() - 1 : vnum - 1;
    nb2 = vnum == pg.size() - 1 ? 0 : vnum + 1;
// decide what type of vertex 'vnum' is
// if both edges go to right then insertionpoint (type 0)
// if both edges come from the left then deletionpoint (type 3)
// otherwise a transitionpoint (type 1 or 2)
    if (pg.order(nb1) < pg.order(vnum))
	vertextype++;
    if (pg.order(nb2) < pg.order(vnum))
	vertextype += 2;
    switch (vertextype) {
    case 0:
    // vnum is insertionpoint
	edgelistp *lp;
	edgelistp       e1, e2;
    // search between which edges 'vnum' lies
/*
	for (lp = &el;
	     *lp != NULL && is_below(pg.vertex(vnum), (*lp)->line);
	     lp = &((*lp)->next));	// EMPTY STATEMENT
*/
	lp = search_edgelist(&el, pg.vertex(vnum));
    // insert the two edges starting in vertex vnum
	e1 = new_edgelist();
	e2 = new_edgelist();
	e2->next = *lp;
	e1->next = e2;
	*lp = e1;
	e1->id = e2->id = pol_id;
	if (cross(pg.vertex(nb1) - pg.vertex(vnum),
		pg.vertex(nb2) - pg.vertex(vnum)) < 0) {
	// vnum-nb1 lies above vnum-nb2
	    e1->line = pl_line(pg.vertex(nb1), pg.direction(nb1));
	    e1->end = nb1;
	    e2->line = pl_line(pg.vertex(vnum), pg.direction(vnum));
	    e2->end = nb2;
	} else {	
	// vnum-nb1 lies below vnum-nb2
	    e1->line = pl_line(pg.vertex(vnum), pg.direction(vnum));
	    e1->end = nb2;
	    e2->line = pl_line(pg.vertex(nb1), pg.direction(nb1));
	    e2->end = nb1;
	}
	break;
    case 1:
    // vnum is transitionpoint: nb1->vnum->nb2
    // zoek in lijst tot je edge met vnum als eindpunt vindt.
	cur = el;
	prev = NULL;
	while (cur->end != vnum || cur->id != pol_id) {
	    prev = cur;
	    cur = cur->next;
	}
    // check of is_below(vnum,bovenbuur)
    // check of above(vnum,benedenbuur)
    // als check faalt return 1;(overlap)
	if (prev != NULL && is_above(pg.vertex(vnum), prev->line))
	    return 1;
	if (cur->next != NULL && is_below(pg.vertex(vnum), cur->next->line))
	    return 1;
    // vervang edge door edge vnum->nb2;
	cur->end = nb2;
	cur->line = pl_line(pg.vertex(vnum), pg.direction(vnum));
	break;
    case 2:	
    // vnum is transitionpoint: nb2->vnum->nb1
	cur = el;
	prev = NULL;
	while (cur->end != vnum || cur->id != pol_id) {
	    prev = cur;
	    cur = cur->next;
	}
	if (prev != NULL && is_above(pg.vertex(vnum), prev->line))
	    return 1;
	if (cur->next != NULL && is_below(pg.vertex(vnum), cur->next->line))
	    return 1;
	cur->end = nb1;
	cur->line = pl_line(pg.vertex(nb1), pg.direction(nb1));
	break;
    case 3:
    // vnum is deletionpoint
    // zoek in lijst tot je edge met vnum als eindpunt vindt.
	cur = el;
	prev = NULL;
	while (cur->end != vnum || cur->id != pol_id) {
	    prev = cur;
	    cur = cur->next;
	}
    // check of benedenbuur ook vnum als eindpunt heeft
	if (cur->next->end != vnum || cur->next->id != pol_id)
	    return 1;
	edgelistp next;
	next = cur->next->next;
    // check of is_below(vnum,bovenbuur)
	if (prev != NULL && is_above(pg.vertex(vnum), prev->line))
	    return 1;
    // check of above(vnum,2*benedenbuur)
	if (next != NULL && is_below(pg.vertex(vnum), next->line))
	    return 1;
    // laat twee edges uit list weg
	delete_edgelist(cur->next);
	delete_edgelist(cur);
	if (prev == NULL)
	    el = next;
	else
	    prev->next = next;
	break;
    }
    return 0;
}

int do_cross(const pl_pgn &pgn1, const pl_pgn &pgn2)
{
// eerst bounding-box-test
    if (pgn1.size() <= 1 || pgn2.size() <= 1)
	return 0;
    if (!do_overlap(pgn1.bbox(), pgn2.bbox()))
	return 0;
    int l1 = 0, l2 = 0;
// cleanup list of edges crossing sweepline
    clear_edgelist();
    while (l1 < pgn1.size() && l2 < pgn2.size()) {
	if (pgn1.velement(l1) < pgn2.velement(l2)) {
	    if (cross_event(pgn1, 1, pgn1.element(l1)))
		return 1;
	    l1++;
	} else {
	    if (cross_event(pgn2, 2, pgn2.element(l2)))
		return 1;
	    l2++;
	}
    }
    return 0;
}

int do_cross(const pl_pgn &pgn, const pl_disc &disc)
{
    int i;
    if (!do_overlap(pgn.bbox(), disc))
	return 0;
    for (i=0; i<pgn.size(); i++)
	if (do_cross(pgn.edge(i), disc))
	    return 1;
    return 0;
}

int do_overlap(const pl_pgn &pgn, const pl_disc &disc)
{
    int i;
    if (!do_overlap(pgn.bbox(), disc))
	return 0;
    for (i=0; i<pgn.size(); i++)
	if (do_overlap(pgn.edge(i), disc))
	    return 1;
    if (lies_inside(disc.center(), pgn))
	return 1;
    return 0;
}

// ******   DISC   ******

int do_cross(const pl_disc &disc1, const pl_disc &disc2)
{
    s_coord l = clearance(disc1.center(), disc2.center());
    s_coord r1 = disc1.radius(), r2 = disc2.radius();
    if (l > r1+r2)
	return 0;
    if (r1 > l+r2)
	return 0;
    if (r2 > l+r1)
	return 0;
    return 1;
}

int do_overlap(const pl_disc &disc1, const pl_disc &disc2)
{
    return clearance(disc1.center(), disc2.center()) <=
	disc1.radius() + disc2.radius();
}

// ***************************
