
#include <plageo.h>


const s_coord scoord = 3;
static s_coord big = max_of_type(scoord);

static void pgn_out_of_box(pl_pgn &pgn, const pl_box &box)
{
// assert: !box.is_empty()
    static pl_vecreray vrr(4);
    vrr[0] = pl_vec(box.xmax(), box.ymax());
    vrr[1] = pl_vec(box.xmin(), box.ymax());
    vrr[2] = pl_vec(box.xmin(), box.ymin());
    vrr[3] = pl_vec(box.xmax(), box.ymin());
    pgn = pl_pgn(vrr);
}

int certain_clearance(const pl_boundingbox &bb1, const pl_boundingbox &bb2,
			s_coord bound)
{
    if (bb1.xmin()-bb2.xmax() >= bound)  return 1;
    if (bb2.xmin()-bb1.xmax() >= bound)  return 1;
    if (bb1.ymin()-bb2.ymax() >= bound)  return 1;
    if (bb2.ymin()-bb1.ymax() >= bound)  return 1;
    return 0;
}

int certain_clearance(const pl_fixedvec &vt, const pl_boundingbox &bb,
			s_coord bound)
{
    if (bb.xmin()-vt.x() >= bound)  return 1;
    if (vt.x()-bb.xmax() >= bound)  return 1;
    if (bb.ymin()-vt.y() >= bound)  return 1;
    if (vt.y()-bb.ymax() >= bound)  return 1;
    return 0;
}

int certain_sqclearance(const pl_boundingbox &bb1, const pl_boundingbox &bb2,
			s_coord sqbound)
{
    s_coord d;
    d = bb1.xmin()-bb2.xmax();
    if (d<0) {
	d = bb2.xmin()-bb1.xmax();
	if (d < 0)
	    d = 0;
    }
    sqbound -= d*d;
    if (sqbound <= 0)
	return 1;
    d = bb1.ymin()-bb2.ymax();
    if (d<0) {
	d = bb2.ymin()-bb1.ymax();
	if (d < 0)
	    d = 0;
    }
    return  d*d >= sqbound;
}

int certain_sqclearance(const pl_fixedvec &vt, const pl_boundingbox &bb,
			s_coord sqbound)
{
    s_coord d;
    d = vt.x()-bb.xmax();
    if (d<0) {
	d = bb.xmin()-vt.x();
	if (d < 0)
	    d = 0;
    }
    sqbound -= d*d;
    if (sqbound <= 0)
	return 1;
    d = vt.y()-bb.ymax();
    if (d<0) {
	d = bb.ymin()-vt.y();
	if (d < 0)
	    d = 0;
    }
    return (d*d) >= sqbound;
}

s_coord sqclearance(const pl_fixedvec &vt, const pl_line &l)
{
    pl_fixedvec vt_on_line(l.ref_vt());
    pl_unitvec dir(l.direction());
    pl_vec diff(vt - vt_on_line);
    return (diff - dot(diff, dir) * dir).sqlength();
}

s_coord sqclearance(const pl_fixedvec &v, const pl_edge &e)
{
    if (e.begin_vt().x() == e.end_vt().x()
	&& e.begin_vt().y() == e.end_vt().y())
	return sqclearance(e.begin_vt(),v);
    if (dot(e.end_vt()-e.begin_vt(),v-e.begin_vt()) <= 0)
	return sqclearance(e.begin_vt(),v);
    if (dot(e.end_vt()-e.begin_vt(),v-e.end_vt()) >= 0)
	return sqclearance(e.end_vt(),v);
    return sqclearance(v, e.supporting_line());
}


// ********* VERTEX ... *********

s_coord clearance(const pl_fixedvec &vt1, const pl_fixedvec &vt2, s_coord bound)
{
    s_coord dx, dy;
    dx = fabs_pp(vt1.x() - vt2.x());
    if (dx>= bound) return bound;
    dy = fabs_pp(vt1.y() - vt2.y());
    if (dy>= bound) return bound;
    return sqrt_pp(dx*dx+dy*dy);
}

s_coord clearance(const pl_fixedvec &vt, const pl_edge &edge)
{
    pl_unitvec e1;
    s_coord l, dx, sqrd;
    l = factorise(e1, edge.end_vt() - edge.begin_vt());
    dx = dot(e1,vt-edge.begin_vt());
    if (dx <= 0)
	return clearance(edge.begin_vt(),vt);
    if (dx >= l)
	return clearance(edge.end_vt(),vt);
    sqrd = sqclearance(edge.begin_vt(), vt) - dx*dx;
    return (sqrd <= 0) ? 0 : sqrt_pp(sqrd);
}

// return clearance between vertex and edge if smaller than bound,
// otherwise bound
s_coord clearance(const pl_fixedvec &v, const pl_edge &e, s_coord bound)
{
    if (certain_clearance(v, e.bbox(), bound))
	return bound;
    s_coord cl = clearance(v, e);
    return (cl<bound) ? cl : bound;
}

s_coord clearance(const pl_fixedvec &vt, const pl_ray &ray)
{
    pl_unitvec e1(ray.direction());
    s_coord dx, sqrd;
    dx = dot(e1,vt-ray.begin_vt());
    if (dx <= 0)
	return clearance(ray.begin_vt(),vt);
    sqrd = sqclearance(ray.begin_vt(),vt) - dx*dx;
    return (sqrd <= 0) ? 0 : sqrt_pp(sqrd);
}

s_coord clearance(const pl_fixedvec &vt, const pl_ray &ray, s_coord bound)
{
    s_coord cl;
    cl = clearance(vt, ray);
    return cl>bound ? bound : cl;
}

s_coord clearance(const pl_fixedvec &vt, const pl_line &l)
{
    const pl_vec &vt_on_line = l.ref_vt();
    pl_unitvec dir(l.direction());
    pl_vec diff(vt - vt_on_line);
    return (diff - dot(diff, dir) * dir).length();
}

s_coord clearance(const pl_fixedvec &vt, const pl_line &line, s_coord bound)
{
    s_coord cl;
    cl = clearance(vt, line);
    return cl>bound ? bound : cl;
}

s_coord clearance(const pl_fixedvec &vt, const pl_arc &arc)
{
    return clearance(vt, arc, big);
}

s_coord clearance(const pl_fixedvec &vt, const pl_arc &arc, s_coord bound)
{
    pl_unitvec uvec;
    s_coord cl, l, sqrd;
    if (certain_clearance(vt, arc.bbox(), bound))
	return bound;
    l = factorise(uvec, pl_vec(vt-arc.center()));
    pl_angle vt_angle = angle_of(uvec);
    s_coord radius = arc.radius();
    pl_angle small_angle = arc.small_angle();
    pl_angle big_angle = arc.big_angle();
    if (vt_angle.lies_between(small_angle, big_angle)) {
	cl = fabs_pp(l-radius);
    } else {
	small_angle = small_angle - vt_angle;
	big_angle = vt_angle-big_angle;
	small_angle.normalise(0);
	big_angle.normalise(0);
	if (big_angle.value() < small_angle.value()) {
	    small_angle = big_angle;
	}
	sqrd = radius*radius + l*l - 2*l*radius*cos_pp(small_angle.value());
	cl = (sqrd <= 0) ? 0 : sqrt_pp(sqrd);
    }
    return (cl > bound) ? bound : cl;
}


s_coord clearance(const pl_fixedvec &vt, const pl_boundingbox &box)
{
    return clearance(box, vt, big);
}

s_coord clearance(const pl_fixedvec &vt, const pl_boundingbox &box, s_coord bound)
{
    s_coord d, dx, dy;
    if (box.is_empty())
	return bound;
    dx = vt.x() - box.xmax();
    if (dx < 0) {
	dx = box.xmin() - vt.x();
	if (dx < 0) dx = 0;
    }
    if (dx > bound)
	return bound;
    dy = vt.y() - box.ymax();
    if (dy < 0) {
	dy = box.ymin() - vt.y();
	if (dy < 0) dy = 0;
    }
    if (dy > bound)
	return bound;
    d = sqrt_pp(dx*dx+dy*dy);
    return d>bound ? bound : d;
}

s_coord clearance(const pl_fixedvec &vt, const pl_pgn &pgn)
{
    return clearance(vt, pgn, big);
}

s_coord clearance(const pl_fixedvec &vt, const pl_pgn &pgn, s_coord bound)
{
    int i;
    s_coord cl;
    if (certain_clearance(vt, pgn.bbox(), bound))
	return bound;
    if (lies_inside(vt, pgn))
	return 0;
    for (i=0; i<pgn.size(); i++) {
	cl = clearance(vt, pgn.edge(i));
	if (cl < bound)
	    bound = cl;
    }
    return bound;
}

s_coord clearance(const pl_fixedvec &vt, const pl_disc &disc)
{
    s_coord c;
    c = clearance(vt, disc.center()) - disc.radius();
    return c>0 ? c : 0;
}

s_coord clearance(const pl_fixedvec &vt, const pl_disc &disc, s_coord bound)
{
    s_coord c;
    c = clearance(vt, disc.center()) - disc.radius();
    if (c<0) c=0;
    return c>bound ? bound : c;
}





// ********* EDGE ... *********

// clearance between two edges:
// two cases are discerned: parallel edges and non parallel edges.
// PARALLEL CASE:
//   discern between distance between supporting lines (sq_acrossdist)
//   and distance of projection of edges on supporting_line (dx).
// NON PARALLEL CASE:
//   The two direction vectors span the space.
s_coord clearance(const pl_edge &edge0, const pl_edge &edge1)
{
    pl_unitvec joindir;
    pl_unitvec e0, e1;
    s_coord l0,l1, sqrd;
    pl_vec joinvec = edge1.begin_vt() - edge0.begin_vt();
    l0 = factorise(e0, edge0.end_vt()-edge0.begin_vt());
    l1 = factorise(e1, edge1.end_vt()-edge1.begin_vt());
    if ( cross(e0, e1) == 0.0F) {
	// parallel lines
	s_coord alongdist = dot(joinvec, e0);
	s_coord dy = dot(joinvec, e0.normal());
	s_coord dx;
	if (dot(e0, e1) > 0) {
	    if( (dx = alongdist - l0) >= 0)
		return sqrt_pp(dy*dy + dx*dx);
	    if( (dx = -(alongdist + l1)) >= 0)
		return sqrt_pp(dy*dy + dx*dx);
	    return fabs_pp(dy); 
	} else {
	    if ( (dx = -alongdist) >= 0)
		return sqrt_pp(dy*dy + dx*dx);
	    if ( (dx = alongdist - l0 - l1) >= 0)
		return sqrt_pp(dy*dy + dx*dx);
	    return fabs_pp(dy); 
	}
    } else {
	s_coord eta[2];
	s_coord min0, min1, x;
    // write joinvec as a combination of e0, e1.
	pl__ontbind2(eta, e0, e1, joinvec);
    // take the intersection point of the supporting lines as new origin,
    // e0 and e1 as (non orthogonal) axes. After this:
    //   edge0 goes from eta[0]*e0  to  (eta[0]+l0)*e0
    //   edge1 goes from eta[1]*e1  to  (eta[1]+l1)*e1
	eta[0] = -eta[0];
    // compute the minimum distance from both edges to new origin.
	if ((min0 = eta[0]) < 0) {
	    if ((min0 += l0) > 0)
		min0 = 0;
	}
	if ((min1 = eta[1]) < 0) {
	    if ((min1 += l1) > 0)
		min1 = 0;
	}
    // if both edges have distance 0 to origin they must intersect there.
	if (min0 == 0 && min1 == 0)
	    return 0;	// crossing edges
    // otherwise take the distance from the closest point that is furthest
    // away to the other edge.
	if (fabs_pp(min0) > fabs_pp(min1)) {
	    x = min0 * dot(e0, e1) - eta[1];
	    if (x < 0)
		return (eta[1]*e1 - min0*e0).length();
	    else if (x > l1)
		return ((eta[1]+l1)*e1 - min0*e0).length();
	    else {
		sqrd = (eta[1]*e1 - min0*e0).sqlength()-x*x;
		return (sqrd <= 0) ? 0 : sqrt_pp(sqrd);
	    }
	} else {
	    x = min1 * dot(e0, e1) - eta[0];
	    if (x < 0)
		return (eta[0]*e0 - min1*e1).length();
	    else if (x > l0)
		return ((eta[0]+l0)*e0 - min1*e1).length();
	    else {
		sqrd = (eta[0]*e0 - min1*e1).sqlength()-x*x;
		return (sqrd <= 0) ? 0 : sqrt_pp(sqrd);
	    }
	}
    }
}

s_coord clearance(const pl_edge &e1, const pl_edge &e2, s_coord bound)
{
    if (certain_clearance(e1.bbox(), e2.bbox(), bound))
	return bound;
    s_coord cl = clearance(e1, e2);
    return (cl<bound) ? cl : bound;
}

// clearance between an edge and a ray.
// If the ray is considered an edge with infinite length, the following
// code follows from the edge-edge case by eliminating superfluous tests.
s_coord clearance(const pl_edge &edge0, const pl_ray &ray)
{
    pl_unitvec joindir;
    pl_unitvec e0, e1;
    s_coord l0, sqrd;
    pl_vec joinvec = ray.begin_vt() - edge0.begin_vt();
    l0 = factorise(e0, edge0.end_vt()-edge0.begin_vt());
    e1 = ray.direction();
    if ( cross(e0, e1) == 0.0F) {
	// parallel lines
	s_coord alongdist = dot(joinvec, e0);
	s_coord dx;
	s_coord sq_acrossdist = (joinvec- alongdist * e0).sqlength();
	if (dot(e0, e1) > 0) {
	    if( (dx = alongdist - l0) >= 0)
		return sqrt_pp(sq_acrossdist + dx*dx);
	    return sqrt_pp(sq_acrossdist); 
	} else {
	    if ( (dx = -alongdist) >= 0)
		return sqrt_pp(sq_acrossdist + dx*dx);
	    return sqrt_pp(sq_acrossdist); 
	}
    } else {
	s_coord eta[2];
	s_coord min0, min1, x;
    // write joinvec as a combination of e0, e1.
	pl__ontbind2(eta, e0, e1, joinvec);
    // take the intersection point of the supporting lines as origin,
    // e0 and e1 as (non orthogonal) axes.
	eta[0] = -eta[0];
    // compute the minimum distance from both edges to new origin.
	if ((min0 = eta[0]) < 0) {
	    if ((min0 += l0) > 0)
		min0 = 0;
	}
	if ((min1 = eta[1]) < 0) {
	    min1 = 0;
	}
    // if both edges have distance 0 to origin they must intersect there.
	if (min0 == 0 && min1 == 0)
	    return 0;	// crossing edges
    // otherwise take the distance from the closest point that is furthest
    // away to the other edge.
	if (fabs_pp(min0) > fabs_pp(min1)) {
	    x = min0 * dot(e0, e1) - eta[1];
	    if (x < 0)
		return (eta[1]*e1 - min0*e0).length();
	    else {
		sqrd = (eta[1]*e1 - min0*e0).sqlength()-x*x;
		return (sqrd <= 0) ? 0 : sqrt_pp(sqrd);
	    }
	} else {
	    x = min1 * dot(e0, e1) - eta[0];
	    if (x < 0)
		return (eta[0]*e0 - min1*e1).length();
	    else if (x > l0)
		return ((eta[0]+l0)*e0 - min1*e1).length();
	    else {
		sqrd = (eta[0]*e0 - min1*e1).sqlength()-x*x;
		return (sqrd <= 0) ? 0 : sqrt_pp(sqrd);
	    }
	}
    }
}

s_coord clearance(const pl_edge &edge, const pl_ray &ray, s_coord bound)
{
    s_coord cl;
    cl = clearance(edge, ray);
    return cl>bound ? bound : cl;
}

// clearance between an edge and a line.
s_coord clearance(const pl_edge &edge0, const pl_line &line)
{
    pl_unitvec joindir;
    pl_unitvec normal(line.normal());
    const pl_vec &vt_on_line(line.ref_vt());
    s_coord dist1 = dot((edge0.begin_vt() - vt_on_line), normal);
    s_coord dist2 = dot((edge0.begin_vt() - vt_on_line), normal);
    if (dist1 >= 0) {
	if (dist2 <= 0)
	    return 0.0;
	return (dist1 < dist2) ? dist1 : dist2;
    } else {
	if (dist2 >= 0)
	    return 0.0;
	return (dist1 > dist2) ? -dist1 : -dist2;
    }
}

s_coord clearance(const pl_edge &edge, const pl_line &line, s_coord bound)
{
    s_coord cl;
    cl = clearance(edge, line);
    return cl>bound ? bound : cl;
}

s_coord clearance(const pl_edge &edge, const pl_arc &arc)
{
    return clearance(edge, arc, big);
}

s_coord clearance(const pl_edge &edge, const pl_arc &arc, s_coord bound)
{
    if (certain_clearance(edge.bbox(), arc.bbox(), bound)) {
	return bound;
    }
// check special cases;
    {
    pl_unitvec dir;
    s_coord l, l1, l2;
    pl_vec Mvec, beg_vec;
    l2 = factorise(dir, edge.end_vt()-edge.begin_vt());
    beg_vec = edge.begin_vt()-arc.center();
    l1 = dot(beg_vec, dir);
    Mvec = beg_vec - l1*dir;
    l = Mvec.length();
    if (l >= arc.radius()) {
// parallel arc-edge distance
	l2 = l2 + l1;
	if (l1 <= 0 && l2 >= 0) {
	    pl_angle Mangle;
	    Mangle = angle_of(Mvec);
	    if (Mangle.lies_between(arc.small_angle(), arc.big_angle())) {
		s_coord cl = l-arc.radius();
		return cl>bound ? bound : cl;
	    }
	}
    } else {
// intersection
	if (do_intersect(edge, arc))
	    return 0;
    }
    }
    bound = clearance(arc.begin_vt(), edge, bound);
    bound = clearance(arc.end_vt(), edge, bound);
    bound = clearance(edge.begin_vt(), arc, bound);
    bound = clearance(edge.end_vt(), arc, bound);
    return bound;
}

s_coord clearance(const pl_edge &edge, const pl_box &box)
{
    return clearance(edge, box, big);
}

s_coord clearance(const pl_edge &edge, const pl_box &box, s_coord bound)
{
    pl_pgn pgn;
    if (box.is_empty()) return bound;
    pgn_out_of_box(pgn, box);
    return clearance(edge, pgn, bound);
}

s_coord clearance(const pl_edge &edge, const pl_pgn &pgn)
{
    return clearance(edge, pgn, big);
}

s_coord clearance(const pl_edge &e, const pl_pgn &pgn, s_coord bound)
{
    int i;
    s_coord cl;
    if (certain_clearance(e.bbox(), pgn.bbox(), bound))
	return bound;
    if (do_overlap(e, pgn))
	return 0;
    for (i=0; i<pgn.size(); i++) {
	cl = clearance(e, pgn.edge(i));
	if (cl < bound)
	    bound = cl;
    }
    return bound;
}

s_coord clearance(const pl_edge &edge, const pl_disc &disc)
{
    s_coord c;
    c = clearance(edge, disc.center()) - disc.radius();
    return c>0 ? c : 0;
}

s_coord clearance(const pl_edge &edge, const pl_disc &disc, s_coord bound)
{
    s_coord cl;
    cl = clearance(edge, disc.center()) - disc.radius();
    if (cl<0) cl=0;
    return cl>bound ? bound : cl;
}

// ********* RAY ... *********

s_coord clearance(const pl_ray &ray1, const pl_ray &ray2)
{
    pl_unitvec joindir;
    pl_unitvec e0, e1;
    s_coord sqrd;
    pl_vec joinvec = ray2.begin_vt() - ray1.begin_vt();
    e0 = ray1.direction();
    e1 = ray2.direction();
    if ( cross(e0, e1) == 0.0F) {
	// parallel lines
	s_coord alongdist = dot(joinvec, e0);
	s_coord dx;
	s_coord sq_acrossdist = (joinvec- alongdist * e0).sqlength();
	if (dot(e0, e1) > 0) {
	    return sqrt_pp(sq_acrossdist); 
	} else {
	    if ( (dx = -alongdist) >= 0)
		return sqrt_pp(sq_acrossdist + dx*dx);
	    return sqrt_pp(sq_acrossdist); 
	}
    } else {
	s_coord eta[2];
	s_coord min0, min1, x;
    // write joinvec as a combination of e0, e1.
	pl__ontbind2(eta, e0, e1, joinvec);
    // take the intersection point of the supporting lines as origin,
    // e0 and e1 as (non orthogonal) axes.
	eta[0] = -eta[0];
    // compute the minimum distance from both edges to new origin.
	if ((min0 = eta[0]) < 0) {
		min0 = 0;
	}
	if ((min1 = eta[1]) < 0) {
	    min1 = 0;
	}
    // if both edges have distance 0 to origin they must intersect there.
	if (min0 == 0 && min1 == 0)
	    return 0;	// crossing edges
    // otherwise take the distance from the closest point that is furthest
    // away to the other edge.
	if (fabs_pp(min0) > fabs_pp(min1)) {
	    x = min0 * dot(e0, e1) - eta[1];
	    if (x < 0)
		return (eta[1]*e1 - min0*e0).length();
	    else {
		sqrd = (eta[1]*e1 - min0*e0).sqlength()-x*x;
		return (sqrd <= 0) ? 0 : sqrt_pp(sqrd);
	    }
	} else {
	    x = min1 * dot(e0, e1) - eta[0];
	    if (x < 0)
		return (eta[0]*e0 - min1*e1).length();
	    else {
		sqrd = (eta[0]*e0 - min1*e1).sqlength()-x*x;
		return (sqrd <= 0) ? 0 : sqrt_pp(sqrd);
	    }
	}
    }
}

s_coord clearance(const pl_ray &ray1, const pl_ray &ray2, s_coord bound)
{
    s_coord cl;
    cl = clearance(ray1, ray2);
    return cl>bound ? bound : cl;
}

s_coord clearance(const pl_ray &ray, const pl_line &line)
{
    s_coord signed_dist, dy;
    pl_unitvec normal(line.normal());
    signed_dist = dot(ray.begin_vt() - line.ref_vt(), normal);
    dy = dot(ray.direction(), normal);
    if (signed_dist >= 0)
	return (dy < 0) ? 0 : signed_dist;
    else
	return (dy > 0) ? 0 : -signed_dist;
}

s_coord clearance(const pl_ray &ray, const pl_line &line, s_coord bound)
{
    s_coord cl;
    cl = clearance(ray, line);
    return cl>bound ? bound : cl;
}

s_coord clearance(const pl_ray &ray, const pl_arc &arc)
{
    return clearance(ray, arc, big);
}

s_coord clearance(const pl_ray &ray, const pl_arc &arc, s_coord bound)
{
// check special cases;
    {
    pl_unitvec dir;
    s_coord l, l1, cl;
    pl_vec Mvec, beg_vec;
    dir = ray.direction();
    beg_vec = ray.begin_vt()-arc.center();
    l1 = dot(beg_vec, dir);
    Mvec = beg_vec - l1*dir;
    l = Mvec.length();
    cl = l-arc.radius();
    if (cl > bound)
	return bound;
    if (cl >= 0) {
// check parallel arc-ray distance
	if (l1 <= 0) {
	    pl_angle Mangle;
	    Mangle = angle_of(Mvec);
	    if (Mangle.lies_between(arc.small_angle(), arc.big_angle())) {
		return cl;
	    }
	}
    } else {
// check on intersection
	if (do_intersect(ray, arc))
	    return 0;
    }
    }
    s_coord cl;
    cl = clearance(arc.begin_vt(), ray);
    if (cl < bound) bound = cl;
    cl = clearance(arc.end_vt(), ray);
    if (cl < bound) bound = cl;
    cl = clearance(ray.begin_vt(), arc);
    if (cl < bound) bound = cl;
    return bound;
}

s_coord clearance(const pl_ray &ray, const pl_box &box)
{
    return clearance(ray, box, big);
}

s_coord clearance(const pl_ray &ray, const pl_box &box, s_coord bound)
{
    pl_pgn pgn;
    if (box.is_empty()) return bound;
    pgn_out_of_box(pgn, box);
    return clearance(ray, pgn, bound);
}

s_coord clearance(const pl_ray &ray, const pl_pgn &pgn)
{
    return clearance(ray, pgn, big);
}

s_coord clearance(const pl_ray &ray, const pl_pgn &pgn, s_coord bound)
{
    int i;
    s_coord cl;
    if (clearance(pgn.bbox(), ray.supporting_line()) >= bound)
	return bound;
    for (i=0; i<pgn.size(); i++) {
	cl = clearance(ray, pgn.edge(i));
	if (cl < bound)
	    bound = cl;
    }
    return bound;
}

s_coord clearance(const pl_ray &ray, const pl_disc &disc)
{
    s_coord c;
    c = clearance(ray, disc.center()) - disc.radius();
    return c>0 ? c : 0;
}

s_coord clearance(const pl_ray &ray, const pl_disc &disc, s_coord bound)
{
    s_coord cl;
    cl = clearance(ray, disc);
    return cl>bound ? bound : cl;
}

// ********* LINE ... *********

s_coord clearance(const pl_line &line, const pl_arc &arc)
{
    return clearance(line, arc, big);
}

s_coord clearance(const pl_line &line, const pl_arc &arc, s_coord bound)
{
// check special cases;
    {
    pl_unitvec dir;
    s_coord l, cl;
    pl_vec Mvec;
    pl_vec M;
    M = placeproject(arc.center(), line);
    Mvec = M - arc.center();
    l = Mvec.length();
    cl = l-arc.radius();
    if (cl > bound)
	return bound;
    if (cl >= 0) {
// check parallel arc-line distance
	pl_angle Mangle;
	Mangle = angle_of(Mvec);
	if (Mangle.lies_between(arc.small_angle(), arc.big_angle())) {
	    return l-arc.radius();
	}
    } else {
// check on intersection
	if (do_intersect(line, arc))
	    return 0;
    }
    }
    s_coord cl = clearance(arc.begin_vt(), line);
    if (cl < bound) bound = cl;
    cl = clearance(arc.end_vt(), line);
    if (cl < bound) bound = cl;
    return bound;
}

s_coord clearance(const pl_line &line, const pl_boundingbox &bb)
{
    pl_vec normal(line.normal());
    const pl_vec &vt(line.ref_vt());
    s_coord x0, x1, y0, y1;
    if (normal.x() >= 0) {
	x0 = bb.xmin(); x1 = bb.xmax();
    } else {
	x0 = bb.xmax(); x1 = bb.xmin();
    }
    if (normal.y() >= 0) {
	y0 = bb.ymin(); y1 = bb.ymax();
    } else {
	y0 = bb.ymax(); y1 = bb.ymin();
    }
    pl_vec vt0(x0,y0);
    pl_vec vt1(x1,y1);
    s_coord sd0, sd1;	// signed distance from vt0/vt1 to line
    // vt0 and vt1 have been chosen so that for any point p of the bbox
    // signed_dist(vt0, line) <= signed_dist(p,line) <= signed_dist(vt1, line)
    if ((sd0 = dot(normal, vt0 - vt)) >= 0)
	return sd0;
    if ((sd1 = dot(normal, vt1 - vt)) <= 0)
	return -sd1;
    return 0;
}

s_coord clearance(const pl_line &line, const pl_boundingbox &bb, s_coord bound)
{
    s_coord cl;
    cl = clearance(line, bb);
    return cl>bound ? bound : cl;
}

s_coord clearance(const pl_line &line, const pl_pgn &pgn)
{
    return clearance(line, pgn, big);
}

s_coord clearance(const pl_line &line, const pl_pgn &pgn, s_coord bound)
{
    int i;
    s_coord cl;
    if (clearance(pgn.bbox(), line) >= bound)
	return bound;
    for (i=0; i<pgn.size(); i++) {
	cl = clearance(line, pgn.edge(i));
	if (cl < bound)
	    bound = cl;
    }
    return bound;
}

s_coord clearance(const pl_line &line, const pl_disc &disc)
{
    s_coord cl;
    cl = clearance(line, disc.center()) - disc.radius();
    return cl>0 ? cl : 0;
}

s_coord clearance(const pl_line &line, const pl_disc &disc, s_coord bound)
{
    s_coord cl;
    cl = clearance(line, disc);
    return cl>bound ? bound : cl;
}

// ********* ARC ... *********

s_coord clearance(const pl_arc &arc1, const pl_arc &arc2)
{
    return clearance(arc1, arc2, big);
}

s_coord clearance(const pl_arc &arc1, const pl_arc &arc2, s_coord bound)
{
    s_coord cl;
    if (certain_clearance(arc1.bbox(), arc2.bbox(), bound))
	return bound;
// check special cases;
    pl_vec join_vec = arc2.center()-arc1.center();
    s_coord dist = join_vec.length();
    if (dist == 0) {
// case I: concentric arcs
// check for overlap of arc-intervals
	s_coord b1 = arc1.small_angle().value();
	s_coord e1, b2, e2;
	e1 = arc1.big_angle().normalised_value(b1);
	b2 = arc2.small_angle().normalised_value(b1);
	e2 = arc2.big_angle().normalised_value(b1);
// only case without overlap: b1 < e1 < b2 < e2
	if (e1 >= b2 || b2 >= e2) {
	    cl = fabs_pp(arc2.radius()-arc1.radius());
	    return cl>bound? bound : cl;
	}
    } else {
    if (arc1.radius() >= arc2.radius() + dist) {
// case II: circle2 inside circle1
	pl_angle join_angle;
	join_angle = angle_of(join_vec);
	if (join_angle.lies_between(arc1.small_angle(), arc1.big_angle()) &&
	      join_angle.lies_between(arc2.small_angle(), arc2.big_angle())) {
	    cl = arc1.radius()-arc2.radius()-dist;
	    return cl>bound? bound : cl;
	}
    } else {
    if (arc2.radius() >= arc1.radius() + dist) {
// case III: circle1 inside circle2
	pl_angle join_angle;
	join_angle = angle_of(-join_vec);
	if (join_angle.lies_between(arc1.small_angle(), arc1.big_angle()) &&
	      join_angle.lies_between(arc2.small_angle(), arc2.big_angle())) {
	    cl = arc2.radius()-arc1.radius()-dist;
	    return cl>bound? bound : cl;
	}
    } else {
    if (arc1.radius()+arc2.radius() <= dist) {
// case IV: circles outside each other
	pl_angle join_angle1, join_angle2;
	join_angle1 = angle_of(-join_vec);
	join_angle2 = angle_of(join_vec);
	if (join_angle1.lies_between(arc1.small_angle(), arc1.big_angle()) &&
	      join_angle2.lies_between(arc2.small_angle(), arc2.big_angle())) {
	    cl = dist-arc2.radius()-arc1.radius();
	    return cl>bound? bound : cl;
	}
    } else {
// case V: intersecting circles
	if (do_intersect(arc1, arc2))
	    return 0;
    }}}}
    bound = clearance(arc1.begin_vt(), arc2, bound);
    bound = clearance(arc1.end_vt(), arc2, bound);
    bound = clearance(arc2.begin_vt(), arc1, bound);
    bound = clearance(arc2.end_vt(), arc1, bound);
    return bound;
}

s_coord clearance(const pl_arc &arc, const pl_box &box)
{
    return clearance(arc, box, big);
}

s_coord clearance(const pl_arc &arc, const pl_box &box, s_coord bound)
{
    pl_pgn pgn;
    if (box.is_empty()) return bound;
    pgn_out_of_box(pgn, box);
    return clearance(arc, pgn, bound);
}

s_coord clearance(const pl_arc &arc, const pl_pgn &pgn)
{
    return clearance(pgn, arc, big);
}

s_coord clearance(const pl_arc &arc, const pl_pgn &pgn, s_coord bound)
{
    int i;
    if (certain_clearance(arc.bbox(), pgn.bbox(), bound))
	return bound;
    if (lies_inside(arc.begin_vt(), pgn))
	return 0;
    for (i=0; i<pgn.size(); i++) {
	bound = clearance(pgn.edge(i), arc, bound);
    }
    return bound;
}

s_coord clearance(const pl_arc &arc, const pl_disc &disc)
{
    return clearance(disc, arc, big);
}

s_coord clearance(const pl_arc &arc, const pl_disc &disc, s_coord bound)
{
    s_coord cl;
    if (clearance(arc.bbox(), disc) >= bound)
	return bound;
// check special cases;
    pl_vec join_vec = arc.center()-disc.center();
    s_coord dist = join_vec.length();
    if (disc.radius() >= arc.radius() + dist) {
// case II: circle2 inside circle1
	return 0;
    } else {
    if (arc.radius() >= disc.radius() + dist) {
// case III: circle1 inside circle2
	pl_angle join_angle;
	join_angle = angle_of(-join_vec);
	if (dist==0 || join_angle.lies_between(arc.small_angle(), arc.big_angle())) {
	    cl = arc.radius()-disc.radius()-dist;
	    return cl>bound? bound : cl;
	}
    } else {
    if (disc.radius()+arc.radius() <= dist) {
// case IV: circles outside each other
	pl_angle join_angle2;
	join_angle2 = angle_of(join_vec);
	if (join_angle2.lies_between(arc.small_angle(), arc.big_angle())) {
	    cl = dist-arc.radius()-disc.radius();
	    return cl>bound? bound : cl;
	}
    } else {
// case V: intersecting circles
	if (do_overlap(disc, arc))
	    return 0;
    }}}
    bound = clearance(arc.begin_vt(), disc, bound);
    bound = clearance(arc.end_vt(), disc, bound);
    return bound;
}

s_coord open_clearance(const pl_arc &arc, const pl_disc &circle)
{
    return open_clearance(circle, arc, big);
}

s_coord open_clearance(const pl_arc &arc, const pl_disc &circle, s_coord bound)
{
    s_coord cl;
    if (clearance(arc.bbox(), circle) >= bound)
	return bound;
// check special cases;
    pl_vec join_vec = arc.center()-circle.center();
    s_coord dist = join_vec.length();
    if (dist == 0) {
// case I: concentric arcs
	cl = fabs_pp(arc.radius()-circle.radius());
	return cl>bound? bound : cl;
    } else {
    if (circle.radius() >= arc.radius() + dist) {
// case II: circle2 inside circle1
	pl_angle join_angle;
	join_angle = angle_of(join_vec);
	if (join_angle.lies_between(arc.small_angle(), arc.big_angle())) {
	    cl = circle.radius()-arc.radius()-dist;
	    return cl>bound? bound : cl;
	}
    } else {
    if (arc.radius() >= circle.radius() + dist) {
// case III: circle1 inside circle2
	pl_angle join_angle;
	join_angle = angle_of(-join_vec);
	if (join_angle.lies_between(arc.small_angle(), arc.big_angle())) {
	    cl = arc.radius()-circle.radius()-dist;
	    return cl>bound? bound : cl;
	}
    } else {
    if (circle.radius()+arc.radius() <= dist) {
// case IV: circles outside each other
	pl_angle join_angle2;
	join_angle2 = angle_of(join_vec);
	if (join_angle2.lies_between(arc.small_angle(), arc.big_angle())) {
	    cl = dist-arc.radius()-circle.radius();
	    return cl>bound? bound : cl;
	}
    } else {
// case V: intersecting circles
	if (do_cross(circle, arc))
	    return 0;
    }}}}
    bound = clearance(arc.begin_vt(), circle, bound);
    bound = clearance(arc.end_vt(), circle, bound);
    return bound;
}
// ********* BOX ... *********

s_coord clearance(const pl_box &box1, const pl_box &box2)
{
    return clearance(box1, box2, big);
}

s_coord clearance(const pl_box &box1, const pl_box &box2, s_coord bound)
{
    s_coord d, dx, dy;
    if (box1.is_empty())
	return bound;
    if (box2.is_empty())
	return bound;
    dx = box1.xmin() - box2.xmax();
    if (dx < 0) {
	dx = box2.xmin() - box1.xmax();
	if (dx < 0) dx = 0;
    }
    if (dx > bound)
	return bound;
    dy = box1.ymin() - box2.ymax();
    if (dy < 0) {
	dy = box2.ymin() - box1.ymax();
	if (dy < 0) dy = 0;
    }
    if (dy > bound)
	return bound;
    d = sqrt_pp(dx*dx+dy*dy);
    return d>bound ? bound : d;
}

s_coord clearance(const pl_box &box, const pl_pgn &pgn)
{
    return clearance(box, pgn, big);
}

s_coord clearance(const pl_box &box, const pl_pgn &pgn, s_coord bound)
{
    pl_pgn boxpgn;
    if (box.is_empty()) return bound;
    pgn_out_of_box(boxpgn, box);
    return clearance(pgn, boxpgn, bound);
}

s_coord clearance(const pl_boundingbox &box, const pl_disc &disc)
{
    return clearance(disc, box, big);
}

s_coord clearance(const pl_box &box, const pl_disc &disc, s_coord bound)
{
    s_coord cl;
    cl = clearance(disc.center(), box, bound+disc.radius()) - disc.radius();
    if (cl < 0) cl = 0;
    return cl;
}


// ********* POLYGON ... *********

s_coord clearance(  const pl_pgn &pgn1, const pl_pgn &pgn2)
{
    return clearance(pgn1, pgn2, big);
}

/**************
 *  superseded by code following it (not tested).
static s_coord pe_v_sqclearance(const pl_fixedvec &v, const pl_edge &e)
{
    if (e.begin_vt().x() == e.end_vt().x() && e.begin_vt().y() == e.end_vt().y())
	return sqclearance(e.begin_vt(),v);
    if (dot(e.end_vt()-e.begin_vt(),v-e.begin_vt()) > 0 && dot(e.end_vt()-e.begin_vt(),v-e.end_vt()) < 0)
	return sqclearance(v, e.supporting_line());
    return sqclearance(e.begin_vt(),v);
}

s_coord clearance(const pl_pgn &pgn1, const pl_pgn &pgn2, s_coord bound)
{
    s_coord d, sqbound;
    int i, j, pgn1size, pgn2size;
    pl_edge e;
    pl_vec v;
    pl_boundingbox bb1, bb2, bbe;
    if (bound < big)
	sqbound = bound*bound;
    else
	sqbound = big;
    bb1 = pgn1.bbox();
    bb2 = pgn2.bbox();
    if (certain_clearance(bb1, bb2, bound))
	return bound;
    if(do_overlap(pgn1,pgn2))
	return 0;
    pgn1size = pgn1.size();
    pgn2size = pgn2.size();
    if (pgn2size > 1)
	for (j=0 ; j < pgn2size; j++) {
	    e = pgn2.edge(j);
	    bbe = e.bbox();
	    for (i=0 ; i < pgn1size; i++) {
		v = pgn1.vertex(i);
		if (certain_clearance(v, bbe, bound))
		    continue;
		d = pe_v_sqclearance(v, e);
		if (d<sqbound) {
		    sqbound=d;
		    bound =sqrt_pp(sqbound);
		}
	    }
	}
    if (pgn1size > 1)
	for (i=0 ; i < pgn1size; i++) {
	    e = pgn1.edge(i);
	    bbe = e.bbox();
	    for (j=0 ; j < pgn2size; j++) {
		v = pgn2.vertex(j);
		if (certain_clearance(v, bbe, bound))
		    continue;
		d = pe_v_sqclearance(v, e);
		if (d<sqbound) {
		    sqbound=d;
		    bound =sqrt_pp(sqbound);
		}
	    }
	}
    if (pgn1size == 1 && pgn2size == 1) {
	d = sqclearance(pgn2.vertex(0), pgn1.vertex(0));
	if (d<sqbound) {
	    sqbound=d;
	    bound =sqrt_pp(sqbound);
	}
    }
    return bound;
}
******* */


static s_coord pe_v_sqclearance(const pl_fixedvec &v,
	    const pl_edge &edge, const pl_line &supporting_line)
{
    if (dot(edge.end_vt()-edge.begin_vt(),v-edge.begin_vt()) > 0
		&& dot(edge.end_vt()-edge.begin_vt(),v-edge.end_vt()) < 0)
	return sqclearance(v, supporting_line);
    return sqclearance(edge.begin_vt(), v);
// The end_vertex will be taken into account as the begin vertex of the
// next polygon edge.
}

// 
static void vts_edges_clearance(const pl_pgn &pgn1, const pl_pgn &pgn2, 
	    s_coord &bound, s_coord &sqbound)
{
    int i, j;
    pl_edge edge;
    pl_boundingbox bbe;
    s_coord d;
    for (j=0 ; j < pgn2.size(); j++) {
	edge = pgn2.edge(j);
	// check for degenerate edge.
	if (edge.begin_vt().x() == edge.end_vt().x() &&
		    edge.begin_vt().y() == edge.end_vt().y())
	    continue;
	bbe = edge.bbox();
	if (certain_clearance(bbe, pgn1.bbox(), bound))
	    continue;
	const pl_line &supporting_line(edge.supporting_line());
	for (i=0 ; i < pgn1.size(); i++) {
	    const pl_vec &vt(pgn1.vertex(i));
	    if (certain_clearance(vt, bbe, bound))
		continue;
	    d = pe_v_sqclearance(vt, edge, supporting_line);
	    if (d<sqbound) {
		sqbound=d;
		bound =sqrt_pp(sqbound);
	    }
	}
    }
}

s_coord clearance(const pl_pgn &pgn1, const pl_pgn &pgn2, s_coord bound)
{
    s_coord d, sqbound;
    if (certain_clearance(pgn1.bbox(), pgn2.bbox(), bound))
	return bound;
    if(do_overlap(pgn1,pgn2))
	return 0;
    if (bound < big)
	sqbound = bound*bound;
    else
	sqbound = big;
    if (pgn2.size() > 1)
	vts_edges_clearance(pgn1, pgn2, bound,  sqbound);
    if (pgn1.size() > 1)
	vts_edges_clearance(pgn2, pgn1, bound,  sqbound);
    if (pgn1.size() == 1 && pgn2.size() == 1) {
	d = sqclearance(pgn2.vertex(0), pgn1.vertex(0));
	if (d<sqbound) {
	    sqbound=d;
	    bound =sqrt_pp(sqbound);
	}
    }
    return bound;
}



s_coord open_clearance(const pl_pgn &pgn1, const pl_pgn &pgn2)
{
    return open_clearance(pgn1, pgn2, big);
}

s_coord open_clearance(const pl_pgn &pgn1, const pl_pgn &pgn2, s_coord bound)
{
    s_coord d, sqbound;
    int i, j, pgn1size, pgn2size;
    pl_edge e;
    pl_vec v;
    pl_boundingbox bb1, bb2, bbe;
    if (bound < big)
	sqbound = bound*bound;
    else
	sqbound = big;
    bb1 = pgn1.bbox();
    bb2 = pgn2.bbox();
    if (certain_clearance(bb1, bb2, bound))
	return bound;
    pgn1size = pgn1.size();
    pgn2size = pgn2.size();
    if (pgn2size > 1)
	for (j=0 ; j < pgn2size; j++) {
	    e = pgn2.edge(j);
	    bbe = e.bbox();
	const pl_line &supporting_line(e.supporting_line());
	    for (i=0 ; i < pgn1size; i++) {
		v = pgn1.vertex(i);
		if (certain_clearance(v, bbe, bound))
		    continue;
		d = pe_v_sqclearance(v, e, supporting_line);
		if (d<sqbound) {
		    sqbound=d;
		    bound =sqrt_pp(sqbound);
		}
	    }
	}
    if (pgn1size > 1)
	for (i=0 ; i < pgn1size; i++) {
	    e = pgn1.edge(i);
	    bbe = e.bbox();
	const pl_line &supporting_line(e.supporting_line());
	    for (j=0 ; j < pgn2size; j++) {
		v = pgn2.vertex(j);
		if (certain_clearance(v, bbe, bound))
		    continue;
		d = pe_v_sqclearance(v, e, supporting_line);
		if (d<sqbound) {
		    sqbound=d;
		    bound =sqrt_pp(sqbound);
		}
	    }
	}
    if (pgn1size == 1 && pgn2size == 1) {
	d = sqclearance(pgn2.vertex(0), pgn1.vertex(0));
	if (d<sqbound) {
	    sqbound=d;
	    bound =sqrt_pp(sqbound);
	}
    }
    return bound;
}

s_coord clearance(const pl_pgn &pgn, const pl_disc &disc)
{
    s_coord c;
    c = clearance(pgn, disc.center()) - disc.radius();
    return c>0 ? c : 0;
}

s_coord clearance(const pl_pgn &pgn, const pl_disc &disc, s_coord bound)
{
    s_coord c;
    if (certain_clearance(disc.center(), pgn.bbox(), bound+disc.radius())) {
	return bound;
    }
    c = clearance(pgn, disc.center()) - disc.radius();
    if (c<0) c=0;
    return c<=bound ? c : bound;
}

// ********* DISC ... *********

s_coord clearance(const pl_disc &disc1, const pl_disc &disc2)
{
    s_coord c;
    c = clearance(disc1.center(), disc2.center()) - disc1.radius() - disc2.radius();
    return c>0 ? c : 0;
}

s_coord clearance(const pl_disc &disc1, const pl_disc &disc2, s_coord bound)
{
    s_coord c;
    c = clearance(disc1.center(), disc2.center()) - disc1.radius() - disc2.radius();
    if (c<0) c=0;
    return c>bound ? bound : c;
}

// *************  ... OBJECT ****************

s_coord clearance(const pl_fixedvec &vt, const pl_object &obj, s_coord bound)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj, bound);
    case PL_LINE:
	return clearance(vt, *(pl_line *) &obj, bound);
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj, bound);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj, bound);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj, bound);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj, bound);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj, bound);
    }
}

s_coord clearance(const pl_fixedvec &vt, const pl_object &obj)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj);
    case PL_LINE:
	return clearance(vt, *(pl_line *) &obj);
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj);
    }
}


s_coord clearance(const pl_line &vt, const pl_object &obj, s_coord bound)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj, bound);
    case PL_LINE:
	return 0;
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj, bound);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj, bound);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj, bound);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj, bound);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj, bound);
    }
}

s_coord clearance(const pl_line &vt, const pl_object &obj)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj);
    case PL_LINE:
	return 0;
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj);
    }
}

s_coord clearance(const pl_ray &vt, const pl_object &obj, s_coord bound)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj, bound);
    case PL_LINE:
	return clearance(vt, *(pl_line *) &obj, bound);
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj, bound);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj, bound);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj, bound);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj, bound);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj, bound);
    }
}

s_coord clearance(const pl_ray &vt, const pl_object &obj)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj);
    case PL_LINE:
	return clearance(vt, *(pl_line *) &obj);
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj);
    }
}

s_coord clearance(const pl_edge &vt, const pl_object &obj, s_coord bound)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj, bound);
    case PL_LINE:
	return clearance(vt, *(pl_line *) &obj, bound);
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj, bound);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj, bound);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj, bound);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj, bound);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj, bound);
    }
}

s_coord clearance(const pl_edge &vt, const pl_object &obj)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj);
    case PL_LINE:
	return clearance(vt, *(pl_line *) &obj);
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj);
    }
}

s_coord clearance(const pl_pgn &vt, const pl_object &obj, s_coord bound)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj, bound);
    case PL_LINE:
	return clearance(vt, *(pl_line *) &obj, bound);
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj, bound);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj, bound);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj, bound);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj, bound);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj, bound);
    }
}

s_coord clearance(const pl_pgn &vt, const pl_object &obj)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj);
    case PL_LINE:
	return clearance(vt, *(pl_line *) &obj);
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj);
    }
}

s_coord clearance(const pl_disc &vt, const pl_object &obj, s_coord bound)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj, bound);
    case PL_LINE:
	return clearance(vt, *(pl_line *) &obj, bound);
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj, bound);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj, bound);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj, bound);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj, bound);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj, bound);
    }
}

s_coord clearance(const pl_disc &vt, const pl_object &obj)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj);
    case PL_LINE:
	return clearance(vt, *(pl_line *) &obj);
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj);
    }
}

s_coord clearance(const pl_arc &vt, const pl_object &obj, s_coord bound)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj, bound);
    case PL_LINE:
	return clearance(vt, *(pl_line *) &obj, bound);
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj, bound);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj, bound);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj, bound);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj, bound);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj, bound);
    }
}

s_coord clearance(const pl_arc &vt, const pl_object &obj)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj);
    case PL_LINE:
	return clearance(vt, *(pl_line *) &obj);
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj);
    }
}

s_coord clearance(const pl_box &vt, const pl_object &obj, s_coord bound)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj, bound);
    case PL_LINE:
	return clearance(vt, *(pl_line *) &obj, bound);
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj, bound);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj, bound);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj, bound);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj, bound);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj, bound);
    }
}

s_coord clearance(const pl_box &vt, const pl_object &obj)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj);
    case PL_LINE:
	return clearance(vt, *(pl_line *) &obj);
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj);
    }
}

s_coord clearance(const pl_object &vt, const pl_object &obj, s_coord bound)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj, bound);
    case PL_LINE:
	return clearance(vt, *(pl_line *) &obj, bound);
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj, bound);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj, bound);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj, bound);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj, bound);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj, bound);
    }
}

s_coord clearance(const pl_object &vt, const pl_object &obj)
{
    switch (obj.type()) {
    case PL_POINT:
	return clearance(vt, *(pl_point *) &obj);
    case PL_LINE:
	return clearance(vt, *(pl_line *) &obj);
    case PL_RAY:
	return clearance(vt, *(pl_ray *) &obj);
    case PL_EDGE:
	return clearance(vt, *(pl_edge *) &obj);
    case PL_POLYGON:
	return clearance(vt, *(pl_pgn *) &obj);
    case PL_DISC:
	return clearance(vt, *(pl_disc *) &obj);
    case PL_ARC:
	return clearance(vt, *(pl_arc *) &obj);
    }
}



// **********************************

s_coord sqclearance(const pl_pgn &pgn1, const pl_pgn &pgn2,
		    s_coord sqbound)
{
    s_coord d;
    int i,j,pgn1size,pgn2size;
    pl_edge e;
    pl_vec v;
    pl_boundingbox bb1, bb2, bbe;
    bb1 = pgn1.bbox();
    bb2 = pgn2.bbox();
    if (certain_sqclearance(bb1, bb2, sqbound))
	return sqbound;
    if(do_overlap(pgn1,pgn2))
	return 0;
    pgn1size = pgn1.size();
    pgn2size = pgn2.size();
    if (pgn2size > 1)
	for (j=0 ; j < pgn2size; j++) {
	    e = pgn2.edge(j);
	    bbe = e.bbox();
	const pl_line &supporting_line(e.supporting_line());
	    for (i=0 ; i < pgn1size; i++) {
		v = pgn1.vertex(i);
		if (!certain_sqclearance(v, bbe, sqbound)) {
		    d = pe_v_sqclearance(v, e, supporting_line);
		    if (d<sqbound)
			sqbound=d;
		}
	    }
	}
    if (pgn1size > 1)
	for (i=0 ; i < pgn1size; i++) {
	    e = pgn1.edge(i);
	    bbe = e.bbox();
	const pl_line &supporting_line(e.supporting_line());
	    for (j=0 ; j < pgn2size; j++) {
		v = pgn2.vertex(j);
		if (!certain_sqclearance(v, bbe, sqbound)) {
		    d = pe_v_sqclearance(v, e, supporting_line);
		    if (d<sqbound)
			sqbound=d;
		}
	    }
	}
    if (pgn1size == 1 && pgn2size == 1) {
	d = sqclearance(pgn2.vertex(0), pgn1.vertex(0));
	if (d<sqbound) {
	    sqbound=d;
	}
    }
    return sqbound;
}

// **********************************
