// $Modified: Wednesday, August 17, 1994 by otfried $
#include <plageo.h>

const s_coord scoord = 3;
static s_coord big = max_of_type(scoord);
static s_coord littl = min_of_type(scoord);

pl_object_type pl_line::type() const
{
    return PL_LINE;
}

pl_box pl_line::bbox() const
{
    static int initialised = 0;
    static pl_box result;
    if (!initialised) {
	initialised = 1;
	result.add_first_point(pl_vec(littl, littl));
	result.add_non_first_point(pl_vec(big, big));
    }
    return result;  
}

void pl_line::transform(const pl_transform &tfm)
{
    _vt = tfm * _vt;
    _d = tfm * _d;
}

void pl_line::translate(const pl_fixedvec &trans)
{
    _vt += trans;
}

void pl_line::scale(s_coord fac)
{
    _vt *= fac;
    if (fac < 0)
	_d = -_d;
}

s_coord pl_line::placeproject(const pl_fixedvec &vec) const
{
    return dot(vec-_vt, _d);
}

s_coord pl_line::dirproject(const pl_fixedvec &vec) const
{
    return dot(vec, _d);
}


pl_object_type pl_ray::type() const
{
    return PL_RAY;
}

pl_box pl_ray::bbox() const
{
    static int initialised = 0;
    static pl_box result;
    if (!initialised) {
	s_coord xextreme,  yextreme;
	initialised = 1;
	result.add_first_point(_begin_vt);
	xextreme = _d.x() > 0 ? big : littl;
	yextreme = _d.y() > 0 ? big : littl;
	result.add_non_first_point(pl_vec(xextreme, yextreme));
    }
    return result;  
}

void pl_ray::transform(const pl_transform &tfm)
{
    _begin_vt = tfm * _begin_vt;
    _d = tfm * _d;
}

void pl_ray::translate(const pl_fixedvec &trans)
{
    _begin_vt += trans;
}

void pl_ray::scale(s_coord fac)
{
    _begin_vt *= fac;
    if (fac < 0)
	_d = -_d;
}

pl_object_type pl_edge::type() const
{
    return PL_EDGE;
}

// return the bounding box of an edge
pl_boundingbox pl_edge::bbox() const
{
    pl_boundingbox     bb;
    bb.add_first_point(begin_vt());
    bb.add_non_first_point(end_vt());
    return bb;
}

void pl_edge::transform(const pl_transform &tfm)
{
    _begin_vt = tfm * _begin_vt;
    _end_vt = tfm * _end_vt;
}

void pl_edge::translate(const pl_fixedvec &disp)
{
    _begin_vt += disp;
    _end_vt += disp;
}

void pl_edge::scale(s_coord fac)
{
    _begin_vt *= fac;
    _end_vt *= fac;
}

// ************    ******************


pl_vec placeproject(const pl_fixedvec &vec, const pl_line &l)
{
    const pl_vec &vt_on_line = l.ref_vt();
    const pl_unitvec &dir = l.direction();
    return vt_on_line + dot(vec-vt_on_line, dir) * dir;
}

pl_vec dirproject(const pl_fixedvec &vec, const pl_line &l)
{
    const pl_unitvec &dir = l.direction();
    return dot(vec, dir) * dir;
}

int is_to_the_left(const pl_fixedvec &vec, const pl_line &line)
{
    return cross(line.direction(), vec-line.ref_vt()) > 0;
}

int is_to_the_right(const pl_fixedvec &vec, const pl_line &line)
{
    return cross(line.direction(), vec-line.ref_vt()) < 0;
}

// does a vertex lie above a line ?
int is_above(const pl_fixedvec &vec, const pl_line & line)
{
    pl_vec dir(line.direction());
    if (dir.x() < 0 || (dir.x() == 0 && dir.y() < 0))
	return (is_to_the_right(vec, line));
    else
	return (is_to_the_left(vec, line));
}

// does a vertex lie below a line ?
int is_below(const pl_fixedvec &vec, const pl_line & line)
{
    pl_vec dir(line.direction());
    if (dir.x() < 0 || (dir.x() == 0 && dir.y() < 0))
	return (is_to_the_left(vec, line));
    else
	return (is_to_the_right(vec, line));
}



// ********** General Intervals on a Line *****************


// On two halflines, starting in the intersectionpoint and making an acute
// angle with each other lie a number of intervals.
// Find the shortest connection between any two intervals on different half
// lines if this connection is shorter than bound.
// cos_enclosed is the cosine of the acute angle.
// The intervals lie in the positive halfline.

static s_coord find_short_connection(const intervals iv[2],
	    s_coord cos_enclosed, s_coord bound)
{
// Find the shortest connection between two sets of intervals, each lying on a
// halfline. The halflines start from a common vertex, forming a wedge.
// The cosine of the angle between the halflines is `cos_enclosed'.
// The angle must be acute, so cos_enclosed must be non-negative.
// The endpoints of the intervals are given by `iv' and are sorted.
// Those values give the distance from the wedge-point and must be >=0.
// If there is an odd number of values, the last interval extends to infinity.
// The value returned is the square of the clearance between the intervals.
// If this value is bigger than `bound', then `bound' is returned instead.
//
// The intervals are treated one by one, from close to the origin to far.
// `close' indicates the halfline on which the interval closest to the origin
// that is not yet treated lies.
// `far' indicates the other halfline.
// `dclose' is the distance from the point that is treated on the close-line
// to the origin.
// `dfar' is the distance from the point that is treated on far to the origin.
// This is always a startpoint of an interval.
// This point is called `the far point' or `far' for short.
// `dftol' is the square of the Distance from the Far point To the Other Line.
// `pfool' is the distance of the Projection of the Far point On the Other Line.
    int close = 0, far = 1;
    s_coord dclose, dfar;
    s_coord sql;
    s_coord dftol; 
    s_coord pfool;
    int cur_index[2];
    cur_index[0] = cur_index[1] = 0;
    int limit[2];
    limit[0] = iv[0].boundaries.size();
    limit[1] = iv[1].boundaries.size();
    if ( cur_index[far] == limit[far])
	return bound;
    dfar =  (iv[far].boundaries[cur_index[far]]);
    while (1) {
    // check if there is another interval.
	if (cur_index[close] == limit[close])
	    return bound;
    // check if close point is further than far point. If so swap.
	dclose = 
		(iv[close].boundaries[cur_index[close]]);
	if ( dclose > dfar) {
	    far = 1-far;
	    close = 1-close;
	    dfar = dclose;
	    continue;
	}
	pfool = dfar * cos_enclosed;
	dftol = dfar*dfar - pfool*pfool;
    // check if there is a chance of finding anything better.
	if ( dftol >= bound)
	    return bound;
    // check if the interval starts beyond the projection of far.
	if (dclose >= pfool) {
	    sql = dclose*dclose + dfar*dfar - 2*dclose*dfar*cos_enclosed;
	    if (sql < bound)
		bound = sql;
	    return bound;
	}
    // now look at the endpoint of the interval.
    // check if the endpoint extends beyond the projection of far.
	cur_index[close] ++;
	if (cur_index[close] == limit[close])	// interval extends to infinity
	    return dftol;
	dclose = iv[close].boundaries[cur_index[close]];
	if (dclose > pfool)
	    return dftol;
    // compare distance of this endpoint to far with bound.
	sql = dclose*dclose + dfar*dfar - 2*dclose*dfar*cos_enclosed;
	if (sql < bound)
	    bound = sql;
    // go to next interval.
	cur_index[close] ++;
    }
}


static void split_in_wedges(intervals wedges[2][2], s_coord *cos_enclosed,
	const pl_dotted_line &iv0, const pl_dotted_line &iv1)
{
    int i0, i1, index0, index1;
    s_coord eta[2];
    pl_unitvec e0(iv0.direction()), e1(iv1.direction());
    pl_vec joinvec = iv1.ref_vt() - iv0.ref_vt();
    int afterorigin[2];
// write joinvec as a combination of e0, e1.
    pl__ontbind2(eta, e0, e1, joinvec);
// take the intersection point of the supporting lines as new origin,
// e0 and e1 as (non orthogonal) axes.
    eta[1] = -eta[1];
// now origin == ref_vt0 + eta[0]*e0
//     origin == ref_vt1 + eta[1]*e1
    afterorigin[0] = iv0.fiv.find_index_after(eta[0]);
    afterorigin[1] = iv1.fiv.find_index_after(eta[1]);
    *cos_enclosed = dot(e0, e1);
    wedges[0][0].starts_at_min_infinity = 0;
    wedges[0][1].starts_at_min_infinity = 0;
    wedges[1][0].starts_at_min_infinity = 0;
    wedges[1][1].starts_at_min_infinity = 0;
// make wedge-arms of first interval
    if (iv0.fiv.is_interval_endpoint(afterorigin[0])) {
	wedges[0][0].boundaries.newsize(iv0.fiv.boundaries.size() -
							    afterorigin[0]+1);
	wedges[1][0].boundaries.newsize(afterorigin[0]+1);
	wedges[0][0].boundaries.replace(0,0.0F);
	wedges[1][0].boundaries.replace(0,0.0F);
	i0 = 1;
	i1 = 1;
    } else {
	wedges[0][0].boundaries.newsize(iv0.fiv.boundaries.size() -
							    afterorigin[0]);
	wedges[1][0].boundaries.newsize(afterorigin[0]);
	i0 = 0;
	i1 = 0;
    }
    for ( index0 = afterorigin[0] ; index0 < iv0.fiv.boundaries.size();
	    index0++, i0++)
	wedges[0][0].boundaries.replace(i0, iv0.fiv.boundaries[index0] - eta[0]);
    for ( index1 = afterorigin[0] - 1; index1 >= 0; index1--, i1++)
	wedges[1][0].boundaries.replace(i1, -eta[0] - iv0.fiv.boundaries[index1]);
// make wedge-arms of second interval
    int one, other;
    if (*cos_enclosed > 0) {
	one = 0; other = 1;
    } else {
	one = 1; other = 0; *cos_enclosed = - *cos_enclosed;
    }
    if (iv1.fiv.is_interval_endpoint(afterorigin[1])) {
	wedges[one][1].boundaries.newsize(iv1.fiv.boundaries.size() -
							afterorigin[1]+1);
	wedges[other][1].boundaries.newsize(afterorigin[1]+1);
	wedges[one][1].boundaries.replace(0,0.0F);
	wedges[other][1].boundaries.replace(0,0.0F);
	i0 = 1;
	i1 = 1;
    } else {
	wedges[one][1].boundaries.newsize(iv1.fiv.boundaries.size() -
							afterorigin[1]);
	wedges[other][1].boundaries.newsize(afterorigin[1]);
	i0 = 0;
	i1 = 0;
    }
    for ( index0 = afterorigin[1] ; index0 < iv1.fiv.boundaries.size();
	    index0++, i0++)
	wedges[one][1].boundaries.replace(i0,
				    iv1.fiv.boundaries[index0] - eta[1]);
    for ( index1 = afterorigin[1] - 1; index1 >= 0; index1--, i1++)
	wedges[other][1].boundaries.replace(i1,
				    -eta[1] - iv1.fiv.boundaries[index1]);
}

// IPE fix: type was missing!
inline s_coord sq(const s_coord &x)
{ return x*x; }

s_coord clearance(const pl_dotted_line &iv0,
	const pl_dotted_line &iv1)
{
    pl_unitvec joindir;
    pl_unitvec e0(iv0.direction()), e1(iv1.direction());
    if ( cross(e0, e1) == 0.0F) {
	// parallel lines
	int i;
	int iv1size = iv1.fiv.boundaries.size();
	s_coord dx,dy;
	intervals tempfiv;
	tempfiv.boundaries.newsize(iv1size);
	tempfiv.starts_at_min_infinity =
			iv1.fiv.is_interval_endpoint(iv1size);
	pl_vec diffvec(iv1.ref_vt() - iv0.ref_vt());
	dx = dot(diffvec, e0);
	if (dot(e0, e1) > 0)
	    for (i = 0; i<iv1size; i++)
		tempfiv.boundaries.replace(i, dx + iv1.fiv.boundaries[i]);
	else
	    for (i = 0; i<iv1size; i++)
	       tempfiv.boundaries.replace(iv1size-i-1,dx-iv1.fiv.boundaries[i]);
	dx = shortest_gap(iv0.fiv, tempfiv);
	dy = dot(diffvec, e0.normal());
	return sqrt_pp(dx*dx + dy*dy);
    } else {
	intervals wedges[2][2];
	s_coord cos_enclosed;
	split_in_wedges(wedges, &cos_enclosed, iv0, iv1);
    // now first calculate bound out of min dist between different wedges.
	s_coord found_sq_dist, sqlen;
	found_sq_dist = big;
	if (wedges[0][0].boundaries.size() > 0
		&& wedges[1][1].boundaries.size() > 0){
	    sqlen = sq(wedges[0][0].boundaries[0]) +
		sq(wedges[1][1].boundaries[0]) +
		2 * cos_enclosed*wedges[0][0].boundaries[0] *
						wedges[1][1].boundaries[0];
	    if (sqlen < found_sq_dist)
		found_sq_dist = sqlen;
	}
	if (wedges[0][1].boundaries.size() > 0
		&& wedges[1][0].boundaries.size() > 0){
	    sqlen = sq(wedges[0][1].boundaries[0]) +
		sq(wedges[1][0].boundaries[0]) +
		2 * cos_enclosed*wedges[0][1].boundaries[0] *
						wedges[1][0].boundaries[0];
	    if (sqlen < found_sq_dist)
		found_sq_dist = sqlen;
	}
    // now compute the distances between the wedge-line intervals
	found_sq_dist=find_short_connection(wedges[0],
						cos_enclosed,found_sq_dist);
	found_sq_dist=find_short_connection(wedges[1],
						cos_enclosed,found_sq_dist);
	return sqrt_pp(found_sq_dist);
    }
}


