#include <limits.h>
#include <plageo.h>

static pl_vec from_polar_coords(s_coord radius, pl_angle angle)
{
    return pl_vec(radius * cos_pp(angle), radius * sin_pp(angle));
}

pl_object_type pl_disc::type() const
{
    return PL_DISC;
}

void pl_disc::transform(const pl_transform &tfm)
{
    _center = tfm * _center;
    _radius = tfm * _radius;
}

void pl_disc::translate(const pl_fixedvec &v)
{
    _center += v;
}

s_coord pl_disc::area() const
{
    return M_PI * _radius * _radius;
}

void pl_disc::scale(s_coord fac)
{
    _center *= fac;
    _radius *= fabs_pp(fac);
}

pl_boundingbox pl_disc::bbox() const
{
    pl_boundingbox bb;
    pl_vec extreme(_radius, _radius);
    bb.add_first_point(_center + extreme);
    bb.add_non_first_point(_center - extreme);
    return bb;
}

void pl_disc::outer_approximation_d(pl_vecreray &result, s_coord epsilon) const
// benader disc door aantal line-segments.
// Verste afstand line-segmnent-disc is epsilon.
{
    // bepaal aantal rechthoekige driehoeken dat nodig is
    // assert(epsilon > 0)
    int n;
    s_coord r = radius();
    s_coord outradius = r + epsilon;
    s_coord nf;
    nf = 2*ceil_pp(fabs_pp(M_PI)/(acos_pp(r/outradius)));
    n = (nf >= INT_MAX) ? INT_MAX : (int) nf;
    outer_approximation_n(result, n);
}

void pl_disc::outer_approximation_n(pl_vecreray &result, int n) const
// benader disc door aantal n line-segments.
{
    int sym, period, i;
    if (n<3) n=3;
    double phi = 2*M_PI/n;
    s_coord outradius = radius()/cos_pp(phi/2);
    result.newsize(n);
    sym = n%4;
    switch (sym) {
    case 0: period = n/4; break;
    case 2: period = n/2; break;
    default: period = n; break;
    }
    double costhe, sinthe, cosphi, sinphi;
    cosphi = cos(phi); sinphi = sin(phi);
    costhe = 1; sinthe = 0;
    result[0] = pl_vec(outradius*costhe, outradius*sinthe);
    for (i=1; i<period; i++) {
	double newcos = costhe*cosphi - sinthe*sinphi;
	sinthe = sinthe*cosphi + costhe*sinphi;
	costhe = newcos;
	result[i] = pl_vec(outradius*costhe, outradius*sinthe);
    }
    if (sym == 0) {
	for (i=0; i<period; i++)
	    result[period+i] = pl_vec(-result[i].y(), result[i].x());
	period = 2*period;
    }
    if (sym == 0 || sym == 2)
	for (i=0; i<period; i++)
	    result[period+i] = pl_vec(-result[i].x(), -result[i].y());
    for (i=0; i<n; i++)
	result[i] = result[i]+center();
}
    
void pl_disc::inner_approximation_d(pl_vecreray &result, s_coord epsilon) const
// benader disc door aantal line-segments.
// Verste afstand line-segmnent-disc is epsilon.
{
    // bepaal aantal rechthoekige driehoeken dat nodig is
    // assert(epsilon > 0)
    int i, n;
    s_coord r = radius();
    if (epsilon >= r)
	n=3;
    else {
	s_coord nf;
	nf = ceil_pp(fabs_pp(M_PI)/(acos_pp((r-epsilon)/r)));
	if (nf >= INT_MAX)
	    n = INT_MAX;
	else {
	    n = (int) nf;
	    if (n<3) n=3;
	}
    }
    pl_angle step = 2*M_PI/n;
    result.newsize(n);
    for (i=0; i<n; i++) {
	result[i] = center() + from_polar_coords(r, i*step);
    }
}
    

pl_object_type pl_arc::type() const
{
    return PL_ARC;
}

pl_boundingbox pl_arc::bbox() const
{
    if (_bbox.is_empty())
	((pl_arc *)this)->make_bbox();
    return _bbox;
}

void pl_arc::make_bbox()
{
    _bbox.add_first_point(begin_vt());
    _bbox.add_non_first_point(end_vt());
    pl_angle smallangle, bigangle;
    if (_angle == 0)
	return;
    smallangle = small_angle();
    bigangle = big_angle();
    if (increasing_angle_order(smallangle, 0, bigangle))
	_bbox.add_point(pl_vec(_disc.center().x()+_disc.radius(),
						    _disc.center().y()));
    if (increasing_angle_order(smallangle, M_PI_2, bigangle))
	_bbox.add_point(pl_vec(_disc.center().x(),
				    _disc.center().y()+_disc.radius()));
    if (increasing_angle_order(smallangle, M_PI, bigangle))
	_bbox.add_point(pl_vec(_disc.center().x()-_disc.radius(),
						    _disc.center().y()));
    if (increasing_angle_order(smallangle, -M_PI_2, bigangle))
	_bbox.add_point(pl_vec(_disc.center().x(),
					_disc.center().y()-_disc.radius()));
}
    
pl_vec pl_arc::begin_vt() const
{
    return _disc.center() + from_polar_coords(_disc.radius(), begin_angle());
}

pl_vec pl_arc::end_vt() const
{
    pl_vec co;
    co = from_polar(pl_vec(_disc.radius(), end_angle()));
    return _disc.center() + co;
}

pl_vec pl_arc::small_vt() const
{
    return (_angle.value() >= 0) ? 
	_disc.center() + from_polar_coords(_disc.radius(), _startangle) :
	_disc.center() +
	    from_polar_coords(_disc.radius(), pl_angle(_startangle+_angle));
}

pl_vec pl_arc::big_vt() const
{
    return (_angle.value() <= 0) ? 
	_disc.center() + from_polar_coords(_disc.radius(), _startangle) :
	_disc.center() +
	    from_polar_coords(_disc.radius(), pl_angle(_startangle+_angle));
}

void pl_arc::transform(const pl_transform &tfm)
{
    _disc.transform(tfm);
    _startangle = tfm * _startangle;
    _bbox.make_empty();
}

void pl_arc::translate(const pl_fixedvec &v)
{
    _disc.translate(v);
    _bbox.make_empty();
}

void pl_arc::scale(s_coord fac)
{
    _disc.scale(fac);
    if (fac < 0) {
	_startangle = _startangle + M_PI;
    }
    _bbox.make_empty();
}

pl_arc::pl_arc(const pl_fixedvec &vt, s_coord radius, pl_angle startangle, 
			    pl_angle enclosed_angle) : _disc(vt, radius)
{
    _startangle = startangle;
// assert -2pi <= enclosed_angle <= 2pi
    _angle = enclosed_angle;
    _bbox.make_empty();
}

pl_arc::pl_arc(const pl_disc &c, pl_angle startangle, 
			    pl_angle enclosed_angle) : _disc(c)
{
    _startangle = startangle;
// assert -2pi <= enclosed_angle <= 2pi
    _angle = enclosed_angle;
    _bbox.make_empty();
}

pl_angle pl_arc::small_angle() const
{
    return (_angle.value() >= 0) ? _startangle : pl_angle(_startangle+_angle);
}

pl_angle pl_arc::big_angle() const
{
    return (_angle >= 0) ? pl_angle(_startangle+_angle) : _startangle;
}

void pl_arc::set_angles(pl_angle startangle, pl_angle diffangle)
{
    _startangle = startangle;
// assert -2pi <= diffangle <= 2pi
    _angle = diffangle;
    _bbox.make_empty();
}

/*
void pl_arc::outer_approximation(pl_vecreray &result, s_coord epsilon) const
// benader arc door aantal line-segments.
// Verste afstand line-segmnent-arc is epsilon.
{
    // bepaal aantal rechthoekige driehoeken dat nodig is
    // assert(epsilon > 0)
    int i, n;
    s_coord r = radius();
    s_coord outradius = r + epsilon;
    s_coord nf;
    nf = 2*ceil_pp(fabs_pp(enclosed_angle())/(2*acos_pp(r/outradius)));
    if (nf >= INT_MAX)
	n = INT_MAX;
    else {
	n = (int) nf;
	if (n<2) n=2;   // shouldn't be necessary
    }
    pl_angle step = enclosed_angle()/n;
    if (n < 3)
	outradius = r/cos_pp(step);
    result.newsize(0);
    result.append(center() + from_polar_coords(r, begin_angle()));
    for (i=1; i<n; i+=2) {
	result.append(center()+
			from_polar_coords(outradius, begin_angle()+i*step));
    }
    result.append(center() + from_polar_coords(r, end_angle()));
}
*/
    
void pl_arc::outer_approximation_d(pl_vecreray &result, s_coord epsilon) const
// benader arc door aantal line-segments.
// Verste afstand line-segmnent-arc is epsilon.
// Uses formulas:   cos(the+phi) = cos(the).cos(phi) - sin(the).sin(phi)
//		    sin(the+phi) = sin(the).cos(phi) + cos(the).sin(phi)
{
    // bepaal aantal rechthoekige driehoeken dat nodig is
    // assert(epsilon > 0)
    int n;
    s_coord r = radius();
    s_coord outradius = r + epsilon;
    s_coord nf;
    nf = ceil_pp(2+fabs_pp(enclosed_angle())/(2*acos_pp(r/outradius)));
    n = (nf >= INT_MAX) ?  INT_MAX :  (int) nf;
    outer_approximation_n(result, n);
}

void pl_arc::outer_approximation_n(pl_vecreray &result, int m) const
{
    // assert(n>=1)
    int i;
    if (m<3) m=3;
    if (m==3 && fabs_pp(enclosed_angle().value()) >= M_PI)
	m = 4;
    double phi = enclosed_angle()/(m-2);
    s_coord outradius = radius()/cos_pp(phi/2);
    result.newsize(m);
    result[0] = begin_vt();
    result.last() = end_vt();
    double mid_angle = begin_angle().value() + enclosed_angle().value()/2;
    if (m%2 == 0) {
	double costhe, sinthe, cosphi, sinphi;
	int midn;
	midn = m/2;
	cosphi = cos(phi); sinphi = sin(phi);
	costhe = cos(mid_angle+phi/2); sinthe = sin(mid_angle+phi/2);
	result[midn] = center()+outradius*pl_vec(costhe, sinthe);
	for (i=midn+1; i<m-1; i++) {
	    double newcos = costhe*cosphi - sinthe*sinphi;
	    sinthe = sinthe*cosphi + costhe*sinphi;
	    costhe = newcos;
	    result[i] = center()+outradius*pl_vec(costhe, sinthe);
	}
	
	midn = m/2-1;
	sinphi = -sinphi;
	costhe = cos(mid_angle-phi/2); sinthe = sin(mid_angle-phi/2);
	result[midn] = center()+outradius*pl_vec(costhe, sinthe);
	for (i=midn-1; i>0; i--) {
	    double newcos = costhe*cosphi - sinthe*sinphi;
	    sinthe = sinthe*cosphi + costhe*sinphi;
	    costhe = newcos;
	    result[i] = center()+outradius*pl_vec(costhe, sinthe);
	}
    } else {
	double costhe, sinthe, cosphi, sinphi;
	int midn = (m-1)/2;
	cosphi = cos(phi); sinphi = sin(phi);
	costhe = cos(mid_angle); sinthe = sin(mid_angle);
	result[midn] = center()+outradius*pl_vec(costhe, sinthe);
	for (i=midn+1; i<m-1; i++) {
	    double newcos = costhe*cosphi - sinthe*sinphi;
	    sinthe = sinthe*cosphi + costhe*sinphi;
	    costhe = newcos;
	    result[i] = center()+outradius*pl_vec(costhe, sinthe);
	}
	sinphi = -sinphi;
	costhe = cos(mid_angle); sinthe = sin(mid_angle);
	for (i=midn-1; i>0; i--) {
	    double newcos = costhe*cosphi - sinthe*sinphi;
	    sinthe = sinthe*cosphi + costhe*sinphi;
	    costhe = newcos;
	    result[i] = center()+outradius*pl_vec(costhe, sinthe);
	}
    }
}
    
void pl_arc::inner_approximation_d(pl_vecreray &result, s_coord epsilon) const
// benader arc door aantal line-segments.
// Verste afstand line-segmnent-arc is epsilon.
{
    // bepaal aantal rechthoekige driehoeken dat nodig is
    // assert(epsilon > 0)
    int n;
    s_coord r = radius();
    if (epsilon >= r)
	n=2;
    else {
	s_coord nf;
	nf = ceil_pp(1+fabs_pp(enclosed_angle())/
					    (2*acos_pp((r-epsilon)/r)));
	n = (nf >= INT_MAX) ? INT_MAX : (int) nf;
    }
    inner_approximation_n(result, n);
}


void pl_arc::inner_approximation_n(pl_vecreray &result, int n) const
{
    int i;
    s_coord r = radius();
    if (n<2) n=2;
    result.newsize(n);
    result.last() = end_vt();
    pl_angle phi = enclosed_angle()/(n-1);
    
    double costhe, sinthe, cosphi, sinphi;
    cosphi = cos(phi); sinphi = sin(phi);
    costhe = cos(begin_angle()); sinthe = sin(begin_angle());
    for (i=0; i<n-1; i++) {
	result[i] = center()+r*pl_vec(costhe, sinthe);
	double newcos = costhe*cosphi - sinthe*sinphi;
	sinthe = sinthe*cosphi + costhe*sinphi;
	costhe = newcos;
    }
}


// ***********************************************
// Here are some routines concerning Axis Centered Ellipses (ACEs)
// ACE have their center in the origin. Their big axis lies along the x-axis
// and the y-axis is the small axis.
// use Newton Rhapson method to approximate closest point on ellipse
// Find the angle that minimises the square of the distance,
// (vt.x()-ell.x()*cos(angle))^2 + (vt.y()-ell.y()*sin(angle))^2
// This is done by finding the angle where the first derivative is zero
// and the second derivative is positive.

s_coord wn_open_clearance(pl_vec &wn, const pl_fixedvec &vt,
	const pl_ACE &ell, s_coord bound)
{
    s_coord angle;
    s_coord e1, e2, e3, prevdiff, diff, nom, denom;
    pl_box bb;
    bb.add_first_point(pl_vec(-ell.x(), -ell.y()));
    bb.add_non_first_point(pl_vec(ell.x(), ell.y()));
    if (certain_clearance(vt, bb, bound))
	return bound;
    e1 = ell.x()*vt.x();
    e2 = ell.y()*vt.y();
    e3 = ell.y()*ell.y()-ell.x()*ell.x();
    angle = (vt.y() >= 0) ? M_PI/2 : -M_PI/2;	// starting angle
    diff = 10.0;		// something big enough : stop condition
    
    s_coord sina, cosa;
    do {
	prevdiff = diff;
	sina = sin_pp(angle);
	cosa = cos_pp(angle);
	nom = (e1*sina-e2*cosa+e3*cosa*sina);	// first derivative
	denom = e1*cosa+e2*sina+e3*(cosa*cosa-sina*sina);   // second derivative
	if (denom > 0) {
	    diff = nom / denom;	// the normal NR step
	    // but do not take too big steps.
	    if (diff > 0.5)
		diff = 0.5;
	    if (diff < -0.5)
		diff = -0.5;
	} else {
	    // the normal NR step would lead to a maximum (f'' < 0)
	    diff = (nom > 0) ? 0.5 : -0.5;
	}
	angle = angle - diff;
	diff = fabs_pp(diff);
    } while (diff<prevdiff || diff>0.1);
    
    wn.set_x(ell.x()*cosa);
    wn.set_y(ell.y()*sina);
    s_coord result = clearance(wn, vt);
    return (result>= bound) ? bound : result;
}

s_coord wn_clearance(pl_vec &wn, const pl_fixedvec &vt, const pl_ACE &ell, s_coord bound)
{
    s_coord x, y;
    if (vt.x() == 0)
	x = 0;
    else if (ell.x() == 0)
	x = 2;
    else
	x = vt.x()/ell.x();
    if (vt.y() == 0)
	y = 0;
    else if (ell.y() == 0)
	y = 2;
    else
	y = vt.y()/ell.y();
    if (x*x+y*y <= 1) {
	wn = vt;
	return 0;
    }
    return wn_open_clearance(wn, vt, ell, bound);
}

s_coord wn_open_clearance(pl_vec &wn, const pl_fixedvec &vt, const pl_ACE &ell)
{
    s_coord angle;
    s_coord e1, e2, e3, prevdiff, diff, nom, denom;
    e1 = ell.x()*vt.x();
    e2 = ell.y()*vt.y();
    e3 = ell.y()*ell.y()-ell.x()*ell.x();
    angle = (vt.y() >= 0) ? M_PI/2 : -M_PI/2;	// starting angle
    diff = 10.0;		// something big enough : stop condition
    
    s_coord sina, cosa;
    do {
	prevdiff = diff;
	sina = sin_pp(angle);
	cosa = cos_pp(angle);
	nom = (e1*sina-e2*cosa+e3*cosa*sina);	// first derivative
	denom = e1*cosa+e2*sina+e3*(cosa*cosa-sina*sina);   // second derivative
	if (denom > 0) {
	    diff = nom / denom;	// the normal NR step
	    // but do not take too big steps.
	    if (diff > 0.5)
		diff = 0.5;
	    if (diff < -0.5)
		diff = -0.5;
	} else {
	    // the normal NR step would lead to a maximum (f'' < 0)
	    diff = (nom > 0) ? 0.5 : -0.5;
	}
	angle = angle - diff;
	diff = fabs_pp(diff);
    } while (diff<prevdiff || diff>0.1);
    
    wn.set_x(ell.x()*cosa);
    wn.set_y(ell.y()*sina);
    return clearance(wn, vt);
}

s_coord wn_clearance(pl_vec &wn, const pl_fixedvec &vt, const pl_ACE &ell)
{
    s_coord x, y;
    if (vt.x() == 0)
	x = 0;
    else if (ell.x() == 0)
	x = 2;
    else
	x = vt.x()/ell.x();
    if (vt.y() == 0)
	y = 0;
    else if (ell.y() == 0)
	y = 2;
    else
	y = vt.y()/ell.y();
    if (x*x+y*y <= 1) {
	wn = vt;
	return 0;
    }
    return wn_open_clearance(wn, vt, ell);
}

s_coord open_clearance(const pl_fixedvec &vt, const pl_ACE &ell, s_coord bound)
{
    s_coord angle;
    s_coord e1, e2, e3, prevdiff, diff, nom, denom;
    pl_box bb;
    bb.add_first_point(pl_vec(-ell.x(), -ell.y()));
    bb.add_non_first_point(pl_vec(ell.x(), ell.y()));
    if (certain_clearance(vt, bb, bound))
	return bound;
    e1 = ell.x()*vt.x();
    e2 = ell.y()*vt.y();
    e3 = ell.y()*ell.y()-ell.x()*ell.x();
    angle = (vt.y() >= 0) ? M_PI/2 : -M_PI/2;	// starting angle
    diff = 10.0;		// something big enough : stop condition
    
    s_coord sina, cosa;
    do {
	prevdiff = diff;
	sina = sin_pp(angle);
	cosa = cos_pp(angle);
	nom = (e1*sina-e2*cosa+e3*cosa*sina);	// first derivative
	denom = e1*cosa+e2*sina+e3*(cosa*cosa-sina*sina);   // second derivative
	if (denom > 0) {
	    diff = nom / denom;	// the normal NR step
	    // but do not take too big steps.
	    if (diff > 0.5)
		diff = 0.5;
	    if (diff < -0.5)
		diff = -0.5;
	} else {
	    // the normal NR step would lead to a maximum (f'' < 0)
	    diff = (nom > 0) ? 0.5 : -0.5;
	}
	angle = angle - diff;
	diff = fabs_pp(diff);
    } while (diff<prevdiff || diff>0.1);
    
    pl_vec wn(ell.x()*cosa, ell.y()*sina);
    return clearance(wn, vt, bound);
}

s_coord clearance(const pl_fixedvec &vt, const pl_ACE &ell, s_coord bound)
{
    s_coord x, y;
    if (vt.x() == 0)
	x = 0;
    else if (ell.x() == 0)
	x = 2;
    else
	x = vt.x()/ell.x();
    if (vt.y() == 0)
	y = 0;
    else if (ell.y() == 0)
	y = 2;
    else
	y = vt.y()/ell.y();
    if (x*x+y*y <= 1) {
	return 0;
    }
    return open_clearance(vt, ell, bound);
}

