/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: Chio.cc,v $
$Revision: 1.10 $
$Date: 1997/07/02 04:45:00 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: Chio.cc,v $
Revision 1.10  1997/07/02 04:45:00  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.9  1997/06/21 02:21:30  andrewm
Checkpoint.  PostScript generator going well. A lot of small tweeks
all over to accomplish this.

Revision 1.8  1997/05/31 21:12:40  andrewm
Checkpoint.  Things are working well.

Revision 1.7  1997/05/15 04:14:39  andrewm
Checkpoint.  Reworked the low level file format stuff to contain
proper lists rather than maps keyed to binary numbers.
This point represents the entire program working with this file format
change.

Revision 1.6  1997/05/11 18:41:05  andrewm
Committing changes so that lists are fundamental values and
parsing of multi line terms is corrected back along them main branch.

Revision 1.5.2.2  1997/05/11 18:38:27  andrewm
Changed the manner in which multi line terms are parsed.  Now, a backslash
escape mechanism is used to determine the end of the term and any
backslashes or close braces ( \ or } ) must be escaped with a backslash.
This will happen normally in the output functions for the multi line
type terms.

Revision 1.5.2.1  1997/05/08 04:53:46  andrewm
Have Chio library working with "lists".  Rather than having all
maps some values of which are keyed by binary numbers, now the
lists are represented as lists internally.

Revision 1.5  1997/03/18 06:51:00  andrewm
Checkpoint.  Mouse select, insert, and delete working.
Some changes to improve robustness in the face of an arbitrary input file.

Revision 1.4  1997/03/04 06:32:49  andrewm
Another check point.  The editor can draw output from files.
The crashing during the dtor for MachineGroup is fixed.

Revision 1.3  1996/12/26 05:55:20  andrewm
Checkpoint, the compiler is working again.

Revision 1.2  1996/12/20 06:11:19  andrewm
Checkpoint.

Revision 1.1  1996/12/15 05:11:06  andrewm
First release of the reworked HIO library.

 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "Chio.h"
#include "CleanName.h"
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <strstream.h>
#include <algorithm>
#include "ascii.h"

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: Chio.cc,v $ $Revision: 1.10 $" ;

/*
STATIC MEMBER DEFINITIONS
*/

/*
FUNCTION DEFINITIONS
*/

/*
=========================================================================
ChioTerm
=========================================================================
*/

ChioTerm::
ChioTerm() :
		string(),
		_type(StringChioTerm),
		_lineno(0)
{
}

ChioTerm::
ChioTerm(
	const string& term,
	ChioTermType type,
	ChioTermSource source,
	int lineno) :
		string(source == ExternalChioTerm ?
			external_to_internal(term, type) : term),
		_type(type),
		_lineno(lineno)
{
	if (_type == UnknownChioTerm)
		_type = determine_term_type((const string &)*this) ;
}

ChioTerm::
ChioTerm(
	const char *term,
	ChioTermType type,
	ChioTermSource source,
	int lineno) :
		string(source == ExternalChioTerm ?
			external_to_internal(string(term), type) : string(term)),
		_type(type),
		_lineno(lineno)
{
	if (_type == UnknownChioTerm)
		_type = determine_term_type((const string &)*this) ;
}

ChioTerm::
ChioTerm(
	unsigned long u) :
		string(numeric(u)),
		_type(WordChioTerm),
		_lineno(0)
{
}

ChioTerm::
ChioTerm(
	unsigned u) :
		string(numeric(u)),
		_type(WordChioTerm),
		_lineno(0)
{
}

ChioTerm::
ChioTerm(
	long l) :
		string(numeric(l)),
		_type(WordChioTerm),
		_lineno(0)
{
}

ChioTerm::
ChioTerm(
	int i) :
		string(numeric(i)),
		_type(WordChioTerm),
		_lineno(0)
{
}

ChioTerm::
ChioTerm(
	double d) :
		string(numeric(d)),
		_type(WordChioTerm),
		_lineno(0)
{
}

ChioTerm::
ChioTerm(
	float f) :
		string(numeric(f)),
		_type(WordChioTerm),
		_lineno(0)
{
}

ChioTerm::
operator unsigned long() const
{
	return strtoul(c_str(), NULL, 0) ;
}

ChioTerm::
operator unsigned () const
{
	return (unsigned)strtoul(c_str(), NULL, 0) ;
}

ChioTerm::
operator long int() const
{
	return strtol(c_str(), NULL, 0) ;
}

ChioTerm::
operator int() const
{
	return (int)strtol(c_str(), NULL, 0) ;
}

ChioTerm::
operator double() const
{
	return strtod(c_str(), NULL) ;
}

ChioTerm::
operator float() const
{
	return (float)strtod(c_str(), NULL) ;
}

string ChioTerm::
var_name() const
{
	string var_string(size(), '\0') ;
	transform(begin(), end(), var_string.begin(), CleanName()) ;
	return var_string ;
}

string ChioTerm::
internal_to_external() const
{
	switch (_type)
	{
	case WordChioTerm:
		return word_internal_to_external() ;

	case StringChioTerm:
		return string_internal_to_external() ;

	default:
		cerr << "\"Unknown term type\"" << endl ;
		// Fall through
	case MultiLineChioTerm:
		return multi_internal_to_external() ;
	}
}

string ChioTerm::
word_internal_to_external() const
{
	string ext_str ;
	ext_str.reserve(length()) ;

	for (string::const_iterator is = begin() ; is != end() ; ++is)
	{
		char c = *is ;
		switch (c)
		{
		case ' ':
		case '\t':
		case '\n':
		case '\r':
		case '\v':
		case '\f':
		case ',':
		case '"':
		case '[':
		case ']':
		case '{':
		case '}':
		case '=':
			ext_str += '\\' ;
			break ;
		}

		ext_str += c ;
	}

	return ext_str ;
}

string ChioTerm::
string_internal_to_external() const
{
	string ext_str ;
	ext_str.reserve(length()) ;

	for (string::const_iterator is = begin() ; is != end() ; ++is)
	{
		char c = *is ;
		switch (c)
		{
		case ASCII_NL:
			ext_str += '\\' ;
			c = 'n' ;
			break ;

		case ASCII_HT:
			ext_str += '\\' ;
			c = 't' ;
			break ;

		case ASCII_VT:
			ext_str += '\\' ;
			c = 'v' ;
			break ;

		case ASCII_BS:
			ext_str += '\\' ;
			c = 'b' ;
			break ;

		case ASCII_CR:
			ext_str += '\\' ;
			c = 'r' ;
			break ;

		case ASCII_DC4:
			ext_str += '\\' ;
			c = 'f' ;
			break ;

		case ASCII_BEL:
			ext_str += '\\' ;
			c = 'a' ;
			break ;

		case '"':
			ext_str += '\\' ;
			break ;

		default:
			if (!isprint(c))
			{
				ostrstream buf_stream ;

				buf_stream << "\\x" << hex << c << ends ;
				ext_str += buf_stream.str() ;
				continue ;
			}
			break ;
		}
		ext_str += c ;
	}
	return ext_str ;
}

string ChioTerm::
multi_internal_to_external() const
{
	string ext_str ;
	ext_str.reserve(length()) ;

	for (string::const_iterator is = begin() ; is != end() ; ++is)
	{
		char c = *is ;
		if (c == '\\' || c == '}')
			ext_str += '\\' ;
		ext_str += c ;
	}
	return ext_str ;
}

string ChioTerm::
external_to_internal(
	const string& str,
	ChioTermType type)
{
	if (type == UnknownChioTerm)
		type = determine_term_type(str) ;

	switch (type)
	{
	case WordChioTerm:
		return word_external_to_internal(str) ;

	case StringChioTerm:
		return string_external_to_internal(str) ;

	case MultiLineChioTerm:
		return multi_external_to_internal(str) ;
	}

	return str ;
}

ChioTerm::ChioTermType ChioTerm::
determine_term_type(
	const string& str)
{
	ChioTermType t = WordChioTerm ;

	int len = str.length() ;
	if (len == 0
		|| (str[0] == '"' && str[len - 1] == '"')
		|| strpbrk(str.c_str(), " \t\f\r\n"))
	{
		t = StringChioTerm ;
	}

	if (strpbrk(str.c_str(), "}"))
	{
		t = MultiLineChioTerm ;
	}

	return t ;
}

string ChioTerm::
word_external_to_internal(
	const string& str)
{
	string int_str ;
	int_str.reserve(str.length()) ;

	for (string::const_iterator is = str.begin() ;
		is != str.end() ; ++is)
	{
		if (*is == '\\')
		{
			++is ;
			if (is == str.end())
				break ;
		}
		int_str += *is ;
	}

	return int_str ;
}

string ChioTerm::
string_external_to_internal(
	const string& str)
{
	string int_str ;
	int_str.reserve(str.length()) ;

	string::const_iterator is = str.begin() ;
	if (*is == '\"')
		is++ ;
	for ( ; is != str.end() ; ++is)
	{
		char c = *is ;
		if (c == '\\')
		{
			++is ;
			if (is == str.end())
				break ;
			c = *is ;
			if (isdigit(c))
			{
				c = strtol(is, (char **)&is, 8) ;
				--is ;
			}
			else
				switch (c)
				{
				case 'n':
					c = ASCII_NL ;
					break ;

				case 't':
					c = ASCII_HT ;
					break ;

				case 'v':
					c = ASCII_VT ;
					break ;

				case 'b':
					c = ASCII_BS ;
					break ;

				case 'r':
					c = ASCII_CR ;
					break ;

				case 'f':
					c = ASCII_DC4 ;
					break ;

				case 'a':
					c = ASCII_BEL ;
					break ;

				case 'x':
					is++ ;
					c = strtol(is, (char **)&is, 16) ;
					--is ;
					break ;
				}
		}
		else if (c == '\"')
			break ;
		if (c)
			int_str += c ;
	}
	return int_str ;
}

string ChioTerm::
multi_external_to_internal(
	const string& str)
{
	string int_str ;
	int_str.reserve(str.length()) ;

	string::const_iterator is = str.begin() ;
	if (*is == '{')
		is++ ;
	for ( ; is != str.end() ; ++is)
	{
		char c = *is ;
		if (c == '\\')
		{
			if (++is == str.end())
				break ;
			c = *is ;
		}
		else if (c == '}')
			break ;
		int_str += c ;
	}
	return int_str ;
}

string ChioTerm::
numeric(
	unsigned long u)
{
	ostrstream out_buf ;
	out_buf << u << ends ;
	return string(out_buf.str()) ;
}

string ChioTerm::
numeric(
	unsigned u)
{
	ostrstream out_buf ;
	out_buf << u << ends ;
	return string(out_buf.str()) ;
}

string ChioTerm::
numeric(
	long l)
{
	ostrstream out_buf ;
	out_buf << l << ends ;
	return string(out_buf.str()) ;
}

string ChioTerm::
numeric(
	int i)
{
	ostrstream out_buf ;
	out_buf << i << ends ;
	return string(out_buf.str()) ;
}

string ChioTerm::
numeric(
	double d)
{
	ostrstream out_buf ;
	out_buf << d << ends ;
	return string(out_buf.str()) ;
}

string ChioTerm::
numeric(
	float f)
{
	ostrstream out_buf ;
	out_buf << f << ends ;
	return string(out_buf.str()) ;
}

ostream&
operator <<(
	ostream& s,
	const ChioTerm& t)
{
	if (t.length() != 0)
	{
		switch (t._type)
		{
		case ChioTerm::WordChioTerm:
			s << t.word_internal_to_external() ;
			break ;

		case ChioTerm::StringChioTerm:
			s << '"' << t.string_internal_to_external() << '"' ;
			break ;

		default:
			cerr << "\"Unknown term type\"" << endl ;
			// Fall through
		case ChioTerm::MultiLineChioTerm:
			s << '{' << t.multi_internal_to_external() << '}' ;
			break ;
		}
	}
		/*
		For cases where we have an and empty string,
		just format it as an empty string.
		*/
	else
	{
		s << "\"\"" ;
	}

	return s ;
}

bool
operator <(
	const ChioTerm& t1,
	const ChioTerm& t2)
{
	return (string&)t1 < (string&)t2 ;
}

bool
operator ==(
	const ChioTerm& t1,
	const ChioTerm& t2)
{
	return (string&)t1 == (string&)t2 ;
}

/*
=========================================================================
ChioValue
=========================================================================
*/

ChioValue::
ChioValue(
	ChioValueType type,
	string comment) :
		_type(type),
		_comment(comment)
{
	switch (_type)
	{
	case EmptyChioValue:
		_empty = NULL ;
		break ;

	case TermChioValue:
		_term = new ChioTerm ;
		break ;

	case MapChioValue:
		_map = new ChioMap ;
		break ;

	case ListChioValue:
		_list = new ChioList ;
		break ;
	}
}

ChioValue::
ChioValue(
	const ChioTerm& term,
	string comment) :
		_term(new ChioTerm(term)),
		_type(TermChioValue),
		_comment(comment)
{
}

ChioValue::
ChioValue(
	ChioTerm *term,
	string comment) :
		_term(term),
		_type(TermChioValue),
		_comment(comment)
{
}

ChioValue::
ChioValue(
	const string& term,
	ChioTerm::ChioTermType type,
	string comment) :
		_term(new ChioTerm(term, type)),
		_type(TermChioValue),
		_comment(comment)
{
}

ChioValue::
ChioValue(
	const char *term,
	ChioTerm::ChioTermType type,
	string comment) :
		_term(new ChioTerm(term, type)),
		_type(TermChioValue),
		_comment(comment)
{
}

ChioValue::
ChioValue(
	const ChioMap& map,
	string comment) :
		_map(new ChioMap(map)),
		_type(MapChioValue),
		_comment(comment)
{
}

ChioValue::
ChioValue(
	ChioMap *map,
	string comment) :
		_map(map),
		_type(MapChioValue),
		_comment(comment)
{
}

ChioValue::
ChioValue(
	const ChioList& list,
	string comment) :
		_list(new ChioList(list)),
		_type(ListChioValue),
		_comment(comment)
{
}

ChioValue::
ChioValue(
	ChioList *list,
	string comment) :
		_list(list),
		_type(ListChioValue),
		_comment(comment)
{
}

ChioValue::
ChioValue(
	const ChioValue& value)
{
	switch (_type = value._type)
	{
	case EmptyChioValue:
		_empty = NULL ;
		break ;

	case TermChioValue:
		_term = new ChioTerm(*value._term) ;
		break ;

	case MapChioValue:
		_map = new ChioMap(*value._map) ;
		break ;

	case ListChioValue:
		_list = new ChioList(*value._list) ;
		break ;
	}
	_comment = value._comment ;
}

ChioValue::
ChioValue(
	unsigned long u) :
		_term(new ChioTerm(u)),
		_type(TermChioValue)
{
}

ChioValue::
ChioValue(
	unsigned u) :
		_term(new ChioTerm(u)),
		_type(TermChioValue)
{
}

ChioValue::
ChioValue(
	long l) :
		_term(new ChioTerm(l)),
		_type(TermChioValue)
{
}

ChioValue::
ChioValue(
	int i) :
		_term(new ChioTerm(i)),
		_type(TermChioValue)
{
}

ChioValue::
ChioValue(
	double d) :
		_term(new ChioTerm(d)),
		_type(TermChioValue)
{
}

ChioValue::
ChioValue(
	float f) :
		_term(new ChioTerm(f)),
		_type(TermChioValue)
{
}

ChioValue::
~ChioValue()
{
	switch (_type)
	{
	case TermChioValue:
		delete _term ;
		break ;

	case MapChioValue:
		delete _map ;
		break ;

	case ListChioValue:
		delete _list ;
		break ;
	}
}

ChioValue& ChioValue::
operator =(
	const ChioValue& value)
{
	if (this != &value)
	{
		switch (_type)
		{
		case TermChioValue:
			delete _term ;
			break ;

		case MapChioValue:
			delete _map ;
			break ;

		case ListChioValue:
			delete _list ;
			break ;
		}

		switch (_type = value._type)
		{
		case EmptyChioValue:
			_empty = NULL ;
			break ;

		case TermChioValue:
			_term = new ChioTerm(*value._term) ;
			break ;

		case MapChioValue:
			_map = new ChioMap(*value._map) ;
			break ;

		case ListChioValue:
			_list = new ChioList(*value._list) ;
			break ;
		}
		_comment = value._comment ;
	}

	return *this ;
}

ChioValue::
operator ChioTerm&()
{
	if (_type == EmptyChioValue)
	{
		_type = TermChioValue ;
		_term = new ChioTerm ;
	}
	assert(_type == TermChioValue) ;
	return *_term ;
}

ChioValue::
operator ChioMap&()
{
	if (_type == EmptyChioValue)
	{
		_type = MapChioValue ;
		_map = new ChioMap ;
	}
	assert(_type == MapChioValue) ;
	return *_map ;
}

ChioValue::
operator ChioList&()
{
	if (_type == EmptyChioValue)
	{
		_type = ListChioValue ;
		_list = new ChioList ;
	}
	assert(_type == ListChioValue) ;
	return *_list ;
}

ChioValue::
operator const string&() const
{
	assert(_type == TermChioValue) ;
	if (_type == TermChioValue)
		return *_term ;
	else
	{
		cerr << "Warning: attempt to cast complex ChioValue to \"string&\""
			<< endl ;
	}
}

ChioValue::
operator unsigned long() const
{
	assert(_type == TermChioValue) ;
	if (_type == TermChioValue)
		return (unsigned long)*_term ;
	else
	{
		cerr
			<< "Warning: attempt to cast complex ChioValue to \"unsigned long\""
			<< endl ;
		return 0 ;
	}
}

ChioValue::
operator unsigned () const
{
	assert(_type == TermChioValue) ;
	if (_type == TermChioValue)
		return (unsigned)*_term ;
	else
	{
		cerr << "Warning: attempt to cast complex ChioValue to \"unsigned\""
			<< endl ;
		return 0 ;
	}
}

ChioValue::
operator long int() const
{
	assert(_type == TermChioValue) ;
	if (_type == TermChioValue)
		return (long int)*_term ;
	else
	{
		cerr << "Warning: attempt to cast complex ChioValue to \"long int\""
			<< endl ;
		return 0 ;
	}
}

ChioValue::
operator int() const
{
	assert(_type == TermChioValue) ;
	if (_type == TermChioValue)
		return (int)*_term ;
	else
	{
		cerr << "Warning: attempt to cast complex ChioValue to \"int\""
			<< endl ;
		return 0 ;
	}
}

ChioValue::
operator double() const
{
	assert(_type == TermChioValue) ;
	if (_type == TermChioValue)
		return (double)*_term ;
	else
	{
		cerr << "Warning: attempt to cast complex ChioValue to \"double\""
			<< endl ;
		return 0 ;
	}
}

ChioValue::
operator float() const
{
	assert(_type == TermChioValue) ;
	if (_type == TermChioValue)
		return (float)*_term ;
	else
	{
		cerr << "Warning: attempt to cast complex ChioValue to \"float\""
			<< endl ;
		return 0 ;
	}
}

ChioValue::
operator const char *() const
{
	assert(_type == TermChioValue) ;
	if (_type == TermChioValue)
		return _term->c_str() ;
	else
	{
		cerr << "Warning: attempt to cast complex ChioValue to \"const char *\""
			<< endl ;
		return 0 ;
	}
}

string ChioValue::
format_comment() const
{
	string comment(ChioMap::level(), '\t') ;

	if (_comment.length() != 0)
	{
		comment += '\t' ;
		comment += _comment ;
		comment += '\n' ;
		comment.append(ChioMap::level(), '\t') ;
	}
	return comment ;
}

ostream&
operator <<(
	ostream& s,
	const ChioValue& v)
{
	switch (v._type)
	{
	case ChioValue::EmptyChioValue:
		s << "[\n" << string(ChioMap::level(), '\t') << ']' ;
		break ;

	case ChioValue::TermChioValue:
		s << *v._term ;
		break ;

	case ChioValue::MapChioValue:
		ChioMap::incr_level() ;
		s << "[\n" << *v._map ;
		ChioMap::decr_level() ;
		s << string(ChioMap::level(), '\t') << ']' ;
		break ;

	case ChioValue::ListChioValue:
		ChioMap::incr_level() ;
		s << "[\n" << *v._list ;
		ChioMap::decr_level() ;
		s << string(ChioMap::level(), '\t') << ']' ;
		break ;
	}
	return s ;
}

/*
=========================================================================
ChioMap
=========================================================================
*/

unsigned ChioMap::_level = 0 ;

istream&
operator >>(
	istream& s,
	ChioMap& m)
{
	extern FILE *hio_in ;
	extern int hio_parse() ;
	extern ChioMap *hio_file_map ;

	hio_in = s.rdbuf() ;

	if (hio_parse())
		cerr << "Error parsing input" << endl ;
	if (hio_file_map)
	{
		m = *hio_file_map ;
		delete hio_file_map ;
		hio_file_map = NULL ;
	}
	return s ;
}

ostream&
operator <<(
	ostream& s,
	const ChioMap& m)
{
	copy(m.begin(), m.end(), ostream_iterator<ChioAssignment>(s)) ;
	return s ;
}

/*
=========================================================================
ChioList
=========================================================================
*/

ostream&
operator <<(
	ostream& s,
	const ChioList& m)
{
	for (ChioListConstIter v_iter = m.begin() ;
		v_iter != m.end() ; ++v_iter)
	{
		s << string(ChioMap::level(), '\t') << *v_iter << ',' << endl ;
	}
	return s ;
}

/*
=========================================================================
ChioAssignment
=========================================================================
*/

ostream&
operator <<(
	ostream& s,
	const ChioAssignment& assign)
{
	return s << assign.second.format_comment()
		<< assign.first << " = " << assign.second << ',' << endl ;
}
