/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:
chio -- Classes that implement a Hierarchical I/O system.

$RCSfile: Chio.h,v $
$Revision: 1.9 $
$Date: 1997/07/02 04:45:00 $

ABSTRACT:
This file contains the definition of a set of classes to implement
Hierarchical I/O from a file.  These classes use the facilities
of the Standard Template Library (STL) to form a hierarchy of
STL maps that contain the contents of an ASCII file.

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: Chio.h,v $
Revision 1.9  1997/07/02 04:45:00  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.8  1997/06/21 02:21:30  andrewm
Checkpoint.  PostScript generator going well. A lot of small tweeks
all over to accomplish this.

Revision 1.7  1997/05/31 21:12:40  andrewm
Checkpoint.  Things are working well.

Revision 1.6  1997/05/15 04:14:39  andrewm
Checkpoint.  Reworked the low level file format stuff to contain
proper lists rather than maps keyed to binary numbers.
This point represents the entire program working with this file format
change.

Revision 1.5  1997/05/11 18:41:06  andrewm
Committing changes so that lists are fundamental values and
parsing of multi line terms is corrected back along them main branch.

Revision 1.4.2.2  1997/05/11 18:38:28  andrewm
Changed the manner in which multi line terms are parsed.  Now, a backslash
escape mechanism is used to determine the end of the term and any
backslashes or close braces ( \ or } ) must be escaped with a backslash.
This will happen normally in the output functions for the multi line
type terms.

Revision 1.4.2.1  1997/05/08 04:53:46  andrewm
Have Chio library working with "lists".  Rather than having all
maps some values of which are keyed by binary numbers, now the
lists are represented as lists internally.

Revision 1.4  1997/03/04 06:32:49  andrewm
Another check point.  The editor can draw output from files.
The crashing during the dtor for MachineGroup is fixed.

Revision 1.3  1996/12/26 05:55:21  andrewm
Checkpoint, the compiler is working again.

Revision 1.2  1996/12/20 06:11:20  andrewm
Checkpoint.

Revision 1.1  1996/12/15 05:11:07  andrewm
First release of the reworked HIO library.

 *--
 */
#ifndef _Chio_h_
#define _Chio_h_

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma interface
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include <iostream.h>
#include <string>
#include <map>
#include <list>

/*
MACRO DEFINITIONS
*/

/*
CLASS FORWARD REFERENCES
*/
class ChioTerm ;
class ChioValue ;
class ChioMap ;
class ChioList ;

/*
CLASS DEFINITIONS
*/

/*
=========================================================================
ChioTerm
=========================================================================
*/
	/*
	The ChioTerm class represents the basic unit of information.
	It is derived from "string" which shows its basic character
	orientation.  Fundamentally all information is retrieved from
	the file and placed in strings.  Additionally some type information
	and the line number in the file where the term started are held.
	This information is useful in constructing the output so that
	it identically matches the input file.
	*/
class ChioTerm :
	public string
{
	friend ostream& operator <<(ostream& s, const ChioTerm& t) ;
	friend bool operator <(const ChioTerm& t1, const ChioTerm& t2) ;
	friend bool operator ==(const ChioTerm& t1, const ChioTerm& t2) ;

public: // types
	enum ChioTermSource
	{
		ExternalChioTerm,
		InternalChioTerm,
	} ;

	enum ChioTermType
	{
		UnknownChioTerm,
		WordChioTerm,
		StringChioTerm,
		MultiLineChioTerm,
	} ;

public: // members
	ChioTerm() ;
	ChioTerm(const string& term, ChioTermType type = WordChioTerm,
		ChioTermSource source = InternalChioTerm, int lineno = 0) ;
	ChioTerm(const char *term, ChioTermType type = WordChioTerm,
		ChioTermSource source = InternalChioTerm, int lineno = 0) ;
	ChioTerm(unsigned long u) ;
	ChioTerm(unsigned u) ;
	ChioTerm(long l) ;
	ChioTerm(int i) ;
	ChioTerm(double d) ;
	ChioTerm(float f) ;

	operator unsigned long() const ;
	operator unsigned () const ;
	operator long int() const ;
	operator int() const ;
	operator double() const ;
	operator float() const ;

	string var_name() const ;
	ChioTermType type() const { return _type ; }
	int lineno() const { return _lineno ; }
	string internal_to_external() const ;

private: // function members
	string word_internal_to_external() const ;
	string string_internal_to_external() const ;
	string multi_internal_to_external() const ;

private: // data members
	ChioTermType _type ;
	int _lineno ;

private: // static functions
	static ChioTermType determine_term_type(const string& str) ;
	static string external_to_internal(const string& str, ChioTermType type) ;
	static string word_external_to_internal(const string& str) ;
	static string string_external_to_internal(const string& str) ;
	static string multi_external_to_internal(const string& str) ;
	static string numeric(unsigned long u) ;
	static string numeric(unsigned u) ;
	static string numeric(long l) ;
	static string numeric(int i) ;
	static string numeric(double d) ;
	static string numeric(float f) ;
} ;

/*
=========================================================================
ChioValue
=========================================================================
*/
	/*
	Corresponding to the key for the map is the value type of the map.
	This class is also a union so that both single terms and entire
	sub maps can be held.  This is what generates the hierarchical
	nature of the scheme.  Consequently the value types have a long
	list of constructors to support values that either a ChioTerm
	or a ChioMap.
	*/
class ChioValue
{
	friend ostream& operator <<(ostream& s, const ChioValue& v) ;

public: // types
	enum ChioValueType
	{
		EmptyChioValue,
		TermChioValue,
		MapChioValue,
		ListChioValue,
	} ;

public: // function members
	ChioValue(
		ChioValueType type = TermChioValue,
		string comment = string()) ;
	ChioValue(const ChioTerm& term, string comment = string()) ;
	ChioValue(ChioTerm *term, string comment = string()) ;
	ChioValue(
		const string& term,
		ChioTerm::ChioTermType type = ChioTerm::UnknownChioTerm,
		string comment = string()) ;
	ChioValue(
		const char *term,
		ChioTerm::ChioTermType type = ChioTerm::UnknownChioTerm,
		string comment = string()) ;
	ChioValue(const ChioMap& map, string comment = string()) ;
	ChioValue(ChioMap *map, string comment = string()) ;
	ChioValue(const ChioList& list, string comment = string()) ;
	ChioValue(ChioList *list, string comment = string()) ;
	ChioValue(const ChioValue& value) ;
	ChioValue(unsigned long u) ;
	ChioValue(unsigned u) ;
	ChioValue(long l) ;
	ChioValue(int i) ;
	ChioValue(double d) ;
	ChioValue(float f) ;
	~ChioValue() ;

	ChioValue& operator =(const ChioValue& value) ;
	operator ChioTerm&() ;
	operator ChioMap&() ;
	operator ChioList&() ;
	operator const string&() const ;
	operator unsigned long() const ;
	operator unsigned () const ;
	operator long int() const ;
	operator int() const ;
	operator double() const ;
	operator float() const ;
	operator const char *() const ;

	string& comment() { return _comment ; }
	string format_comment() const ;

	bool is_term_value() { return _type == TermChioValue ; }
	bool is_map_value() { return _type == MapChioValue ; }

private: // data members
	union
	{
		ChioTerm *_term ;
		ChioMap *_map ;
		ChioList *_list ;
		void *_empty ;
	} ;
	ChioValueType _type ;
	string _comment ;
} ;

/*
=========================================================================
ChioMap
=========================================================================
*/
class ChioMap :
	public map<ChioTerm, ChioValue, less<ChioTerm> >
{
	/*
	A level indicator is provided for the benefit of output so that
	the hierarchy can be pretty printed with the proper indentation.
	*/
public: // static function members
	static unsigned level() { return _level ; }
	static void incr_level() { ++_level ; }
	static void decr_level() { --_level ; }

private: // static data members
	static unsigned _level ;
} ;
typedef ChioMap::iterator ChioMapIter ;
typedef ChioMap::const_iterator ChioMapConstIter ;

istream& operator >>(istream& s, ChioMap& m) ;
ostream& operator <<(ostream& s, const ChioMap& m) ;

extern char *hio_source ;

/*
=========================================================================
ChioList
=========================================================================
*/
class ChioList :
	public list<ChioValue>
{
private:
	ChioList& operator =(const ChioList& vect) ; // no assigments
} ;
typedef ChioList::iterator ChioListIter ;
typedef ChioList::const_iterator ChioListConstIter ;

ostream& operator <<(ostream& s, const ChioList& m) ;


/*
=========================================================================
ChioAssignment
=========================================================================
*/
	/*
	The value type for the map container is given a cleaner typedef.
	As it turns out, the association of key to value that the map
	value type performs does have some significance in this context
	over and above the necessities of feeding the map.  The key/value
	pair is really an "assignment" in our context that associates
	an attribute to a value.
	*/
typedef ChioMap::value_type ChioAssignment ;
ostream& operator <<(ostream& s, const ChioAssignment& assign) ;


#endif /* _Chio_h_ */
