/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: DocPageState.cc,v $
$Revision: 1.9 $
$Date: 1997/07/02 04:45:02 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: DocPageState.cc,v $
Revision 1.9  1997/07/02 04:45:02  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.8  1997/06/06 04:34:24  andrewm
Checkpoint.  Changed the Subject / Observer code to include a
pointer to the changed subject in the call to "update".  This allowed
a given observer to observe muliple subjects.  Then I modified
various dialogs, especially those in that control the various fields
in the state machine to observe multiple subjects as necessary to
insure that all their fields were updated properly.

Revision 1.7  1997/05/20 05:15:29  andrewm
Checkpoint.  Improved the structure of the "State" class and this
had quite some ripple effects.  However, now there is an abstract
class "State" with two concrete classes "PseudoState" to represent
error and ignore and "RealState" to represent the user specified
states.  Also improved the text display of event names on the transitions.

Revision 1.6  1997/05/15 04:14:43  andrewm
Checkpoint.  Reworked the low level file format stuff to contain
proper lists rather than maps keyed to binary numbers.
This point represents the entire program working with this file format
change.

Revision 1.5  1997/04/24 03:20:46  andrewm
Checkpoint.  All features in.  Starting test cycle.

Revision 1.4  1997/04/16 04:06:13  andrewm
Checkpoint, as last major dialog is in.

Revision 1.3  1997/03/30 02:07:26  andrewm
Checkpoint.  Things working well. About to change the way
machines are stored in the backing file.

Revision 1.2  1997/03/18 06:51:01  andrewm
Checkpoint.  Mouse select, insert, and delete working.
Some changes to improve robustness in the face of an arbitrary input file.

Revision 1.1  1997/03/12 03:13:04  andrewm
Checkpoint.  Things are working rather well.

 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "Application.h"
#include "MainWindow.h"
#include "DrawingArea.h"
#include "EventSelectionDialog.h"
#include "GsmDocument.h"
#include "Smachine.h"
#include "State.h"
#include "MachEvent.h"
#include "Transition.h"
#include "RubberBand.h"
#include "RubberLine.h"
#include "CSutils.h"
#include "DocPageState.h"

#include <assert.h>
#include <iostream.h>
#include <X11/cursorfont.h>

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: DocPageState.cc,v $ $Revision: 1.9 $" ;

/*
STATIC MEMBER DEFINITIONS
*/

/*
FUNCTION DEFINITIONS
*/

DocPageState::
DocPageState(
	GsmDocument& doc) :
		_doc(doc),
		_current_machine(doc.current_machine()),
		_state_at_cursor(NULL),
		_trans_at_cursor(NULL),
		_current_state(NULL)
{
	assert(_current_machine != NULL) ;
	_current_machine->select_all(false) ;
	_doc.attach(this) ;
}

DocPageState::
~DocPageState()
{
	_doc.detach(this) ;
}

void DocPageState::
update(
	const Subject *subject)
{
	assert(subject == &_doc) ;
	Smachine *old_machine = _current_machine ;
	_current_machine = _doc.current_machine() ;
	if (_current_machine != old_machine)
		_current_state = NULL ;
}

void DocPageState::
subject_deleted(
	Subject *subject)
{
	assert(subject == &_doc) ;
	delete this ;
}

void DocPageState::
define_cursor(
	Cursor cursor)
{
	Application *appl = Application::instance() ;
	Display *display = appl->display() ;
	Window window = XtWindow(appl->main_window()->drawing_area()->widget()) ;
	XDefineCursor(display, window, cursor) ;
}

/*=====================================================================
Change
*/

DocPageChangeState::
DocPageChangeState(
	GsmDocument& doc) :
		DocPageState(doc),
		_rubber_band(NULL)
{
}

DocPageChangeState::
~DocPageChangeState()
{
	delete _rubber_band ;
}

void DocPageChangeState::
button_1_press(
	const Point& p)
{
	_ref_point = p ;

	_state_at_cursor = _current_machine->find_state_at_location(p) ;
	if (!_state_at_cursor)
		_trans_at_cursor = _current_machine->find_transition_at_location(p) ;

	if (!(_state_at_cursor || _trans_at_cursor))
	{
		_current_machine->select_all(false) ;
		_rubber_band = new RubberBand() ;
		_rubber_band->start_banding(p) ;
	}
}

void DocPageChangeState::
button_1_release(
	const Point& p)
{
	if (_state_at_cursor)
	{
		State *current_state = _current_machine->find_state_at_location(p) ;
		if (_state_at_cursor == current_state
			&& _state_at_cursor->selected() == false)
		{
			_current_machine->select_all(false) ;
			_state_at_cursor->selected(true) ;
		}
	}
	else if (_trans_at_cursor)
	{
		Transition *current_trans =
			_current_machine->find_transition_at_location(p) ;
		if (_trans_at_cursor == current_trans
			&& _trans_at_cursor->selected() == false)
		{
			_current_machine->select_all(false) ;
			_trans_at_cursor->selected(true) ;
		}
	}
	else if (_rubber_band)
	{
		_rubber_band->stop_banding(p) ;
		_current_machine->select_if_within(_rubber_band->rect()) ;
		delete _rubber_band ;
		_rubber_band = NULL ;
	}
	else
	{
		_current_machine->select_all(false) ;
	}
	_current_state = _state_at_cursor ;
	notify() ;
	_state_at_cursor = NULL ;
	_trans_at_cursor = NULL ;
}

void DocPageChangeState::
button_1_motion(
	const Point& p)
{
	if (_state_at_cursor)
	{
		if (_state_at_cursor->selected() == true)
		{
			_current_machine->move_relative(p - _ref_point) ;
			_ref_point = p ;
		}
		else
		{
			_state_at_cursor->move(p) ;
		}
	}
	else if (_trans_at_cursor)
	{
		_trans_at_cursor->move(p) ;
	}
	else if (_rubber_band)
	{
		_rubber_band->new_band(p) ;
	}
}

void DocPageChangeState::
button_2_press(
	const Point& p)
{
	_ref_point = p ;

	_state_at_cursor = _current_machine->find_state_at_location(p) ;
	_trans_at_cursor = _current_machine->find_transition_at_location(p) ;

	if (!(_state_at_cursor || _trans_at_cursor))
	{
		_rubber_band = new RubberBand() ;
		_rubber_band->start_banding(p) ;
	}
}

void DocPageChangeState::
button_2_release(
	const Point& p)
{
	if (_state_at_cursor)
	{
		State *current_state = _current_machine->find_state_at_location(p) ;
		if (_state_at_cursor == current_state)
		{
			_state_at_cursor->selected(!_state_at_cursor->selected()) ;
		}
	}
	else if (_trans_at_cursor)
	{
		Transition *current_trans =
			_current_machine->find_transition_at_location(p) ;
		if (_trans_at_cursor == current_trans)
		{
			_trans_at_cursor->selected(!_trans_at_cursor->selected()) ;
		}
	}
	else if (_rubber_band)
	{
		_rubber_band->stop_banding(p) ;
		_current_machine->select_if_within(_rubber_band->rect()) ;
		delete _rubber_band ;
		_rubber_band = NULL ;
	}
	_current_state = _state_at_cursor ;
	notify() ;
	_state_at_cursor = NULL ;
	_trans_at_cursor = NULL ;
}

void DocPageChangeState::
button_2_motion(
	const Point& p)
{
	button_1_motion(p) ;
}

void DocPageChangeState::
button_3_press(
	const Point& p)
{
}

void DocPageChangeState::
button_3_release(
	const Point& p)
{
}

void DocPageChangeState::
button_3_motion(
	const Point& p)
{
}

/*=====================================================================
Insert
*/

DocPageInsertState::
DocPageInsertState(
	GsmDocument& doc) :
		DocPageState(doc),
		_insert_cursor(XCreateFontCursor(Application::instance()->display(),
			XC_circle)),
		_begin_state(NULL),
		_end_state(NULL),
		_rl(NULL),
		_select(NULL)
{
	define_cursor(_insert_cursor) ;
	_select = new EventSelectionDialog("event_select", Application::instance(),
		_current_machine) ;
	_select->ok_command(new SimpleXmCommand<DocPageInsertState>(this,
		DocPageInsertState::event_ok)) ;
	_select->cancel_command(new SimpleXmCommand<DocPageInsertState>(this,
		DocPageInsertState::event_cancel)) ;
}

DocPageInsertState::
~DocPageInsertState()
{
	delete _rl ;
	delete _select ;
	if (_insert_cursor)
		XFreeCursor(Application::instance()->display(), _insert_cursor) ;
	define_cursor(None) ;
}

void DocPageInsertState::
button_1_press(
	const Point& p)
{
	_state_at_cursor = _current_machine->find_state_at_location(p) ;
	if (!_state_at_cursor)
		_trans_at_cursor = _current_machine->find_transition_at_location(p) ;
}

void DocPageInsertState::
button_1_release(
	const Point& p)
{
	if (_state_at_cursor)
	{
		_current_machine->select_all(false) ;
		_state_at_cursor->selected(true) ;
	}
	else if (_trans_at_cursor)
	{
		Transition *current_trans =
			_current_machine->find_transition_at_location(p) ;
		if (_trans_at_cursor == current_trans)
		{
			if (_trans_at_cursor->selected() == true)
				_trans_at_cursor->insert_at_location(p) ;
			else
			{
				_current_machine->select_all(false) ;
				_trans_at_cursor->selected(true) ;
			}
		}
	}
	else
	{
		_current_machine->select_all(false) ;
		if (_state_at_cursor = _current_machine->add_state(p))
			_state_at_cursor->selected(true) ;
	}
	_current_state = _state_at_cursor ;
	notify() ;
	_state_at_cursor = NULL ;
	_trans_at_cursor = NULL ;
}

void DocPageInsertState::
button_1_motion(
	const Point& p)
{
	if (_state_at_cursor)
	{
		_state_at_cursor->move(p) ;
	}
	else if (_trans_at_cursor)
	{
		_trans_at_cursor->move(p) ;
	}
}

void DocPageInsertState::
button_2_press(
	const Point& p)
{
	_begin_state = _current_machine->find_state_at_location(p) ;
	if (_begin_state)
	{
		_rl = new RubberLine(p) ;
	}
}

void DocPageInsertState::
button_2_release(
	const Point& p)
{
	if (_rl)
	{
		_rl->new_band(p) ;
		_end_state = _current_machine->find_state_at_location(p) ;
		if (_end_state)
			_select->manage() ;
		else
		{
			_begin_state = NULL ;
			delete _rl ;
			_rl = NULL ;
		}
	}
	else
		_begin_state = NULL ;
}

void DocPageInsertState::
button_2_motion(
	const Point& p)
{
	if (_rl)
		_rl->new_band(p) ;
}

void DocPageInsertState::
button_3_press(
	const Point& p)
{
}

void DocPageInsertState::
button_3_release(
	const Point& p)
{
}

void DocPageInsertState::
button_3_motion(
	const Point& p)
{
}

void DocPageInsertState::
event_ok(
	XmAnyCallbackStruct *callback)
{
	if (_rl)
	{
		delete _rl ;
		_rl = NULL ;
	}

	XmSelectionBoxCallbackStruct *cbs =
		(XmSelectionBoxCallbackStruct *)callback ;
	char *event_name = CSToText(cbs->value) ;
	if (*event_name)
	{
		MachEvent *event = _current_machine->find_event(event_name) ;
		if (!event)
			event = _current_machine->add_event(event_name) ;
		if (event)
		{
			Transition *trans = _begin_state->find_transition(event) ;
			if (trans)
				cerr << "A transition out of state \"" << *_begin_state
					<< "\" via event \"" << *event << "\" already exists."
					<< endl ;
			else if (_begin_state == _end_state)
				_begin_state->add_loop_transition(event) ;
			else
				_begin_state->add_segment_transition(event, _end_state) ;
		}
	}
	else
	{
		cerr << "Event names may not be empty" << endl ;
	}
	XtFree(event_name) ;
	_begin_state = NULL ;
	_end_state = NULL ;
}

void DocPageInsertState::
event_cancel(
	XmAnyCallbackStruct *callback)
{
	_begin_state = NULL ;
	_end_state = NULL ;
	if (_rl)
	{
		delete _rl ;
		_rl = NULL ;
	}
}

/*=====================================================================
Delete
*/

DocPageDeleteState::
DocPageDeleteState(
	GsmDocument& doc) :
		DocPageState(doc),
		_delete_cursor(XCreateFontCursor(Application::instance()->display(),
			XC_pirate))
{
	define_cursor(_delete_cursor) ;
}

DocPageDeleteState::
~DocPageDeleteState()
{
	if (_delete_cursor)
		XFreeCursor(Application::instance()->display(), _delete_cursor) ;
	define_cursor(None) ;
}

void DocPageDeleteState::
button_1_press(
	const Point& p)
{
	_state_at_cursor = _current_machine->find_state_at_location(p) ;
	if (!_state_at_cursor)
		_trans_at_cursor = _current_machine->find_transition_at_location(p) ;
	_current_state = _state_at_cursor ;
}

void DocPageDeleteState::
button_1_release(
	const Point& p)
{
	if (_state_at_cursor)
	{
		State *current_state = _current_machine->find_state_at_location(p) ;
		if (_state_at_cursor == current_state)
		{
			if (_state_at_cursor->selected() == true)
			{
				delete _state_at_cursor ;
				_current_state = NULL ;
			}
			else
			{
				_current_machine->select_all(false) ;
				_state_at_cursor->selected(true) ;
				_current_state = _state_at_cursor ;
			}
			notify() ;
		}
	}
	else if (_trans_at_cursor)
	{
		Transition *current_trans =
			_current_machine->find_transition_at_location(p) ;
		if (_trans_at_cursor == current_trans)
		{
			if (_trans_at_cursor->selected() == true)
			{
				bool vertex_deleted = _trans_at_cursor->delete_at_location(p) ;
				if (vertex_deleted == false)
					delete _trans_at_cursor ;
			}
			else
			{
				_current_machine->select_all(false) ;
				_trans_at_cursor->selected(true) ;
			}
		}
	}
	else
	{
		_current_machine->select_all(false) ;
	}

	_state_at_cursor = NULL ;
	_trans_at_cursor = NULL ;
}

void DocPageDeleteState::
button_1_motion(
	const Point& p)
{
}

void DocPageDeleteState::
button_2_press(
	const Point& p)
{
	button_1_press(p) ;
}

void DocPageDeleteState::
button_2_release(
	const Point& p)
{
	button_1_release(p) ;
}

void DocPageDeleteState::
button_2_motion(
	const Point& p)
{
	button_1_motion(p) ;
}

void DocPageDeleteState::
button_3_press(
	const Point& p)
{
}

void DocPageDeleteState::
button_3_release(
	const Point& p)
{
}

void DocPageDeleteState::
button_3_motion(
	const Point& p)
{
}
