/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: ParameterDialog.cc,v $
$Revision: 1.7 $
$Date: 1997/07/02 04:45:08 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: ParameterDialog.cc,v $
Revision 1.7  1997/07/02 04:45:08  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.6  1997/06/06 04:34:26  andrewm
Checkpoint.  Changed the Subject / Observer code to include a
pointer to the changed subject in the call to "update".  This allowed
a given observer to observe muliple subjects.  Then I modified
various dialogs, especially those in that control the various fields
in the state machine to observe multiple subjects as necessary to
insure that all their fields were updated properly.

Revision 1.5  1997/05/31 21:12:41  andrewm
Checkpoint.  Things are working well.

Revision 1.4  1997/05/20 05:15:30  andrewm
Checkpoint.  Improved the structure of the "State" class and this
had quite some ripple effects.  However, now there is an abstract
class "State" with two concrete classes "PseudoState" to represent
error and ignore and "RealState" to represent the user specified
states.  Also improved the text display of event names on the transitions.

Revision 1.3  1997/05/15 04:14:43  andrewm
Checkpoint.  Reworked the low level file format stuff to contain
proper lists rather than maps keyed to binary numbers.
This point represents the entire program working with this file format
change.

Revision 1.2  1997/04/12 21:15:00  andrewm
Checkpoint after adding the event dialog.

Revision 1.1  1997/04/08 04:34:30  andrewm
Checkpoint as dialogs are added.
Decided to make sure that the name part of a name/value pair in
the hierarchical file is not used for id purposes.  Specifically
state machine names need to be contained within the machine not as
the key to the machine.

 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "ParameterDialog.h"
#include "Parameter.h"
#include "State.h"
#include "MachEvent.h"

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: ParameterDialog.cc,v $ $Revision: 1.7 $" ;

/*
STATIC MEMBER DEFINITIONS
*/

/*
FUNCTION DEFINITIONS
*/

ParameterDialog::
ParameterDialog(
	const char *name,
	UIComponent *parent) :
		TwoPartSelectionBox(name, parent)
{
}

void ParameterDialog::
show_parameter(
	Parameter *param)
{
	add_item(param->type().c_str(), param->name().c_str(), 0) ;
}

StateParameterDialog::
StateParameterDialog(
	const char *name,
	UIComponent *parent) :
		ParameterDialog(name, parent),
		_state(NULL)
{
	ok_command(new SimpleCommand<StateParameterDialog>(this,
		StateParameterDialog::add_parameter)) ;
	apply_command(new SimpleCommand<StateParameterDialog>(this,
		StateParameterDialog::change_parameter)) ;
	cancel_command(new SimpleCommand<StateParameterDialog>(this,
		StateParameterDialog::delete_parameter)) ;
}

void StateParameterDialog::
add_parameter()
{
	if (_state)
	{
		ChioTerm type(selection_text1()) ;
		ChioTerm name(selection_text2()) ;
		if (type.size() != 0 && name.size() != 0)
		{
			_state->add_parameter(type, name) ;
		}
	}
}

void StateParameterDialog::
change_parameter()
{
	if (_state)
	{
		ChioTerm type(selected_1()) ;
		ChioTerm name(selected_2()) ;
		if (type.size() != 0 && name.size() != 0)
		{
			StateParameter *sp = _state->find_parameter(type, name) ;
			if (sp)
			{
				ChioTerm new_type(selection_text1()) ;
				ChioTerm new_name(selection_text2()) ;
				if (new_type.size() != 0 && new_name.size() != 0)
				{
					if (_state->change_parameter(sp, new_type, new_name)
						== false)
					{
						cerr << "Parameter with type " << new_type
							<< " and name " << new_name
							<< " already exists for state " << *_state << "."
							<< endl ;
					}
				}
			}
		}
	}
}

void StateParameterDialog::
delete_parameter()
{
	if (_state)
	{
		ChioTerm type(selection_text1()) ;
		ChioTerm name(selection_text2()) ;
		if (type.size() != 0 && name.size() != 0)
		{
			StateParameter *sp = _state->find_parameter(type, name) ;
			delete sp ;
		}
	}
}

void StateParameterDialog::
update(
	const Subject *subject)
{
	if (_state)
	{
		assert(subject == _state) ;

		clear_list() ;
		const State::ParameterList& p_list = _state->parameter_list() ;
		for (State::ParameterListConstIter p_iter = p_list.begin() ;
			p_iter != p_list.end() ; ++p_iter)
		{
			show_parameter(*p_iter) ;
		}
	}
}

void StateParameterDialog::
subject_deleted(
	Subject *subject)
{
	assert(subject == _state) ;
	_state = NULL ;
	clear_list() ;
}

void StateParameterDialog::
state(
	State *new_state)
{
	State *old_state = _state ;
	_state = new_state ;
	if (old_state != _state)
	{
		if (old_state) old_state->detach(this) ;
		if (_state) _state->attach(this) ;
	}
	update(_state) ;
}

EventParameterDialog::
EventParameterDialog(
	const char *name,
	UIComponent *parent) :
		ParameterDialog(name, parent),
		_event(NULL)
{
	ok_command(new SimpleCommand<EventParameterDialog>(this,
		EventParameterDialog::add_parameter)) ;
	apply_command(new SimpleCommand<EventParameterDialog>(this,
		EventParameterDialog::change_parameter)) ;
	cancel_command(new SimpleCommand<EventParameterDialog>(this,
		EventParameterDialog::delete_parameter)) ;
}

void EventParameterDialog::
event(
	MachEvent *new_event)
{
	MachEvent *old_event = _event ;
	_event = new_event ;
	if (old_event != _event)
	{
		if (old_event) old_event->detach(this) ;
		if (_event) _event->attach(this) ;
	}
	update(_event) ;
}

void EventParameterDialog::
add_parameter()
{
	if (_event)
	{
		ChioTerm type(selection_text1()) ;
		ChioTerm name(selection_text2()) ;
		if (type.size() != 0 && name.size() != 0)
		{
			_event->add_parameter(type, name) ;
		}
	}
}

void EventParameterDialog::
change_parameter()
{
	if (_event)
	{
		ChioTerm type(selected_1()) ;
		ChioTerm name(selected_2()) ;
		if (type.size() != 0 && name.size() != 0)
		{
			EventParameter *ep = _event->find_parameter(type, name) ;
			if (ep)
			{
				ChioTerm new_type(selection_text1()) ;
				ChioTerm new_name(selection_text2()) ;
				if (new_type.size() != 0 && new_name.size() != 0)
				{
					if (_event->change_parameter(ep, new_type, new_name)
						== false)
					{
						cerr << "Parameter with type " << new_type
							<< " and name " << new_name
							<< " already exists for event " << *_event << "."
							<< endl ;
					}
				}
			}
		}
	}
}

void EventParameterDialog::
delete_parameter()
{
	if (_event)
	{
		ChioTerm type(selection_text1()) ;
		ChioTerm name(selection_text2()) ;
		if (type.size() != 0 && name.size() != 0)
		{
			EventParameter *ep = _event->find_parameter(type, name) ;
			delete ep ;
		}
	}
}

void EventParameterDialog::
update(
	const Subject *subject)
{
	clear_list() ;
	if (_event)
	{
		assert(subject == _event) ;

		MachEvent::ParameterList& p_list = _event->parameter_list() ;
		for (MachEvent::ParameterListIter p_iter = p_list.begin() ;
			p_iter != p_list.end() ; ++p_iter)
		{
			show_parameter(*p_iter) ;
		}
	}
}

void EventParameterDialog::
subject_deleted(
	Subject *subject)
{
	assert(subject == _event) ;
	_event = NULL ;
	clear_list() ;
}
