/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: SmachineDialogDirector.cc,v $
$Revision: 1.8 $
$Date: 1997/07/02 04:45:10 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: SmachineDialogDirector.cc,v $
Revision 1.8  1997/07/02 04:45:10  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.7  1997/06/06 04:34:26  andrewm
Checkpoint.  Changed the Subject / Observer code to include a
pointer to the changed subject in the call to "update".  This allowed
a given observer to observe muliple subjects.  Then I modified
various dialogs, especially those in that control the various fields
in the state machine to observe multiple subjects as necessary to
insure that all their fields were updated properly.

Revision 1.6  1997/05/31 21:12:42  andrewm
Checkpoint.  Things are working well.

Revision 1.5  1997/05/20 05:15:30  andrewm
Checkpoint.  Improved the structure of the "State" class and this
had quite some ripple effects.  However, now there is an abstract
class "State" with two concrete classes "PseudoState" to represent
error and ignore and "RealState" to represent the user specified
states.  Also improved the text display of event names on the transitions.

Revision 1.4  1997/05/15 04:14:44  andrewm
Checkpoint.  Reworked the low level file format stuff to contain
proper lists rather than maps keyed to binary numbers.
This point represents the entire program working with this file format
change.

Revision 1.3  1997/04/24 03:20:46  andrewm
Checkpoint.  All features in.  Starting test cycle.

Revision 1.2  1997/04/16 04:06:15  andrewm
Checkpoint, as last major dialog is in.

Revision 1.1  1997/04/08 04:34:31  andrewm
Checkpoint as dialogs are added.
Decided to make sure that the name part of a name/value pair in
the hierarchical file is not used for id purposes.  Specifically
state machine names need to be contained within the machine not as
the key to the machine.

 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "GsmDocument.h"
#include "Application.h"
#include "Command.h"
#include "MachineGroup.h"
#include "Smachine.h"
#include "State.h"
#include "DialogForm.h"
#include "MenuButton.h"
#include "TextEditor.h"
#include "SmachineDialogDirector.h"

#include <Xm/Separator.h>
#include <Xm/Label.h>
#include <Xm/RowColumn.h>
#include <Xm/TextF.h>
#include <XmCxx/CxxXmString.h>


/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/
typedef SimpleCommand<SmachineDialogDirector> SDD_Command ;

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: SmachineDialogDirector.cc,v $ $Revision: 1.8 $" ;

/*
STATIC MEMBER DEFINITIONS
*/

/*
FUNCTION DEFINITIONS
*/

SmachineDialogDirector::
SmachineDialogDirector(
	const char *name,
	UIComponent *parent,
	GsmDocument& document) :
		_document(document),
		_machine(NULL),
		_initial_state(NULL),
		_terminal_state(NULL),
		_default_state(NULL),
		_error_state(NULL),
		_ignore_state(NULL),
		_form(new DialogForm(name, parent)),
		_prolog_button(NULL),
		_prolog_editor(NULL),
		_epilog_button(NULL),
		_epilog_editor(NULL)
{
	Arg args[10] ;
	Cardinal n ;

	n = 0 ;
	XtSetArg(args[n], XmNverticalSpacing, 10) ; n++ ;
	XtSetArg(args[n], XmNhorizontalSpacing, 10) ; n++ ;
	XtSetArg(args[n], XmNautoUnmanage, False) ; n++ ;
	_form->setValues(args, n) ;

	n = 0 ;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM) ; n++ ;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM) ; n++ ;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM) ; n++ ;
	XtSetArg(args[n], XmNorientation, XmHORIZONTAL) ; n++ ;
	Widget name_work = XmCreateRowColumn(_form->widget(), "name_work",
		args, n) ;
	n = 0 ;
	Widget name_label = XmCreateLabel(name_work, "name_label", args, n) ;
	XtManageChild(name_label) ;
	n = 0 ;
	_name_entry = XmCreateTextField(name_work, "name_entry", args, n) ;
	XtAddCallback(_name_entry, XmNactivateCallback, name_entry_cb,
		(XtPointer)this) ;
	XtManageChild(_name_entry) ;
	XtManageChild(name_work) ;

	n = 0 ;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET) ; n++ ;
	XtSetArg(args[n], XmNtopWidget, name_work) ; n++ ;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM) ; n++ ;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM) ; n++ ;
	XtSetArg(args[n], XmNorientation, XmHORIZONTAL) ; n++ ;
	Widget initial_work = XmCreateRowColumn(_form->widget(), "initial_work",
		args, n) ;
	n = 0 ;
	Widget initial_label = XmCreateLabel(initial_work, "initial_label",
		args, n) ;
	XtManageChild(initial_label) ;
	n = 0 ;
	_initial_state_entry = XmCreateTextField(initial_work,
		"initial_state_entry", args, n) ;
	XtAddCallback(_initial_state_entry, XmNactivateCallback,
		initial_state_entry_cb, (XtPointer)this) ;
	XtManageChild(_initial_state_entry) ;
	XtManageChild(initial_work) ;

	n = 0 ;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET) ; n++ ;
	XtSetArg(args[n], XmNtopWidget, initial_work) ; n++ ;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM) ; n++ ;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM) ; n++ ;
	XtSetArg(args[n], XmNorientation, XmHORIZONTAL) ; n++ ;
	Widget terminal_work = XmCreateRowColumn(_form->widget(), "terminal_work",
		args, n) ;
	n = 0 ;
	Widget terminal_label = XmCreateLabel(terminal_work, "terminal_label",
		args, n) ;
	XtManageChild(terminal_label) ;
	n = 0 ;
	_terminal_state_entry = XmCreateTextField(terminal_work,
		"terminal_state_entry", args, n) ;
	XtAddCallback(_terminal_state_entry, XmNactivateCallback,
		terminal_state_entry_cb, (XtPointer)this) ;
	XtManageChild(_terminal_state_entry) ;
	XtManageChild(terminal_work) ;

	n = 0 ;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET) ; n++ ;
	XtSetArg(args[n], XmNtopWidget, terminal_work) ; n++ ;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM) ; n++ ;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM) ; n++ ;
	XtSetArg(args[n], XmNorientation, XmHORIZONTAL) ; n++ ;
	Widget default_work = XmCreateRowColumn(_form->widget(), "default_work",
		args, n) ;
	n = 0 ;
	Widget default_label = XmCreateLabel(default_work, "default_label",
		args, n) ;
	XtManageChild(default_label) ;
	n = 0 ;
	_default_state_entry = XmCreateTextField(default_work,
		"default_state_entry", args, n) ;
	XtAddCallback(_default_state_entry, XmNactivateCallback,
		default_state_entry_cb, (XtPointer)this) ;
	XtManageChild(_default_state_entry) ;
	XtManageChild(default_work) ;

	n = 0 ;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET) ; n++ ;
	XtSetArg(args[n], XmNtopWidget, default_work) ; n++ ;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM) ; n++ ;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM) ; n++ ;
	Widget sep1 = XmCreateSeparator(_form->widget(), "sep2", args, n) ;
	XtManageChild(sep1) ;

	Application *appl = Application::instance() ;

	_error_code_button = new MenuButton("edit_error_code", _form) ;
	n = 0 ; 
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET) ; n++ ;
	XtSetArg(args[n], XmNtopWidget, sep1) ; n++ ;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM) ; n++ ;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM) ; n++ ;
	_error_code_button->setValues(args, n) ;
	_error_code_button->command(new SDD_Command(this,
		SmachineDialogDirector::edit_error_code)) ;
	_error_code_button->manage() ;
	_error_code_editor = new TextEditor("Error State", appl) ;
	_error_code_editor->save_command(new SDD_Command(this,
		SmachineDialogDirector::set_error_code)) ;

	_ignore_code_button = new MenuButton("edit_ignore_code", _form) ;
	n = 0 ; 
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET) ; n++ ;
	XtSetArg(args[n], XmNtopWidget, _error_code_button->widget()) ; n++ ;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM) ; n++ ;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM) ; n++ ;
	_ignore_code_button->setValues(args, n) ;
	_ignore_code_button->command(new SDD_Command(this,
		SmachineDialogDirector::edit_ignore_code)) ;
	_ignore_code_button->manage() ;
	_ignore_code_editor = new TextEditor("Ignore State", appl) ;
	_ignore_code_editor->save_command(new SDD_Command(this,
		SmachineDialogDirector::set_ignore_code)) ;

	_prolog_button = new MenuButton("edit_prolog", _form) ;
	n = 0 ; 
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET) ; n++ ;
	XtSetArg(args[n], XmNtopWidget, _ignore_code_button->widget()) ; n++ ;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM) ; n++ ;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM) ; n++ ;
	_prolog_button->setValues(args, n) ;
	_prolog_button->command(new SDD_Command(this,
		SmachineDialogDirector::edit_prolog)) ;
	_prolog_button->manage() ;
	_prolog_editor = new TextEditor("Machine Prolog", appl) ;
	_prolog_editor->save_command(new SDD_Command(this,
		SmachineDialogDirector::set_prolog)) ;

	_epilog_button = new MenuButton("edit_epilog", _form) ;
	n = 0 ; 
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET) ; n++ ;
	XtSetArg(args[n], XmNtopWidget, _prolog_button->widget()) ; n++ ;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM) ; n++ ;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM) ; n++ ;
	XtSetArg(args[n], XmNbottomOffset, 10) ; n++ ;
	_epilog_button->setValues(args, n) ;
	_epilog_button->command(new SDD_Command(this,
		SmachineDialogDirector::edit_epilog)) ;
	_epilog_button->manage() ;
	_epilog_editor = new TextEditor("Machine Epilog", appl) ;
	_epilog_editor->save_command(new SDD_Command(this,
		SmachineDialogDirector::set_epilog)) ;

	_document.attach(this) ;
}

SmachineDialogDirector::
~SmachineDialogDirector()
{
	delete _error_code_button ;
	delete _error_code_editor ;
	delete _ignore_code_button ;
	delete _ignore_code_editor ;
	delete _prolog_button ;
	delete _prolog_editor ;
	delete _epilog_button ;
	delete _epilog_editor ;
	delete _form ;
	_document.detach(this) ;
	if (_machine)
		_machine->detach(this) ;
	detach_states() ;
}

void SmachineDialogDirector::
show_dialog()
{
	update(&_document) ;
	_form->manage() ;
}

void SmachineDialogDirector::
dismiss_dialog()
{
	_form->unmanage() ;
}

void SmachineDialogDirector::
update(
	const Subject *subject)
{
	if (subject == &_document)
		update_document() ;
	else if (subject == _machine)
		update_machine() ;
	else if (subject == _initial_state)
		update_initial_state() ;
	else if (subject == _terminal_state)
		update_terminal_state() ;
	else if (subject == _default_state)
		update_default_state() ;
	else if (subject == _error_state)
		update_error_state() ;
	else if (subject == _ignore_state)
		update_ignore_state() ;
}

void SmachineDialogDirector::
subject_deleted(
	Subject *subject)
{
	if (subject == _machine)
	{
		_machine = NULL ;
		update_machine() ;
	}
	else if (subject == _initial_state)
	{
		_initial_state = NULL ;
		update_initial_state() ;
	}
	else if (subject == _terminal_state)
	{
		_terminal_state = NULL ;
		update_terminal_state() ;
	}
	else if (subject == _default_state)
	{
		_default_state = NULL ;
		update_default_state() ;
	}
	else if (subject == _error_state)
	{
		_error_state = NULL ;
		update_error_state() ;
	}
	else if (subject == _ignore_state)
	{
		_ignore_state = NULL ;
		update_ignore_state() ;
	}
	// else (subject == &_document) and "this" is deleted in the
	// destructor for the GsmDocument.
}

bool SmachineDialogDirector::
modified() const
{
	bool mod = false ;

	if (mod == false)
		mod = _error_code_editor->modified() ;
	if (mod == false)
		mod = _ignore_code_editor->modified() ;
	if (mod == false)
		mod = _prolog_editor->modified() ;
	if (mod == false)
		mod = _epilog_editor->modified() ;

	return mod ;
}

void SmachineDialogDirector::
modified(
	bool new_modified)
{
	_error_code_editor->modified(new_modified) ;
	_ignore_code_editor->modified(new_modified) ;
	_prolog_editor->modified(new_modified) ;
	_epilog_editor->modified(new_modified) ;
}

void SmachineDialogDirector::
set_name()
{
	char *name_entered = XmTextFieldGetString(_name_entry) ;
	_machine->rename(ChioTerm(name_entered,
		ChioTerm::StringChioTerm, ChioTerm::ExternalChioTerm)) ;
	XtFree(name_entered) ;
}

bool SmachineDialogDirector::
set_initial_state()
{
	bool status = true ;
	char *state_entered = XmTextFieldGetString(_initial_state_entry) ;
	if (*state_entered)
	{
		ChioTerm se(state_entered, ChioTerm::StringChioTerm,
			ChioTerm::ExternalChioTerm) ;
		State *state = _machine->find_state(se) ;
		if (state)
		{
			if (state->is_pseudo_state())
				cerr << "The initial state may not be set to \"" << *state
					<< "\"." << endl ;
			else
				_machine->initial_state(state) ;
		}
		else
		{
			cerr << "State " << se << " does not exist." << endl ;
			status = false ;
			update_initial_state() ;
		}
	}
	else
	{
		_machine->initial_state(NULL) ;
	}
	XtFree(state_entered) ;

	return status ;
}

bool SmachineDialogDirector::
set_terminal_state()
{
	bool status = true ;
	char *state_entered = XmTextFieldGetString(_terminal_state_entry) ;
	if (*state_entered)
	{
		ChioTerm se(state_entered, ChioTerm::StringChioTerm,
			ChioTerm::ExternalChioTerm) ;
		State *state = _machine->find_state(se) ;
		if (state)
		{
			if (state->is_pseudo_state())
				cerr << "The terminal state may not be set to \"" << *state
					<< "\"." << endl ;
			else
				_machine->terminal_state(state) ;
		}
		else
		{
			cerr << "State " << se << " does not exist." << endl ;
			status = false ;
			update_terminal_state() ;
		}
	}
	else
	{
		_machine->terminal_state(NULL) ;
	}
	XtFree(state_entered) ;

	return status ;
}

bool SmachineDialogDirector::
set_default_state()
{
	bool status = true ;
	char *state_entered = XmTextFieldGetString(_default_state_entry) ;
	ChioTerm se(state_entered, ChioTerm::StringChioTerm,
		ChioTerm::ExternalChioTerm) ;
	State *state = _machine->find_state(se) ;
	if (state)
		_machine->default_state(state) ;
	else
	{
		cerr << "State " << se << " does not exist." << endl ;
		status = false ;
		update_default_state() ;
	}
	XtFree(state_entered) ;

	return status ;
}

void SmachineDialogDirector::
set_error_code()
{
	_error_state->code(ChioTerm(_error_code_editor->get_text(),
		ChioTerm::MultiLineChioTerm)) ;
	_error_code_editor->modified(false) ;
}

void SmachineDialogDirector::
set_ignore_code()
{
	_ignore_state->code(ChioTerm(_ignore_code_editor->get_text(),
		ChioTerm::MultiLineChioTerm)) ;
	_ignore_code_editor->modified(false) ;
}

void SmachineDialogDirector::
set_prolog()
{
	_machine->prolog(ChioTerm(_prolog_editor->get_text(),
		ChioTerm::MultiLineChioTerm)) ;
	_prolog_editor->modified(false) ;
}

void SmachineDialogDirector::
set_epilog()
{
	_machine->epilog(ChioTerm(_epilog_editor->get_text(),
		ChioTerm::MultiLineChioTerm)) ;
	_epilog_editor->modified(false) ;
}

void SmachineDialogDirector::
edit_error_code()
{
	_error_code_editor->manage() ;
}

void SmachineDialogDirector::
edit_ignore_code()
{
	_ignore_code_editor->manage() ;
}

void SmachineDialogDirector::
edit_prolog()
{
	_prolog_editor->manage() ;
}

void SmachineDialogDirector::
edit_epilog()
{
	_epilog_editor->manage() ;
}

void SmachineDialogDirector::
update_document()
{
	Smachine *old_machine = _machine ;
	_machine = _document.current_machine() ;
	if (old_machine != _machine)
	{
		if (old_machine)
		{
			old_machine->detach(this) ;
			if (_error_code_editor->modified())
				set_error_code() ;
			if (_ignore_code_editor->modified())
				set_ignore_code() ;
			if (_prolog_editor->modified())
			{
				old_machine->prolog(ChioTerm(_prolog_editor->get_text(),
					ChioTerm::MultiLineChioTerm)) ;
				_prolog_editor->modified(false) ;
			}
			if (_epilog_editor->modified())
			{
				old_machine->epilog(ChioTerm(_epilog_editor->get_text(),
					ChioTerm::MultiLineChioTerm)) ;
				_epilog_editor->modified(false) ;
			}
		}
		detach_states() ;
		if (_machine) _machine->attach(this) ;
		update_machine() ;
	}
}

void SmachineDialogDirector::
update_machine()
{
	if (_machine != NULL)
	{
		XmTextFieldSetString(_name_entry,
			(char *)_machine->name().internal_to_external().c_str()) ;
		_prolog_editor->set_text(_machine->prolog().c_str()) ;
		_epilog_editor->set_text(_machine->epilog().c_str()) ;

		State *new_state = _machine->initial_state() ;
		if (new_state != _initial_state)
		{
			if (_initial_state)
				_initial_state->detach(this) ;
			if (_initial_state = new_state)
				_initial_state->attach(this) ;
			update_initial_state() ;
		}

		new_state = _machine->terminal_state() ;
		if (new_state != _terminal_state)
		{
			if (_terminal_state)
				_terminal_state->detach(this) ;
			if (_terminal_state = new_state)
				_terminal_state->attach(this) ;
			update_terminal_state() ;
		}

		new_state = _machine->default_state() ;
		if (new_state != _default_state)
		{
			if (_default_state)
				_default_state->detach(this) ;
			if (_default_state = new_state)
				_default_state->attach(this) ;
			update_default_state() ;
		}

		new_state = _machine->find_state("error") ;
		if (new_state != _error_state)
		{
			if (_error_state)
				_error_state->detach(this) ;
			if (_error_state = new_state)
				_error_state->attach(this) ;
			update_error_state() ;
		}

		new_state = _machine->find_state("ignore") ;
		if (new_state != _ignore_state)
		{
			if (_ignore_state)
				_ignore_state->detach(this) ;
			if (_ignore_state = new_state)
				_ignore_state->attach(this) ;
			update_ignore_state() ;
		}
	}
	else
	{
		XmTextFieldSetString(_name_entry, "") ;
		_prolog_editor->set_text("") ;
		_epilog_editor->set_text("") ;
		detach_states() ;
	}
}

void SmachineDialogDirector::
update_initial_state()
{
	XmTextFieldSetString(_initial_state_entry,
		_initial_state ? 
		(char *)_initial_state->name().internal_to_external().c_str() : "") ;
}

void SmachineDialogDirector::
update_terminal_state()
{
	XmTextFieldSetString(_terminal_state_entry,
		_terminal_state ? 
		(char *)_terminal_state->name().internal_to_external().c_str() : "") ;
}

void SmachineDialogDirector::
update_default_state()
{
	XmTextFieldSetString(_default_state_entry,
		_default_state ? 
		(char *)_default_state->name().internal_to_external().c_str() : "") ;
}

void SmachineDialogDirector::
update_error_state()
{
	_error_code_editor->set_text(
		_error_state ? _error_state->code().c_str() : "") ;
}

void SmachineDialogDirector::
update_ignore_state()
{
	_ignore_code_editor->set_text(
		_ignore_state ? _ignore_state->code().c_str() : "") ;
}

void SmachineDialogDirector::
detach_states()
{
	if (_initial_state)
	{
		_initial_state->detach(this) ;
		_initial_state = NULL ;
		update_initial_state() ;
	}
	if (_terminal_state)
	{
		_terminal_state->detach(this) ;
		_terminal_state = NULL ;
		update_terminal_state() ;
	}
	if (_default_state)
	{
		_default_state->detach(this) ;
		_default_state = NULL ;
		update_default_state() ;
	}
	if (_error_state)
	{
		_error_state->detach(this) ;
		_error_state = NULL ;
		update_error_state() ;
	}
	if (_ignore_state)
	{
		_ignore_state->detach(this) ;
		_ignore_state = NULL ;
		update_ignore_state() ;
	}
}

void SmachineDialogDirector::
name_entry_cb(
	Widget widget,
	XtPointer closure,
	XtPointer callback_data)
{
	SmachineDialogDirector *dd = (SmachineDialogDirector *)closure ;
	dd->set_name() ;
	XmProcessTraversal(widget, XmTRAVERSE_NEXT_TAB_GROUP) ;
}

void SmachineDialogDirector::
initial_state_entry_cb(
	Widget widget,
	XtPointer closure,
	XtPointer callback_data)
{
	SmachineDialogDirector *dd = (SmachineDialogDirector *)closure ;
	if (dd->set_initial_state())
		XmProcessTraversal(widget, XmTRAVERSE_NEXT_TAB_GROUP) ;
}

void SmachineDialogDirector::
terminal_state_entry_cb(
	Widget widget,
	XtPointer closure,
	XtPointer callback_data)
{
	SmachineDialogDirector *dd = (SmachineDialogDirector *)closure ;
	if (dd->set_terminal_state())
		XmProcessTraversal(widget, XmTRAVERSE_NEXT_TAB_GROUP) ;
}

void SmachineDialogDirector::
default_state_entry_cb(
	Widget widget,
	XtPointer closure,
	XtPointer callback_data)
{
	SmachineDialogDirector *dd = (SmachineDialogDirector *)closure ;
	if (dd->set_default_state())
		XmProcessTraversal(widget, XmTRAVERSE_NEXT_TAB_GROUP) ;
}
