/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: MachEvent.cc,v $
$Revision: 1.15 $
$Date: 1997/07/02 04:45:11 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: MachEvent.cc,v $
Revision 1.15  1997/07/02 04:45:11  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.14  1997/05/31 21:12:42  andrewm
Checkpoint.  Things are working well.

Revision 1.13  1997/05/20 05:15:33  andrewm
Checkpoint.  Improved the structure of the "State" class and this
had quite some ripple effects.  However, now there is an abstract
class "State" with two concrete classes "PseudoState" to represent
error and ignore and "RealState" to represent the user specified
states.  Also improved the text display of event names on the transitions.

Revision 1.12  1997/05/15 04:14:44  andrewm
Checkpoint.  Reworked the low level file format stuff to contain
proper lists rather than maps keyed to binary numbers.
This point represents the entire program working with this file format
change.

Revision 1.11  1997/04/24 03:20:47  andrewm
Checkpoint.  All features in.  Starting test cycle.

Revision 1.10  1997/04/12 21:15:01  andrewm
Checkpoint after adding the event dialog.

Revision 1.9  1997/03/04 06:32:54  andrewm
Another check point.  The editor can draw output from files.
The crashing during the dtor for MachineGroup is fixed.

Revision 1.8  1997/02/23 23:44:12  andrewm
Checkpoint.  Things seem to be working reasonably well.

Revision 1.7  1996/12/26 05:55:24  andrewm
Checkpoint, the compiler is working again.

Revision 1.6  1996/12/24 05:20:10  andrewm
Checkpoint.

Revision 1.5  1996/09/22 01:18:22  andrewm
pre-alpha release

// Revision 1.4  1996/08/18  17:57:45  andrewm
// checkpoint
//
// Revision 1.3  1996/08/05  01:25:42  andrewm
// checkpoint
//
// Revision 1.2  1996/06/26  03:14:43  andrewm
// checkpoint
//
// Revision 1.1  1996/06/15  23:53:13  andrewm
// Initial revision
//
 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "MachEvent.h"
#include "Smachine.h"
#include "State.h"
#include "Transition.h"
#include "Parameter.h"
#include "CleanName.h"

#include <assert.h>
#include <algorithm>

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: MachEvent.cc,v $ $Revision: 1.15 $" ;

/*
STATIC MEMBER DEFINITIONS
*/

/*
FUNCTION DEFINITIONS
*/

MachEvent::
MachEvent(
	Smachine *parent,
	const ChioTerm& name) :
		_parent(parent),
		_name(name),
		_modified(true)
{
		// Add your own name to the parent's map of state machines.
	ChioMap& parent_map = _parent->events() ;
	ChioMap::pair_iterator_bool result = parent_map.insert(
		ChioAssignment(name, ChioValue(ChioValue::ListChioValue))) ;
	assert(result.second == true) ;
}

MachEvent::
MachEvent(
	Smachine *parent,
	ChioMapIter place) :
		_parent(parent),
		_modified(false)
{
	_name = (*place).first ;
	ChioList& file_def = (*place).second ;

	for (ChioListIter e_iter = file_def.begin() ;
		e_iter != file_def.end() ; ++e_iter)
	{
		_param_list.push_back(new EventParameter(this, e_iter)) ;
	}
}

MachEvent::
~MachEvent(void)
{
	if (_parent)
	{
			/*
			Delete all the transitions that are caused by this event.
			*/
		Smachine::StateList& s_list = _parent->state_list() ;
		for (Smachine::StateListIter s_iter = s_list.begin() ;
			s_iter != s_list.end() ; ++s_iter)
		{
			State *state = *s_iter ;
			Transition *trans = state->find_transition(this) ;
			if (trans)
				delete trans ;
		}

		_parent->events().erase(find_place()) ;
		_parent->remove_event(this) ;
	}

	for (ParameterListIter param_iter = _param_list.begin() ;
		param_iter != _param_list.end() ; param_iter++)
	{
		EventParameter *param = *param_iter ;
		param->orphan() ;
		delete param ;
	}
}

bool MachEvent::
modified(void) const
{
	bool needs = false ;
	for (ParameterListConstIter param_iter = _param_list.begin() ;
		param_iter != _param_list.end() ; param_iter++)
	{
		EventParameter *param = *param_iter ;
		needs = needs || param->modified() ;
	}

	return (_modified || needs) ;
}

void MachEvent::
modified(
	bool new_modified)
{
	for (ParameterListIter param_iter = _param_list.begin() ;
		param_iter != _param_list.end() ; param_iter++)
	{
		EventParameter *param = *param_iter ;
		param->modified(new_modified) ;
	}
	_modified = new_modified ;
}

void MachEvent::
sync()
{
	for (ParameterListConstIter param_iter = _param_list.begin() ;
		param_iter != _param_list.end() ; param_iter++)
	{
		EventParameter *param = *param_iter ;
		param->sync() ;
	}
}

void MachEvent::
rename(
	const ChioTerm& new_name)
{
	MachEvent *event = _parent->find_event(new_name) ;
	if (event)
		cerr << "An event by the name of " << new_name << " exists." << endl ;
	else
	{
		ChioMapIter place = find_place() ;
		ChioValue& old_value = (*place).second ;
		ChioMap& event_map = _parent->events() ;
		ChioMap::pair_iterator_bool result = event_map.insert(
			ChioAssignment(new_name, old_value)) ;
		assert(result.second == true) ;

		event_map.erase(place) ;
		_name = new_name ;
		_modified = true ;
		notify() ;
	}
}

EventParameter *MachEvent::
find_parameter(
	const ChioTerm& type,
	const ChioTerm& name) const
{
	for (ParameterListConstIter param_iter = _param_list.begin() ;
		param_iter != _param_list.end() ; param_iter++)
	{
		EventParameter *param = *param_iter ;
		if (param->type() == type && param->name() == name)
			return param ;
	}
	return NULL ;
}

EventParameter *MachEvent::
add_parameter(
	const ChioTerm& type,
	const ChioTerm& name)
{
	EventParameter *parameter = find_parameter(type, name) ;
	if (!parameter)
	{
		parameter = new EventParameter(this, parameters().end(), type, name) ;
		_param_list.push_back(parameter) ;
		_modified = true ;
		notify() ;
	}
	return parameter ;
}

EventParameter *MachEvent::
insert_parameter(
	EventParameter *current,
	const ChioTerm& type,
	const ChioTerm& name)
{
	EventParameter *parameter = find_parameter(type, name) ;
	if (parameter)
		return NULL ;

	parameter = new EventParameter(this, current->find_place(), type, name) ;
	ParameterListIter p_pos = find(_param_list.begin(), _param_list.end(),
		current) ;
	assert(p_pos != _param_list.end()) ;
	_param_list.insert(p_pos, parameter) ;
	_modified = true ;
	notify() ;
	return parameter ;
}

bool MachEvent::
change_parameter(
	EventParameter *parameter,
	const ChioTerm& type,
	const ChioTerm& name)
{
	EventParameter *p = find_parameter(type, name) ;
	if (p)
		return false ;

	parameter->type(type) ;
	parameter->name(name) ;
	notify() ;
	return true ;
}

void MachEvent::
remove_parameter(
	EventParameter *parameter)
{
	parameter->orphan() ;
	_param_list.remove(parameter) ;
	_modified = true ;
	notify() ;
}

ChioList& MachEvent::
parameters()
{
	ChioMapIter place = find_place() ;
	return (ChioList&)((*place).second) ;
}

ChioMapIter MachEvent::
find_place()
{
	ChioMap& event_map = _parent->events() ;
	ChioMapIter found = event_map.find(_name) ;
	assert(found != event_map.end()) ;
	return found ;
}

ostream&
operator<<(
	ostream& stream,
	MachEvent& event)
{
	stream << event._name.c_str() << '(' ;

	const char *leader = "" ;
	for (MachEvent::ParameterListIter param_iter = event._param_list.begin() ;
		param_iter != event._param_list.end() ; param_iter++)
	{
		EventParameter *param = *param_iter ;
		stream << leader << *param ;
		leader = ", " ;
	}

	stream << ')' ;

	return stream ;
}
