/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: State.h,v $
$Revision: 1.19 $
$Date: 1997/07/02 04:45:13 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: State.h,v $
Revision 1.19  1997/07/02 04:45:13  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.18  1997/05/31 21:12:43  andrewm
Checkpoint.  Things are working well.

Revision 1.17  1997/05/20 05:15:34  andrewm
Checkpoint.  Improved the structure of the "State" class and this
had quite some ripple effects.  However, now there is an abstract
class "State" with two concrete classes "PseudoState" to represent
error and ignore and "RealState" to represent the user specified
states.  Also improved the text display of event names on the transitions.

Revision 1.16  1997/05/15 04:14:46  andrewm
Checkpoint.  Reworked the low level file format stuff to contain
proper lists rather than maps keyed to binary numbers.
This point represents the entire program working with this file format
change.

Revision 1.15  1997/04/24 03:20:49  andrewm
Checkpoint.  All features in.  Starting test cycle.

Revision 1.14  1997/03/18 06:51:03  andrewm
Checkpoint.  Mouse select, insert, and delete working.
Some changes to improve robustness in the face of an arbitrary input file.

Revision 1.13  1997/03/12 03:13:07  andrewm
Checkpoint.  Things are working rather well.

Revision 1.12  1997/03/04 06:32:55  andrewm
Another check point.  The editor can draw output from files.
The crashing during the dtor for MachineGroup is fixed.

Revision 1.11  1997/02/23 23:44:13  andrewm
Checkpoint.  Things seem to be working reasonably well.

Revision 1.10  1997/02/08 04:37:40  andrewm
Checkpoint before returning to work on the GUI portion.

Revision 1.9  1997/01/23 06:20:29  andrewm
Checkpoint as base and graphics classes are operating together.

Revision 1.8  1996/12/26 05:55:25  andrewm
Checkpoint, the compiler is working again.

Revision 1.7  1996/12/24 05:20:11  andrewm
Checkpoint.

Revision 1.6  1996/10/01 04:39:14  andrewm
checkpoint and revision

 * Revision 1.5  1996/09/22  01:18:22  andrewm
 * pre-alpha release
 *
 * Revision 1.4  1996/08/18  17:57:45  andrewm
 * checkpoint
 *
 * Revision 1.3  1996/07/15  01:20:40  andrewm
 * checkpoint again
 *
 * Revision 1.2  1996/06/26  03:14:43  andrewm
 * checkpoint
 *
 * Revision 1.1  1996/06/15  23:53:13  andrewm
 * Initial revision
 *
 *--
 */
#ifndef _State_h_
#define _State_h_

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma interface
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "Chio.h"
#include "Observer.h"
#include "Geom2d.h"
#include "MachineVisitor.h"
#include "Smachine.h"
#include <list>

/*
MACRO DEFINITIONS
*/

/*
CLASS FORWARD REFERENCES
*/

class MachEvent ;
class StateParameter ;
class Transition ;
class PseudoTransition ;
class LoopTransition ;
class SegmentTransition ;

/*
CLASS DEFINITIONS
*/

class State :
	public Subject
{
	friend ostream& operator <<(ostream& stream, const State& state) ;

public:
	typedef list<StateParameter *> ParameterList ;
	typedef list<StateParameter *>::iterator ParameterListIter ;
	typedef list<StateParameter *>::const_iterator ParameterListConstIter ;
	typedef list<Transition *> TransitionList ;
	typedef list<Transition *>::iterator TransitionListIter ;
	typedef list<Transition *>::const_iterator TransitionListConstIter ;

public:
	State(Smachine *parent, const ChioTerm& name) ;
	State(Smachine *parent, ChioMapIter place) ;
	virtual ~State() ;

	Smachine *parent() const { return _parent ; }
	void orphan() { _parent = NULL ; }
	const ChioTerm& name() const { return _name ; }
	const ChioTerm& code() const { return _code ; }
	void code(const ChioTerm& new_code) ;
	TransitionList get_inbound_transitions() ;
	bool is_initial_state() { return _parent->initial_state() == this ; }
	bool is_terminal_state() { return _parent->terminal_state() == this ; }
	bool is_default_state() { return _parent->default_state() == this ; }

	virtual bool rename(const ChioTerm& new_name) = 0 ;
	virtual bool modified() const = 0 ;
	virtual void modified(bool new_modified) = 0 ;
	virtual void sync() = 0 ;
	virtual bool is_pseudo_state() = 0 ;
	virtual bool is_real_state() = 0 ;
	virtual void print_state(ostream& s) const = 0 ;

	virtual StateParameter *find_parameter(const ChioTerm& type,
		const ChioTerm& name) const = 0 ;
	virtual StateParameter *add_parameter(const ChioTerm& type,
		const ChioTerm& name) = 0 ;
	virtual StateParameter *insert_parameter(StateParameter *current,
		const ChioTerm& type, const ChioTerm& name) = 0 ;
	virtual bool change_parameter(StateParameter *parameter,
		const ChioTerm& type, const ChioTerm&name) = 0 ;
	virtual void remove_parameter(StateParameter *parameter) = 0 ;
	virtual ParameterList parameter_list() = 0 ;
	virtual ChioList& parameters() = 0 ;

	virtual Transition *find_transition(MachEvent *event) const = 0 ;
	virtual Transition *find_transition_at_location(
		const Point& loc) const = 0 ;
	virtual PseudoTransition *add_pseudo_transition(MachEvent *event,
		State *destination) = 0 ;
	virtual LoopTransition *add_loop_transition(MachEvent *event) = 0 ;
	virtual SegmentTransition *add_segment_transition(MachEvent* event,
		State *destination) = 0 ;
	virtual void remove_transition(Transition *transition) = 0 ;
	virtual void resolve_transitions() = 0 ;
	virtual TransitionList transition_list() = 0 ;
	virtual ChioMap& transitions() = 0 ;

	virtual const Circle& circle() const = 0 ;
	virtual void adjust_radius(float new_radius) = 0 ;
	virtual void move(const Point& p) = 0 ;
	virtual void move_relative(const Point& p) = 0 ;
	virtual bool contains(const Point& p) = 0 ;
	virtual Rectangle bounding_box() const = 0 ;
	virtual void selected(bool new_selected) = 0 ;
	virtual bool selected() const = 0 ;
	virtual void select_all(bool new_selected) = 0 ;
	virtual void select_if_within(const Rectangle& box) = 0 ;

	virtual void accept_visitor(MachineVisitor& visitor) = 0 ;

protected:
	Smachine *_parent ;
	ChioTerm _name ;
	ChioTerm _code ;
	bool _modified ;

protected:
	ChioMapIter find_place() ;

protected:
	static const char _code_name[] ;

private:
	State& operator =(const State& state) ; // no assignments
} ;

class PseudoState :
	public State
{
public:
	PseudoState(Smachine *parent, const ChioTerm& name) ;
	PseudoState(Smachine *parent, ChioMapIter place) ;
	virtual ~PseudoState() ;

	virtual bool rename(const ChioTerm& new_name) ;
	virtual bool modified() const ;
	virtual void modified(bool new_modified) ;
	virtual void sync() ;
	virtual bool is_pseudo_state() ;
	virtual bool is_real_state() ;
	virtual void print_state(ostream& s) const ;

	virtual StateParameter *find_parameter(const ChioTerm& type,
		const ChioTerm& name) const ;
	virtual StateParameter *add_parameter(const ChioTerm& type,
		const ChioTerm& name) ;
	virtual StateParameter *insert_parameter(StateParameter *current,
		const ChioTerm& type, const ChioTerm& name) ;
	virtual bool change_parameter(StateParameter *parameter,
		const ChioTerm& type, const ChioTerm&name) ;
	virtual void remove_parameter(StateParameter *parameter) ;
	virtual ParameterList parameter_list() ;
	virtual ChioList& parameters() ;

	virtual Transition *find_transition(MachEvent *event) const ;
	virtual Transition *find_transition_at_location(const Point& loc) const ;
	virtual PseudoTransition *add_pseudo_transition(MachEvent *event,
		State *destination) ;
	virtual LoopTransition *add_loop_transition(MachEvent *event) ;
	virtual SegmentTransition *add_segment_transition(MachEvent* event,
		State *destination) ;
	virtual void remove_transition(Transition *transition) ;
	virtual void resolve_transitions() ;
	virtual TransitionList transition_list() ;
	virtual ChioMap& transitions() ;

	virtual const Circle& circle() const ;
	virtual void adjust_radius(float new_radius) ;
	virtual void move(const Point& p) ;
	virtual void move_relative(const Point& p) ;
	virtual bool contains(const Point& p) ;
	virtual Rectangle bounding_box() const ;
	virtual void selected(bool new_selected) ;
	virtual bool selected() const ;
	virtual void select_all(bool new_selected) ;
	virtual void select_if_within(const Rectangle& box) ;

	virtual void accept_visitor(MachineVisitor& visitor) ;

private:
	PseudoState& operator =(const PseudoState& state) ; // no assignments
} ;


class RealState :
	public State
{
public:
	RealState(Smachine *parent, const ChioTerm& name) ;
	RealState(Smachine *parent, ChioMapIter place) ;
	virtual ~RealState() ;

	virtual bool rename(const ChioTerm& new_name) ;
	virtual bool modified() const ;
	virtual void modified(bool new_modified) ;
	virtual void sync() ;
	virtual bool is_pseudo_state() ;
	virtual bool is_real_state() ;
	virtual void print_state(ostream& s) const ;

	virtual StateParameter *find_parameter(const ChioTerm& type,
		const ChioTerm& name) const ;
	virtual StateParameter *add_parameter(const ChioTerm& type,
		const ChioTerm& name) ;
	virtual StateParameter *insert_parameter(StateParameter *current,
		const ChioTerm& type, const ChioTerm& name) ;
	virtual bool change_parameter(StateParameter *parameter,
		const ChioTerm& type, const ChioTerm&name) ;
	virtual void remove_parameter(StateParameter *parameter) ;
	virtual ParameterList parameter_list() ;
	virtual ChioList& parameters() ;

	virtual Transition *find_transition(MachEvent *event) const ;
	virtual Transition *find_transition_at_location(const Point& loc) const ;
	virtual PseudoTransition *add_pseudo_transition(MachEvent *event,
		State *destination) ;
	virtual LoopTransition *add_loop_transition(MachEvent *event) ;
	virtual SegmentTransition *add_segment_transition(MachEvent* event,
		State *destination) ;
	virtual void remove_transition(Transition *transition) ;
	virtual void resolve_transitions() ;
	virtual TransitionList transition_list() ;
	virtual ChioMap& transitions() ;

	virtual const Circle& circle() const ;
	virtual void adjust_radius(float new_radius) ;
	virtual void move(const Point& p) ;
	virtual void move_relative(const Point& p) ;
	virtual bool contains(const Point& p) ;
	virtual Rectangle bounding_box() const ;
	virtual void selected(bool new_selected) ;
	virtual bool selected() const ;
	virtual void select_all(bool new_selected) ;
	virtual void select_if_within(const Rectangle& box) ;

	virtual void accept_visitor(MachineVisitor& visitor) ;

private:
	ParameterList _param_list ;
	TransitionList _trans_list ;
		// Graphic representation
	Circle _circle ;
	bool _selected ;

private:
	void notify_src_trans() ;
	void notify_dest_trans() ;
	RealState& operator =(const RealState& state) ; // no assignments

private:
	static const char _parameters_name[] ;
	static const char _transitions_name[] ;
	static const char _circle_name[] ;
	static const char _center_name[] ;
	static const char _radius_name[] ;
	static const float default_radius ;
} ;



#endif /* _State_h_ */
