/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: Transition.h,v $
$Revision: 1.15 $
$Date: 1997/07/02 04:45:14 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: Transition.h,v $
Revision 1.15  1997/07/02 04:45:14  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.14  1997/05/20 05:15:35  andrewm
Checkpoint.  Improved the structure of the "State" class and this
had quite some ripple effects.  However, now there is an abstract
class "State" with two concrete classes "PseudoState" to represent
error and ignore and "RealState" to represent the user specified
states.  Also improved the text display of event names on the transitions.

Revision 1.13  1997/05/15 04:14:46  andrewm
Checkpoint.  Reworked the low level file format stuff to contain
proper lists rather than maps keyed to binary numbers.
This point represents the entire program working with this file format
change.

Revision 1.12  1997/04/24 03:20:49  andrewm
Checkpoint.  All features in.  Starting test cycle.

Revision 1.11  1997/03/18 06:51:04  andrewm
Checkpoint.  Mouse select, insert, and delete working.
Some changes to improve robustness in the face of an arbitrary input file.

Revision 1.10  1997/03/12 03:13:07  andrewm
Checkpoint.  Things are working rather well.

Revision 1.9  1997/03/04 06:32:56  andrewm
Another check point.  The editor can draw output from files.
The crashing during the dtor for MachineGroup is fixed.

Revision 1.8  1997/02/23 23:44:14  andrewm
Checkpoint.  Things seem to be working reasonably well.

Revision 1.7  1997/01/23 06:20:30  andrewm
Checkpoint as base and graphics classes are operating together.

Revision 1.6  1996/12/26 05:55:25  andrewm
Checkpoint, the compiler is working again.

Revision 1.5  1996/12/24 05:20:12  andrewm
Checkpoint.

Revision 1.4  1996/09/22 01:18:22  andrewm
pre-alpha release

 * Revision 1.3  1996/08/18  17:57:45  andrewm
 * checkpoint
 *
 * Revision 1.2  1996/07/15  01:20:40  andrewm
 * checkpoint again
 *
 * Revision 1.1  1996/06/26  03:14:43  andrewm
 * Initial revision
 *
 *--
 */
#ifndef _Transition_h_
#define _Transition_h_

/*
PRAGMAS
*/

/*
INCLUDE FILES
*/
#include "Chio.h"
#include "Observer.h"
#include "Geom2d.h"
#include "MachineVisitor.h"
#include "State.h"
#include <vector>

/*
MACRO DEFINITIONS
*/

/*
CLASS DEFINITIONS
*/

class MachEvent ;

class Transition :
	public Subject
{
	friend ostream& operator <<(ostream& stream, const Transition& transition) ;

public:
	Transition(State *source, MachEvent *event) ;
	Transition(State *source, ChioMapIter place) ;
	virtual ~Transition() ;

	State *source() const { return _source ; }
	void orphan() { _source = 0 ;}
	MachEvent *event() const { return _event ; }
	bool modified() const { return _modified ; }
	void modified(bool new_modified) { _modified = new_modified ; }
	virtual State *destination() const = 0 ;
	virtual void print_transition(ostream& s) const = 0 ;
	virtual bool is_pseudo_transition() = 0 ;
	virtual bool is_loop_transition() = 0 ;
	virtual void sync() = 0 ;
	virtual void selected(bool new_selected) ;
	virtual bool selected() const { return _selected ; }
	virtual void select_if_within(const Rectangle& box) ;
	virtual void insert_at_location(const Point& p) ;
	virtual bool delete_at_location(const Point& p) ;

	virtual void move(const Point& p) = 0 ;
	virtual void move_relative(const Point& p) = 0 ;
	virtual bool contains(const Point& p) = 0 ;
	virtual Rectangle bounding_box() = 0 ;

	void accept_visitor(MachineVisitor& visitor) { visitor.visit(this) ; }

public:
	static ChioTerm& determine_destination(ChioMap& trans_map) ;

protected:
	State *_source ;
	MachEvent *_event ;
	bool _modified ;
	bool _selected ;

protected:
	ChioMapIter find_place() ;

protected:
	static const char _destination_name[] ;
} ;

class PseudoTransition :
	public Transition
{
public:
	PseudoTransition(State *source, MachEvent *event, State *destination) ;
	PseudoTransition(State *source, ChioMapIter place) ;

	virtual State *destination() const { return _destination ; }
	virtual void print_transition(ostream& s) const ;
	virtual bool is_pseudo_transition() { return true ; }
	virtual bool is_loop_transition() { return false ; }
	virtual void sync() ;
	virtual void select_if_within(const Rectangle& box) {}

	virtual void move(const Point& p) {}
	virtual void move_relative(const Point& p) {}
	virtual bool contains(const Point& p) { return false ; }
	virtual Rectangle bounding_box() { return Rectangle() ; }

private:
	State *_destination ;
} ;


class LoopTransition :
	public Transition
{
public:
	LoopTransition(State *source, MachEvent *event,
		float attach_angle = 0.0) ;
	LoopTransition(State *source, ChioMapIter place) ;

	virtual State *destination() const { return _source ; }
	virtual void print_transition(ostream& s) const ;
	virtual bool is_pseudo_transition() { return false ; }
	virtual bool is_loop_transition() { return true ; }
	virtual void sync() ;

	virtual void move(const Point& p) ;
	virtual void move_relative(const Point& p) ;
	virtual bool contains(const Point& p) ;
	virtual Rectangle bounding_box() ;

	float attach_angle() const { return _attach_angle ; }
	void attach_angle(float new_angle) ;
	Circle compute_loop_circle() const ;

private:
	float _attach_angle ; // in radians

private:
	static const char _attach_name[] ;
} ;


class SegmentTransition :
	public Transition
{
public:
	typedef vector<Point> TransitionVertices ;
	typedef vector<Point>::iterator TransitionVerticesIter ;
	typedef vector<Point>::const_iterator TransitionVerticesConstIter ;

public:
	SegmentTransition(State *source, MachEvent *event, State *destination,
		const TransitionVertices& vertices = TransitionVertices()) ;
	SegmentTransition(State *source, ChioMapIter place) ;

	virtual State *destination() const { return _destination ; }
	virtual void print_transition(ostream& s) const ;
	virtual bool is_pseudo_transition() { return false ; }
	virtual bool is_loop_transition() { return false ; }
	virtual void sync() ;
	virtual void selected(bool new_selected) ;
	virtual bool selected() const { return Transition::selected() ; }
	virtual void insert_at_location(const Point& p) ;
	virtual bool delete_at_location(const Point& p) ;

	virtual void move(const Point& p) ;
	virtual void move_relative(const Point& p) ;
	virtual bool contains(const Point& p) ;
	virtual Rectangle bounding_box() ;

	TransitionVertices& vertices() { return _vertices ; }
	TransitionVertices compute_segments(
		float line_width,
		float arrow_length) const ;

private:
	State *_destination ;
	TransitionVertices _vertices ;
	TransitionVerticesIter _current_vertex ;

private:
	TransitionVerticesIter find_vertex_near(const Point& p) ;
	TransitionVerticesIter find_vertex_after(const Point& p) ;

private:
	static const char _path_name[] ;
} ;

#endif /* _Transition_h_ */
