/*
 *++
MODULE:

$RCSfile: HIOReader.cc,v $
$Revision: 1.6 $
$Date: 1996/10/01 04:39:31 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: HIOReader.cc,v $
Revision 1.6  1996/10/01 04:39:31  andrewm
checkpoint and revision

// Revision 1.5  1996/09/22  01:18:43  andrewm
// pre-alpha release
//
// Revision 1.4  1996/08/18  17:58:26  andrewm
// checkpoint
//
// Revision 1.3  1996/07/27  20:57:24  andrewm
// checkpoint
//
// Revision 1.2  1996/06/26  03:17:38  andrewm
// checkpoint
//
// Revision 1.1  1996/06/15  23:53:35  andrewm
// Initial revision
//
 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include <stdlib.h>
#include <iostream.h>
#include <errno.h>

#include "HIOReader.h"
#include "SMAbstractBuilder.h"
#include "DataArrayIter.h"

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: HIOReader.cc,v $ $Revision: 1.6 $" ;

/*
STATIC MEMBER DEFINITIONS
*/

/*
FUNCTION DEFINITIONS
*/

HIOReader::
~HIOReader(void)
{
	if (_file_list)
		hio_delete_assignment_list(&_file_list) ;
}

bool HIOReader::
read_file(
	const char *filename)
{
	if (filename == NULL || strlen(filename) == 0)
	{
		cerr << "HIOReader: NULL file name" << endl ;
		return false ;
	}
	FILE *infile = fopen(filename, "r") ;
	if (infile == NULL)
	{
		cerr << "Cannot open \"" << filename << "\" for reading: "
			<< strerror(errno) << endl ;
		return false ;
	}

	hio_source = (char *)filename ;
	int parse_errors = hio_read(infile, &_file_list) ;
	hio_source = NULL ;
	fclose(infile) ;
	if (parse_errors != 0)
	{
		cerr << "Unable to correctly parse input file \"" << filename
			<< "\"" << endl ;
		return false ;
	}

	return true ;
}

void HIOReader::
build(void)
{
	if (_file_list == NULL)
		return ;

	AssignmentList file_list(&_file_list) ;

	_builder->build_start() ;

	_builder->build_author(file_list["author"]) ;
	_builder->build_revision(file_list["revision"]) ;
	read_machines(file_list) ;

	_builder->build_end() ;
}

void HIOReader::
read_machines(
	AssignmentList list)
{
	Assignment sm_assign(list.find_assignment("state_machines")) ;
	if (sm_assign)
	{
		AssignmentArray *machines_array =
			sm_assign.valueList().list_assignments() ;
		IteratorPtr<Assignment> machine_iter(
			new DataArrayIter<Assignment>(*machines_array)) ;
		for (machine_iter->first() ; !machine_iter->done() ;
			machine_iter->next())
		{
			Assignment machine_assign(machine_iter->current()) ;
			AssignmentList machine_value(machine_assign.valueList()) ;

			_builder->build_machine(machine_assign.attributeString()) ;

			read_events(machine_value) ;
			read_states(machine_value) ;

			_builder->build_counter(machine_value["counter"]) ;

			Assignment t_assign = machine_value.find_assignment("prolog") ;
			if (t_assign)
			{
				Term term(t_assign.valueTerm()) ;
				_builder->build_prolog(term.string(), term.line()) ;
			}
			t_assign = machine_value.find_assignment("epilog") ;
			if (t_assign)
			{
				Term term(t_assign.valueTerm()) ;
				_builder->build_epilog(term.string(), term.line()) ;
			}
			t_assign = machine_value.find_assignment("error_action") ;
			if (t_assign)
			{
				Term term(t_assign.valueTerm()) ;
				_builder->build_error_action(term.string(), term.line()) ;
			}
			t_assign = machine_value.find_assignment("ignore_action") ;
			if (t_assign)
			{
				Term term(t_assign.valueTerm()) ;
				_builder->build_ignore_action(term.string(), term.line()) ;
			}

			_builder->build_initial_state(machine_value["initial_state"]) ;
			_builder->build_terminal_state(machine_value["terminal_state"]) ;
			_builder->build_default_state(machine_value["default_state"]) ;
		}
		delete machines_array ;
	}
	else
	{
		cerr << "State Machine file contains no \"state_machines\" section"
			<< endl ;
	}
}

void HIOReader::
read_events(
	AssignmentList list)
{
		/*
		In the HIO file a single event looks like:
		"event name" = ["int b", "char *c",  ...  ],

		So we have a natural assignment which has a value that is a list of
		ordered assignments each one of which contains a value that is
		the parameter declaration.
		*/
	Assignment events(list.find_assignment("events")) ;
	if (events)
	{
		AssignmentArray *events_array = events.valueList().list_assignments() ;
		IteratorPtr<Assignment> event_iter(
			new DataArrayIter<Assignment>(*events_array)) ;
		for (event_iter->first() ; !event_iter->done() ; event_iter->next())
		{
			Assignment event_assign(event_iter->current()) ;
			_builder->build_event(event_assign.attributeString()) ;
			read_event_parameters(event_assign.valueList()) ;
		}
		delete events_array ;
	}
	else
	{
		cerr << "State Machine file contains no \"events\" section" << endl ;
	}
}

void HIOReader::
read_event_parameters(
	AssignmentList list)
{
	AssignmentArray *params_array = list.list_assignments() ;
	IteratorPtr<Assignment> param_iter(
		new DataArrayIter<Assignment>(*params_array)) ;
	for (param_iter->first() ; !param_iter->done() ; param_iter->next())
	{
		Assignment parameter(param_iter->current()) ;
		AssignmentList param_value(parameter.valueList()) ;
		_builder->build_event_parameter(param_value["type"],
			param_value["name"]) ;
	}
	delete params_array ;
}

void HIOReader::
read_states(
	AssignmentList list)
{
	Assignment states(list.find_assignment("states")) ;
	if (states)
	{
			/*
			First build the states without any transition information.
			*/
		AssignmentArray *states_array =
			states.valueList().list_assignments() ;
		IteratorPtr<Assignment> state_iter(
			new DataArrayIter<Assignment>(*states_array)) ;
		for (state_iter->first() ; !state_iter->done() ; state_iter->next())
		{
			Assignment state_assign(state_iter->current()) ;
			AssignmentList state_value(state_assign.valueList()) ;

			_builder->build_state(state_assign.attributeString()) ;
			Assignment t_assign = state_value.find_assignment("code") ;
			if (t_assign)
			{
				Term term(t_assign.valueTerm()) ;
				_builder->build_code(term.string(), term.line()) ;
			}
			read_state_parameters(state_value) ;
			read_state_graphic_center(state_value) ;
			_builder->build_state_graphic_radius(state_value["radius"]) ;
		}
			/*
			Then read in the transitions.
			*/
		for (state_iter->first() ; !state_iter->done() ; state_iter->next())
		{
			Assignment state_assign(state_iter->current()) ;
				/*
				Rebuilding the state is important as it serves to reset
				the current state pointer in the builder so that the
				transitions get added to the correct state.
				*/
			_builder->build_state(state_assign.attributeString()) ;
			read_transitions(state_assign.valueList()) ;
		}
		delete states_array ;
	}
	else
	{
		cerr << "State Machine file contains no \"states\" section" << endl ;
	}
}

void HIOReader::
read_state_parameters(
	AssignmentList list)
{
	Assignment parameters(list.find_assignment("parameters")) ;
	if (parameters)
	{
		AssignmentArray *params_array =
			parameters.valueList().list_assignments() ;
		IteratorPtr<Assignment> param_iter(
			new DataArrayIter<Assignment>(*params_array)) ;
		for (param_iter->first() ; !param_iter->done() ; param_iter->next())
		{
			Assignment parameter(param_iter->current()) ;
			AssignmentList param_value(parameter.valueList()) ;
			_builder->build_state_parameter(param_value["type"],
				param_value["name"]) ;
		}
		delete params_array ;
	}
}

void HIOReader::
read_state_graphic_center(
	AssignmentList list)
{
	Assignment center(list.find_assignment("center")) ;
	if (center)
	{
		_builder->build_state_graphic_center(center[0], center[1]) ;
	}
}

void HIOReader::
read_transitions(
	AssignmentList list)
{
	Assignment trans(list.find_assignment("transitions")) ;
	if (trans)
	{
		AssignmentArray *trans_array = trans.valueList().list_assignments() ;
		IteratorPtr<Assignment> trans_iter(
			new DataArrayIter<Assignment>(*trans_array)) ;
		for (trans_iter->first() ; !trans_iter->done() ; trans_iter->next())
		{
			Assignment trans_assign(trans_iter->current()) ;
			AssignmentList trans_value(trans_assign.valueList()) ;

			const char *event = trans_assign.attributeString() ;
			Assignment destination(trans_value.find_assignment("destination")) ;
			if (destination)
			{
				_builder->build_transition(event, destination.valueString()) ;

				_builder->build_loop_graphic(trans_value["attach"]) ;
				read_segment_graphics(trans_value) ;
			}
			else
			{
				cerr << "Transition contains no \"destination\" assignment"
					<< endl ;
			}
		}
		delete trans_array ;
	}
}

void HIOReader::
read_segment_graphics(
	AssignmentList list)
{
	Assignment path(list.find_assignment("path")) ;
	if (path)
	{
		AssignmentArray *vertices_array = path.valueList().list_assignments() ;
		IteratorPtr<Assignment> vertices_iter(
			new DataArrayIter<Assignment>(*vertices_array)) ;
		for (vertices_iter->first() ; !vertices_iter->done() ;
			vertices_iter->next())
		{
			Assignment path_component(vertices_iter->current()) ;
			_builder->build_segment_vertex(path_component[0],
				path_component[1]) ;
		}
		delete vertices_array ;
	}
}
