/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:
	Geom2d.h -- header file for 2 dimensional geometry classes

$RCSfile: Geom2d.h,v $
$Revision: 1.15 $
$Date: 1997/07/02 04:45:15 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: Geom2d.h,v $
Revision 1.15  1997/07/02 04:45:15  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.14  1997/06/21 02:21:34  andrewm
Checkpoint.  PostScript generator going well. A lot of small tweeks
all over to accomplish this.

Revision 1.13  1997/06/12 03:20:26  andrewm
Checkpoint.  Crude version of the PostScript output running.  Need
to change the way text is manipulated in order to make PostScript generation
better.

Revision 1.12  1997/03/18 06:51:04  andrewm
Checkpoint.  Mouse select, insert, and delete working.
Some changes to improve robustness in the face of an arbitrary input file.

Revision 1.11  1997/03/12 03:13:08  andrewm
Checkpoint.  Things are working rather well.

Revision 1.10  1997/02/23 23:44:15  andrewm
Checkpoint.  Things seem to be working reasonably well.

Revision 1.9  1997/02/08 04:37:41  andrewm
Checkpoint before returning to work on the GUI portion.

Revision 1.8  1997/01/23 06:19:49  andrewm
Checkpoint as base and graphics classes stabilize.
Observer/subject pattern operating between base and graphic classes.

Revision 1.7  1996/11/27 01:25:53  andrewm
Another checkpoint before I go off and figure out what to do about
the graphical representations so that I'll be able to get some post script
output one of these days.

Revision 1.6  1996/11/17 21:50:52  andrewm
Converting to the new version of g++ (2.7.2) and now everything
is under CVS.

 * Revision 1.5  1996/09/22  01:22:58  andrewm
 * *** empty log message ***
 *
 * Revision 1.5  1996/09/22  01:22:58  andrewm
 * *** empty log message ***
 *
 * Revision 1.4  1996/08/18  17:57:31  andrewm
 * checkpoint
 *
 * Revision 1.4  1996/08/18  17:57:31  andrewm
 * checkpoint
 *
 * Revision 1.3  1996/07/27  20:56:24  andrewm
 * checkpoint
 *
 * Revision 1.2  1996/07/15  01:20:12  andrewm
 * checkpoint again
 *
 * Revision 1.1  1996/06/26  03:12:53  andrewm
 * Initial revision
 *
 * Revision 1.1  1996/05/26  22:27:43  andrewm
 * Initial revision
 *
 * Revision 1.5  1996/04/01  03:25:19  andrewm
 * Checkpoint after extensive rework of the graphics aspects.
 *
 * Revision 1.4  1996/02/27  04:52:23  andrewm
 * Checkpoint before going in and straightening out some of the
 * class architecture.
 *
 * Revision 1.3  1996/02/19  19:41:00  andrewm
 * Checkpoint as things seem to be working well.
 *
 * Revision 1.2  1996/01/28  20:22:10  andrewm
 * Checkpoint after straightening out the templates.
 *
 * Revision 1.1  1996/01/23  06:22:17  andrewm
 * Initial revision
 *
 *--
 */
#ifndef _Geom2d_h_
#define _Geom2d_h_

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma interface
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include <iostream.h>
#include <math.h>
#include <minmax.h>
#include <builtin.h>

/*
MACRO DEFINITIONS
*/

/*
CLASS DEFINITIONS
*/

enum line_test
{
	not_on_line = 0,
	on_ray_P,
	in_segment_PQ,
	on_ray_Q
} ;

template<class T> class Point2D ;
template<class T> class Line2D ;
template<class T> class Circle2D ;

template<class T>
class Point2D
{
public:
	Point2D(T x_coord = 0 , T y_coord = 0) ;

	T& x() { return _x ; }
	const T& x() const { return _x ; }
	T& y() { return _y ; }
	const T& y() const { return _y ; }

	Point2D<T>& operator +=(const Point2D<T>& p) ;
	Point2D<T>& operator -=(const Point2D<T>& p) ;
	Point2D<T>& operator *=(T scalar) ;
	Point2D<T>& operator /=(T scalar) ;
	bool operator ==(const Point2D<T>& p) const ;
	bool operator !=(const Point2D<T>& p) const ;

	friend Point2D<T> operator +(const Point2D<T>& p1, const Point2D<T>& p2) ;
	friend Point2D<T> operator -(const Point2D<T>& p1, const Point2D<T>& p2) ;
	friend T operator *(const Point2D<T>& p1, const Point2D<T>& p2) ;
	friend Point2D<T> operator *(const Point2D<T>& p, T scalar) ;
	friend Point2D<T> operator *(T scalar, const Point2D<T>& p) ;
	friend Point2D<T> operator /(const Point2D<T>& p, T scalar) ;

	T length() const ;
	T distance(const Point2D<T>& p) const ;
	Point2D<T>& normalize() ;
	Point2D<T> perpendicular() const ;
	Point2D<T> reflect() const ;
	Point2D<T>& x_complement(T range) ;
	Point2D<T>& y_complement(T range) ;

	friend line_test point_on_line(const Point2D<T>& p, const Point2D<T>& q,
		const Point2D<T>& t) ;

	friend ostream& operator <<(ostream& s, const Point2D<T>& p) ;

private:
	T _x ;
	T _y ;
} ;

typedef Point2D<float> Point ;
typedef Point2D<int> IPoint ;

template<class T> Point2D<T>::
Point2D(
	T x_coord,
	T y_coord) :
		_x(x_coord), _y(y_coord)
{
}

template<class T> Point2D<T>& Point2D<T>::
operator +=(
	const Point2D<T>& p)
{
	_x += p._x ;
	_y += p._y ;
	return *this ;
}

template<class T> Point2D<T>& Point2D<T>::
operator -=(
	const Point2D<T>& p)
{
	_x -= p._x ;
	_y -= p._y ;
	return *this ;
}

template<class T> Point2D<T>& Point2D<T>::
operator *=(
	T scalar)
{
	_x *= scalar ;
	_y *= scalar ;
	return *this ;
}

template<class T> Point2D<T>& Point2D<T>::
operator /=(
	T scalar)
{
	if (scalar != 0)
	{
		_x /= scalar ;
		_y /= scalar ;
	}
	else
	{
		_x = T(HUGE) ;
		_y = T(HUGE) ;
	}
	return *this ;
}

template<class T> bool Point2D<T>::
operator ==(
	const Point2D<T>& p) const
{
	return (_x == p._x && _y == p._y) ;
}

template<class T> bool Point2D<T>::
operator !=(
	const Point2D<T>& p) const
{
	return (_x != p._x || _y != p._y) ;
}

template<class T> Point2D<T>
operator +(
	const Point2D<T>& p1,
	const Point2D<T>& p2)
{
	return Point2D<T>(p1._x + p2._x, p1._y + p2._y) ;
}

template<class T> Point2D<T>
operator -(
	const Point2D<T>& p1,
	const Point2D<T>& p2)
{
	return Point2D<T>(p1._x - p2._x, p1._y - p2._y) ;
}

template<class T> T
operator *(
	const Point2D<T>& p1,
	const Point2D<T>& p2)
{
	return (p1._x * p2._x + p1._y * p2._y) ;
}

template<class T> Point2D<T>
operator *(
	const Point2D<T>& p,
	T scalar)
{
	return Point2D<T>(p._x * scalar, p._y * scalar) ;
}

template<class T> Point2D<T>
operator *(
	T scalar,
	const Point2D<T>& p)
{
	return Point2D<T>(p._x * scalar, p._y * scalar) ;
}

template<class T> Point2D<T>
operator /(
	const Point2D<T>& p,
	T scalar)
{
	return Point2D<T>(p._x / scalar, p._y / scalar) ;
}

template<class T> T Point2D<T>::
length() const
{
	return (T)sqrt(*this * *this) ;
}

template<class T> T Point2D<T>::
distance(
	const Point2D<T>& p) const
{
	Point2D<T> d(*this - p) ;
	return d.length() ;
}

template<class T> Point2D<T>& Point2D<T>::
normalize()
{
	*this /= this->length() ;
	return *this ;
}

template<class T> Point2D<T> Point2D<T>::
perpendicular() const
{
	return Point2D<T>(-_y, _x) ;
}

template<class T> Point2D<T> Point2D<T>::
reflect() const
{
	return Point2D<T>(-_x, -_y) ;
}

template<class T> Point2D<T>& Point2D<T>::
x_complement(
	T range)
{
	_x = range - _x ;
	return *this ;
}

template<class T> Point2D<T>& Point2D<T>::
y_complement(
	T range)
{
	_y = range - _y ;
	return *this ;
}

template<class T> line_test
point_on_line(
	const Point2D<T>& p,
	const Point2D<T>& q,
	const Point2D<T>& t)
{
	const T margin = 500 ;

	if (abs((q._y - p._y) * (t._x - p._x) - (t._y - p._y) * (q._x - p._x))
		>= max(abs(q._x - p._x), abs(q._y - p._y)) + margin)
	{
		return not_on_line ;
	}
	if ((q._x < p._x && p._x < t._x) || (q._y < p._y && p._y < t._y))
	{
		return on_ray_P ;
	}
	if ((t._x < p._x && p._x < q._x) || (t._y < p._y && p._y < q._y))
	{
		return on_ray_P ;
	}
	if ((p._x < q._x && q._x < t._x) || (p._y < q._y && q._y < t._y))
	{
		return on_ray_Q ;
	}
	if ((t._x < q._x && q._x < p._x) || (t._y < q._y && q._y < p._y))
	{
		return on_ray_Q ;
	}
	return in_segment_PQ ;
}

template<class T> ostream&
operator <<(
	ostream& s,
	const Point2D<T>& p)
{
	return s << '(' << p.x() << ", " << p.y() << ')' ;
}

template<class T> Point2D<T> Line2D<T>::
point_on_line_nearest_origin() const
{
	return point_on_line_nearest(Point2D<T>()) ;
}

template<class T>
class Line2D
{
public:
	Line2D(const Point2D<T>& n, T c) ;
	Line2D(const Point2D<T>& n, T c, const Point2D<T>& u, const Point2D<T>& v) ;
	Line2D(const Point2D<T>& a, const Point2D<T>& b) ;

	Line2D<T>& to_explicit() ;
	Line2D<T>& to_implicit() ;
	Line2D<T>& normalize() ;
	bool normalized() const { return _normalized ; }
	Point2D<T> point_along(T scalar) ;
	T distance_from_point(const Point2D<T>& p) const ;
	Point2D<T> point_on_line_nearest(const Point2D<T>& p) const ;
	Point2D<T> point_on_line_nearest_origin() const ;
	Line2D<T> perpendicular_through_point(const Point2D<T>& p) const ;

	friend bool circle_line_intersect(const Circle2D<T>& c, const Line2D& l,
		Point2D<T> *p1, Point2D<T> *p2) ;
	friend bool intersection(
		const Line2D& l1, const Line2D& l2, Point2D<T> *p) ;
	friend T cosine_between(const Line2D& l, const Line2D& m) ;

private:
	Point2D<T> _n ; // implicit representation
	T _c ;
	Point2D<T> _u ; // explicit representation
	Point2D<T> _v ;
	bool _normalized ;
} ;

typedef Line2D<float> Line ;

template<class T> Line2D<T>::
Line2D(
	const Point2D<T>& n,
	T c) :
		_n(n),
		_c(c),
		_normalized(false)
{
	to_explicit() ;
}

template<class T> Line2D<T>::
Line2D(
	const Point2D<T>& n,
	T c,
	const Point2D<T>& u,
	const Point2D<T>& v) :
		_n(n),
		_c(c),
		_u(u),
		_v(v),
		_normalized(false)
{
}

template<class T> Line2D<T>::
Line2D(
	const Point2D<T>& a,
	const Point2D<T>& b)
{
	_u = a ;
	_v = b - a ;
	_v.normalize() ;
	to_implicit() ;
	_normalized = false ;
}

template<class T> Line2D<T>& Line2D<T>::
to_implicit()
{
	_n = _v.perpendicular() ;
	_c = _n * _u ;
	return *this ;
}

template<class T> Line2D<T>& Line2D<T>::
to_explicit()
{
	_u = point_on_line_nearest(Point2D<T>()) ;
	_v = _n.reflect().perpendicular() ;
	return *this ;
}

template<class T> Line2D<T>& Line2D<T>::
normalize()
{
	_c /= _n.length() ;
	_n.normalize() ;
	_v.normalize() ;
	_normalized = true ;
}

template<class T> Point2D<T> Line2D<T>::
point_along(
	T scalar)
{
	if (!_normalized)
		normalize() ;
	return (_u + scalar * _v) ;
}

template<class T> T Line2D<T>::
distance_from_point(
	const Point2D<T>& p) const
{
	Point2D<T> q(this->point_on_line_nearest(p)) ;
	return q.distance(p) ;
}

template<class T> Point2D<T> Line2D<T>::
point_on_line_nearest(
	const Point2D<T>& p) const
{
	T q = _c + (_n * p) ;
	if (!_normalized)
		q /= _n.length() ;

	return Point2D<T>(p - q * _n) ;
}

template<class T> Line2D<T> Line2D<T>::
perpendicular_through_point(
	const Point2D<T>& p) const
{
	Point2D<T> ppn(_n.perpendicular()) ;
	return Line2D(ppn, (-1 * ppn) * p, p, _n) ;
}

template<class T>
class Circle2D
{
public:
	Circle2D() : _center(Point2D<T>(0, 0)), _radius(0) {}
	Circle2D(const Point2D<T>& c, T r) : _center(c), _radius(r) {}

	Point2D<T>& center() { return _center ; }
	const Point2D<T>& center() const { return _center ; }
	T& radius() { return _radius ; }
	const T& radius() const { return _radius ; }

	Point2D<T> on_circle_nearest(const Point2D<T>& point) const ;
	Point2D<T> out_from_circle(const Point2D<T>& point, T delta) const ;
	Point2D<T> on_circle_at_angle(T radians) const ;
	Line2D<T> tangent_line_at_point(Point2D<T>& point) ;
	bool is_point_within(const Point2D<T>& point) const ;

	friend bool circle_line_intersect(const Circle2D<T>& c, const Line2D<T>& l,
		Point2D<T> *p1, Point2D<T> *p2) ;

	friend ostream& operator <<(ostream& s, const Circle2D<T>& c) ;

private:
	Point2D<T> _center ;
	T _radius ;
} ;

typedef Circle2D<float> Circle ;

template<class T> Point2D<T> Circle2D<T>::
on_circle_nearest(
	const Point2D<T>& point) const
{
	Point2D<T> q(point - _center) ;
	q.normalize() ;
	return Point2D<T>(_center + _radius * q) ;
}

template<class T> Point2D<T> Circle2D<T>::
out_from_circle(
	const Point2D<T>& point,
	T delta) const
{
	Point2D<T> q(point - _center) ;
	q.normalize() ;
	return Point2D<T>(_center + (_radius + delta) * q) ;
}

template<class T> Point2D<T> Circle2D<T>::
on_circle_at_angle(
	T radians) const
{
	T x(_radius * cos(radians)) ;
	T y(_radius * sin(radians)) ;
		// In X Window land, the Y axis runs opposite the usual conventions.
	return Point(_center.x() + x, _center.y() - y) ;
}

template<class T> Line2D<T> Circle2D<T>::
tangent_line_at_point(
	Point2D<T>& point)
{
	Point2D<T> l_n(point - _center) ;
	T l_c(-(l_n * point)) ;
	Line2D<T> tangent(l_n, l_c, point, l_n.perpendicular().reflect()) ;
	tangent.normalize() ;
	return tangent ;
}

template<class T> bool Circle2D<T>::
is_point_within(
	const Point2D<T>& point) const
{
	return (point.distance(_center) <= _radius) ;
}

template<class T> bool
circle_line_intersect(
	const Circle2D<T>& circle,
	const Line2D<T>& line,
	Point2D<T> *p1,
	Point2D<T> *p2)
{
	Point2D<T> g(line._explicit.u - circle._center) ;
	T a(line._explicit.v * line._explicit.v) ;
	T b(line._explicit.v * g) ;
	b *= 2 ;
	T c(g * g - circle._radius * circle._radius) ;
	T d(b * b - 4 * a * c) ;

	if (d < 0)
		return false ;

	T t1((-b + sqrt(d)) / (2 * a)) ;
	T t2((-b - sqrt(d)) / (2 * a)) ;
	*p1 = Point2D<T>(line._explicit.u + t1 * line._explicit.v) ;
	*p2 = Point2D<T>(line._explicit.u + t2 * line._explicit.v) ;

	return true ;
}

template<class T> ostream&
operator <<(
	ostream& s,
	const Circle2D<T>& c)
{
	return s << "Circle(" << c._center << ", " << c._radius << ')' ;
}

template<class T>
class Rectangle2D
{
public:
	Rectangle2D() : _origin(Point2D<T>(0, 0)), _width(0), _height(0) {}
	Rectangle2D(const Point2D<T>& origin, T width, T height) :
		_origin(origin), _width(width), _height(height) {}

	Point2D<T>& origin() { return _origin ; }
	const Point2D<T>& origin() const { return _origin ; }
	T& width() { return _width ; }
	const T& width() const { return _width ; }
	T& height() { return _height ; }
	const T& height() const { return _height ; }

	Point2D<T> northwest() const { return _origin ; }
	Point2D<T> northeast() const { return _origin + Point2D<T>(_width, 0) ; }
	Point2D<T> southwest() const { return _origin + Point2D<T>(0, _height) ; }
	Point2D<T> southeast() const {
		return _origin + Point2D<T>(_width, _height) ; }

	Rectangle2D<T>& enclose(const Rectangle2D<T>& rect) ;
	bool contains(const Rectangle2D<T>& rect) const ;

	friend ostream& operator <<(ostream& s, const Rectangle2D<T>& p) ;

private:
	Point2D<T> _origin ;
	T _width ;
	T _height ;
} ;

typedef Rectangle2D<float> Rectangle ;
typedef Rectangle2D<int> IRectangle ;

template<class T>
Rectangle2D<T>& Rectangle2D<T>::
enclose(
	const Rectangle2D<T>& rect)
{
		// Only include rectangles that have dimension in
		// both directions
	if (rect._width > 0 && rect._height > 0)
	{
		T min_x = _origin.x() ;
		T min_y = _origin.y() ;
		T max_x = min_x + _width ;
		T max_y = min_y + _height ;

		Point2D<T> rect_origin(rect.origin()) ;
		T min_rect_x = rect_origin.x() ;
		T min_rect_y = rect_origin.y() ;
		T max_rect_x = min_rect_x + rect._width ;
		T max_rect_y = min_rect_y + rect._height ;

		min_x = min(min_x, min_rect_x) ;
		max_x = max(max_x, max_rect_x) ;
		min_y = min(min_y, min_rect_y) ;
		max_y = max(max_y, max_rect_y) ;

		_origin = Point2D<T>(min_x, min_y) ;
		_width = max_x - min_x ;
		_height = max_y - min_y ;
	}
	return *this ;
}

template<class T>
bool Rectangle2D<T>::
contains(
	const Rectangle2D<T>& rect) const
{
	T min_x = _origin.x() ;
	T min_y = _origin.y() ;
	T max_x = min_x + _width ;
	T max_y = min_y + _height ;

	Point2D<T> rect_origin(rect.origin()) ;
	T min_rect_x = rect_origin.x() ;
	T min_rect_y = rect_origin.y() ;
	T max_rect_x = min_rect_x + rect._width ;
	T max_rect_y = min_rect_y + rect._height ;

	return min_x <= min_rect_x && max_x >= max_rect_x
		&& min_y <= min_rect_y && max_y >= max_rect_y ;
}

template<class T> ostream&
operator <<(
	ostream& s,
	const Rectangle2D<T>& p)
{
	return s << "Rectangle(" << p._origin << ", " <<
		p._width << ", " << p._height << ')' ;
}


template<class T>
bool
intersection(
	const Line2D<T>& l,
	const Line2D<T>& m,
	Point2D<T> *p)
{
	T d(l._implicit.n * m._explicit.v) ;
	if (d == 0)
		return false ;

	T t((l._implicit.n * m._explicit.u + l._implicit.c) / d) ;
	*p = Point2D<T>(m._explicit.u - t * m._explicit.v) ;
	return true ;
}

template<class T>
T
cosine_between(
	const Line2D<T>& l,
	const Line2D<T>& m)
{
	T d(l._v * m._v) ;

	if (!(l.normalized() && m.normalized()))
		d /= l._v.length() * m._v.length() ;

	return d ;
}

template<class T>
Rectangle2D<T>
enclose_circle(
	const Circle2D<T>& circle,
	T space)
{
	Point2D<T> center(circle.center()) ;
	T radius = circle.radius() ;
	Point2D<T> corner(
		center.x() - radius - space,
		center.y() - radius - space) ;
	T width = 2 * (radius + space) ;
	return Rectangle2D<T>(corner, width, width) ;
}



#endif /* _Geom2d_h_ */
