/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: StateClassDeclVisitor.cc,v $
$Revision: 1.7 $
$Date: 1997/07/02 04:45:24 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: StateClassDeclVisitor.cc,v $
Revision 1.7  1997/07/02 04:45:24  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.6  1997/05/20 05:15:37  andrewm
Checkpoint.  Improved the structure of the "State" class and this
had quite some ripple effects.  However, now there is an abstract
class "State" with two concrete classes "PseudoState" to represent
error and ignore and "RealState" to represent the user specified
states.  Also improved the text display of event names on the transitions.

Revision 1.5  1997/05/15 04:14:48  andrewm
Checkpoint.  Reworked the low level file format stuff to contain
proper lists rather than maps keyed to binary numbers.
This point represents the entire program working with this file format
change.

Revision 1.4  1997/01/12 02:17:47  andrewm
Changed the compiler so that state machine is defined as nested
classes in order to prevent polluting the global class name space.

Revision 1.3  1996/12/27 02:11:02  andrewm
Checkpoint before attempting to restructure the compiler to generate
nested class declarations.

Revision 1.2  1996/12/26 05:55:29  andrewm
Checkpoint, the compiler is working again.

Revision 1.1  1996/09/22 01:18:50  andrewm
pre-alpha release

 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "StateClassDeclVisitor.h"
#include "MachineGroup.h"
#include "Smachine.h"
#include "MachEvent.h"
#include "State.h"
#include "Parameter.h"

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: StateClassDeclVisitor.cc,v $ $Revision: 1.7 $" ;

/*
STATIC MEMBER DEFINITIONS
*/

/*
FUNCTION DEFINITIONS
*/

StateClassDeclVisitor::
StateClassDeclVisitor(
	ostream& stream) :
		_headerfile(stream)
{
}

void StateClassDeclVisitor::
visit(
	Smachine *machine)
{
	_prefix_name = machine->name().var_name() ;
	_event_class = string(_prefix_name + "_Event") ;
	string state_class(_prefix_name + "_State") ;
	State *default_state = machine->default_state() ;
	if (default_state == NULL)
		_default_action = "error_action(machine, event) ;" ;
	else
	{
		_default_action = default_state->name().var_name() ;
		_default_action += "_action(machine, event) ;" ;
	}
	State *initial_state = machine->initial_state() ;

	_headerfile << "\n\t// Abstract State declaration for " <<
		_prefix_name << endl ;
	_headerfile <<
		"\tclass " << state_class << endl <<
		"\t{" <<
			"\n\t\tfriend ostream& operator <<(ostream& s, " <<
				state_class << "& state) ;" << endl ;
	if (initial_state)
	{
		_headerfile <<
			"\n\tpublic:\n"
				"\t\tstatic " << state_class << "& initial_state() "
					"{ return _" << initial_state->name().var_name() << " ; }"
					<< endl ;
	}
	_headerfile <<
		"\n\tpublic:\n"
			"\t\tvirtual const char *name() const = 0 ;\n" ;

	if (initial_state)
	{
		_headerfile <<
			"\t\tvirtual void initial_event(" <<
				_prefix_name << "& machine, " <<
				"const " << _event_class << "& event" ;

		const State::ParameterList& param_list =
			initial_state->parameter_list() ;
		for (State::ParameterListConstIter param_iter = param_list.begin() ;
			param_iter != param_list.end() ; param_iter++)
		{
			Parameter *param = *param_iter ;
			param->accept_visitor(*this) ;
		}

		_headerfile << ')' << endl <<
			"\t\t{ error_action(machine, event) ; }" << endl ;
	}

	Smachine::MachEventList& event_list = machine->event_list() ;
	for (Smachine::MachEventListIter event_iter = event_list.begin() ;
		event_iter != event_list.end() ; event_iter++)
	{
		MachEvent *ev = *event_iter ;
		ev->accept_visitor(*this) ;
	}

	_headerfile <<
		"\n\tprotected:\n"
			"\t\tvoid error_action(" << _prefix_name << "& machine, " <<
				"const " << _event_class << "& event) ;\n"
			"\t\tvoid ignore_action(" << _prefix_name << "& machine, " <<
				"const " << _event_class << "& event) ;\n"
			"\t\tvoid set_state(" << _prefix_name << "& machine, " <<
				state_class << "* state) ;\n"
			"\t\tvoid log(" << _prefix_name << "& machine, " <<
				"const " << _event_class << "& event, " <<
				state_class << "& new_state) ;\n"
		"\n\tprotected:" << endl ;

	Smachine::StateList& state_list = machine->state_list() ;
	for (Smachine::StateListIter st_iter = state_list.begin() ;
		st_iter != state_list.end() ; st_iter++)
	{
		State *st = *st_iter ;
		st->accept_visitor(*this) ;
	}

	_headerfile << "\t} ;" << endl ;
}

void StateClassDeclVisitor::
visit(
	MachEvent *event)
{
	string event_name = event->name().var_name() ;

	_headerfile <<
		"\t\tvirtual void " << event_name << '(' <<
			_prefix_name << "& machine, " <<
			"const " << _event_class << "& event" ;

	MachEvent::ParameterList& param_list = event->parameter_list() ;
	for (MachEvent::ParameterListIter param_iter = param_list.begin() ;
		param_iter != param_list.end() ; param_iter++)
	{
		Parameter *param = *param_iter ;
		param->accept_visitor(*this) ;
	}

	_headerfile << ')' << endl <<
		"\t\t\t{ " << _default_action << " }" << endl ;
}

void StateClassDeclVisitor::
visit(
	State *state)
{
	string state_name = state->name().var_name() ;
	string state_class_name = _prefix_name + "_" + state_name + "_State" ;

	_headerfile <<
		"\t\tstatic " << state_class_name << " _" << state_name << " ;"
		<< endl ;
}

void StateClassDeclVisitor::
visit(
	Parameter *parameter)
{
	_headerfile << ", " << (const string&)parameter->type() << ' '
		<< (const string&)parameter->name() ;
}
