/*------------------------------------------------------------------*\
 
 mca_calc.c: calculates values & hence colours for a non-linear CA. 
 Copyright (C) 02000 Luke Schubert

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 

 The author may be contacted at luke@dspace.com.au

 file:     mandelca.c
 function: Implements a non-linear CA.
 Author:   Luke Schubert 01/02/00
 modified to:
 file:     mca_calc.c
 function: Calculates values & hence colours for a non-linear CA.
 Author:   Luke Schubert 13/02/00
           $Id: mca_calc.c,v 1.8 2000/03/28 09:51:17 luke Exp $

\*------------------------------------------------------------------*/

/*------------------------------------------------------------------*\

                              INCLUDES

\*------------------------------------------------------------------*/

#include <stdio.h>
#include <math.h>

#include "mca_calc.h"

/*------------------------------------------------------------------*\

                              FUNCTIONS

\*------------------------------------------------------------------*/

/*------------------------------------------------------------------*\

 function: mca_calc
 Author:   Luke Schubert
 Date:     01/02/02000
 
 Calculates new values for colours of Mandelbrot Cellular Automaton

\*------------------------------------------------------------------*/

colour_struct mca_calc(int no_gens)
{
  static double old_cell_x[CA_WIDTH][CA_HEIGHT];
  static double old_cell_y[CA_WIDTH][CA_HEIGHT];
  static double orig_cell_x[CA_WIDTH][CA_HEIGHT];
  static double orig_cell_y[CA_WIDTH][CA_HEIGHT];
  double new_cell_x[CA_WIDTH][CA_HEIGHT];
  double new_cell_y[CA_WIDTH][CA_HEIGHT];
  int i,j, di, dj;
  colour_struct ca_struct;
#if MAND_LIFE
  int zero_nb, hi_nb;
  float avg_x, avg_y, hi_avg_x, hi_avg_y;
#endif
#if USE_POWER
  float old_mag, old_ang;
#endif

  if (no_gens == 0) {
    /* init_ca */
    for (i = 0; i < CA_WIDTH; i++) {
      for (j = 0; j < CA_HEIGHT; j++) { 
	old_cell_x[i][j] = (float)LEFT_X + (float)i*CPLX_WD/(float)CA_WIDTH;
	old_cell_y[i][j] = (float)TOP_Y  + (float)j*CPLX_HT/(float)CA_HEIGHT;
	orig_cell_x[i][j] = old_cell_x[i][j];
	orig_cell_y[i][j] = old_cell_y[i][j];
      }
    }
  }
  ca_struct.no_gens = no_gens;

  for (i = 0; i < CA_WIDTH; i++) {
    for (j = 0; j < CA_HEIGHT; j++) { 
      new_cell_x[i][j] = (old_cell_x[i][j] * old_cell_x[i][j] 
			  - old_cell_y[i][j] * old_cell_y[i][j]);
      new_cell_y[i][j] = 2.0*(old_cell_x[i][j] * old_cell_y[i][j]);

#if USE_POWER
      old_mag = sqrt((old_cell_x[i][j] * old_cell_x[i][j]) 
	+ (old_cell_y[i][j] * old_cell_y[i][j]));
      old_ang = atan2(old_cell_y[i][j], old_cell_x[i][j]);
      new_cell_x[i][j] = pow(old_mag, POWER) 
	* cos(old_ang * POWER);
      new_cell_y[i][j] = pow(old_mag, POWER)
	* sin(old_ang * POWER);
#endif

#if AVG_NBHD
      /* average values of neighbours and add */
      for (di = -1; di <= 1; di++) {
	for (dj = -1; dj <= 1; dj++) {
	  if ((di+i >= 0) && (di+i < CA_WIDTH)
	      && (dj + j >= 0) && (dj + j < CA_HEIGHT)) {
	     	    new_cell_x[i][j] += old_cell_x[i][j]/(9*NBHD_FACTOR);
		    new_cell_y[i][j] += old_cell_y[i][j]/(9*NBHD_FACTOR); 
	  }
	}
      } 
#endif

      /* new_cell_x[i][j] += old_cell_x[i][j]; */

      new_cell_x[i][j] += orig_cell_x[i][j];
      new_cell_y[i][j] += orig_cell_y[i][j];
#if 0
      new_cell_x[i][j] /= 2.0;
      new_cell_y[i][j] /= 2.0;
#endif

    }
  }
  
  for (i = 0; i < CA_WIDTH; i++) {
    for (j = 0; j < CA_HEIGHT; j++) {
      old_cell_x[i][j] = new_cell_x[i][j];
      old_cell_y[i][j] = new_cell_y[i][j];
    }
  }

  /* assign colours & return */
  for (i = 0; i < CA_WIDTH; i++) {
    for (j = 0; j < CA_HEIGHT; j++) {
      ca_struct.colour_array[i][j] = (int)((old_cell_x[i][j] 
					    * old_cell_x[i][j] 
		   + old_cell_y[i][j] * old_cell_y[i][j]) 
			/ COL_GRAD);		  
      if (ca_struct.colour_array[i][j] >= MAX_COLOURS) {
	ca_struct.colour_array[i][j] = MAX_COLOURS - 1;
#if OFLOW_RESET	
	old_cell_x[i][j] = 0.0;
	old_cell_y[i][j] = 0.0;
#else
	old_cell_x[i][j] = sqrt((double)(MAX_COLOURS * COL_GRAD));
	old_cell_y[i][j] = sqrt((double)(MAX_COLOURS * COL_GRAD)); 
#endif
      }
    }
  }

#if MAND_LIFE  
  for (i = 0; i < CA_WIDTH; i++) {
    for (j = 0; j < CA_HEIGHT; j++) { 
      zero_nb = 0;
      avg_x = 0.0;
      avg_y = 0.0;
      for (di = -1; di <= 1; di++) {
	for (dj = -1; dj <= 1; dj++) {
	  if ((di+i >= 0) && (di+i < CA_WIDTH)
	      && (dj + j >= 0) && (dj + j < CA_HEIGHT)) {	    
	    if (ca_struct.colour_array[di+i][dj+j] < LIFE_COL) {
	      zero_nb++;
	      /* avg_x += orig_cell_x[di+i][dj+j];
		 avg_y += orig_cell_y[di+i][dj+j]; */
	      avg_x += old_cell_x[di+i][dj+j];
	      avg_y += old_cell_y[di+i][dj+j];
	    }
	    if (ca_struct.colour_array[di+i][dj+j] > MAX_COLOURS/2) {
	      hi_nb++;
	      hi_avg_x += orig_cell_x[di+i][dj+j];
	      hi_avg_y += orig_cell_y[di+i][dj+j];
	    }
	  }
	}	
      }
      if ((zero_nb == 3) || (zero_nb == 3)) { 
	/* orig_cell_x[i][j] = avg_x/zero_nb;
	   orig_cell_y[i][j] = avg_y/zero_nb; */
	/* old_cell_x[i][j] = avg_x/zero_nb;
	   old_cell_y[i][j] = avg_y/zero_nb; */
	orig_cell_x[i][j] *= LIFE_SCALE;
	orig_cell_y[i][j] *= LIFE_SCALE;
	old_cell_x[i][j] = 0.0;
	old_cell_y[i][j] = 0.0;
      } 
      /* if ((hi_nb == 6)) { */
	/* orig_cell_x[i][j] = hi_avg_x/hi_nb;
	   orig_cell_y[i][j] = hi_avg_y/hi_nb; */
      /* old_cell_x[i][j] *= 1.4;
	 old_cell_y[i][j] *= 1.4;
	 } */
    }
  }
#endif

  return ca_struct;
}





/***************************************************
 *           $Log: mca_calc.c,v $
 *           Revision 1.8  2000/03/28 09:51:17  luke
 *           Added email address.
 *
 *           Revision 1.7  2000/03/28 09:46:52  luke
 *           Added copyright & license notice
 *
 *           Revision 1.6  2000/03/27 10:00:42  luke
 *           Cleaned up, changed code for MAND_LIFE option ... now orig_cell value is reset
 *           to somewhere closer in ...
 *
 *           Revision 1.5  2000/03/03 23:20:59  luke
 *           Added option of old overflow behaviour if "OFLOW_RESET" is clear
 *
 *           Revision 1.4  2000/03/03 23:03:32  luke
 *           Added POWER options (for z^POWER + c); also changed behaviour when magnitude
 *           too large - now resets to zero (so periodic colour cycling outside the
 *           Mandelbrot set).
 *
 *           Revision 1.3  2000/02/14 12:09:43  luke
 *           Added more colour schemes & MAND_LIFE option.
 *
 *           Revision 1.2  2000/02/13 08:43:51  luke
 *           Added AVG_NBHD as a compile-time option.
 *
 *           Revision 1.1  2000/02/13 02:33:10  luke
 *           Initial revision
 *
 ***************************************************/
