/* draw.c - Mandelbrot Explorer (drawing window setup and resize)
 *
 * Released under version 2 of the Gnu Public License.
 * By Chris Brady, cbrady@sgi.com
 */ 

#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Cardinals.h>
#include <X11/Xaw/Simple.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/Paned.h>
#include <X11/Shell.h>
#include "mxp.h"
#ifdef MPI
#include <mpi.h>
#endif

/* externals from colors.c */
extern Colormap cmap;

/* externals from menu.c */
extern struct menud *size_menu;

/* externals from file.c */
extern struct setup set;

/* externals from mandel.c */
extern struct zoomd *zoomp;
extern void mandel();
extern void event_proc();

/* externals from xmandelf.c */
extern Widget toplevel;
extern Dimension wwidth;
extern Dimension wheight;
extern Pixel black_pix;
extern Pixmap marker;
extern short depth;
extern void set_wm_hints();
extern Cursor crosshair_xcr;

Widget drawform;
Window draw_win;
GC draw_gc;
Screen *draw_Screen;
Display *draw_d;
Visual *draw_v;
XSizeHints sizehints;
XImage *xip;
XImage *xipa;
char *dp;
int *dpall;
char *dpref;
int aspect_opt;
short bypp;
#ifdef MPI
extern int mpi_size;
struct wtab *workt;
int lines;
int msg_len;
int *msg_buf;
#endif

void create_draw();
void create_images();
void resize();
void place_draw();
void set_aspect();

/*
 * Resize drawform window
 * Called from the size menu with a size index
 */
void menu_resize(w, index)
Widget w;
int index;
{
	Dimension wd=0, ht=0;

	switch((int)index) {
	case 0:
		wd = 128;
		ht = 96;
		break;
	case 1:
		wd = 192;
		ht = 144;
		break;
	case 2:
		wd = 320;
		ht = 240;
		break;
	case 3:
		wd = 512;
		ht = 384;
		break;
	case 4:
		wd = 640;
		ht = 480;
		break;
	case 5:
		wd = 800;
		ht = 600;
		break;
	case 6:
		wd = 896;
		ht = 672;
		break;
	case 7:
		wd = 1024;
		ht = 768;
		break;
	case 8:
		wd = 1280;
		ht = 1024;
		break;
	case 9:
		wd = 1600;
		ht = 1200;
		break;
	case 10:
		wd = 4000;
		ht = 2730;
		break;
	case 11:
		wd = 8000;
		ht = 5470;
		break;
	}
	if (wwidth == wd && wheight == ht) {
		return;
	}
	resize(wd, ht, index);

	/*
	 * All done with the re-size, so re-create the picture
	 */
	mandel(M_CLEAR);
}

/*
 * Resize the drawform window with the specified size
 * Called by menu_resize (above) or as a result of a mouse triggered
 * resize event
 */
void resize(w, h, sz)
Dimension w;
Dimension h;
int sz;
{
	Arg arg[2];
	static int prv_size = -1;
	double asp, adj;

	/*
	 * Adjust size menu marks to reflect the new size.
	 * If index = -1 then this was a mouse triggered resize
	 * therefore just clear the previous mark, if any.
	 */
	if (prv_size >= 0) {
		XtSetArg(arg[0], XtNleftBitmap, None);
		XtSetValues(size_menu->entry[prv_size].button, arg, 1);
	}
	prv_size = sz;
	if (sz >= 0) {
		XtSetArg(arg[0], XtNleftBitmap, marker);
		XtSetValues(size_menu->entry[prv_size].button, arg, 1);
	}

	/*
	 * Set width and height and adjust aspect ratio in the zoom
	 * structure. This keeps the mandelbrot image from being squashed
	 * or stretched as the aspect ratio changes.
	 */
	wwidth = w;
	wheight = h;
	asp = (double)wwidth / (double)wheight;
	adj = (((asp / set.aspect)-1.0) * (zoomp->ux - zoomp->lx)) / 2;
	zoomp->ux += adj;
	zoomp->lx -= adj;
	set.aspect = asp;
	zoomp->aspect = asp;

	XResizeWindow(draw_d, draw_win, w, h);

	/*
	 * Free the images and related data buffers
	 */
#ifdef MPI
	free(workt);
	free(msg_buf);
#endif
	free(dpall);
	XDestroyImage(xipa);
	XDestroyImage(xip);

	/*
	 * Re-create images and data buffers with the new sizes, place
	 * the window and set the aspect hints.
	 */
	create_images();
	place_draw(0);
	set_aspect(0, aspect_opt+13);
}

/*
 * Allocate memory and XImages needed to draw and refresh the pictures
 */
void create_images()
{
#ifdef MPI
	int segs;
#endif

	bypp = depth/8;
	if (bypp == 3) {  /* Bytes per pixel must be even */
		bypp = 4;
	}

#ifdef MPI
        segs = mpi_size * 8;
        if ((wheight / segs) < 24) {
                segs = wheight / 24;
        }
        lines = wheight / segs;
        segs++;
 
        if ((workt = (struct wtab *)malloc(segs*sizeof(struct wtab))) == NULL) {
                printf("malloc 0 failed\n");
                return;
        }
	msg_len = wwidth * lines;
        if ((msg_buf = (int *)malloc(msg_len * sizeof(int))) == NULL) {
                printf("malloc 0 failed\n");
                return;
        }
#endif
		
	if ((dpref = (char *)malloc(wwidth*wheight*bypp)) == NULL) {
		printf("malloc 0 failed\n"); 
		return; 
	}
	if ((dpall = (int *)malloc(wwidth*wheight*sizeof(int))) == NULL) {
		printf("malloc 1 failed\n"); 
		return; 
	}
	if ((dp = (char *)malloc(wwidth*bypp)) == NULL) {
		printf("malloc 2 failed\n"); 
		return; 
	}
	xip = XCreateImage(draw_d, draw_v, depth, ZPixmap, 0, (char *)dp,
	    wwidth, 1, bypp*8, 0);
	xip->byte_order = MSBFirst;
	xip->bitmap_bit_order = MSBFirst;
	xip->bits_per_pixel = depth;
	xipa = XCreateImage(draw_d, draw_v, depth, ZPixmap, 0, dpref,
	    wwidth, wheight, bypp*8, 0);
	xipa->byte_order = MSBFirst;
	xipa->bitmap_bit_order = MSBFirst;
	xipa->bits_per_pixel = depth;
}

/*
 * Place drawform in the best possible location
 */
void place_draw(first)
int first;
{
	int i;
	Arg arg[2];
	Dimension twid, border_w, scr_w, scr_h;
	Position pos_x, pos_y, tmp, title;
	Position top_x, top_y;

	scr_w = WidthOfScreen(XtScreen(toplevel));
	scr_h = HeightOfScreen(XtScreen(toplevel));

	i = 0;
        XtSetArg (arg[i], XtNwidth, &twid); i++;
        XtGetValues(toplevel, arg, i);

	/*
	 * If the widow has already been mapped then the x and y 
	 * co-ordinates include the title bar and the border added by
	 * the window manager, and these only need to be factored in
	 * the first time. I don't know how to get these dimensions
	 * so I am hard coding my defaults. KLUDGE!!!
	 */
	border_w = BORDER_WIDTH;
	title = TITLE_HEIGHT;

	XtTranslateCoords(toplevel, 0, 0, &top_x, &top_y);

	/*
	 * Place the new window on the screen
	 * First try placement on the right size of the command window
	 * Next try placement on the left size of the command window
	 * Last just put it against the right edge
	 */
	if ((top_x + wwidth + (border_w * 2) + twid) <= scr_w) {
		if (first) {
			pos_x = top_x + twid + border_w;
		} else {
			pos_x = top_x + twid + (border_w * 2);
		}
	} else {
		/* Won't fit on right side of command window */
		if ((wwidth + (border_w * 2)) <= top_x) {
			if (first) {
				pos_x = top_x - wwidth - (border_w * 3);
			} else {
				pos_x = top_x - wwidth - (border_w * 2);
			}
		} else {
			/* Won't fit on either side of command window */
			if (first) {
				pos_x = scr_w - wwidth - (border_w * 2);
			} else {
				pos_x = scr_w - wwidth - border_w;
			}
		}
	}

	/*
	 * Now find the best vertical placement.
	 */
	tmp = top_y + wheight + border_w;
	if (tmp <= scr_h) {
		if (first) {
			pos_y = top_y - border_w - title;
		} else {
			pos_y = top_y;
		}
	} else {
		if (first) {
			pos_y = scr_h - wheight - (border_w * 2) - title;
		} else {
			pos_y = scr_h - wheight - border_w;
		}
	}
	i = 0;
       	XtSetArg (arg[i], XtNx, pos_x); i++;
        XtSetArg (arg[i], XtNy, pos_y); i++;
        XtSetValues(drawform, arg, i);
}

/*
 * Create and configure the widget (drawform) that will be used to
 * display the pictures
 */
void create_draw()
{
	int i;
	Arg arg[6];

	i = 0;
	XtSetArg( arg[i], XtNheight, (XtArgVal) wheight); i++;
	XtSetArg( arg[i], XtNwidth, (XtArgVal) wwidth); i++;
	XtSetArg( arg[i], XtNtransient, (XtArgVal) FALSE); i++;
	XtSetArg( arg[i], XtNbackground, black_pix); i++;
	drawform = XtCreatePopupShell("drawform",
		transientShellWidgetClass, toplevel, arg, i);
	XtRealizeWidget(drawform);
	XtAddEventHandler(drawform, (ExposureMask|ButtonPressMask|
		ButtonReleaseMask|LeaveWindowMask|PointerMotionMask|
		StructureNotifyMask), FALSE, event_proc, 0);

	draw_d = XtDisplay(drawform);
        draw_win = XtWindow(drawform);
        draw_Screen = XtScreen(drawform);
        draw_gc = draw_Screen->default_gc;
        draw_v = draw_Screen->root_visual;

	place_draw(1);

	sizehints.flags = PResizeInc|USPosition;
	sizehints.width_inc = RESIZE_INC;
	sizehints.height_inc = RESIZE_INC;

	set_wm_hints(drawform, "Mxp", "Mxp",
		&sizehints, crosshair_xcr);	

        XtPopup(drawform, XtGrabNone);

	set_aspect(0, aspect_opt+13);
}

/*
 * Set the the aspect ration option to fixed or variable
 * Called at initialization and from the size menu. An "opt" of
 * 13 means fixed. This value also correlates with the menu option
 * position and is used to adjust the menu markers.
 */
void set_aspect(w, opt)
Widget w;
int opt;
{
	Arg arg[1];

	XtSetArg(arg[0], XtNleftBitmap, None);
	XtSetValues(size_menu->entry[aspect_opt+13].button, arg, 1);
        XtSetArg(arg[0], XtNleftBitmap, marker);
        XtSetValues(size_menu->entry[opt].button, arg, 1); 

	if (opt == 13) {
		aspect_opt = 0;
		sizehints.min_aspect.y = 1000; 
		sizehints.min_aspect.x = 1333; 
		sizehints.max_aspect.y = 1000; 
		sizehints.max_aspect.x = 1333; 
	} else {
		aspect_opt = 1;
		sizehints.min_aspect.x = 1; 
		sizehints.min_aspect.y = 20; 
		sizehints.max_aspect.x = 20; 
		sizehints.max_aspect.y = 1; 
	}

	/*
	 * The hints flags need to be preserved so just or in the new bit
	 */
	sizehints.flags |= PAspect;
	XSetWMNormalHints(draw_d, draw_win, &sizehints);
}
