/*
 * Copyright (C) 1995 by Tom Pfau (pfau@cnj.digex.net)
 *
 * This file may be distributed under the terms of the GPL.
 */
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <libsx.h>
#include "mandelbrot.h"
#include "hlsrgb.h"
#include "expand.h"
#include "setcolor.h"
#include "freq.h"
#include "submit.h"

#define M_INACT 0
#define M_START 1
#define M_ACTIVE 2
#define M_SELECTED 3
typedef struct {
    int State;
    int startX;
    int startY;
    int endX;
    int endY;
} MouseInfo;

#define TIMER_INTERVAL 100 /* 0.1 seconds */

static Widget wQuit,wLoad,wReload,wFname,wDraw;
static Widget wLeft,wTop,wHeight,wLimit,wIter,wProcs;
static Widget wCycle,wReset;
static unsigned char *colors;
static struct PARAMS params;
static MouseInfo jerry = { M_INACT, 0, 0, 0, 0 };
static char curfnam[255];
static unsigned char cmap_r[256],cmap_g[256],cmap_b[256];
static int rotate_flag = 0;

void quit_cb(Widget w,void *data)
{
    exit(0);
}

void rotate_one(unsigned char *arr)
{
    unsigned char c;
    c = arr[2];
    memmove(&arr[2],&arr[3],253);
    arr[255] = c;
}

void rotate(void *data)
{
    rotate_one(cmap_r);
    rotate_one(cmap_g);
    rotate_one(cmap_b);
    SetMyColorMap(256,cmap_r,cmap_g,cmap_b);
    if (rotate_flag)
	AddTimeOut(TIMER_INTERVAL,rotate,NULL);
}

void cycle_cb(Widget w, void *data)
{
    if (rotate_flag)
    {
	rotate_flag = 0;
	return;
    }
    rotate_flag = 1;
    rotate(NULL);
}

void redraw(Widget w,int width,int height,void *data)
{
    DrawImage(colors,0,0,SCREEN_WIDTH,SCREEN_HEIGHT);
}

void CalcColormap(void)
{
    int i;
    cmap_r[0] = cmap_g[0] = cmap_b[0] = -1;
    cmap_r[1] = cmap_g[1] = cmap_b[1] = 0;
    for (i=2;i<256;i++)
    {
	float a,fr,fg,fb;
	a = (float) (i-2) * 360.0 / 254.0;
	hls_to_rgb(a,0.5,0.8,&fr,&fg,&fb);
	cmap_r[i] = (int)(fr * 255.0);
	cmap_g[i] = (int)(fg * 255.0);
	cmap_b[i] = (int)(fb * 255.0);
    }
}

void reset_cb(Widget w, void *data)
{
    CalcColormap();
    SetMyColorMap(256,cmap_r,cmap_g,cmap_b);
}

void makebox(int x1,int y1,int x2,int y2)
{
    SetColor(255);
    SetDrawMode(GXinvert);
    DrawBox(x1,y1,x2-x1,y2-y1);
    SetDrawMode(GXcopy);
}

void start_new_plot(void)
{
    double w,h;
    double l,t;
    w = jerry.startX - jerry.endX;
    h = jerry.startY - jerry.endY;
    if (jerry.startX > jerry.endX) jerry.startX = jerry.endX;
    if (jerry.startY > jerry.endY) jerry.startY = jerry.endY;
    l = params.left + (double)jerry.startX * params.height / SCREEN_WIDTH;
    t = params.top + (double)jerry.startY * params.height / SCREEN_HEIGHT;
    if (w < 0) w = -w;
    if (h < 0) h = -h;
    if (w > h) h = w;
    h = params.height * h / SCREEN_HEIGHT;
    submit_plot(l,t,h,params.iterations,params.limit);
}

void button_down(Widget w, int which, int x, int y, void *data)
{
    switch (jerry.State)
    {
      case M_INACT:
	if (which == 1)
	{
	    jerry.State = M_START;
	    jerry.startX = x;
	    jerry.startY = y;
	}
	return;
      case M_SELECTED:
	if (which == 3)
	    start_new_plot();
      case M_ACTIVE:
	makebox(jerry.startX,jerry.startY,jerry.endX,jerry.endY);
      case M_START:
	jerry.State = M_INACT;
	return;
    }
}

void button_up(Widget w, int which, int x, int y, void *data)
{
    if (jerry.State != M_ACTIVE)
    {
	jerry.State = M_INACT;
	return;
    }
    makebox(jerry.startX,jerry.startY,jerry.endX,jerry.endY);
    jerry.endX = x;
    jerry.endY = y;
    makebox(jerry.startX,jerry.startY,jerry.endX,jerry.endY);
    jerry.State = M_SELECTED;
}

void movement(Widget w, int x, int y, void *data)
{
    if ((jerry.State == M_INACT) || (jerry.State == M_SELECTED))
	return;
    if (jerry.State == M_ACTIVE)
    	makebox(jerry.startX,jerry.startY,jerry.endX,jerry.endY);
    jerry.endX = x;
    jerry.endY = y;
    makebox(jerry.startX,jerry.startY,jerry.endX,jerry.endY);
    jerry.State = M_ACTIVE;
}

void loadimage(char *fnam)
{
    short int buf[SCREEN_WIDTH+1];
    short int tbuf[SCREEN_WIDTH];
    char obuf[SCREEN_WIDTH+1];
    int number_colors=256;
    int r;
    FILE *log;

    rotate_flag = 0;
    if (fnam)
    {
    	if ((log = fopen(fnam,"rb")) == NULL)
    	{
	    perror(fnam);
            exit(2);
    	}
    	SetLabel(wFname,fnam);
	strcpy(curfnam,fnam);
    }
    else
	if ((log = fopen(curfnam,"rb")) == NULL)
	{
	    perror(curfnam);
	    exit(2);
	}

    jerry.State = M_INACT;

    fread(&params,sizeof(struct PARAMS),1,log);

    sprintf(obuf,"%-18.16g",params.left);
    SetLabel(wLeft,obuf);
    sprintf(obuf,"%-18.16g",params.top);
    SetLabel(wTop,obuf);
    sprintf(obuf,"%-18.16g",params.height);
    SetLabel(wHeight,obuf);
    sprintf(obuf,"%-5f",params.limit);
    SetLabel(wLimit,obuf);
    sprintf(obuf,"%-5d",params.iterations);
    SetLabel(wIter,obuf);
    if (!params.processes)
	params.processes = 1;
    sprintf(obuf,"%-2d",params.processes);
    SetLabel(wProcs,obuf);

    memset(colors,'\1',SCREEN_HEIGHT * SCREEN_WIDTH);
    for (r=0;r<SCREEN_HEIGHT;r++)
    {
	if (fread(buf,sizeof(short int),1,log) == 0)
	    break;
	fread(&buf[1],sizeof(short int),buf[0],log);
	expand(buf,tbuf);
	set_color(tbuf,&colors[r*SCREEN_WIDTH],params.iterations,number_colors);
    }
    fclose(log);

    DrawImage((char *)colors,0,0,SCREEN_WIDTH,SCREEN_HEIGHT);
}

void reload_cb(Widget w, void *data)
{
    loadimage(NULL);
}

void load_cb(Widget w, void *data)
{
    char *fnam;
    rotate_flag = 0;
    fnam = GetFile(NULL);
    if (fnam)
	loadimage(fnam);
}

void init_display(void)
{
    Widget wTemp[10];

    wTemp[0] = MakeForm(TOP_LEVEL_FORM,NO_CARE,NULL,NO_CARE,NULL);
    wQuit = MakeButton("Quit",quit_cb,NULL);
    wReload = MakeButton("Reload",reload_cb,NULL);
    SetWidgetPos(wReload,PLACE_RIGHT,wQuit,0,NULL);
    wLoad = MakeButton("Load...",load_cb,NULL);
    SetWidgetPos(wLoad,PLACE_RIGHT,wReload,0,NULL);
    wTemp[1] = MakeLabel("File Name: ");
    SetWidgetPos(wTemp[1],PLACE_RIGHT,wLoad,0,NULL);
    wFname = MakeLabel("xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx");
    SetWidgetPos(wFname,PLACE_RIGHT,wTemp[1],0,NULL);

    wTemp[1] = MakeForm(TOP_LEVEL_FORM,PLACE_UNDER,wTemp[0],NO_CARE,NULL);
    wTemp[2] = MakeLabel("Left: ");
    wLeft = MakeLabel(" 0.0000000000000000e-00");
    SetWidgetPos(wLeft,PLACE_RIGHT,wTemp[2],0,NULL);
    wTemp[3] = MakeLabel("Top: ");
    SetWidgetPos(wTemp[3],PLACE_RIGHT,wLeft,0,NULL);
    wTop = MakeLabel(" 0.0000000000000000e-00");
    SetWidgetPos(wTop,PLACE_RIGHT,wTemp[3],0,NULL);
    wTemp[4] = MakeLabel("Height: ");
    SetWidgetPos(wTemp[4],PLACE_RIGHT,wTop,0,NULL);
    wHeight = MakeLabel(" 0.0000000000000000e-00");
    SetWidgetPos(wHeight,PLACE_RIGHT,wTemp[4],0,NULL);

    wTemp[2] = MakeForm(TOP_LEVEL_FORM,PLACE_UNDER,wTemp[1],NO_CARE,NULL);
    wTemp[3] = MakeLabel("Limit: ");
    wLimit = MakeLabel("002.0000");
    SetWidgetPos(wLimit,PLACE_RIGHT,wTemp[3],0,NULL);
    wTemp[4] = MakeLabel("Iterations: ");
    SetWidgetPos(wTemp[4],PLACE_RIGHT,wLimit,0,NULL);
    wIter = MakeLabel("00000");
    SetWidgetPos(wIter,PLACE_RIGHT,wTemp[4],0,NULL);
    wTemp[5] = MakeLabel("Processes: ");
    SetWidgetPos(wTemp[5],PLACE_RIGHT,wIter,0,NULL);
    wProcs = MakeLabel("00");
    SetWidgetPos(wProcs,PLACE_RIGHT,wTemp[5],0,NULL);

    wTemp[3] = MakeForm(TOP_LEVEL_FORM,
			PLACE_UNDER,wTemp[1],PLACE_RIGHT,wTemp[2]);
    wTemp[4] = MakeLabel("Colors");
    wCycle = MakeButton("Cycle",cycle_cb,NULL);
    SetWidgetPos(wCycle,PLACE_RIGHT,wTemp[4],NO_CARE,NULL);
    wReset = MakeButton("Reset",reset_cb,NULL);
    SetWidgetPos(wReset,PLACE_RIGHT,wCycle,NO_CARE,NULL);

    SetForm(TOP_LEVEL_FORM);

    colors = malloc(SCREEN_HEIGHT * SCREEN_WIDTH);
    if (colors == NULL)
    {
	perror("malloc");
	exit(4);
    }
    memset(colors,'\1',SCREEN_HEIGHT * SCREEN_WIDTH);
    wDraw = MakeDrawArea(SCREEN_WIDTH,SCREEN_HEIGHT,redraw,NULL);
    SetWidgetPos(wDraw,PLACE_UNDER,wTemp[2],0,NULL);
    SetButtonDownCB(wDraw,button_down);
    SetButtonUpCB(wDraw,button_up);
    SetMouseMotionCB(wDraw,movement);
}

void main(int argc,char *argv[])
{
    if (!OpenDisplay(argc,argv))
	exit(2);

    init_display();
    ShowDisplay();

    CalcColormap();
    GetAllColors();
    SetMyColorMap(256,cmap_r,cmap_g,cmap_b);

    if (argc > 1)
	loadimage(argv[1]);
    MainLoop();
}
