#include "iter.h"
#include <stdlib.h>
#include <math.h>
#include "image.h"
#include "mmath.h"
#include "gi.h"
#include "input.h"
#include <stdio.h>
#include "info.h"

#define MAX_STEP 12

#define W1 4
#define H1 4

static int x_1, x_2;
static int y_1, y_2;

static int step_size=MAX_STEP;
static int scan_size=MAX_STEP;

extern ginter gi;
extern struct info info;
extern struct input input;


static int maxcolor_table = 0;
static unsigned long *colors = NULL;
static unsigned long color_table[] = {
	0x000000,
	0x000020,
	0x000040,
	0x000060,
	0x000080,
	0x0000A0,
	0x0000C0,
	0x0020A0,
	0x004080,
	0x006060,
	0x008040,
	0x00A020,
	0x00C000,
	0x20A000,
	0x408000,
	0x606000,
	0x804000,
	0xA02000,
	0xC00000
};


static unsigned long getcolor(unsigned long c)
{
	return rgb_conv((c >> 16) & 0xff, (c >> 8) & 0xff, c & 0xff);
}

static int init_color(void)
{
	unsigned long *t;
	int i;
	maxcolor_table = sizeof(color_table) / sizeof(color_table[0]);

	t = realloc(colors,
		    (size_t) (sizeof(unsigned long)) * maxcolor_table);
	if (t == NULL) {
		if (colors != NULL)
			free(colors);
		return -1;
	}
	colors = t;
	for (i = 0; i < maxcolor_table; i++) {
		colors[i] = getcolor(color_table[i]);
	}
	return 0;
}

int set_iter(void)
{
	x_2 = image_width() / 2;
	y_2 = image_height() / 2;
	x_1 = x_2;
	y_1 = y_2;

	if (init_color() != 0)
		return -1;
	return 0;
}

__inline__ static double t_xb(int xa, double dx, int xh, double s)
{
	double x = (double) (xa - xh) / s - (double) dx;
	return x;
}

__inline__ static double t_yb(int ya, double dy, int yh, double s)
{
	double y = -(double) (ya - yh) / s - (double) dy;
	return y;
}


__inline__ static double t_xa(int xb, double dx, int xh, double s)
{
	double x = ((double) xb + dx) * s + (double) xh;
	return x;
}

__inline__ static double t_ya(int yb, double dy, int yh, double s)
{
	double y = -((double) yb + dy) * s + (double) yh;
	return y;
}


#ifdef ASM_OPT

extern int itplot(double x, double y, int miter, double mrage);

#else

__inline__ static int itplot(double x, double y, int miter, double mrage)
{
	int res = 0;
	double sqra = SQR(x);
	double sqrb = SQR(y);
	double a = x;
	double b = y;
	for (res = 0; mrage > (sqra + sqrb); res++) {
		if (res > miter)
			return 0;
		b = 2 * a * b + y;
		a = sqra + x - sqrb;
		sqra = SQR(a);
		sqrb = SQR(b);
	}
	return res;
}

#endif

__inline__ static void clear_old(int xa1, int xa2, int ya1, int ya2)
{
	int x1 = x_1, y1 = y_1, x2 = x_2, y2 = y_2;

	if (y1 < ya1) {
		y2 = MIN(y2, ya1);
		set_area(x1, y1, x2, y2);
		clear_data(x1, y1, x2, y2);
		if (y2 == y_2)
			return;
		y1 = y2;
		y2 = y_2;
	}
	if (x1 < xa1) {
		x2 = MIN(x2, xa1);
		set_area(x1, y1, x2, y2);
		clear_data(x1, y1, x2, y2);
		if (x2 == x_2)
			return;
		x1 = x2;
		x2 = x_2;
	}
	if (y2 > ya2) {
		y1 = MAX(y1, ya2);
		set_area(x1, y1, x2, y2);
		clear_data(x1, y1, x2, y2);
		if (y1 == y_1)
			return;
		y2 = y1;
		y1 = y_1;
	}
	if (x2 > xa2) {
		x1 = MAX(x1, xa2);
		set_area(x1, y1, x2, y2);
		clear_data(x1, y1, x2, y2);
	}
}


__inline__ static void iter_area(int x0, int y0, int x1, int y1, double xb,
				 double yb, double dxb, double dyb,
				 int miter, double mrage)
{
	for (; x0 < x1; x0++, xb += dxb) {
		double tyb = yb;
		int ty0 = y0;
		for (; ty0 < y1; ty0++, tyb += dyb) {
			unsigned long c;
			int i = itplot(xb, tyb, miter, mrage);
			i = i % maxcolor_table;
			c = colors[i];
			points(x0, ty0, c);
		}
	}
}


__inline__ int iter_a(double dx, double dy, double s, 
		      int miter, double mrage, int xa1, int ya1, int xa2,
		      int ya2)
{
	unsigned long int x, y;
	double dxb = 1.0 / s;
	double dyb = -1.0 / s;
	double xb0 = t_xb(xa1, dx, image_width() / 2, s);
	double yb = t_yb(ya1, dy, image_height() / 2, s);
	set_area(xa1, ya1, xa2, ya2);
	if(step_size==1){
		iter_area(xa1,ya1,xa2,ya2,xb0,yb,dxb,dyb,miter,mrage);
		return 1;
	}
	for (y = ya1; y < ya2; y += step_size, yb += dyb * step_size) {
		double xb = xb0;
		for (x = xa1; x < xa2; x += step_size, xb += dxb * step_size) {

			unsigned long c;
			int i = itplot(xb, yb, miter, mrage);
			i = i % maxcolor_table;
			/*int i = itplot(xb,yb); */
			c = colors[i];
			{
				int kx = MIN(x+step_size,xa2);
				int ky = MIN(y+step_size,ya2);
				arena(x, y, kx-x, ky-y, c);
			}
		}
	}
	return step_size;
}

__inline__ static void iter_scan(double dx, double dy,double s, int miter, double mrage,int xa1,int ya1,int xa2,int ya2 ){
	int ka=scan_size;
	unsigned long int x, y;
	double dxb = 1.0 / s;
	double dyb = -1.0 / s;
	double xb0 = t_xb(xa1, dx, image_width() / 2, s);
	double yb = t_yb(ya1, dy, image_height() / 2, s);
	for (y = ya1; y < ya2; y += ka, yb += dyb * ka) {
		double xb = xb0;
		for (x = xa1; x < xa2;
			     x += ka, xb += dxb * ka) {
			unsigned long int c0 = getpoints(x, y);
			if (x + ka >= xa2
			    || y + ka >= ya2
			    || c0 != getpoints(x + ka+1, y)
			    || c0 != getpoints(x, y + ka+1)
			    || c0 != getpoints(x + ka+1, y + ka+1)
			    || c0 != getpoints(x + ka / 2, y + ka / 2)){
			    	unsigned int xend=MIN(x + ka, xa2);
				unsigned int yend=MIN(y + ka, ya2);
				iter_area(x, y, xend, yend, xb, yb, dxb, dyb, miter, mrage);
				set_area(x, y, xend, yend);
			}
		}
	}
}



static void stepsize (int fps){
	if(fps > 30 && step_size > 1){
		step_size--;
	} else if(fps < 15 && step_size < MAX_STEP){
		step_size++;
	}	
}

__inline__ void iter(double dx, double dy, double s,
		    int miter, double mrage, int fps,int mode)
{
	double tmp_a =
	    MAX(0, t_xa(-W1 / 2, dx, (int) image_width() / 2, s));
	int xa1 = (int) tmp_a;
	int xa2;
	int ya1;
	int ya2;
	tmp_a =
	    MIN(image_width(),
		t_xa(W1 / 2, dx, (int) image_width() / 2, s));
	xa2 = (int) tmp_a;
	tmp_a = MAX(0, t_ya(H1 / 2, dy, (int) image_height() / 2, s));
	ya1 = (int) tmp_a;
	tmp_a =
	    MIN(image_height(),
		t_ya(-H1 / 2, dy, (int) image_height() / 2, s));
	ya2 = (int) tmp_a;

/*	x_1=min(xa1,x_1);
	y_1=min(ya1,y_1);
	x_2=max(xa2,x_2);
	y_2=max(ya2,y_2);
*/

	if(mode==2 && scan_size==1)
	                return;

	if(mode==1 && fps!=0)
		stepsize (fps);
		
	if (xa1 < image_width() && xa2 >= 0
	    && ya1 < image_height() && ya2 >= 0) {
		if (mode == 1) {
			iter_a(dx, dy, s, miter, mrage, xa1, ya1, xa2, ya2);
			image_to_swap(1);	
			scan_size=MAX(step_size,scan_size);
		} else if (mode == 2) {
			image_to_swap(1);
			if(scan_size > 8){
				int tmp=step_size; 
				step_size=7; 
				iter_a(dx, dy, s, miter, mrage, xa1, ya1, xa2, ya2); 
				scan_size=7; 
				step_size=tmp;
			} else if(scan_size > 1) {
				iter_scan(dx, dy, s, miter, mrage, xa1, ya1, xa2, ya2);
				if(scan_size / 2 > 2)
					scan_size /= 2;
				else if(scan_size == 2)
					scan_size = 1;
				else	
					scan_size = 2;
			} else {
				int tmp=step_size;
				step_size=1;
				iter_a(dx, dy, s, miter, mrage, xa1, ya1, xa2, ya2);
				scan_size=0;
				step_size=tmp;
			}
		}
	} else {
		xa1 = 0;
		xa2 = 0;
		ya1 = 0;
		ya2 = 0;
	}

	clear_old(xa1, xa2, ya1, ya2);

	x_1 = xa1;
	y_1 = ya1;
	x_2 = xa2;
	y_2 = ya2;

	
}


void stop_iter()
{
	if (colors != NULL)
		free(colors);
}
