/* Tab=4, Linewrap=off */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "gtnpic.h"
#include "gtnpicfont.h"

/* Private */
static void version(void);
static void usage(void);
static void addtext(int xpos, int ypos, char *str, int xsiz);
static void addpic(unsigned char *inimage, int xpos, int ypos, int xsiz);

int main(int argc, char **argv)
{
	int total=0;                            /* Total files to be indexed   */
	int f;                                  /* Current file to be indexed  */
	div_t cxy;
	int cx;
	int cy;
	int xsiz;                               /* Total output image xsize    */
	int ysiz;                               /* Total output image ysize    */
	int blahtype;                           /* Image format type           */
	unsigned char *bigimage=NULL;           /* Pointer to current image    */
	unsigned char *smallimage=NULL;         /* Pointer to resized image    */
	unsigned char *palrgb=NULL;             /* Pointer to image palette    */
	char str[1024],*org;                    /* Text information            */
	int supported;                          /* Is this format supported?   */
	int x=0;
	int y=0;
	int  opt_flag_quality=0;                /* Option flag                 */
	int  opt_flag_bg=0;                     /* Option flag                 */
	int  opt_flag_filename=0;               /* Option flag                 */
	int  opt_quality;                       /* JPEG Quality                */
	char *opt_bg=NULL;                      /* Blackground color           */
	char *opt_filename=NULL;                /* Output filename             */


	/* Check command args */
	if(argc < 2)
		{
		usage();
		exit(EXIT_FAILURE);
		}
	while( -1 != (x=getopt(argc,argv,"vhq:b:o:")) )
		{
		switch(x)
			{
			case 'v':
				version();
				exit(EXIT_FAILURE);
				break;
			case 'h':
				usage();
				exit(EXIT_FAILURE);
				break;
			case 'q':
				sscanf(optarg,"%d",&opt_quality);
				if(opt_quality < 1 || opt_quality > 100)
					{
					printf("Invalid quality range: %d\n",opt_quality);
					exit(EXIT_FAILURE);
					}
				opt_flag_quality=1;
				break;
			case 'b':
				opt_bg=optarg;
				if( !strcmp(opt_bg,"white"))
					background=WHITE;
				else if( !strcmp(opt_bg,"gray"))
					background=GRAY;
				else if( !strcmp(opt_bg,"black"))
					background=BLACK;
				else
					{
					printf("Unknown background color: %s\n",opt_bg);
					exit(EXIT_FAILURE);
					}
				opt_flag_bg=1;
				break;
			case 'o':
				opt_filename=optarg;
				opt_flag_filename=1;
				break;
			case '?':
				exit(EXIT_FAILURE);
				break;
			default:
				printf("Unknown error. Exiting.\n");
				exit(EXIT_FAILURE);
				break;
			}
		}
	for(x=optind;x<argc;x++)
		total++;
	if(total<1)
		{
		usage();
		exit(EXIT_FAILURE);
		}

	/* Set the defaults */
	if(!opt_flag_quality)
		opt_quality=70;
	if(!opt_flag_bg)
		background=GRAY;
	if(!opt_flag_filename)
		{
		if( NULL == (opt_filename=(char *) malloc(1024)) )
			{
			printf("Failed to malloc memory for filename.\n");
			exit(EXIT_FAILURE);
			}
		strcpy(opt_filename,"00_index.jpg");
		}

	/* Do some calculations */
	cxy=div(total,PICS_ACROSS);
	cx=cxy.rem;
	xsiz=((cxy.quot+1)==1) ? cx*SCRNWIDE : (PICS_ACROSS*SCRNWIDE);
	if(total>1)
		cxy=div(total-1,PICS_ACROSS);
	ysiz=(cy=cxy.quot+1)*(SCRNHIGH+TEXTHI);

	/* Calloc memory to hold thumbnail index file */
	if( NULL == (outimage=calloc(1,xsiz*ysiz*3)) )
		{
		printf("calloc: Error.\n");
		if(opt_filename) free(opt_filename);
		exit(EXIT_FAILURE);
		}

	/* Set the background color */
	switch(background)
		{
		case BLACK:
			for(x=0;x<(xsiz*ysiz*3);x++)
				outimage[x]=BLACK;
			break;
		case GRAY:
			for(x=0;x<(xsiz*ysiz*3);x++)
				outimage[x]=GRAY;
			break;
		case WHITE:
			for(x=0;x<(xsiz*ysiz*3);x++)
				outimage[x]=WHITE;
			break;
		default: /* Set default background to GRAY - shouldn't get here */
			for(x=0;x<(xsiz*ysiz*3);x++)
				outimage[x]=GRAY;
			break;
		}

	/* The big loop */
	printf("Total Files:%d.\n",total);
	for(f=optind;f<argc;f++)
		{
		cxy=div(y,PICS_ACROSS);
		y++;
		printf("Reading %s...\n",argv[f]);
		blahtype=get_imagetype(argv[f]);
		switch(blahtype)
			{
			case UNKNOWN:
				printf("%s : Unknown imagetype, skipping.\n",argv[f]);
				supported=0;
				break;
			case PNG:
				printf("%s : PNG format not yet supported, skipping.\n",argv[f]);
				supported=0;
				break;
			case BMP:
				printf("%s : BMP format not supported, skipping.\n",argv[f]);
				supported=0;
				break;
			case GIF:
				printf("%s : GIF format not supported, skipping.\n",argv[f]);
				printf("Consider converting this image to jpeg or png format.\n");
				supported=0;
				break;
			case JPEG:
				/* Returns a pointer to the original image or NULL on error */
				bigimage=read_JPEG_file(argv[f], &palette);
				if( NULL == bigimage)
					{
					printf("read_JPEG_file: Error\n");
					if(opt_filename) free(opt_filename);
					if(outimage) free(outimage);
					exit(EXIT_FAILURE);
					}
				palrgb=palette;
				supported=1;
				break;
			default:
				printf("Unknown Error.\n");
				if(opt_filename) free(opt_filename);
				if(outimage) free(outimage);
				exit(EXIT_FAILURE);
				break;
			}

		if(supported)
			{
			/* Returns a pointer to the resized image or NULL on error */
			smallimage=resizepic(bigimage,palrgb,palrgb+256,
                                 palrgb+512,width,height,SCRNWIDE,SCRNHIGH);
			if(NULL == smallimage)
				{
				printf("resizepic: Error\n");
				if(opt_filename) free(opt_filename);
				if(bigimage) free(bigimage);
				if(outimage) free(outimage);
				exit(EXIT_FAILURE);
				}
	
			/* Add the resized image to the index image */
			addpic(smallimage,((cxy.rem)*SCRNWIDE),
                   ((cxy.quot)*(SCRNHIGH+TEXTHI)),xsiz);

			/* Add in the image comments */
			if(strrchr(argv[f],'/')!=NULL)
				org=strrchr(argv[f],'/')+1;
			else
				org=argv[f];
			strcpy(str,org);
			if(strlen(str)>16) str[16]=0;
			addtext((cxy.rem)*SCRNWIDE,(cxy.quot)*(SCRNHIGH+TEXTHI)+SCRNHIGH+3,
                    str,xsiz);
			sprintf(str,"%d x %d",width,height);
			addtext((cxy.rem)*SCRNWIDE,(cxy.quot)*(SCRNHIGH+TEXTHI)+SCRNHIGH+10,
                    str,xsiz);

			free(palrgb);
			free(bigimage);
			free(smallimage);
			}
		supported=0;
		}
	printf("Writing index file...\n");
	write_JPEG_file(opt_filename,xsiz,ysiz,opt_quality);
	free(opt_filename);
	free(outimage);
	printf("Done.\n");

return EXIT_SUCCESS;
}

static void addtext(int xpos, int ypos, char *str, int xsiz)
{
	int a,b,c,f,x,loc;

	x=xpos;
	for(f=0;f<strlen(str);f++)
		{
		c=toupper(str[f])-32;
		if((c<0)||(c>63))
			c='_'-32;
		for(b=0;b<5;b++)
		for(a=0;a<tnpicfont[LETTERSIZ*c];a++)
			if(tnpicfont[LETTERSIZ*c+1+b*8+a]=='x')
				{
				loc=3*((ypos+b)*xsiz+x+a);
				switch(background)
					{
					case BLACK:
						outimage[loc  ]=WHITE;
						outimage[loc+1]=WHITE;
						outimage[loc+2]=WHITE;
						break;
					case GRAY:
						outimage[loc  ]=WHITE;
						outimage[loc+1]=WHITE;
						outimage[loc+2]=WHITE;
						break;
					case WHITE:
						outimage[loc  ]=GRAY;
						outimage[loc+1]=GRAY;
						outimage[loc+2]=GRAY;
						break;
					default: /* Default bg is GRAY */
						outimage[loc  ]=WHITE;
						outimage[loc+1]=WHITE;
						outimage[loc+2]=WHITE;
						break;
					}
				}
		x+=tnpicfont[LETTERSIZ*c]+1;
		}
}

static void addpic(unsigned char *inimage, int xpos, int ypos, int xsiz)
{
	int x,y;
	for(y=0;y<SCRNHIGH;y++)
	for(x=0;x<SCRNWIDE;x++)
		{
		*(outimage+3*((ypos+y)*xsiz+xpos+x)  )=*(inimage+3*(y*SCRNWIDE+x)  );
		*(outimage+3*((ypos+y)*xsiz+xpos+x)+1)=*(inimage+3*(y*SCRNWIDE+x)+1);
		*(outimage+3*((ypos+y)*xsiz+xpos+x)+2)=*(inimage+3*(y*SCRNWIDE+x)+2);
		}
}

static void usage(void)
{
	version();
	printf("Usage: gtnpic [-vh] [-q nn] [-b bgcolor] [-o outfile] file_1..file_n\n\n");
	printf("Examples:\n");
	printf("To create a thumbnail index containing all png\n");
	printf("and jpeg files in the current directory:\n\n");
	printf("    gtnpic *.png *.jpg\n\n");
	printf("To set the index image backgound to white:\n\n");
	printf("    gtnpic -b white *.jpg\n\n");
	printf("See gtnpic(1) for more details.\n\n");
}

static void version(void)
{
	printf("gtnpic : Version %s\n",GTNPIC_VERSION);
	printf("Copyright (c) 1997 Willie Daniel <gith@quicklink.net> \n\n");
}
