/*
    pmosc - recreates an image out of smaller image tiles
    Copyright (C) 1998 Jens Vaasjo <jvaasjo@iname.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include<stdio.h>
#include<stdlib.h>
#include<limits.h>
#include<float.h>
#include<math.h>
#include"image.h"
#include"pmosc.h"
#include"misc.h"

static double rgbdif(rgb_t c1,rgb_t c2)
{
	double dif;

	dif  = ((double)c2.r - (double)c1.r) * ((double)c2.r - (double)c1.r);
	dif += ((double)c2.g - (double)c1.g) * ((double)c2.g - (double)c1.g);
	dif += ((double)c2.b - (double)c1.b) * ((double)c2.b - (double)c1.b);

	return sqrt(dif);
}

static double deviation(const image_t *bigimg,const image_t *img,
				unsigned long xn,unsigned long yn)
{
	unsigned long xt,yt,x,y;
	double dif = 0.0;

	xt = xn * img->width;
	yt = yn * img->height;

	for(y=0;y<img->height;y++)
	{
		for(x=0;x<img->width;x++)
		{
			dif += rgbdif(img->data[x+y*img->width],
				bigimg->data[x+xt+(y+yt)*bigimg->width]);
		}
	}

	return dif / (double)(img->width * img->height);
}

static unsigned long insert(image_t **imgs,const image_t *bigimg,image_t *img,
			double *difs,double **imgdifs,double *idifs,
			unsigned long nx,unsigned long ny)
{
	unsigned long bestx,besty,x,y;
	double bestdif,dif,*tmpdifs;
	image_t *imgtmp;
	unsigned fill=0;

	if(idifs[0] == DBL_MAX) fill = 1;

	bestx = besty = ~0;
	bestdif = DBL_MAX;

	for(y=0;y<ny;y++)
	{
		for(x=0;x<nx;x++)
		{
			if(fill)
			{
				dif = deviation(bigimg,img,x,y);
				idifs[x+y*nx] = dif;
			}
			else dif = idifs[x+y*nx];

			if( (dif < difs[x+y*nx]) && (dif < bestdif) )
			{
				bestx = x;
				besty = y;
				bestdif = dif;
			}
		}
	}

	if(bestdif == DBL_MAX)
	{
		free(idifs);
		imageFree(img);
		return ~0;
	}

	difs[bestx + besty*nx] = bestdif;

	imgtmp = imgs[bestx + besty*nx];
	imgs[bestx + besty*nx] = img;

	tmpdifs = imgdifs[bestx + besty*nx];
	imgdifs[bestx + besty*nx] = idifs;

	if(imgtmp) insert(imgs,bigimg,imgtmp,difs,imgdifs,tmpdifs,nx,ny);

	return bestx + besty * nx;
}

static void mkfinal(image_t *final,image_t **imgs,
		unsigned long nx,unsigned long ny)
{
	unsigned long i,x,y;

	for(y=0;y<ny;y++)
	{
		for(x=0;x<nx;x++)
		{
			i = x+y*nx;
			imageCopy(final,imgs[i],
				x*imgs[i]->width,y*imgs[i]->height,
				0,0,imgs[i]->width,imgs[i]->height);
		}
	}
}

static image_t *imosc(image_t *bigimg,char **file,
		unsigned long nx,unsigned long ny,
		unsigned long width,unsigned long height)
{
	image_t *final=NULL,*img=NULL,*tmp=NULL;
	image_t **imgs=NULL;
	double **imgdifs=NULL;
	double *difs=NULL,*idifs=NULL;
	unsigned long count,passes;
	unsigned long i,n,l;
	double avg;

	for(count=0;file[count];count++);
	passes = (unsigned long)ceil((double)(nx * ny) / (double)count);
	fprintf(stderr,"Processing passes: %lu\n",passes);

	difs = (double*)xmalloc(nx * ny * sizeof(double));
	for(n=0;n<(nx*ny);n++) difs[n] = DBL_MAX;

	imgdifs = (double**)xcalloc(nx*ny,sizeof(double*));
	imgs = (image_t**)xcalloc(nx*ny,sizeof(image_t*));

	for(n=0;n<count;n++)
	{
		fprintf(stderr,"Processing: %lu/%lu: %s\n",
					n+1,count,file[n]);
		img = imageLoad(file[n],width,height);
		for(i=0,l=~0;i<passes;i++)
		{
			tmp = imageDup(img);
			idifs = (double*)xmalloc(nx * ny * sizeof(double));

			if(l == ~0) idifs[0] = DBL_MAX;
			else memcpy(idifs,imgdifs[l],nx * ny * sizeof(double));

			l = insert(imgs,bigimg,tmp,difs,imgdifs,idifs,nx,ny);
		}
		imageFree(img);
	}

	fprintf(stderr,"Building final image...\n");
	mkfinal(bigimg,imgs,nx,ny);

	avg = 0;
	for(n=0;n<(nx*ny);n++) avg += difs[n];
	avg /= (double)(nx*ny);
	avg /= sqrt(255.0*255.0*3.0);

	fprintf(stderr,"Normalized Average Deviation: %f\n",avg);

	final = bigimg;

	free(difs);

	for(n=0;n<(nx*ny);n++) free(imgdifs[n]);
	free(imgdifs);

	for(n=0;n<(nx*ny);n++) imageFree(imgs[n]);
	free(imgs);

	return final;
}

int pmosc(const char *output,const char *input,
	unsigned long nx,unsigned long ny,
	unsigned long width,unsigned long height,char **file)
{
	image_t *initial,*final;

	fprintf(stderr,"Loading: %s\n",input);
	initial = imageLoad(input,nx * width,ny * height);

	final = imosc(initial,file,nx,ny,width,height);

	fprintf(stderr,"Writing: %s\n",output);
	imageWrite(output,final);

	imageFree(initial);
	return 0;
}

