/*GPL*START*
 * 
 * Copyright (C) 1998 by Johannes Overmann <overmann@iname.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * *GPL*END*/  

#include <limits.h>
#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qtimer.h>
#include <qapp.h>


#include "fractwid.h"
#include "tappconfig.h"
#include "tstring.h"
#include "tminmax.h"



#define NOPIX



//returns number of iters
double calc_asm(int *inf, int infwidth, 
					 int ix, int iy, int maxx, int maxy, int dx, int dy,
					 int maxiter,
					 double c_x, double c_y, 
					 double dxx, double dxy, double dyx, double dyy,
					 bool julia, double julia_x, double julia_y, bool force) {
  double ax, ay, cx, cy, qx, qy, t, iter=0;
  int x, y, i, xx, yy;

  for(y=iy, yy=0; yy < maxy; yy+=dy, y+=dy) {
	 for(x=ix, xx=0; xx < maxx; xx+=dx, x+=dx) {
		if(force || inf[xx + yy*infwidth]<0) {
		  ax = c_x + x*dxx + y*dyx;
		  ay = c_y + x*dxy + y*dyy;	 
		  if(!julia) {
			 cx = ax;
			 cy = ay;
		  } else {
			 cx = julia_x;
			 cy = julia_y;
		  }	    
		  for(i = 0; i < maxiter; i++) { // inner loop of all inner loops !!!
			 qx = ax*ax;
			 qy = ay*ay;
			 if(qx+qy >= 4.0) break;
			 t = qx-qy + cx;
			 ay = 2*ax*ay + cy;
			 ax = t; 	    
		  }		
		  iter += i;
		  inf[xx + yy*infwidth] = i;
		}
	 }
  }  
  return iter;
}


FractWid::FractWid(QWidget *parent, bool julia): QWidget(parent), isdirty(true), pix(0),
phase(0), simple_interlace(16),
center_x(0.0), center_y(0.0), zoom(40.0), rotate(20.0), 
julia_x(0.0), julia_y(0.0), julia(julia),
cross(false), maxiter(5), guessing(true), guessdeb(false), altcol(false), guess_width(16),
inflen(0), infwidth(0), infheight(0), inf(0), infoff(0), fastasm(false)
{
   // init colors 
   numcol = 6 * 20;
   col = new QColor[numcol];
   int col_w = numcol/6;
   int col_w1 = col_w - 1;
   for(int i=0; i<numcol; i++) {
      int cup=(i%col_w)*(255/col_w1);
      int cdn=(col_w1-(i%col_w))*(255/col_w1);
      switch(i/col_w) {
       case 0: 
	 col[i].setRgb(255,cup,0  );
	 break;
       case 1: 
	 col[i].setRgb(cdn,255,0  );
	 break;
       case 2: 
	 col[i].setRgb(0  ,255,cup);
	 break;
       case 3: 
	 col[i].setRgb(0  ,cdn,255);
	 break;
       case 4: 
	 col[i].setRgb(cup,0  ,255);
	 break;
       case 5: 
	 col[i].setRgb(255,0  ,cdn);
	 break;
      }
   }
   lake_col.setRgb(0,0,32);
   guess_col.setRgb(200,200,200);
   
   //
	inf = new int[qApp->desktop()->width() * qApp->desktop()->height() * 2];
#ifdef NOPIX
   drawable = this;
#else     
   drawable = pix = new QPixmap(10,10);
#endif
   updateRotate();
   full_update();
}


FractWid::~FractWid() {
  delete col;
  delete inf;
  if(pix) delete pix;
}


double FractWid::calc_simple(int phase, int phaselen, int maxiter, 
									  QRect r, QPaintDevice *pd,
									  double c_x, double c_y, 
									  double dxx, double dxy, double dyx, double dyy,
									  bool julia, double julia_x, double julia_y) {
  QPainter p(pd);
  int x, y, i, old_i, old_x;
  double iter = 0.0;

  if(fastasm) {
    iter = calc_asm(inf, r.width(), r.x(), r.y() + phase, r.width(), r.height(), 
		    1, phaselen,
		    maxiter, c_x, c_y, dxx, dxy, dyx, dyy, 
		    julia, julia_x, julia_y, true);
    for(y = r.y() + phase; y < r.y() + r.height(); y += phaselen) {	 
      for(x = r.x(); x < r.x() + r.width(); x++) {
	i = inf[(x-r.x()) + (y-r.y()-phase)*r.width()];
	if(x==r.x()) {
	  old_i = i;
	  old_x = x;
	}
	if(old_i != i) {
	  p.setPen(iter2col(old_i));
	  if((x - old_x) == 1) {
	    if(phase==phaselen-1) p.drawLine(old_x, y, old_x, y-(phaselen-1));
	       else p.drawPoint(old_x, y);
	  } else {
	    if(phase==phaselen-1) p.fillRect(old_x, y-(phaselen-1), x-old_x, phaselen, iter2col(old_i));
	    else p.drawLine(old_x, y, x-1, y);
	  }
	  old_i = i;
	  old_x = x;
	}
      }
      p.setPen(iter2col(old_i));
      if((x - old_x) == 1) {
	if(phase==phaselen-1) p.drawLine(old_x, y, old_x, y-(phaselen-1));
	else p.drawPoint(old_x, y);
      } else {
	if(phase==phaselen-1) p.fillRect(old_x, y-(phaselen-1), x-old_x, phaselen, iter2col(old_i));
	else p.drawLine(old_x, y, x-1, y);
      }
    }
  } else {
    double ax, ay, cx, cy, qx, qy, t;
    
    for(y = r.y() + phase; y < r.y() + r.height(); y += phaselen) {	 
      for(x = r.x(); x < r.x() + r.width(); x++) {
	ax = c_x + x*dxx + y*dyx;
	ay = c_y + x*dxy + y*dyy;	 
	if(!julia) {
	  cx = ax;
	  cy = ay;
	} else {
	  cx = julia_x;
	  cy = julia_y;
	}	    
	for(i = 0; i < maxiter; i++) { // inner loop of all inner loops !!!
	  qx = ax*ax;
	  qy = ay*ay;
	  if(qx+qy >= 4.0) break;
	  t = qx-qy + cx;
	  ay = 2*ax*ay + cy;
	  ax = t; 	    
	}
	if(x==r.x()) {
	  old_i = i;
	  old_x = x;
	}
	if(old_i != i) {
	  p.setPen(iter2col(old_i));
	  if((x - old_x) == 1) {
	    if(phase==phaselen-1) p.drawLine(old_x, y, old_x, y-(phaselen-1));
	    else p.drawPoint(old_x, y);
	  } else {
	    if(phase==phaselen-1) p.fillRect(old_x, y-(phaselen-1), x-old_x, phaselen, iter2col(old_i));
	    else p.drawLine(old_x, y, x-1, y);
	  }
	  old_i = i;
	  old_x = x;
	}
	iter += i;
      }
      p.setPen(iter2col(old_i));
      if((x - old_x) == 1) {
	if(phase==phaselen-1) p.drawLine(old_x, y, old_x, y-(phaselen-1));
	else p.drawPoint(old_x, y);
      } else {
	if(phase==phaselen-1) p.fillRect(old_x, y-(phaselen-1), x-old_x, phaselen, iter2col(old_i));
	else p.drawLine(old_x, y, x-1, y);
      }
    }	
  }
  return iter;
}


void FractWid::full_update() {
  isdirty = true;
  repaint(false);
}


double FractWid::calc_init_grid(int phase, int maxiter, 
				QRect r, QPaintDevice *pd,
				double c_x, double c_y, 
				double dxx, double dxy, double dyx, double dyy,
				bool julia, double julia_x, double julia_y) {
   double iter=0;
   int x, y, i, old_i, old_x, pha;
   int w = r.width();
   int h = r.height();
   QPainter p(pd);

   // init guessdebug
   if(guessdeb) {
      p.fillRect(r, guess_col);
   }
   
   // init inf
   infwidth  = w + 4*phase;
   infheight = h + 4*phase;
   infoff = infwidth*phase + phase - r.x() - (r.y()*infwidth);
   memset(inf, -1, infwidth*infheight*sizeof(int)); // invalidate all
   
   // calc initial grid
   if(fastasm) {
     iter = calc_asm(inf + infwidth*phase + phase, infwidth, 
		     r.x(), r.y(), r.width(), r.height(), 
		     phase, phase,
		     maxiter, c_x, c_y, dxx, dxy, dyx, dyy, 
		     julia, julia_x, julia_y, false);  
     for(y = r.y(); y < r.y() + h; y += phase) {
       if(y+phase > r.y()+h) pha = (r.y()+h)-y;
       else pha = phase;
       for(x = r.x(); x < r.x() + w; x += phase) {
	 i = info(x,y);
	 if(x==r.x()) {
	   old_i = i;
	   old_x = x;
	 }
	 if(guessdeb) {
	   p.setPen(iter2col(i));
	   p.drawPoint(x,y);
	 } else {
	   if(old_i != i) {
	     p.fillRect(old_x, y, x-old_x, pha, iter2col(old_i));
	     old_i = i;
	     old_x = x;
	   }
	 }
       }
       if(!guessdeb) p.fillRect(old_x, y, (r.x()+w) - old_x, pha, iter2col(old_i));
     }
     
   } else {
     
     double ax, ay, cx, cy, qx, qy, t, iter=0;
     
     for(y = r.y(); y < r.y() + h; y += phase) {
       if(y+phase > r.y()+h) pha = (r.y()+h)-y;
       else pha = phase;
       for(x = r.x(); x < r.x() + w; x += phase) {
	 ax = c_x + x*dxx + y*dyx;
	 ay = c_y + x*dxy + y*dyy;	 
	 if(!julia) {
	   cx = ax;
	   cy = ay;
	 } else {
	   cx = julia_x;
	   cy = julia_y;
	 }	    
	 for(i = 0; i < maxiter; i++) { // inner loop of all inner loops !!!
	   qx = ax*ax;
	   qy = ay*ay;
	   if(qx+qy >= 4.0) break;
	   t = qx-qy + cx;
	   ay = 2*ax*ay + cy;
	   ax = t; 	    
	 }
	 info(x,y) = i;
	 if(x==r.x()) {
	   old_i = i;
	   old_x = x;
	 }
	 if(guessdeb) {
	    p.setPen(iter2col(i));
	    p.drawPoint(x,y);
	 } else {
	   if(old_i != i) {
	     p.fillRect(old_x, y, x-old_x, pha, iter2col(old_i));
	     old_i = i;
	     old_x = x;
	   }
	 }
	 iter += i;
       }
      if(!guessdeb) p.fillRect(old_x, y, (r.x()+w)-old_x, pha, iter2col(old_i));
     }	 
   }

   return iter;   
}


double FractWid::calc_next_grid(int phase, int maxiter, 
										  QRect r, QPaintDevice *pd,
										  double c_x, double c_y, 
										  double dxx, double dxy, double dyx, double dyy,
										  bool julia, double julia_x, double julia_y) {
  double iter=0;
  int x, y, i, old_i, old_x, pha;
  int w = r.width();
  int h = r.height();
  QPainter p(pd);

#if 1 
  // guess some pixel
  int p2 = 2*phase;
  int p4 = 4*phase;
  for(y = r.y(); y < r.y() + h; y += p2) {
    for(x = r.x(); x < r.x() + w; x += p2) {
      if(info(x, y) < 0) {
	i = info(x, y);
	if((info(x+p4, y+p4)==i)&&
	   (info(x+p4, y+p2)==i)&&
	   (info(x+p4, y   )==i)&&
	   (info(x+p4, y-p2)==i)&&
	   
	   (info(x+p2, y+p4)==i)&&
	   (info(x+p2, y+p2)==i)&&
	   (info(x+p2, y   )==i)&&
	   (info(x+p2, y-p2)==i)&&
	   
	   (info(x   , y+p4)==i)&&
	   (info(x   , y+p2)==i)&&
	   (info(x   , y-p2)==i)&&
	   
	   (info(x-p2, y+p4)==i)&&
	   (info(x-p2, y+p2)==i)&&
	   (info(x-p2, y   )==i)&&
	   (info(x-p2, y-p2)==i)) {
	  info(x+phase, y) = i;
	  info(x+phase, y+phase) = i;
	  info(x+p2, y+phase) = i;
	  info(x+phase, y+p2) = i;
	  info(x, y+phase) = i;
	}	    
      }
    }
  }
#endif
  
  // calc next grid
  if(fastasm) {
    iter = calc_asm(inf + infwidth*guess_width + guess_width, infwidth, 
		    r.x(), r.y(), r.width(), r.height(), 
		    phase, phase,
		    maxiter, c_x, c_y, dxx, dxy, dyx, dyy, 
		    julia, julia_x, julia_y, false);  
    for(y = r.y(); y < r.y() + h; y += phase) {
      if(y+phase > r.y()+h) pha = (r.y()+h)-y;
      else pha = phase;
      for(x = r.x(); x < r.x() + w; x += phase) {
	i = info(x,y);
	if(x==r.x()) {
	  old_i = i;
	  old_x = x;
	}
	if(old_i != i) {
	  if(phase==1) {
	    p.setPen(iter2col(old_i));
	    if((x-old_x)==1)
	      p.drawPoint(old_x, y);
	    else
	      p.drawLine(old_x, y, x-1, y);
	  } else p.fillRect(old_x, y, x-old_x, pha, iter2col(old_i));
	  old_i = i;
	  old_x = x;
	}
      }
      if(phase==1) {
	p.setPen(iter2col(old_i));
	if(((r.x()+w)-old_x)==1)
	  p.drawPoint(old_x, y);
	else
	  p.drawLine(old_x, y, r.x()+w-1, y);
      } else p.fillRect(old_x, y, (r.x()+w)-old_x, pha, iter2col(old_i));
    }
    
  } else {
    
    double ax, ay, cx, cy, qx, qy, t, iter=0;
    
    for(y = r.y(); y < r.y() + h; y += phase) {
      if(y+phase > r.y()+h) pha = (r.y()+h)-y;
      else pha = phase;
      for(x = r.x(); x < r.x() + w; x += phase) {
	if(info(x,y)<0) { // dirty pixel
	  ax = c_x + x*dxx + y*dyx;
	  ay = c_y + x*dxy + y*dyy;	 
	  if(!julia) {
	    cx = ax;
	    cy = ay;
	  } else {
	    cx = julia_x;
	    cy = julia_y;
	  }	    
	  for(i = 0; i < maxiter; i++) { // inner loop of all inner loops !!!
	    qx = ax*ax;
	    qy = ay*ay;
	    if(qx+qy >= 4.0) break;
	    t = qx-qy + cx;
	    ay = 2*ax*ay + cy;
	    ax = t; 	    
	  }
	  info(x,y) = i;
	  iter += i;
	  if(guessdeb) {
	    p.setPen(iter2col(i));
	    p.drawPoint(x,y);
	  } 
	} else i = info(x,y);
	if(x==r.x()) {
	  old_i = i;
	  old_x = x;
	}
	if(!guessdeb) {
	  if(old_i != i) {
	    if(phase==1) {
	      p.setPen(iter2col(old_i));
	      if((x-old_x)==1)
		p.drawPoint(old_x, y);
	      else
		p.drawLine(old_x, y, x-1, y);
	    } else p.fillRect(old_x, y, x-old_x, pha, iter2col(old_i));
	    old_i = i;
	    old_x = x;
	  }
	}
      }
      if(!guessdeb) {
	if(phase==1) {
	  p.setPen(iter2col(old_i));
	  if(((r.x()+w)-old_x)==1)
	    p.drawPoint(old_x, y);
	  else
	    p.drawLine(old_x, y, r.x()+w-1, y);
	} else p.fillRect(old_x, y, (r.x()+w)-old_x, pha, iter2col(old_i));
      }
    }	 
  }
  
  return iter;   
}


void FractWid::calc_sync_all_phases(QRect r) {
   int pha = guess_width;
   
   if(guessing) {
      calc_init_grid(pha, maxiter, r, drawable,
		     center_x-(width()/2)*rot_dxx-(height()/2)*rot_dyx, 
		     center_y-(width()/2)*rot_dxy-(height()/2)*rot_dyy,	       
		     rot_dxx, rot_dxy, rot_dyx, rot_dyy,
		     julia, julia_x, julia_y);
      for(pha/=2;pha ;pha/=2) {
	 calc_next_grid(pha, maxiter, r, drawable,
			center_x-(width()/2)*rot_dxx-(height()/2)*rot_dyx, 
			center_y-(width()/2)*rot_dxy-(height()/2)*rot_dyy,	       
			rot_dxx, rot_dxy, rot_dyx, rot_dyy,
			julia, julia_x, julia_y);	 
      }
   } else {
      calc_simple(0, 1, maxiter, r, drawable,
		  center_x-(width()/2)*rot_dxx-(height()/2)*rot_dyx, 
		  center_y-(width()/2)*rot_dxy-(height()/2)*rot_dyy,	       
		  rot_dxx, rot_dxy, rot_dyx, rot_dyy,
		  julia, julia_x, julia_y);
   }
}
 

void FractWid::calc_phase() {
   if(phase<=0) return;
   
   // calc
   if(guessing) {
      if(phase==guess_width) {
	 calc_init_grid(phase, maxiter, cur_rect, drawable,
			center_x-(width()/2)*rot_dxx-(height()/2)*rot_dyx, 
			center_y-(width()/2)*rot_dxy-(height()/2)*rot_dyy,	       
			rot_dxx, rot_dxy, rot_dyx, rot_dyy,
			julia, julia_x, julia_y);
      } else {
	 calc_next_grid(phase, maxiter, cur_rect, drawable,
			center_x-(width()/2)*rot_dxx-(height()/2)*rot_dyx, 
			center_y-(width()/2)*rot_dxy-(height()/2)*rot_dyy,	       
			rot_dxx, rot_dxy, rot_dyx, rot_dyy,
			julia, julia_x, julia_y);	 
      }
      phase /= 2;
   } else {
//      printf("pre_phase(%d)\n", phase);
      phase--;
      calc_simple(phase, simple_interlace, maxiter, cur_rect, drawable,
		  center_x-(width()/2)*rot_dxx-(height()/2)*rot_dyx, 
		  center_y-(width()/2)*rot_dxy-(height()/2)*rot_dyy,	       
		  rot_dxx, rot_dxy, rot_dyx, rot_dyy,
		  julia, julia_x, julia_y);
   }

   // end
   if(cross) drawCross(drawable, width()/2, height()/2);
#ifndef NOPIX
   bitBlt(this, cur_rect.topLeft(), pix, cur_rect);
#endif
   qApp->processEvents();
}


void FractWid::updateRotate() {
   rot_dxx=+cos(rotate*M_PI/180.0)/zoom;
   rot_dyx=+sin(rotate*M_PI/180.0)/zoom;
   rot_dyy=+cos(rotate*M_PI/180.0)/zoom;
   rot_dxy=-sin(rotate*M_PI/180.0)/zoom;
}


void FractWid::calc_next_phase() {
   if(phase<=0) return;   
   calc_phase();   
   if(phase>0)
     QTimer::singleShot(0, this, SLOT(calc_next_phase()));
}


void FractWid::init_calc(QRect r) {
   if(phase>0) {
      cur_rect = cur_rect.unite(r);
   } else cur_rect = r;
   if(guessing)
     phase = guess_width;
   else
     phase = simple_interlace;
   QTimer::singleShot(0, this, SLOT(calc_next_phase()));
}


void FractWid::setMaxIter(int max) {
   if(maxiter!=max) {
      maxiter = max;
      full_update();
   }
}


void FractWid::setGuessing(bool on) {
   if(on!=guessing) {
      guessing = on;
      full_update();
   }
}


void FractWid::setGuessDebug(bool on) {
   if(on!=guessdeb) {
      guessdeb = on;
      full_update();
   }
}


void FractWid::setAltColors(bool on) {
   if(on!=altcol) {
      altcol = on;
      full_update();
   }
}


void FractWid::setAsm(bool on) {
   if(on!=fastasm) {
      fastasm = on;
      full_update();
   }
}


void FractWid::setCrosshairs(bool on) {
   if(on!=cross) {
      cross = on;
      full_update();
   }
}


void FractWid::setJuliaCoords(double x, double y) {
   if(julia && ((x!=julia_x) || (y!=julia_y))) {
      julia_x = x;
      julia_y = y;
      full_update();
   }
}


void FractWid::setPosRotZoom(double cen_x, double cen_y, double rot, double z_oom) {
   if((center_x!=cen_x)||
      (center_y!=cen_y)||
      (rotate!=rot)||
      (zoom!=z_oom)) {
      
      // full update	    
      center_x = cen_x;
      center_y = cen_y;
      rotate = rot;
      zoom = z_oom;
      updateRotate();
      full_update();
      emit posRotZoomChanged(center_x, center_y, rotate, zoom);
      emit centerChanged(center_x, center_y);
   }      
}


void FractWid::forcePosRotZoomChanged() {
   emit posRotZoomChanged(center_x, center_y, rotate, zoom);
}


void FractWid::paintEvent(QPaintEvent *p) {
#ifndef NOPIX
   if(isdirty) {
      if(pix==0) pix = new QPixmap(rect().size());
      if(rect()!=pix->rect()) pix->resize(size());
      init_calc(p->rect());
   }
   bitBlt(this, p->rect().topLeft(), pix, p->rect());
#else
   init_calc(p->rect());
#endif
}


const QColor& FractWid::iter2col(int iter) {
   if(iter==maxiter) return lake_col;
   if(!altcol) return col[iter%numcol];
   else {
      if(iter&1) return col[numcol - (iter%numcol)];
      else return col[iter%numcol];
   }
}


void FractWid::drawCross(QPaintDevice *pd, int cx, int cy) {
   QPainter p(pd);
   p.setPen(black);
   p.drawPoint(cx-2, cy);
   p.drawPoint(cx-4, cy);
   p.drawPoint(cx-6, cy);
   p.drawPoint(cx+2, cy);
   p.drawPoint(cx+4, cy);
   p.drawPoint(cx+6, cy);
   p.drawPoint(cx, cy-2);
   p.drawPoint(cx, cy-4);
   p.drawPoint(cx, cy-6);
   p.drawPoint(cx, cy+2);
   p.drawPoint(cx, cy+4);
   p.drawPoint(cx, cy+6);
   p.setPen(white);
   p.drawPoint(cx-3, cy);
   p.drawPoint(cx-5, cy);
   p.drawPoint(cx-7, cy);
   p.drawPoint(cx+3, cy);
   p.drawPoint(cx+5, cy);
   p.drawPoint(cx+7, cy);
   p.drawPoint(cx, cy-3);
   p.drawPoint(cx, cy-5);
   p.drawPoint(cx, cy-7);
   p.drawPoint(cx, cy+3);
   p.drawPoint(cx, cy+5);
   p.drawPoint(cx, cy+7);
}


void FractWid::resizeEvent(QResizeEvent *) {
   isdirty = true;
}


void FractWid::mousePressEvent(QMouseEvent *e) {
   // start action
   from_x = e->pos().x();
   from_y = e->pos().y();
   switch(e->button()) {
    case LeftButton:
      mousemode = DRAG;
      break;
   
    case RightButton:
      mousemode = ROTATE;
      break;

    case MidButton:
      emit midButtonPressed(e->pos());
      break;
      
    default:
//      mousemode = NONE;
      break;
   }   
}


void FractWid::mouseReleaseEvent(QMouseEvent *e) {
   if(mousemode!=NONE) {
      mousemove(e->pos().x() - from_x, e->pos().y() - from_y);
      mousemode = NONE;
   }
}


void FractWid::mouseMoveEvent(QMouseEvent *e) {
   if(mousemode!=NONE) {
      mousemove(e->pos().x() - from_x, e->pos().y() - from_y);
   }
}


void FractWid::mousemove(int dx, int dy) {
   // move
   int cx = width()/2;
   int cy = height()/2;
   int w = width();
   int h = height();      
   switch(mousemode) {
    case DRAG:
      center_x -= dx*rot_dxx + dy*rot_dyx;
      center_y -= dx*rot_dxy + dy*rot_dyy;
      emit centerChanged(center_x, center_y);
#if 1
      bitBlt(this, dx, dy, this, 0, 0, -1, -1);
      if(dx>0) calc_sync_all_phases(QRect(0, 0, dx, h));
      if(dx<0) calc_sync_all_phases(QRect(w+dx, 0, -dx, h));
      if(dy>0) calc_sync_all_phases(QRect(0, 0, w, dy));
      if(dy<0) calc_sync_all_phases(QRect(0, h+dy, w, -dy));
#else 
      full_update();
#endif
      break;
      
    case ROTATE:
      zoom *= radius(cx, cy, from_x+dx, from_y+dy)/radius(cx, cy, from_x, from_y);
      rotate -= argphi(cx, cy, from_x+dx, from_y+dy)-argphi(cx, cy, from_x, from_y);
      if(rotate<0.0)    rotate += 360.0;
      if(rotate>=360.0) rotate -= 360.0;
      updateRotate();
      emit posRotZoomChanged(center_x, center_y, rotate, zoom);
      full_update();
      break;
      
    default:
      break;
   }
   
   // end
   from_x += dx;
   from_y += dy;
}


double FractWid::radius(int cx, int cy, int x, int y) const {
   double xx = x - cx;
   double yy = cy - y;
   return sqrt(xx*xx+yy*yy);
}


double FractWid::argphi(int cx, int cy, int x, int y) const {
   double xx = x - cx;
   double yy = cy - y;
   if(xx!=0.0) {
      double p = atan(yy/fabs(xx))*180.0/M_PI;
      if(xx<0.0) p=180.0-p;
      return p;
   } else {
      if(yy>=0.0) return 90.0;
      else return 270.0;
   }
}
