/*****************************************************************************
 * Name:            aeropov2.inc
 * Author:	    Hartmut Keller
 * Last Change:     12. Apr 94
 * AERO version:    V1.5
 * 
 * Description:
 * ------------
 * This file will automatically be included in all POV-Ray V2.x scene files 
 * which are created by AERO. So don't forget to set the POV-Ray options in
 * a way that this file can be found by the POV-Ray Raytracer (namely "+L" 
 * option).
 *
 * Here, the camera, light sources and all basic objects (including links) are
 * defined. They have default sizes and positions and will be scaled, rotated
 * and translated within the scene files according to the AERO objects'
 * data. Also, the textures of links and some background effects are defined
 * in this file and the materials textures definition file is included.
 *
 * You may change some of the parameters here as indicated (e.g. the material
 * file name). But be careful or you'll get wrong looking results.
 *
 *
 * AERO objects parameters (DO NEVER CHANGE!)
 * ------------------------------------------
 *    box:        <-0.5, -0.5, -0.5>, <0.5, 0.5, 0.5>
 *    cylinder:   <0.0, 0.0, -0.5>, <0.0, 0.0, 0.5>, radius 1.0
 *    sphere:     <0.0, 0.0, 0.0>, radius 1.0
 *    plane:      <0.0, 1.0, 0.0>, 0.0
 *
 * AERO links parameters (radius may be changed, but this is not recommended)
 * --------------------------------------------------------------------------
 *    rod:        <0.0, 0.0, 0.0>, <0.0, 0.0, 1.0>, radius 0.01
 *    damper:     <0.0, 0.0, 0.0>, <0.0, 0.0, 1.0>, radius 0.01 to 0.045
 *    springpart: (spring assembled from small parts) (radius 0.05)
 *    joint:      <0.0, 0.0, 0.0>, <0.0, 0.0, 1.0>, radius 0.001
 * 
 *****************************************************************************/


/*---------------------------------------------------------------------------*/


/***************************************************************************
 * Include the material file. This name may be changed freely as you like.
 * You may have to set a "+L" POV-Ray option for POV-Ray to find this file.
 ***************************************************************************/

#include "aeromat2.inc"


/*---------------------------------------------------------------------------*/


/***********************************************************
 ***********************************************************
 ***							 ***
 ***  Definition of Camera and Light Sources  	 	 ***
 ***  ======================================		 ***
 ***  You can change the aspect ratio of the images in   ***
 ***  the camera definition.  You may also add further   ***
 ***  light sources or use  different  types of lights   ***
 ***  like area lights or spot lights.			 ***
 ***							 ***
 ***********************************************************
 ***********************************************************/


/******************************************************************************
 * CAMERA:
 * To be as much independent of AERO as possible even the camera is defined
 * here. AERO declares 5 identifiers in conjunction with camera (and light
 * sources). These are:
 *   "aerocameraposition": position of the camera (vector)
 *   "aerocamerax"	 : right hand side direction (vector, length 1)
 *   "aerocameray"       : up direction (vector, length 1)
 *   "aerocameraz"       : front, viewing direction (vector, length 1)
 *   "aerocamerazoom"	 : zoom factor of the object lens (float value)
 * You can use these identifiers also to place any object relative to the
 * camera. This is usually used when defining light sources mounted on the
 * camera.
 * The zoom factor is calculated to get the same segment of the window
 * raytraced as shown in the AERO window at default size. Also the usual
 * aspect ratio of TV screens is 4:3. The aspect ratio should be seen in
 * conjunction with the image size at which the frames are rendered. If you
 * use an image size 4:3 (width:height, e.g. 640x480 pixels), you should also
 * use 4:3 here. If you use another size, also change the fraction value here.
 *****************************************************************************/

#declare aeroaspectratio = 4/3			/* aspect ratio 4:3 */

camera
{
    location  aerocameraposition
    direction aerocameraz * aerocamerazoom	/* zoom goes here */
    up        aerocameray
    right     aerocamerax * aeroaspectratio
}



/******************************************************************************
 * LIGHT_SOURCE:
 * There are no light sources defined in the frame scene files created by
 * AERO. You can define any number of fixed light sources or, using the
 * declared identifiers "aerocameraposition", "aerocamerax", "aerocameray" and
 * "aerocameraz", you can position them relative to the camera. In the latter
 * way they move along with the camera.
 * By default there are two light sources positioned left and above of the
 * camera. They are in one and the same place to enhance light intensity. As
 * any lights are actually defined here, you can change their parameters as
 * you like.
 *****************************************************************************/


light_source				/* LIGHT_SOURCE #1 */
{
    aerocameraposition
	+ (aerocamerax * -3)		/* 3m left */
	+ (aerocameray * 5)		/* 5m above */
	+ (aerocameraz * 0)		/* 0m in front of the camera */
    color red 1.0 green 1.0 blue 1.0	/* white light, may be changed */
}					/* add other light features here */

light_source				/* LIGHT_SOURCE #2 */
{
    aerocameraposition
	+ (aerocamerax * -3)		/* 3m left */
	+ (aerocameray * 5)		/* 5m above */
	+ (aerocameraz * 0)		/* 0m in front of the camera */
    color red 1.0 green 1.0 blue 1.0	/* white light, may be changed */
}					/* add other light features here */


/*---------------------------------------------------------------------------*/


/***********************************************************
 ***********************************************************
 ***							 ***
 ***  Definition of Background and Effects  	 	 ***
 ***  ====================================		 ***
 ***  Here, various modifications can be done to achieve ***
 ***  different effects. You have to remove the comment	 ***
 ***  signs to activate any of these effects.            ***
 ***							 ***
 ***********************************************************
 ***********************************************************/



/******************************************************************************
 * FOG:
 * Subtle use of fog sometimes gives a more realistic impression, especially
 * when using sophisticated light effects. But remember: fog slows down
 * rendering speed and usually it's not needed at all. So by default it's
 * switched off. If you want to use fog, remove the comment signs and edit the
 * fog to your own taste.
 *****************************************************************************/

/*
fog 
{ 
    color red 0.7 green 0.7 blue 0.7		/* gray fog */
    distance 200				/* 100% fog in 200m distance */
}
*/


/******************************************************************************
 * BACKGROUND:
 * Here you can specify your own background color. It is used on those pixels
 * not showing any object of the scene. But be careful: when you use bright
 * colors this often looks flat as there is no shading.  If you don't use any
 * background color, POV-Ray uses black as default. The use of a background
 * color does not affect rendering speed.
 *****************************************************************************/

/*
background {color red 0.01 green 0.01 blue 0.1}  /* very dark blue */
*/


/******************************************************************************
 * BACKGROUND SKY SPHERE:
 * This is a nice background. It shows a sky with moving clouds (according to
 * the frame number of the sequence, aeroframe). This is only useful when you
 * have a plane serving as ground and no other sky elements (e.g. a vertical
 * plane in the background). You may modify the amount of movement and the
 * moving direction. The sky is realized by an enormous sphere surrounding the
 * whole scene including the camera. The sphere has a radius of 20000m, but
 * the clear height above ground is only 3000m.
 *****************************************************************************/

/*
#declare moveamount = 0.01	/* 0.01 deg per frame */
#declare moveoffset = 100	/* 100 deg initial offset */
#declare diagonal   = 30.0	/* 30 deg deviation from straight forward */

sphere
{ 
    <0.0, 0.0, 0.0>, 20000
    texture
    {
      	pigment
      	{
	    bozo			/* cloud texture */
	    turbulence 0.6
	    color_map 
	    {
	        [0.5   color red 0.25 green 0.25 blue 0.7]
                [0.6   color red 0.65  green 0.65  blue 0.65]
                [1.0   color red 0.55  green 0.55  blue 0.55]
            }
	    scale 1150
	}
	finish
	{
	    ambient 0.8
	}
        rotate <0.0, diagonal, 0.0>
	rotate <moveamount * aeroframe + moveoffset, 0.0, 0.0>
	translate <0.0, -500, 0.0>
    }
    translate <0.0, -17000, 0.0>
}
*/


/******************************************************************************
 * ARBITRARY BACKGROUND:
 * Here you may insert a whole POV-Ray scene to create an arbitrary
 * background. The best way to do this is to include an own file.
 *****************************************************************************/

/*
#include "backgrnd.inc"		/* Name may be chosen freely */
*/


/*---------------------------------------------------------------------------*/


/******************************************************************
 ******************************************************************
 ***								***
 ***  Definition of Basic Shapes				***
 ***  ==========================				***
 ***  Never change sizes, orientations, or positions.  You may  ***
 ***  change the shape itself, e.g. to add a texture or create  ***
 ***  a hole.							***
 ***								***
 ******************************************************************
 ******************************************************************/



/***************************************************************
 * CUBOID:
 * Unit edge length 1, position of center: <0, 0, 0>
 ***************************************************************/

#declare aerobox      = box      { <-.5, -.5, -.5>, <.5, .5, .5> }


/***************************************************************
 * CYLINDER:
 * Unit length 1, unit radius 1, position of center: <0, 0, 0>
 ***************************************************************/

#declare aerocylinder = cylinder { <0.0, 0.0, -0.5>, <0.0, 0.0, 0.5>, 1.0 }


/***************************************************************
 * SPHERE:
 * Unit radius 1, position of center: <0, 0, 0>
 ***************************************************************/

#declare aerosphere   = sphere   { <0.0, 0.0, 0.0>, 1.0 }


/***************************************************************
 * PLANE:
 * Parallel to x-z-plane (normal <0, 1, 0>), position <0, 0, 0>
 ***************************************************************/

#declare aeroplane    = plane    { <0.0, 1.0, 0.0>, 0.0 }


/*---------------------------------------------------------------------------*/


/******************************************************************
 ******************************************************************
 ***								***
 ***  Definition of Basic Link Shapes				***
 ***  ===============================				***
 ***  Never change sizes, orientations, or positions.  You may  ***
 ***  change the radius of the links (not recommended).		***
 ***								***
 ******************************************************************
 ******************************************************************/



/**************************************************************************
 * ROD:
 * Unit length 1, radius: 0.01, on z-axis from <0, 0, 0>
 *************************************************************************/

#declare aerorod = cylinder { <0.0, 0.0, 0.0>, <0.0, 0.0, 1.0>, 0.01 }



/**************************************************************************
 * DAMPER:
 * Unit length 1, radius: 0.01 to 0.45, on z-axis from <0, 0, 0>
 *************************************************************************/

#declare aerodamper =
union
{
    object { aerorod }
    cylinder { <0.0, 0.0, 0.3>, <0.0, 0.0, 0.7>, 0.04 }
    difference
    {
	cylinder { <0.0, 0.0, 0.5>, <0.0, 0.0, 0.7>, 0.045 }
	cylinder { <0.0, 0.0, 0.5>, <0.0, 0.0, 0.6>, 0.043 }
    }
}



/**************************************************************************
 * JOINT:
 * Unit length 1, radius 0.001, on z-axis from <0, 0, 0>.
 * As a joint usually just connects 2 points together, it normally can't be
 * seen. But due to AERO these points may be different. To show their
 * connectivity despite of this, a joint can be shown as (#1) a thin rod or
 * (#2) as a thin rod with small spheres at the ends. Select one of these and
 * put the other into a POV-Ray comment.
 **************************************************************************/

/* Joint #1: a thin rod */

#declare aerojoint = cylinder { <0.0, 0.0, 0.0>, <0.0, 0.0, 1.0>, 0.001 }


/* Joint #2: a thin rod with small spheres at the ends */
/*
#declare aerojoint =
union
{
    cylinder { <0.0, 0.0, 0.0>, <0.0, 0.0, 1.0>, 0.001 }
    sphere { <0.0 0.0 0.0> 0.2 } 
    sphere { <0.0 0.0 1.0> 0.2 } 
}
*/



/**************************************************************************
 * SPRING:
 * A spring is assembled from many small cylinders. Use same radius as in
 * aerorod. This part: unit length 1, radius 0.01, lying on y-axis from
 * <0, -0.5, 0> to <0, 0.5, 0>
 * As a spring is very complicated we use a bounding cylinder for speeding up
 * rendering: unit length 1, radius 1, lying on z-axis from <0, 0, 0>
 *************************************************************************/

#declare aerospringpart  = cylinder { <0.0, -0.5, 0.0>, <0.0, 0.5, 0.0>, 0.01 }
#declare aerospringbound = cylinder { <0.0, 0.0, 0.0> <0.0, 0.0, 1.0>, 1.0 }


/*---------------------------------------------------------------------------*/


/******************************************************************
 ******************************************************************
 ***								***
 ***  Definition of Link Textures				***
 ***  ===========================				***
 ***  These textures may be modified freely.  But bear in mind  ***
 ***  that links may vary in length, and springs are assembled	***
 ***  from small parts.						***
 ***								***
 ******************************************************************
 ******************************************************************/



/**************************************************
 * ROD:
 * Metallic design with a touch of blue-green
 **************************************************/

#declare aerorodtexture =
texture
{
    pigment {color red 0.3 green 0.476555 blue 0.39}
    finish
    {
	metallic
	brilliance 5.0
	phong .3
	ambient .15
    }
}


/**************************************************
 * DAMPER:
 * Metallic design with a touch of red
 **************************************************/

#declare aerodampertexture =
texture
{
    pigment {color red 0.531243 green 0.3 blue 0.3}
    finish
    {
	metallic
	brilliance 5.0
	phong .3
	ambient .15
    }
}


/**************************************************
 * SPRING:
 * Metallic design with a touch of yellow
 **************************************************/

#declare aerospringtexture =
texture
{
    pigment {color red 0.476555 green 0.476555 blue 0.3}
    finish
    {
	ambient .15
	metallic
	brilliance 5.0
	roughness 0.001
	specular 1.0
    }
}


/**************************************************
 * JOINT:
 * Metallic design with a touch of blue
 **************************************************/

#declare aerojointtexture =
texture
{
    pigment {color red 0.33 green 0.33 blue 0.37}
    finish
    {
	ambient .15
	metallic
	phong .3
    }
}
