/* this program is copyright 1995 Sam Trenholme   

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/* program to design flowers to export to a pov file 

   the first flower I will attempt to draw is a daisy 

   this is a program to make petals */

#include <math.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>

/* some preset values. These preset values make a reasonable daisy on
   a machine with 32megs ram+swap, while running X-windows, and I am
   wishing I installed the kswap patches :-) */

#define MIDPOINT (float)0
#define XMAX (float)0
#define XINC (float)(0.25)
#define YINC (float)(1)
#define RIPPLE (float)0.03
#define WIDTH (float)0.03

#define NAME_SIZE 30

#define TRUE 1
#define FALSE 0
#define boolean short /* this really should be standard */
#undef  SMOOTH        /* alas, I have not gotton smooth_triangle to work */

float midpoint=MIDPOINT, xmax=XMAX, xinc=XINC, yinc=YINC;
float ripple=RIPPLE, width=WIDTH;

char petal_name[NAME_SIZE];

boolean smooth(z1,z2,z3)
float z1,z2,z3;
{
#ifdef SMOOTH
if(cos(z1)>SMOOTH || cos(z1)<-SMOOTH || cos(z2)>SMOOTH || cos(z2)<-SMOOTH || 
   cos(z3)>SMOOTH || cos(z3)<-SMOOTH)
     return TRUE;
#endif
return FALSE;
}

/* the function z=f(x,y) determines the height of the petal off of the
   "ground". We also need functions to return the partial derviatives
   with respect to the x and y zxis to obtain the vector normal to any 
   given point of our petal */

float zfunction(x,y)
float x,y;
{
return (cos(x) * ripple + -y * y * (float).01);
}

/* delta y over delta z
   although this equations are unaffected by x, I include it to make
   the source easier to modify */
 
float yslope(x,y)
float x,y;
{
return (y * (float)-.02);
}

/* this is a bit of complication I added. Basiclly, The length varies so we
   can make a function above fairly simple, and then fo this stuff to make
   it more pedal like (make the petal sin(foo) in one axis, a gentle parabola
   in another axis */

float lfunction(y)
float y;
{
if(y<-10 || y>10)
   return 0; /* no imaginary numbers ! */
else
   return sqrt((float)100 - y * y) * width;

}

/* delta x over delta z */

float xslope(x,y)
float x,y;
{
if(0==lfunction(y))
  return 0; /* gotta avoid those divide by 0 errors */

return ((float) -1 * sin(x) * ripple / (float)lfunction(y));
}

/* assumming for x=0 to x_max, we get a point between 
   {+/-}xmax * lvalue / 2 */

float ltransform(l_value, x, mid_point, x_max)
float l_value, x, mid_point, x_max;
{
return (mid_point + (x - (x_max/(float)2)) * l_value);
}

/* given a point on our flower mesh that is a point on a normal grid, i.e.
   a grid of graph paper, perform a transformation so that the point becomes
   a point as would be on the petal of a flower */

/* this is a porcedure that modifies new_x, new_y, new_z */

void grid_point(old_x, old_y, new_x, new_y, new_z)
float old_x, old_y, *new_x, *new_y, *new_z;
{
*new_z = zfunction(old_x, old_y);
*new_x = ltransform(lfunction(old_y), old_x, midpoint, xmax);
*new_y = old_y;
}

/* this is a procedure to determine a vector normal to the point described
   above */

void grid_normal(old_x, old_y, normal_x, normal_y, normal_z)
float old_x, old_y, *normal_x, *normal_y, *normal_z;
{
*normal_z = 1;
*normal_y = yslope(old_x, old_y);
*normal_x = xslope(old_x, old_y);
}
    
main(argc,argv)
int argc;
char **argv;
{
float x,y,xp1,yp1,zp1,xp2,yp2,zp2,xp3,yp3,zp3;
float xn1,yn1,zn1,xn2,yn2,zn2,xn3,yn3,zn3;

int c;

/* command line options:
   p midPoint
   m xMax
   x Xinc
   y Yinc
   s Smooth -- currently unimplemented
   r Ripple
   w Width
   n Name of petal object
   t type of flower [only ds1 supported (daisy type one)
   o output format [onlt p2i supported povray2.2 include]
*/

strcpy(petal_name,"petal");

opterr = 0;
while( (c = getopt(argc, argv, "p:m:x:y:r:w:n:t:o:")) != -1)
   {
   switch (c)
     {
     case 'p':
       {
       midpoint = atof(optarg);
       break;
       }

     case 'm':
       {
       xmax = atof(optarg);
       break;
       }

     case 'x':
       {
       xinc = atof(optarg);
       if(xinc<=0)
         xinc = XINC;
       break;
       }

     case 'y':
       {
       yinc = atof(optarg);
       if(yinc<=0)
         yinc = YINC;
       break;
       }

     case 'r':
       {
       ripple = atof(optarg);
       break;
       }

     case 'w':
       {
       width = atof(optarg);
       if(width<=0)
         width=WIDTH;
       break;
       }

     case 'n':
       { 
       strncpy(petal_name,optarg,NAME_SIZE);
       break;
       }

     case 't':
       {
       if(strcmp("ds1",optarg))
         {
         fprintf(stderr,"Error: Flower type %s unsupported.\n",optarg);
         exit(2);
         }
       break;
       }

     case 'o':
       {
       if(strcmp("p2i",optarg))
         {
         fprintf(stderr,"Error: Output format %s unsupported.\n",optarg);
         exit(3);
         }
       break;
       }
        
     case '?':
       {
       fprintf(stderr,"WARNING: Invalid option %c ignored. Valid options: pmxyrwnot\n", c);
       break;
       }

     }
   }

printf("// this object was rendered with the flower renderer by Sam Trenholme.\n\n");
printf("#max_trace_level 50\n"); /* this one is nasty in POV,
                                    since POV neglects to warn you when
                                    it goes above max_trace_level */
printf("#declare %s = union {\n",petal_name);

for(y = -10; y <= 6 ; y+=yinc)
   for(x = -10; x < 10; x+=xinc)
      {
      grid_point((float)x,(float)y,&xp1,&yp1,&zp1);
      grid_point((float)(x+xinc),(float)y,&xp2,&yp2,&zp2);
      grid_point((float)x,(float)(y+yinc),&xp3,&yp3,&zp3);
#ifdef SMOOTH
      if(smooth(x,x,x+xinc))
         {
         grid_normal((float)x,(float)y,&xn1,&yn1,&zn1);
         grid_normal((float)(x+xinc),(float)y,&xn2,&yn2,&zn2);
         grid_normal((float)x,(float)(y+yinc),&xn3,&yn3,&zn3);
         printf("smooth_triangle {<%f, %f, %f>, <%f, %f, %f>,\n",xp1,yp1,zp1,xn1,yn1,zn1);
         printf("<%f, %f, %f>, <%f, %f, %f>,\n",xp2,yp2,zp2,xn2,yn2,zn2);
         printf("<%f, %f, %f>, <%f, %f, %f>}\n",xp3,yp3,zp3,xn3,yn3,zn3);
         }
       else
#endif /* SMOOTH */
         printf("triangle {<%f,%f,%f>,<%f,%f,%f>,<%f,%f,%f>}\n",xp1,yp1,zp1,xp2,yp2,zp2,xp3,yp3,zp3);
      grid_point((float)(x+xinc),(float)(y+yinc),&xp1,&yp1,&zp1);
#ifdef SMOOTH
      if(smooth(x+XINC,x,x))
         {
         grid_normal((float)(x+xinc),(float)(y+yinc),&xn1,&yn1,&zn1);
         printf("smooth_triangle {<%f, %f, %f>, <%f, %f, %f>,\n",xp1,yp1,zp1,xn1,yn1,zn1);
         printf("<%f, %f, %f>, <%f, %f, %f>,\n",xp2,yp2,zp2,xn2,yn2,zn2);
         printf("<%f, %f, %f>, <%f, %f, %f>}\n",xp3,yp3,zp3,xn3,yn3,zn3);
         }
      else
#endif /* SMOOTH */
         printf("triangle {<%f,%f,%f>,<%f,%f,%f>,<%f,%f,%f>}\n",xp1,yp1,zp1,xp2,yp2,zp2,xp3,yp3,zp3);
      }

printf("}\n");
 
}
