
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: viewer.c,v 1.7 2000/04/28 20:46:34 dr Exp $ 
*/


#include "viewer.h"
#include "zimage.h"

#define DEFAULT_TIMEOUT   1000

static void               viewer_init (Viewer *viewer);
static void               viewer_destroy (GtkWidget *widget, Viewer *viewer);

static gint               viewer_button_press (GtkWidget *widget, GdkEventButton *event,
					       Viewer *viewer);
static void               viewer_menu_next (GtkWidget *widget, Viewer *viewer);
static void               viewer_menu_previous (GtkWidget *widget, Viewer *viewer);
static void               viewer_menu_diaporama (GtkWidget *widget, Viewer *viewer);
static void               viewer_menu_fullscreen (GtkWidget *widget, Viewer *viewer);
static gint               viewer_diaporama (Viewer *viewer);

static GtkItemFactoryEntry viewer_menu_factory [] = {
  { "/_File", NULL, NULL, 0, "<Branch>" },
  { "/_View", NULL, NULL, 0, "<Branch>" },
  { "/View/_Next image", "<control>N", viewer_menu_next, 0, NULL },
  { "/View/_Previous image", "<control>P", viewer_menu_previous, 0, NULL },
  { "/View/sep1", NULL, NULL, 0, "<Separator>" },
  { "/View/_Diaporama", "<control>D", viewer_menu_diaporama, 0, NULL },
  { "/View/sep2", NULL, NULL, 0, "<Separator>" },
  { "/View/_Full screen", NULL, viewer_menu_fullscreen, 0, NULL} 
};
static int viewer_menu_factory_count = sizeof (viewer_menu_factory) / sizeof (GtkItemFactoryEntry);

Viewer *
viewer_new () {
  Viewer *viewer;

  viewer                     = g_new (Viewer, 1);
  viewer->zimage             = zimage_new (NULL);
  viewer->diaporama_timeout  = DEFAULT_TIMEOUT;
  viewer->diaporama_id       = 0;
  viewer->is_fullscreen      = 0;
  viewer->menu               = NULL;

  viewer_init (viewer);

  return viewer;
}

static void
viewer_init (Viewer *viewer) {
  GtkWidget *window;
  GtkWidget *vbox;
  GtkWidget *swindow;
  GtkItemFactory *item_factory;
  GtkAccelGroup *accel;

  accel = gtk_accel_group_new ();

  item_factory = gtk_item_factory_new (GTK_TYPE_MENU, "<viewer>", accel);
  gtk_item_factory_create_items_ac (item_factory, 
				    viewer_menu_factory_count, viewer_menu_factory,
				    viewer, 2);
  viewer->menu = gtk_item_factory_get_widget (item_factory, "<viewer>");

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_signal_connect (GTK_OBJECT(window), "destroy",
		      GTK_SIGNAL_FUNC(viewer_destroy), viewer);
  gtk_window_set_default_size (GTK_WINDOW(window), 640, 480);
  gtk_accel_group_attach (accel, GTK_OBJECT(window));

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER(window), vbox);
  gtk_widget_show (vbox);

  /*
  menubar = gtk_item_factory_get_widget (item_factory, "<viewer>");
  gtk_box_pack_start (GTK_BOX(vbox), menubar, FALSE, TRUE, 0);
  gtk_widget_show (menubar);
  */

  swindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(swindow),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER(vbox), swindow);
  gtk_widget_show (swindow);

  gtk_container_add (GTK_CONTAINER(swindow), viewer->zimage);
  gtk_signal_connect (GTK_OBJECT (viewer->zimage), "button_press_event", 
		      (GtkSignalFunc) viewer_button_press, viewer);
  gtk_widget_show (viewer->zimage);

  gtk_widget_show (window);
}

static void
viewer_destroy (GtkWidget *widget, Viewer *viewer) {
  if (viewer->diaporama_id)
    gtk_timeout_remove (viewer->diaporama_id);

}

void
viewer_set_images (Viewer *viewer, ImageSet *set, Image *image) {

  g_return_if_fail (viewer && set);

  viewer->set = set;
  viewer->current_image = 0;

  viewer_set_image (viewer, image);
}

void
viewer_set_image (Viewer *viewer, Image *image) {
  int index;
  g_return_if_fail (viewer);

  index = image_set_get_index (viewer->set, image);
  g_return_if_fail (index != -1);

  viewer->current_image = index;
  zimage_set_image (ZIMAGE(viewer->zimage), image);
}


static gint               
viewer_button_press (GtkWidget *widget, GdkEventButton *event,
		     Viewer *viewer) {
  
  if (event->button == 3) 
    gtk_menu_popup (GTK_MENU (viewer->menu), NULL, NULL, NULL, NULL,
		    event->button, event->time);

  return FALSE;
}

static void               
viewer_menu_next (GtkWidget *widget, Viewer *viewer) {
  g_return_if_fail (viewer);
  
  viewer->current_image ++;
  if (viewer->current_image == image_set_size (viewer->set))
    viewer->current_image = 0;

  viewer_set_image (viewer, image_set_get_image (viewer->set, viewer->current_image));
}

static void               
viewer_menu_previous (GtkWidget *widget, Viewer *viewer) {
  g_return_if_fail (viewer);
  
  viewer->current_image --;
  if (viewer->current_image < 0 )
    viewer->current_image = image_set_size (viewer->set) - 1;

  viewer_set_image (viewer, image_set_get_image (viewer->set, viewer->current_image));

}

static void               
viewer_menu_diaporama (GtkWidget *widget, Viewer *viewer) {
  g_return_if_fail (viewer);

  if (viewer->diaporama_id) {
    gtk_timeout_remove (viewer->diaporama_id);
    viewer->diaporama_id = 0;
  } else
    viewer->diaporama_id = gtk_timeout_add (viewer->diaporama_timeout, 
					    (GtkFunction) viewer_diaporama, viewer);
}


static void               
viewer_menu_fullscreen (GtkWidget *widget, Viewer *viewer) {
  GtkWidget *toplevel;

  g_return_if_fail (viewer);

  toplevel = gtk_widget_get_toplevel (viewer->zimage);

  if (viewer->is_fullscreen) {
    viewer->is_fullscreen = FALSE;

    gdk_window_set_decorations (toplevel->window, GDK_DECOR_ALL);
    gdk_window_set_functions (toplevel->window, GDK_FUNC_ALL);
    gdk_window_move_resize (toplevel->window, 
			    viewer->spos [0],
			    viewer->spos [1],
			    viewer->spos [2],
			    viewer->spos [3]);
	    
  } else {
    viewer->is_fullscreen = TRUE;

    gdk_window_get_position (toplevel->window,
			     &viewer->spos [0],
			     &viewer->spos [1]);
    gdk_window_get_size (toplevel->window,
			 &viewer->spos [2],
			 &viewer->spos [3]);

    gdk_window_set_decorations (toplevel->window, 0);
    gdk_window_set_functions (toplevel->window, 0);
    gdk_window_move_resize (toplevel->window, 
			    0, 0, 
			    gdk_screen_width (),
			    gdk_screen_height ());
  }
}

static gint
viewer_diaporama (Viewer *viewer) {
  /* timeouts are called outside of the gtk lock */
  gdk_threads_enter ();

  if (zimage_loaded (ZIMAGE(viewer->zimage)))
    viewer_menu_next (NULL, viewer);

  gdk_threads_leave ();

  return TRUE;
}

