
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: image.h,v 1.3 2000/04/28 20:46:34 dr Exp $ 
*/

#ifndef _IMAGE_H_
#define _IMAGE_H_

#include <glib.h>
#include <gdk/gdk.h>


typedef struct _Image             Image;
typedef struct _ImageSet          ImageSet;
typedef struct _ImageCache        ImageCache;
typedef struct _ImagePixmap       ImagePixmap;
typedef struct _ImageTypeInfo     ImageTypeInfo;

typedef enum {
  IMAGE_SET_SORT_BY_NAME,
  IMAGE_SET_SORT_BY_SIZE,
  IMAGE_SET_SORT_BY_MTIME
} ImageSetSortType;

struct _ImageTypeInfo {
  guint        type;

  GdkColor     color;
  ImagePixmap  *pixmap;
};

struct _Image {
  /* assumes guint is atomic */
  guint       refcount;

  char        *file_name;
  char        *directory;

  guint32     file_size;
  guint32     file_mtime;
  int         width, height, bpp;

  GList       *caches;
  GList       *pixmaps;

  ImageTypeInfo type_info;
};

struct _ImageSet {
  GArray      *images;
  GCompareFunc compare;
};

struct _ImageCache {
  guint   refcount;
  int     width, height, bpp;
  guchar  *data;
};

struct _ImagePixmap {
  guint     refcount;
  int       width, height;
  GdkPixmap *pixmap;
  GdkBitmap *mask;
};

int                       is_readable_image (char *file_name, char *directory);
Image *                   image_new (char *file_name, char *directory);
Image *                   image_ref (Image *image);
void                      image_unref (Image *image);
char *                    image_get_path (Image *image);

ImageCache *              image_cache_new (Image *image, int width, int height, int bpp);
void                      image_cache_add (Image *image, ImageCache *cache);
ImageCache *              image_cache_ref (Image *image, ImageCache *cache);
void                      image_cache_unref (Image *image, ImageCache *cache);
ImageCache *              image_cache_get (Image *image, int width, int height);

ImagePixmap *             image_pixmap_new (Image *image, int width, int height);
ImagePixmap *             image_pixmap_ref (Image *image, ImagePixmap *ipix);
void                      image_pixmap_unref (Image *image, ImagePixmap *ipix);
ImagePixmap *             image_pixmap_get (Image *image, int width, int height);

int                       image_compare_by_name (const Image **i1, const Image **i2);
int                       image_compare_by_size (const Image **i1, const Image **i2);
int                       image_compare_by_mtime (const Image **i1, const Image **i2);

ImageSet *                image_set_new ();
ImageSet *                image_set_copy (ImageSet *set);
void                      image_set_free (ImageSet *set);
void                      image_set_clear (ImageSet *set);
void                      image_set_add (ImageSet *set, Image *image);
void                      image_set_remove (ImageSet *set, Image *image);
int                       image_set_get_index (ImageSet *set, Image *image);
void                      image_set_sort (ImageSet *set, ImageSetSortType type);

#define                   image_set_get_image(set, index) \
                            g_array_index ((set)->images, Image *, (index))
#define                   image_set_size(set) \
                            (set)->images->len

void                      image_print_leaks ();

#endif /* _IMAGE_H_ */


