
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: mcache.c,v 1.5 2000/04/28 20:46:34 dr Exp $ 
*/


#include "mcache.h"
#include "dcache.h"

static void               mcache_class_init (ModuleClass *class);
static void               mcache_init (Module *module);
static int                mcache_filter_init (Filter *filter);
static int                mcache_filter_process_tile (Filter *filter, FilterTile *src, FilterTile *out);
static void               mcache_filter_end (Filter *filter);

static void               mcache_free_outputs (Filter *filter);

static int                initialized = 0;
static Param *            p_output_cache = NULL;

ModuleType 
mcache_get_type () {
  static ModuleType  type = 0;

  if (!type) {
    static ModuleTypeInfo cache_info = {
      "MCache",
      NULL,
      NULL,
      sizeof (MCache),
      sizeof (MCacheClass),
      mcache_init,
      mcache_class_init,
      /* reserved1 */ NULL,
      /* reserved2 */ NULL,
    };

    type = module_type_unique (filter_get_type (), &cache_info);
  }

  return type;
}


static void               
mcache_class_init (ModuleClass *class) {
  FilterClass *filter_class;

  filter_class             = (FilterClass *) class;

  filter_class->filter_init    = mcache_filter_init;
  filter_class->filter_process_tile   = mcache_filter_process_tile;
  filter_class->filter_end     = mcache_filter_end;
  
  filter_class->free_outputs   = mcache_free_outputs;

  if (!initialized) {
    initialized = 1;
    p_output_cache = 
      param_new ("cache",
		 PARAM_OUTPUT,
		 NULL,
		 "The ouput image cache",
		 PARAM_TYPE_POINTER,
		 (ParamValue) (gpointer) NULL);
    module_class_add_param (class, p_output_cache);
  }
}

static void               
mcache_init (Module *module) {

}

Module *
mcache_new (int dcache) {
  MCache *mc;

  mc = (MCache *) module_type_new (mcache_get_type ());
  mc->dcache = dcache;

  return (Module *) mc;
}

static int
mcache_filter_init (Filter *filter) {
  MCache *mc;

  mc = (MCache *) filter;
  
  mc->cache            = image_cache_new (filter->image,
					  filter->src_width,
					  filter->src_height,
					  filter->src_bpp);
  mc->row_stride       = filter->src_width * filter->src_bpp;
  mc->cache_pos        = mc->cache->data;
  mc->rows_read        = 0;

  module_set_param_value (MODULE (filter),
			  p_output_cache->name,
			  p_output_cache->type,
			  (ParamValue) (gpointer) mc->cache);

  return 1;
}

static int
mcache_filter_process_tile (Filter *filter, FilterTile *src, FilterTile *out) {
  MCache *mc;

  mc = (MCache *) filter;

  g_return_val_if_fail (filter && mc->cache && src, 0);
  g_return_val_if_fail (mc->rows_read + src->height <= filter->src_height, 0);

  memcpy (mc->cache_pos, src->data, mc->row_stride * src->height);
  mc->cache_pos += mc->row_stride * src->height;
  mc->rows_read += src->height;

  return 1;
}

static void
mcache_filter_end (Filter *filter) {
  MCache *mc;

  mc = (MCache *) filter;
  g_return_if_fail (mc && mc->cache);

  if (mc->dcache)
    dcache_put (filter->image, mc->cache);

  mc->cache_pos = NULL;
}

static void
mcache_free_outputs (Filter *filter) {
  MCache *mc;

  mc = (MCache *) filter;
  g_return_if_fail (mc);

  if (mc->cache) {
    module_clear_param_value (MODULE (filter), 
			      p_output_cache->name);
    image_cache_unref (filter->image, mc->cache);
    mc->cache = NULL;
  }
}

ImageCache *
mcache_get_cache (MCache *mc) {
  g_return_val_if_fail (mc, NULL);

  return mc->cache;
}

